/* ---------------------------------------------------------- 
%   (C)1992 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
/*
  pdb_io.c
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#include "pdb.h"
#include "atom.h"
#include "method.h"

#include "../3dview/boolean.h"

#define HELIX "HELIX "
#define SHEET "SHEET "
#define TURN  "TURN  "
#define ATOM  "ATOM  "
#define PTALK "Root"

#define CHAIN 'Z'

extern int user_main(int argc,char** argv);
extern void load_initial_file();
extern int yyparse();
extern void initialize();
extern int is_read_eof;
extern char parse_buffer[BUFSIZ];
extern char* parse_point;
extern const char *ver;
extern const char *date;
extern const char* sys_name;

extern int is_not_load_initial_file;
extern int is_only_pr_talk;

extern FILE *fp_in;

extern secondStructure   helixTable;
extern secondStructure   turnTable;
extern secondStructure   sheetTable;
extern chain   chainTable;
extern unsigned helixCount;
extern unsigned chainCount;

extern unsigned long gj;
extern double gx, gy, gz;

const char *pdb_color[12] = 
{":Carbon",":Oxygen",":Nitrogen",":Sulfur",":Hydrogen",":R",
   ":CarbonWire",":OxygenWire",":NitrogenWire",":SulfurWire",":HydrogenWire",":RWire"};
const char *pdb_radius[6] =
{"CarbonRadius","OxygenRadius","NitrogenRadius","SulfurRadius",
   "HydrogenRadius","RRadius"};

int Remote(char key)
{
  int i = 0;
  switch(key){
    case ' ': i = 0 ; break;
    case 'A': i = 1 ; break;     //alpha
    case 'B': i = 2 ; break;     //beta
    case 'G': i = 3 ; break;     //gamma
    case 'D': i = 4 ; break;     //delta
    case 'E': i = 5 ; break;     //epsilon
    case 'Z': i = 6 ; break;     //zeta
    case 'H': i = 7 ; break;     //eta
    case 'X': i = 99; break;     //terminal amino
  }
  return(i);
}

char RRemote(int key)
{
  char i;
  switch(key){
    case  0: i = ' '; break;
    case  1: i = 'A'; break;     //alpha
    case  2: i = 'B'; break;     //beta
    case  3: i = 'G'; break;     //gamma
    case  4: i = 'D'; break;     //delta
    case  5: i = 'E'; break;     //epsilon
    case  6: i = 'Z'; break;     //zeta
    case  7: i = 'H'; break;     //eta
    case 99: i = 'X'; break;     //terminal amino
  }
  return(i);
}

int Branch(char key)
{
  int i = 0;
  switch(key){
    case ' ': i = 0 ; break;
    case 'T': i = 99; break;
    default : i = (key - 0x30); break;
  }
  return(i);
}

char RBranch(int key)
{
  char i;
  switch(key){
    case  0: i = ' '; break;
    case 99: i = 'T'; break;
    default : i = (key + 0x30); break;
  }
  return(i);
}

int ReadPdb(void)
{
  char record [128];
  residue* newResidue = (residue*)0;
  int rCount = 0;
  chain*   newChain  = (chain*)0;
  char chainame = '$';
  
  parse_point = parse_buffer;
  *parse_point = '\0';

  while(fgets(record, 128, fp_in) != NULL){
    /* make a HELIX table */
    if(!strncmp(record, HELIX, 6)){
      int begin=0, end=0;
      char oligo;
      char tmp[10];
      memset(tmp, 0, 10);
      begin = atoi(strncpy(tmp,record+20,5));
      end   = atoi(strncpy(tmp,record+32,5));
      if(record[19] != ' ') oligo = record[19];
      else oligo = CHAIN;
      helixTable.append(begin,end,oligo);
      helixCount ++;
    }
    /* make a SHEET table */
    else if(!strncmp(record, SHEET, 6)){
      int begin=0, end=0;
      char oligo;
      char tmp[10];
      memset(tmp, 0, 10);
      begin = atoi(strncpy(tmp,record+22,4));
      end   = atoi(strncpy(tmp,record+34,4));
      if(record[21] != ' ') oligo = record[21];
      else oligo = CHAIN;
#ifdef DEBUG
      fprintf(stderr,"oligo:[%c] begin:[%d] end:[%d]\n",oligo,begin,end);
#endif
      sheetTable.append(begin,end,oligo);
    }
    /* make a TURN table */
    else if(!strncmp(record, TURN, 6)){
      int begin=0, end=0;
      char oligo;
      char tmp[10];
      memset(tmp, 0, 10);
      begin = atoi(strncpy(tmp,record+20,4));
      end   = atoi(strncpy(tmp,record+31,4));
      if(record[19] != ' ') oligo = record[19];
      else oligo = CHAIN;
      turnTable.append(begin,end,oligo);
    }
    /* make a ATOM data list */
    else if(!strncmp(record, ATOM, 6) && (record[16] <= 'A')){
      int res=0;
      static int resOld=0;
      char tmp[10];
      memset(tmp, 0, 10);
      coodinate posion =  coodinate(atof(strncpy(tmp,record+30,8)),
		       atof(strncpy(tmp,record+38,8)),
		       atof(strncpy(tmp,record+46,8)));
      
      if(chainame != record[21]){
	chainame = record[21];
	if(rCount != 0) {
	  newChain->nums(rCount);
	  int* tmp = new int[rCount];
	  newChain->residueTables(tmp);
	  residue* searchResidue = newChain->firstResidues();
	  for(int i = 0; i<rCount; ++i){
	    newChain->residueTables(i, (int)searchResidue);
	    searchResidue = (residue*)searchResidue->Next();
	  }
	}
	/* put a chainName when no chain */
	if(chainame != ' ') newChain = chainTable.Append(chainame);
	else newChain = chainTable.Append(CHAIN);
	chainCount++;
	rCount = 0;
      }

      res = atoi(strncpy(tmp,record+22, 4));
      strncpy(tmp,record+17,3);
      if(rCount == 0) {
	newResidue = newChain->Append(tmp,res);
	rCount ++;
	resOld = res;
      }

      int argRemote = Remote(record[14]);
      int argBranch = Branch(record[15]);

      if(res != resOld){
	newResidue = newChain->Append(tmp,res);
	newResidue->AppendAtom(record[13],posion,argRemote,argBranch,res);
	rCount ++;
	resOld = res;
      }
      else newResidue->AppendAtom(record[13],posion,argRemote,argBranch,res);
    }
    else if(!strncmp(record, PTALK, 4)){
      fprintf(stderr,"pr_talk mode!!!\n");
      strcpy(parse_buffer, record);
      is_only_pr_talk = FALSE;
      int yy_ans = yyparse();
      if (!is_only_pr_talk) {
	/*ʸȤС3dviewή*/
	printf("%s\n",parse_buffer);
	fflush(stdout);
      }
      is_only_pr_talk = FALSE;
      parse_point = parse_buffer;
      *parse_point = '\0';
    }
  }

  /* MakeResidueTable Last */

  int* tmp2 = new int[rCount];

  chain* searchChain = chainTable.bottoms();
  searchChain->nums(rCount);
  searchChain->residueTables(tmp2);
  residue* searchResidue = searchChain->firstResidues();
  for(int i = 0; i<rCount; ++i){
    searchChain->residueTables(i, (int)searchResidue);
    searchResidue = (residue*)searchResidue->Next();
  }
  return 0;
}


/* connect bond by BackBone(N-CA-C-O) */
void BackBone(void)
{
  chain* searchChain = chainTable.tops();
  while(searchChain != NULL){
    residue* searchResidue = searchChain->firstResidues();
    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      while(searchAtom != NULL){
	atom* localAtom = searchAtom;
	char symbol = searchAtom->names();
	int  remote = searchAtom->remotenesses();
	int  branch = searchAtom->branchs();

	/***    back bone  N-CA-C-O    ***/

	if(symbol == 'N' && remote == 0 && branch == 0){
	  while(localAtom != NULL){
	    localAtom = (atom*)localAtom->Next();
	    symbol = localAtom->names();
	    remote = localAtom->remotenesses();
	    branch = localAtom->branchs();
	    if(symbol == 'C' && remote == 1 && branch == 0){
	      bonds(searchAtom,localAtom);
	      localAtom->bonds(1,searchAtom);
	      break;
	    }
	  }
	}
	else if(symbol == 'C' && remote == 1 && branch == 0){
	  while(localAtom != NULL){
	    localAtom = (atom*)localAtom->Next();
	    symbol = localAtom->names();
	    remote = localAtom->remotenesses();
	    branch = localAtom->branchs();
	    if(symbol == 'C' && remote == 0 && branch == 0){
	      bonds(searchAtom,localAtom);
	      localAtom->bonds(1,searchAtom);
	      break;
	    }
	  }
	}
	else if(symbol == 'C' && remote == 0 && branch == 0){
	  while(1){
	    if((localAtom = (atom*)localAtom->Next()) == NULL) break;
	    symbol = localAtom->names();
	    remote = localAtom->remotenesses();
	    branch = localAtom->branchs();
	    if(symbol == 'O' && remote == 0 && branch == 0){
	      bonds(searchAtom,localAtom);
	      localAtom->bonds(1,searchAtom);
	    }
	    if(symbol == 'O' && remote == 99 && branch == 99){
	      bonds(searchAtom,localAtom);
	      localAtom->bonds(1,searchAtom);
	      break;
	    }
	  }
	}
	searchAtom = (atom*)searchAtom->Next();
      }
      searchResidue = (residue*)searchResidue->Next();      
    }
    searchChain = (chain*)searchChain->Next();
  }
}

/* connect bond by SideChain */
void SideChain(void)
{
  chain* searchChain = chainTable.tops();
  while(searchChain != NULL){
    residue* searchResidue = searchChain->firstResidues();
    while(searchResidue != NULL){

      /**** Special Case ****/
      unsigned connectFlag = 0;
      if(strncmp(searchResidue->aminoIDs(),"HYP",3) == 0 ||
	 strncmp(searchResidue->aminoIDs(),"PRO",3) == 0) connectFlag = 1;
      else if(strncmp(searchResidue->aminoIDs(),"TRP",3) == 0) connectFlag=4;
      else if(strncmp(searchResidue->aminoIDs(),"HIS",3) == 0) connectFlag=2;

      atom* searchAtom = searchResidue->first();
      atom* tmpAtom = (atom*)0;
      while(searchAtom != NULL){
	char symbol = searchAtom->names();
	int  remote = searchAtom->remotenesses();
	int  branch = searchAtom->branchs();

	if(connectFlag == 1 && symbol == 'N' && remote == 0){
	  tmpAtom = searchAtom;
	}

	if(remote >= 1){
	  int remoNum = remote;
	  int branNum = branch;
	  atom* localAtom = searchResidue->first();
	  atom* subLocalAtom = searchAtom;
	  while(localAtom != NULL){
	    symbol = localAtom->names();
	    remote = localAtom->remotenesses();
	    branch = localAtom->branchs();
	    /* not side chain */
	    if(remote == 0) {
	      localAtom = (atom*)localAtom->Next();
	      continue;
	    }

	    /* connect algorithmic sequence */
	    int sare = remote - remoNum;
	    int saba = branch - branNum;
	    if(sare == 1){
	      if((branch != 0 && saba == 0) || branNum==0 || branch==0 ||
		 (remote==5 && branch==3 && branNum==2)){
		bonds(subLocalAtom,localAtom);
		if(localAtom->bonds(1) == NULL)
		  localAtom->bonds(1,subLocalAtom);
		else bonds(localAtom, subLocalAtom);
	      }
	    }	      
	    
	    /**** Special Case ****/
	    if(connectFlag == 1 && symbol == 'C' && remote == 4){
	      bonds(tmpAtom,localAtom);
	      bonds(localAtom,tmpAtom);
	      connectFlag = 0;
	    }
	    else if(connectFlag== 2 && symbol=='C' && remote==5){
	      tmpAtom = localAtom;
	      connectFlag = 3;
	    }
	    else if(connectFlag== 3 && symbol=='N' && remote==5){
	      bonds(tmpAtom,localAtom);
	      bonds(localAtom,tmpAtom);
	      connectFlag = 0;
	    }
	    else if(connectFlag== 4 && symbol=='N' && remote==5 && branch==1){
	      tmpAtom = localAtom;
	      connectFlag = 5;
	    }
	    else if(connectFlag== 5 && symbol=='C' && remote==5 && branch==2){
	      bonds(tmpAtom,localAtom);
	      bonds(localAtom,tmpAtom);
	      connectFlag = 6;
	    }
	    else if(connectFlag== 6 && remote==6 && branch==3){
	      tmpAtom = localAtom;
	      connectFlag = 7;
	    }
	    else if(connectFlag== 7 && remote==7 ){
	      bonds(tmpAtom,localAtom);
	      bonds(localAtom,tmpAtom);
	      connectFlag = 0;
	    }
	    localAtom = (atom*)localAtom->Next();
	  }
	}
	searchAtom = (atom*)searchAtom->Next();
      }
      searchResidue = (residue*)searchResidue->Next();      
    }
    searchChain = (chain*)searchChain->Next();
  }
}

/* ƻĴϢ뤹 */
void ConnectBackBone(void)
{
  chain* searchChain = chainTable.tops();
  while(searchChain != NULL){
    residue* searchResidue = searchChain->firstResidues();
    atom* tmpAtom;
    int connectFlag = 0;
    int newResID = 0, oldResID = 0;
    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      while(searchAtom != NULL){
	char symbol = searchAtom->names();
	int  remote = searchAtom->remotenesses();
	int  branch = searchAtom->branchs();
	
	if(symbol == 'C' && remote == 0 && branch == 0 && connectFlag==0){
	  tmpAtom = searchAtom;
	  oldResID = searchResidue->resIDs();
	  connectFlag = 1;
	}
	else if(symbol == 'N' && remote==0 && branch==0 && connectFlag==1){
	  newResID = searchResidue->resIDs();
	  if((newResID - oldResID) == 1){
	    bonds(tmpAtom,searchAtom);
	    searchAtom->bonds(1,tmpAtom);
	  }
	  else fprintf(stderr, "Warning: The Residue from %3d to %3d in chain %c are missed.\n",
		       oldResID+1, newResID-1, searchChain->chainIDs());
	  connectFlag = 0;
	}
	searchAtom = (atom*)searchAtom->Next();
      }
      searchResidue = (residue*)searchResidue->Next();      
    }
    searchChain = (chain*)searchChain->Next();
  }
}

void ReConnectHydrogen(void)
{
  chain* searchChain = chainTable.tops();
  while(searchChain != NULL){
    residue* searchResidue = searchChain->firstResidues();
    atom* tmpAtomN;
    atom* tmpAtomCA;

    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      while(searchAtom != NULL){
	char symbol = searchAtom->names();
	int  remote = searchAtom->remotenesses();
	int  branch = searchAtom->branchs();

	if(symbol == 'C' && remote == 1 && branch == 0 ){
	  tmpAtomCA = searchAtom;
	}
	else if(symbol == 'N' && remote==0 && branch==0 ){
	  tmpAtomN = searchAtom;
	}
	else if(symbol == 'H' && remote==0 && branch==2 ){
	  bonds(tmpAtomN,searchAtom);
	  searchAtom->bonds(1,tmpAtomN);
	}
	else if(symbol == 'H' && remote==0 && branch==1 ){
	  bonds(tmpAtomCA,searchAtom);
	  searchAtom->bonds(1,tmpAtomCA);
	}
	searchAtom = (atom*)searchAtom->Next();
      }
      searchResidue = (residue*)searchResidue->Next();      
    }
    searchChain = (chain*)searchChain->Next();
  }
}

void TestPrint(char* amino)
{
  PushGroup("Root");
  MakeGroupObject("Residue");
  PushGroup("Residue");

  chain* searchChain = chainTable.tops();
  while(searchChain != NULL){
    char tmp[20];
    sprintf(tmp,"%c",searchChain->chainIDs());
    
    /* ˥롼ԥ */
    MakeGroupObject(tmp);
    PushGroup(tmp);

    residue* searchResidue = searchChain->firstResidues();
    int allFlag = 0;

    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      char* aminoID = searchResidue->aminoIDs();
      int resID = searchResidue->resIDs();
      if(strcmp(amino,"ALL")==0) {
	allFlag = 1;
      }
      if(strncmp(aminoID,amino,3) == 0 || allFlag == 1){
	
	/* ߥλ˥롼ԥ */
	sprintf(tmp,"Res%d", resID);
	MakeGroupObject(tmp);
	PushGroup(tmp);

	while(searchAtom != NULL){
	  double ax = 0.0, ay = 0.0, az = 0.0;
	  int colorID = 0;
	  char symbol = searchAtom->names();
	  int remo = searchAtom->remotenesses();
	  int bran = searchAtom->branchs();

	  if((modeFlag & 0x20) != 0) {     /* main chain only */
	    if(remo > 1 ){
	      searchAtom = (atom*)searchAtom->Next();
	      continue;
	    }
	  }
	  if(remo == 0 && bran == 0) sprintf(tmp,"%c",symbol);
	  else if(remo == 0 && bran != 0) 
	    sprintf(tmp,"%c%c",symbol, RBranch(bran));
	  else if(remo != 0 && bran == 0)
	    sprintf(tmp,"%c%c",symbol, RRemote(remo));
	  else if(remo != 0 && bran != 0)
	    sprintf(tmp,"%c%c%c",symbol, RRemote(remo), RBranch(bran));
	  MakeSphereObject(tmp);
	  for(int i=1; i<5 ;++i){
	    colorID = 0;
	    if(symbol == 'C') colorID = 0;
	    else if(symbol == 'O') colorID = 1;
	    else if(symbol == 'N') colorID = 2;
	    else if(symbol == 'S') colorID = 3;
	    else if(symbol == 'H') colorID = 4;
	    else if(symbol == 'R') colorID = 5;

	    if(searchAtom->bonds(i) != NULL){
	      char tmp2[20];
	      if(remo == 0 && bran == 0) sprintf(tmp2,"%c%d",symbol,i);
	      else if(remo == 0 && bran != 0)
		sprintf(tmp2,"%c%c%d",symbol, RBranch(bran),i);
	      else if(remo != 0 && bran == 0)
		sprintf(tmp2,"%c%c%d",symbol, RRemote(remo),i);
	      else if(remo != 0 && bran != 0)
		sprintf(tmp2,"%c%c%c%d",symbol,RRemote(remo),RBranch(bran),i);
	      MakePipe0Object(tmp2);
	      TopObject(tmp2, ":BondRadius");
	      BottomObject(tmp2, ":BondRadius");
	      ColorObject(tmp2, pdb_color[colorID]);

	      ax = searchAtom->axizs().CoodinateX();
	      ay = searchAtom->axizs().CoodinateY();
	      az = searchAtom->axizs().CoodinateZ();
	      double bx = searchAtom->bonds(i)->axizs().CoodinateX();
	      double by = searchAtom->bonds(i)->axizs().CoodinateY();
	      double bz = searchAtom->bonds(i)->axizs().CoodinateZ();
	      bx = (ax + bx) / 2.0;
	      by = (ay + by) / 2.0;
	      bz = (az + bz) / 2.0;
	      FromObject(tmp2, ax, ay, az);
	      ToObject(tmp2, bx, by, bz);
	    }
	    if(i==1){
	      ColorObject(tmp, pdb_color[colorID]);
	      RadiusObject(tmp, pdb_radius[colorID]);
	    }

	  }
	  MoveObject(tmp,ax,ay,az);
	  searchAtom = (atom*)searchAtom->Next();
	}
      }
      PopGroup();
      searchResidue = (residue*)searchResidue->Next();      
    }
    PopGroup();
    searchChain = (chain*)searchChain->Next();
  }
  PopGroup();
  PopGroup();
}

void DisplayLineResidue(char* amino)
{
  PushGroup("Root");
  if(ObjectExistence("Line") > 0) {
    DrawAll("Line");
    PopGroup();
    return;
  }
  MakeGroupObject("Line");
  PushGroup("Line");

  chain* searchChain = chainTable.tops();
  while(searchChain != NULL){
    char tmp[20];
    sprintf(tmp,"%c",searchChain->chainIDs());
    
    /* ˥롼ԥ */
    MakeGroupObject(tmp);
    PushGroup(tmp);

    residue* searchResidue = searchChain->firstResidues();
    int allFlag = 0;

    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      char* aminoID = searchResidue->aminoIDs();
      int resID = searchResidue->resIDs();
	
      /* ߥλ˥롼ԥ */
      sprintf(tmp,"%s%d",aminoID, resID);
      MakeGroupObject(tmp);
      PushGroup(tmp);

      while(searchAtom != NULL){
	double ax = 0.0, ay = 0.0, az = 0.0;
	int colorID = 0;
	char symbol = searchAtom->names();
	int remo = searchAtom->remotenesses();
	int bran = searchAtom->branchs();

	if((modeFlag & 0x20) != 0) {     /* main chain only */
	  if(remo > 1 ){
	    searchAtom = (atom*)searchAtom->Next();
	    continue;
	  }
	}

	for(int i=1; i<5 ;++i){
	  colorID = 0;
	  if(symbol == 'C') colorID = 6;
	  else if(symbol == 'O') colorID = 7;
	  else if(symbol == 'N') colorID = 8;
	  else if(symbol == 'S') colorID = 9;
	  else if(symbol == 'H') colorID = 10;
	  else if(symbol == 'R') colorID = 11;

	  if(searchAtom->bonds(i) != NULL){
	    char tmp2[20];
	    sprintf(tmp2,"%c%d%d%dl", symbol,remo,bran, i);
	    MakeLineObject(tmp2);
	    ColorObject(tmp2, pdb_color[colorID]);

	    ax = searchAtom->axizs().CoodinateX();
	    ay = searchAtom->axizs().CoodinateY();
	    az = searchAtom->axizs().CoodinateZ();
	    double bx = searchAtom->bonds(i)->axizs().CoodinateX();
	    double by = searchAtom->bonds(i)->axizs().CoodinateY();
	    double bz = searchAtom->bonds(i)->axizs().CoodinateZ();
	    bx = (ax + bx) / 2.0;
	    by = (ay + by) / 2.0;
	    bz = (az + bz) / 2.0;
	    FromObject(tmp2, ax, ay, az);
	    ToObject(tmp2, bx, by, bz);
	  }

	}
	searchAtom = (atom*)searchAtom->Next();
      }
      PopGroup();
      searchResidue = (residue*)searchResidue->Next();      
    }
    PopGroup();
    searchChain = (chain*)searchChain->Next();
  }
  PopGroup();
  PopGroup();
}

/* ǸҤΰ֤׻ */
void HydrogenSearch(void)
{
  chain* searchChain = chainTable.tops();
  while(searchChain != NULL){
    residue* searchResidue = searchChain->firstResidues();
    while(searchResidue != NULL){
      atom *caAtom, *nAtom;
      coodinate N[4], CA[4], H1, H2;
      atom* searchAtom = searchResidue->first();
      while(searchAtom != NULL){
	char symbol = searchAtom->names();
	int  remote = searchAtom->remotenesses();
	int  branch = searchAtom->branchs();

	if(symbol == 'N' && remote==0 && branch==0){
	  nAtom = searchAtom;
	}
	else if(symbol == 'C' && remote==1 && branch==0){
	  caAtom = searchAtom;
	}
	searchAtom = (atom*)searchAtom->Next();
      }

      /* CA ATOM begin*/
      for(int i = 1; i<=4; ++i){
	if(caAtom->bonds(i) == NULL) break;
	CA[i-1] = caAtom->bonds(i)->axizs();
	CA[i-1] = caAtom->axizs() - CA[i-1];
	CA[i-1].normalize();
      }

      H1  = CA[0] + CA[1];
      H1  = H1 + CA[2]; 
      H1.normalize();
      H1  = H1 + caAtom->axizs();
      searchResidue->AppendAtom('H',H1,0,1,caAtom->atomIDs());
      bonds(caAtom, searchResidue->last());
      searchResidue->last()->bonds(1,caAtom);
      /* CA ATOM end*/

      /* N ATOM begin*/
      for(i = 1; i<=4; ++i){
	if(nAtom->bonds(i) == NULL) break;
	N[i-1] = nAtom->bonds(i)->axizs();
	N[i-1] = nAtom->axizs() - N[i-1];
	N[i-1].normalize();
      }
      if(i!=1){
	H2  = N[0] + N[1];
	H2.normalize();
	H2  = H2 + nAtom->axizs();
	searchResidue->AppendAtom('H',H2,0,2,nAtom->atomIDs());
	bonds(nAtom, searchResidue->last());
	searchResidue->last()->bonds(1,nAtom);
      }
      /* N ATOM end*/

      searchResidue = (residue*)searchResidue->Next();      
    }
    searchChain = (chain*)searchChain->Next();
  }
}

void HelixSearch()
{
  secondStructure* testHelix = helixTable.heads();
  while(testHelix != NULL){
    int begin = testHelix->BeginID();
    int end   = testHelix->EndID();
    coodinate top, bottom;
    chain* searchChain = chainTable.tops();
    //  while(searchChain != NULL){
      residue* searchResidue = searchChain->firstResidues();
      while(searchResidue != NULL){
	atom* searchAtom = searchResidue->first();
	atom* searchAtom2 = searchResidue->first();
	int  resID = searchResidue->resIDs();
	
	if(resID >= begin && resID <= begin+2){
	  while(searchAtom != NULL){
	    char symbol = searchAtom->names();
	    int  remote = searchAtom->remotenesses();
	    
	    if(symbol=='N' && remote==0){
	      top = top + searchAtom->axizs();
	    }
	    searchAtom = (atom*)searchAtom->Next();
	  }
	}
	if(resID <= end && resID >= end - 2){
	  while(searchAtom2 != NULL){
	    char symbol = searchAtom2->names();
	    int  remote = searchAtom2->remotenesses();

	    if(symbol == 'C' && remote == 0) {
	      bottom = bottom+searchAtom2->axizs();
	    }
	    searchAtom2 = (atom*)searchAtom2->Next();
	  }
	}
	searchResidue = (residue*)searchResidue->Next();      
      }
      //    searchChain = (chain*)searchChain->Next();
      //  }
  
    top = top / 3.0;
    bottom = bottom / 3.0;
    
    testHelix->HelixSetBegin(top);
    testHelix->HelixSetEnd(bottom);

    testHelix = (secondStructure*)testHelix->Next();
  }
}

/* إåɽ */
void DisplayHelix(void)
{
  secondStructure* searchHelix = helixTable.heads();
  while(searchHelix != NULL){
    int begin = searchHelix->BeginID();
    printf(":Root:Helix%d is a cylinder0;\n",begin);
    printf(":Root:Helix%d top is :HelixRadius;\n",begin);
    printf(":Root:Helix%d bottom is :HelixRadius;\n",begin);
    printf(":Root:Helix%d color is :Helix;\n",begin);
    printf(":Root:Helix%d from{%f,%f,%f};\n",begin,
	   searchHelix->Begin().CoodinateX(),
	   searchHelix->Begin().CoodinateY(),
	   searchHelix->Begin().CoodinateZ());
    printf(":Root:Helix%d to{%f,%f,%f};\n",begin,
	   searchHelix->End().CoodinateX(),
	   searchHelix->End().CoodinateY(),
	   searchHelix->End().CoodinateZ());
    searchHelix = (secondStructure*)searchHelix->Next();
  }
}

void DisplayRibbonAll(char* groupName)
{
  chain* searchChain = chainTable.tops();
  while(searchChain != NULL){
    char cname = searchChain->chainIDs();
    if(strncmp(groupName,"Root",4) != 0 && groupName[0] != cname){
      searchChain = (chain*)searchChain->Next();
      continue;
    }
    residue* searchResidue = searchChain->firstResidues();

    PushGroup("Root");
    MakeGroupObject("Ribbon");
    PushGroup("Ribbon");
    char tmp[20];
    sprintf(tmp,"%c",cname);
    MakeGroupObject(tmp);
    PushGroup(tmp);
    MakeListObject("COpoints");
    
    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      coodinate carbon;
      coodinate oxigen;

      while(searchAtom != NULL){
	char symbol = searchAtom->names();
	int  remote = searchAtom->remotenesses();
	if(symbol == 'C' && remote == 0){
	  carbon = searchAtom->axizs();
	}
	else if(symbol == 'O' && remote==0){
	  oxigen = searchAtom->axizs();
	}
	searchAtom = (atom*)searchAtom->Next();
      }

      InsertList("COpoints", carbon, oxigen);

      searchResidue = (residue*)searchResidue->Next();      
    }

    MakePolygonObject("Ribbon");
    ColorObject("Ribbon", ":RibbonColor");
    PointsObject("Ribbon", "COpoints");
    NurbsObject("Ribbon");
    PopGroup();
    PopGroup();
    PopGroup();

    searchChain = (chain*)searchChain->Next();
  }
}

int DisplayRibbonHelix(char* groupName)
{
  secondStructure* testHelix = helixTable.heads();
  if(testHelix == NULL) return(-1);
  PushGroup("Root");
  if(ObjectExistence("Helix") > 0) {
    DrawAll("Helix");
    PopGroup();
    return(1);
  }
  PopGroup();
  while(testHelix != NULL){
    int begin = testHelix->BeginID();
    int end   = testHelix->EndID();
    char oligo = testHelix->Oligo();
    if(strncmp(groupName,"Root",4) != 0 && groupName[0] != oligo){
      testHelix = (secondStructure*)testHelix->Next();
      continue;
    }

    PushGroup("Root");
    MakeGroupObject("Helix");
    PushGroup("Helix");
    char tmp[20];
    sprintf(tmp,"%c",oligo);

    MakeGroupObject(tmp);

    chain* searchChain = chainTable.tops();
    while(searchChain != NULL){
      char cname = searchChain->chainIDs();
      if(oligo != cname){
	searchChain = (chain*)searchChain->Next();
	continue;
      }

      sprintf(tmp,"%c",cname);
      PushGroup(tmp);
      sprintf(tmp,"Hp%d",begin);
      MakeListObject(tmp);

      residue* searchResidue = searchChain->firstResidues();
      while(searchResidue != NULL){
	atom* searchAtom = searchResidue->first();
	int  resID = searchResidue->resIDs();
	coodinate C, O, O2;
      
	if(resID >= begin && resID <= end) {
	  while(searchAtom != NULL){
	    char symbol = searchAtom->names();
	    int  remote = searchAtom->remotenesses();

	    if(symbol == 'C' && remote == 0) C = searchAtom->axizs();
	    else if(symbol == 'O' && remote==0) O = searchAtom->axizs();
	    searchAtom = (atom*)searchAtom->Next();
	  }

	  O2 = ((C * 2) - O);

	  InsertList(tmp, O2, O);
	}
	searchResidue = (residue*)searchResidue->Next();      
      }
      char tmp2[20];
      sprintf(tmp2,"Hex%d",begin);
      MakePolygonObject(tmp2);
      ColorObject(tmp2, ":PolygonHelixColor");
      PointsObject(tmp2, tmp);
      NurbsObject(tmp2);
      PopGroup();
      PopGroup();
      PopGroup();

      searchChain = (chain*)searchChain->Next();
    }
  
    testHelix = (secondStructure*)testHelix->Next();
  }
  return(1);
}

/* SHEETܥɽ롣SHEET̵ä硢-1֤ */
int DisplayRibbonSheet(char* groupName)
{
  secondStructure* testSheet = sheetTable.heads();
  if(testSheet == NULL) return(-1);
  PushGroup("Root");
  if(ObjectExistence("Sheet") > 0) {
    DrawAll("Sheet");
    PopGroup();
    return(1);
  }
  PopGroup();

  while(testSheet != NULL){
    int begin = testSheet->BeginID();
    int end   = testSheet->EndID();
    char oligo = testSheet->Oligo();
    if(strncmp(groupName,"Root",4) != 0 && groupName[0] != oligo){
      testSheet = (secondStructure*)testSheet->Next();
      continue;
    }

    PushGroup("Root");
    MakeGroupObject("Sheet");
    PushGroup("Sheet");
    char tmp[20];
    sprintf(tmp,"%c",oligo);

    MakeGroupObject(tmp);

    chain* searchChain = chainTable.tops();
    while(searchChain != NULL){
      char cname = searchChain->chainIDs();
      if(oligo != cname){
	searchChain = (chain*)searchChain->Next();
	continue;
      }

      sprintf(tmp,"%c",cname);
      PushGroup(tmp);
      sprintf(tmp,"Sp%d",begin);
      MakeListObject(tmp);

      residue* searchResidue = searchChain->firstResidues();
      int clock = 0;

      while(searchResidue != NULL){
	atom* searchAtom = searchResidue->first();
	int  resID = searchResidue->resIDs();
	coodinate C, O, O2;

	if(resID >= begin && resID <= end){
	  while(searchAtom != NULL){
	    char symbol = searchAtom->names();
	    int  remote = searchAtom->remotenesses();
	    if(symbol == 'C' && remote == 0) C = searchAtom->axizs();
	    else if(symbol == 'O' && remote==0) O = searchAtom->axizs();
	    searchAtom = (atom*)searchAtom->Next();
	  }
	  O2 = ((C * 2) - O);
	  coodinate first, second;
	  /* last point */
	  if(resID == end){
	    first = C;
	    second = C;
	  }
	  else if(clock == 0){
	    first = O2;
	    second = O;
	    clock = 1;
	  }
	  else {
	    second = O2;
	    first = O;
	    clock = 0;
	  }
	  InsertList(tmp, first, second);

	  /* arrow generate */
	  if(resID >= end -1){
	    InsertList(tmp, first, second);
	    if(resID == end - 1){
	      O2 = ((O * 2) - C);
	      if(clock == 0) first = O2;
	      else second = O2;

	      O2 = ((C * 2) - O2);
	      if(clock == 0) second = O2;
	      else first = O2;

	      InsertList(tmp, first, second);
	      InsertList(tmp, first, second);
	    }
	  }
	}
	searchResidue = (residue*)searchResidue->Next();      
      }
      char tmp2[20];
      sprintf(tmp2,"Sheet%d",begin);
      MakePolygonObject(tmp2);
      ColorObject(tmp2, ":PolygonSheetColor");
      PointsObject(tmp2, tmp);
      NurbsObject(tmp2);
      PopGroup();
      PopGroup();
      PopGroup();
      
      searchChain = (chain*)searchChain->Next();
    }
  
    testSheet = (secondStructure*)testSheet->Next();
  }
  return(1);
}

int DisplayRibbonTurn(char* groupName)
{
  secondStructure* testTurn = turnTable.heads();
  if(testTurn == NULL) return(-1);
  PushGroup("Root");
  if(ObjectExistence("Turn") > 0) {
    DrawAll("Turn");
    PopGroup();
    return(1);
  }
  PopGroup();
  while(testTurn != NULL){
    int begin = testTurn->BeginID();
    int end   = testTurn->EndID();
    char oligo = testTurn->Oligo();
    if(strncmp(groupName,"Root",4) != 0 && groupName[0] != oligo){
      testTurn = (secondStructure*)testTurn->Next();
      continue;
    }

    PushGroup("Root");
    MakeGroupObject("Turn");
    PushGroup("Turn");
    char tmp[20];
    sprintf(tmp,"%c",oligo);
    
    MakeGroupObject(tmp);
    
    chain* searchChain = chainTable.tops();

    while(searchChain != NULL){
      char cname = searchChain->chainIDs();
      if(oligo != cname){
	searchChain = (chain*)searchChain->Next();
	continue;
      }

      sprintf(tmp,"%c",cname);
      PushGroup(tmp);
      sprintf(tmp,"Tp%d",begin);
      MakeListObject(tmp);

      residue* searchResidue = searchChain->firstResidues();

      while(searchResidue != NULL){
	atom* searchAtom = searchResidue->first();
	int  resID = searchResidue->resIDs();
	coodinate C, O, O2;
	
	if(resID >= begin && resID <= end){
	  while(searchAtom != NULL){
	    char symbol = searchAtom->names();
	    int  remote = searchAtom->remotenesses();

	    if(symbol == 'C' && remote == 0) C = searchAtom->axizs();
	    else if(symbol == 'O' && remote==0) O = searchAtom->axizs();
	    searchAtom = (atom*)searchAtom->Next();
	  }
	  O2 = ((C * 2) - O);
	  InsertList(tmp, O2, O);

	}
	searchResidue = (residue*)searchResidue->Next();      
      }
      char tmp2[20];
      sprintf(tmp2,"Turn%d",begin);
      MakePolygonObject(tmp2);
      ColorObject(tmp2, ":PolygonTurnColor");
      PointsObject(tmp2, tmp);
      NurbsObject(tmp2);
      PopGroup();
      PopGroup();
      PopGroup();

      searchChain = (chain*)searchChain->Next();
    }
    testTurn = (secondStructure*)testTurn->Next();
  }
  return(1);
}

void DisplayRibbonOther(void)
{
  secondStructure* searchHelix = helixTable.heads();
  secondStructure* searchSheet = sheetTable.heads();
  secondStructure* searchTurn  = turnTable.heads();
  int i =0, hFlag = 0, sFlag=0, tFlag = 0;

  PushGroup("Root");

  chain* searchChain = chainTable.tops();
  while(searchChain != NULL){
    char cname = searchChain->chainIDs();

    char tmp[20];
    sprintf(tmp,"%c",cname);
    MakeGroupObject(tmp);
    PushGroup(tmp);
    sprintf(tmp,"Op%d",i);
    MakeListObject(tmp);

    residue* searchResidue = searchChain->firstResidues();
    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      int  resID = searchResidue->resIDs();
      coodinate first, second;

      int hStart = 0, hEnd = 0, tStart = 0, tEnd = 0,sStart = 0, sEnd = 0;
      char soligo, holigo, toligo;
      if(searchHelix != NULL){
	hStart = searchHelix->BeginID() +2;
	hEnd   = searchHelix->EndID()   -2;
	holigo = searchHelix->Oligo();
      }
      if(searchSheet != NULL){
	sStart = searchSheet->BeginID() +2;
	sEnd   = searchSheet->EndID()   -2;
	soligo = searchSheet->Oligo();
      }
      if(searchTurn != NULL){
	tStart = searchTurn->BeginID()  +2;
	tEnd   = searchTurn->EndID()    -2;
	toligo = searchTurn->Oligo();
      }

      if((hEnd-hStart) < 0) {
	int tmp=hStart;
	hStart = hEnd;
	hEnd = tmp;
      }	
      if((sEnd-sStart) < 0) {
	int tmp=sStart;
	sStart = sEnd;
	sEnd = tmp;
      }	
      if((tEnd-tStart) < 0) {
	int tmp=tStart;
	tStart = tEnd;
	tEnd = tmp;
      }	

      if(resID == hStart && hFlag == 0 && cname == holigo) {
	OutPolygon(i);
	searchResidue = (residue*)searchResidue->Next();
	if(resID == hEnd) {
	  i++;
	  sprintf(tmp,"Op%d",i);
	  MakeListObject(tmp);
	  if(searchHelix->Next() != NULL)
	    searchHelix = (secondStructure*)searchHelix->Next();
	  hFlag = 0;
	  continue;
	}
	hFlag = 1;
	continue;
      }
      else if(resID == sStart && sFlag == 0 && cname == soligo){
	OutPolygon(i);
	searchResidue = (residue*)searchResidue->Next();
	if(resID == sEnd) {
	  i++;
	  sprintf(tmp,"Op%d",i);
	  MakeListObject(tmp);
	  if(searchSheet->Next() != NULL)
	    searchSheet = (secondStructure*)searchSheet->Next();
	  sFlag = 0;
	  continue;
	}
	sFlag = 1;
	continue;
      }
      else if(resID == tStart && tFlag == 0 && cname == toligo){
	OutPolygon(i);
	searchResidue = (residue*)searchResidue->Next();
	if(resID == tEnd) {
	  i++;
	  sprintf(tmp,"Op%d",i);
	  MakeListObject(tmp);
	  if(searchTurn->Next() != NULL)
	    searchTurn = (secondStructure*)searchTurn->Next();
	  tFlag = 0;
	  continue;
	}
	tFlag = 1;
	continue;
      }
      else if(resID == hEnd && hFlag == 1 && cname == holigo) {
	i++;
	sprintf(tmp,"Op%d",i);
	MakeListObject(tmp);
	searchResidue = (residue*)searchResidue->Next();      
	if(searchHelix->Next() != NULL)
	  searchHelix = (secondStructure*)searchHelix->Next();
	hFlag = 0;
	continue;
      }
      else if(resID == sEnd && sFlag == 1 && cname == soligo) {
	i++;
	sprintf(tmp,"Op%d",i);
	MakeListObject(tmp);
	searchResidue = (residue*)searchResidue->Next();      
	if(searchSheet->Next() != NULL)
	  searchSheet = (secondStructure*)searchSheet->Next();
	sFlag = 0;
	continue;
      }
      else if(resID == tEnd && tFlag == 1 && cname == toligo) {
	i++;
	sprintf(tmp,"Op%d",i);
	MakeListObject(tmp);
	searchResidue = (residue*)searchResidue->Next();      
	if(searchTurn->Next() != NULL) 
	  searchTurn = (secondStructure*)searchTurn->Next();
	tFlag = 0;
	continue;
      }
      else if((resID >= hStart && resID <= hEnd && cname == holigo) || 
	      (resID > sStart && resID < sEnd && cname == soligo) ||
	      (resID > tStart && resID < tEnd && cname == toligo) ){
	searchResidue = (residue*)searchResidue->Next();      
	continue;
      }
      while(searchAtom != NULL){
	char symbol = searchAtom->names();
	int  remote = searchAtom->remotenesses();
	if(symbol == 'C' && remote == 0)    first  = searchAtom->axizs();
	else if(symbol == 'O' && remote==0) second = searchAtom->axizs();
	searchAtom = (atom*)searchAtom->Next();
      }
      sprintf(tmp,"Op%d",i);
      InsertList(tmp, first, second);

      searchResidue = (residue*)searchResidue->Next();      
    }
    OutPolygon(i);
    searchChain = (chain*)searchChain->Next();
    PopGroup();
  }
  PopGroup();
}

void OutPolygon(int i)
{
  char tmp2[20];
  sprintf(tmp2,"Ribbon%d",i);
  MakePolygonObject(tmp2);
  ColorObject(tmp2, ":PolygonColor");
  char tmp[20];
  sprintf(tmp,"Op%d",i);
  PointsObject(tmp2, tmp);
  NurbsObject(tmp2);

}

int AtomIdCheck(char* name, int* remo, int* bra)
{
  int i = 0;
  if((i=strlen(name)) == 1) *remo = *bra = 0;
  else if(i == 2) {
    *remo = Remote(name[1]);
    *bra = 0;
  }
  else if(i == 3) {
    *remo = Remote(name[1]);
    *bra  = Branch(name[2]);
  }
  else {
    fprintf(stderr,"err : illegal atom name.\n");
    exit(1);
  }

  if(*remo > 1) return(1);
  else return(0);
}

int Rotation(char chainname,int aminoNo,char* atomA,char* atomB,int degree)
{
  coodinate A, B;
  chain* searchChain = chainTable.tops();
  int  remoA = 0,remoB= 0,braA = 0,braB = 0;
  char idA = atomA[0];
  char idB = atomB[0];
  AtomIdCheck(atomA, &remoA, &braA);
  AtomIdCheck(atomB, &remoB, &braB);

  while(searchChain != NULL){
    if(searchChain->chainIDs() == chainname) break;
    searchChain = (chain*)searchChain->Next();
  }
//  residue* searchResidue = searchChain->GetResidue(aminoNo);
  residue* searchResidue = searchChain->GetResidue(aminoNo-1);
  atom* searchAtom = searchResidue->first();
  
  while(searchAtom != NULL){
    if(searchAtom->names() == idA && searchAtom->remotenesses() == remoA &&
       searchAtom->branchs() == braA){
      A = searchAtom->axizs();
    }
    if(searchAtom->names() == idB && searchAtom->remotenesses() == remoB &&
       searchAtom->branchs() == braB){
      B = searchAtom->axizs();
    }
    searchAtom = (atom*)searchAtom->Next();
  }

  int sideChainFlag = 0;
  if(remoA >= 2 || remoB >= 2) {
    if(remoA >= remoB) sideChainFlag = remoA;
    else if(remoB > remoA) sideChainFlag = remoB;
  }
  int cFlag = 0;
  if((idA == 'C' && remoA == 0) || (idB == 'C' && remoB == 0)) cFlag = 1;

  if(strncmp(searchResidue->aminoIDs(),"PHE",3) == 0 || /* Phenylalanine */
     strncmp(searchResidue->aminoIDs(),"TYR",3) == 0 || /* Tyrosine      */
     strncmp(searchResidue->aminoIDs(),"TRP",3) == 0 || /* Tryptophan    */
     strncmp(searchResidue->aminoIDs(),"HIS",3) == 0){  /* Histidine     */
    if(sideChainFlag > 3){
      fprintf(stderr,"err(proview) : select bond (benzen)\n");
      return(0);
    }
  }
  if(strncmp(searchResidue->aminoIDs(),"PRO",3) == 0 || /* Proline        */
     strncmp(searchResidue->aminoIDs(),"HYP",3) == 0){  /* Hydroxyproline */
    if(sideChainFlag > 1){
      fprintf(stderr,"err(proview) : select bond (benzen)\n");
      return(0);
    }
  }

  int hFlag = 0;
  while(searchResidue != NULL){
    atom* searchAtom = searchResidue->first();
    while(searchAtom != NULL){
      if(sideChainFlag != 0 && searchAtom->remotenesses() < sideChainFlag){
	searchAtom = (atom*)searchAtom->Next();
	continue;
      }
      char symbol = searchAtom->names();
      int bran = searchAtom->branchs();
      if(cFlag != 0 && hFlag == 0){
	if(symbol != 'O'){
	  searchAtom = (atom*)searchAtom->Next();
	  continue;
	}
      }
      if((modeFlag & 0x08) != 0 && hFlag == 0) {
	if(symbol == 'H' && bran == 2) {
	  searchAtom = (atom*)searchAtom->Next();
	  continue;
	}
      }

      searchAtom->axizs().Rotate(A, B, degree);      /* Rotation */
      searchAtom = (atom*)searchAtom->Next();
    }
    hFlag ++;
    if(sideChainFlag != 0) break;
    searchResidue = (residue*)searchResidue->Next();
    if(searchResidue != NULL)	searchAtom = searchResidue->first();
  }
  return(sideChainFlag);
}

void RamachandranPhi(char chainname,int aminoNo,double phi)
{
  coodinate A, B, C, E;
  chain* searchChain = chainTable.tops();

  while(searchChain != NULL){
    if(searchChain->chainIDs() == chainname) break;
    searchChain = (chain*)searchChain->Next();
  }

  residue* searchResidue = searchChain->GetResidue(aminoNo-1);
  atom* searchAtom = searchResidue->first();
  
  while(searchAtom != NULL){
    if(searchAtom->names() == 'C' && searchAtom->remotenesses() == 0 &&
       searchAtom->branchs() == 0){
      E = searchAtom->axizs();
    }
    searchAtom = (atom*)searchAtom->Next();
  }

  searchResidue = (residue*)searchResidue->Next();
  searchAtom = searchResidue->first();

  while(searchAtom != NULL){
    if(searchAtom->names() == 'C' && searchAtom->remotenesses() == 1 &&
       searchAtom->branchs() == 0){
      B = searchAtom->axizs();
    }
    if(searchAtom->names() == 'N' && searchAtom->remotenesses() == 0 &&
       searchAtom->branchs() == 0){
      A = searchAtom->axizs();
    }
    if(searchAtom->names() == 'C' && searchAtom->remotenesses() == 0 &&
       searchAtom->branchs() == 0){
      C = searchAtom->axizs();
    }
    searchAtom = (atom*)searchAtom->Next();
  }

  phi = C.Rotate2(A, B, E, phi);
  phi = phi / (M_PI/180);
#ifdef DEBUG
  fprintf(stderr,"Phi: %f\n", phi);
#endif

  searchAtom = searchResidue->first();
  while(searchAtom != NULL){
    char symbol = searchAtom->names();
    int remo = searchAtom->remotenesses();
    int bran = searchAtom->branchs();

    if(symbol == 'C' && remo == 1 && bran == 0) {
      searchAtom = (atom*)searchAtom->Next();
      continue;
    }
    else if(symbol == 'N' && remo == 0 && bran == 0) {
      searchAtom = (atom*)searchAtom->Next();
      continue;
    }
    else searchAtom->axizs().Rotate(A, B, phi);      /* Rotation */
    searchAtom = (atom*)searchAtom->Next();
  }
  searchResidue = (residue*)searchResidue->Next();

  while(searchResidue != NULL){
    searchAtom = searchResidue->first();

    while(searchAtom != NULL){
      searchAtom->axizs().Rotate(A, B, phi);      /* Rotation */
      searchAtom = (atom*)searchAtom->Next();
    }

    searchResidue = (residue*)searchResidue->Next();
  }
}

void RamachandranPsi(char chainname,int aminoNo,double psi)
{
  coodinate A, B, C, E;
  chain* searchChain = chainTable.tops();

  while(searchChain != NULL){
    if(searchChain->chainIDs() == chainname) break;
    searchChain = (chain*)searchChain->Next();
  }

  residue* searchResidue = searchChain->GetResidue(aminoNo);
  atom* searchAtom = searchResidue->first();
  
  while(searchAtom != NULL){
    if(searchAtom->names() == 'C' && searchAtom->remotenesses() == 1 &&
       searchAtom->branchs() == 0){
      A = searchAtom->axizs();
    }
    if(searchAtom->names() == 'N' && searchAtom->remotenesses() == 0 &&
       searchAtom->branchs() == 0){
      E = searchAtom->axizs();
    }
    if(searchAtom->names() == 'C' && searchAtom->remotenesses() == 0 &&
       searchAtom->branchs() == 0){
      B = searchAtom->axizs();
    }
    searchAtom = (atom*)searchAtom->Next();
  }

  searchResidue = (residue*)searchResidue->Next();
  searchAtom = searchResidue->first();

  while(searchAtom != NULL){
    if(searchAtom->names() == 'N' && searchAtom->remotenesses() == 0 &&
       searchAtom->branchs() == 0){
      C = searchAtom->axizs();
    }
    searchAtom = (atom*)searchAtom->Next();
  }

  psi = C.Rotate2(A, B, E, psi);
  psi = psi / (M_PI/180);
  fprintf(stderr,"Psi: %f\n",psi);

  searchAtom = searchResidue->first();
  while(searchAtom != NULL){
    char symbol = searchAtom->names();
    int remo = searchAtom->remotenesses();
    int bran = searchAtom->branchs();

    if(symbol == 'O' && remo == 0 && bran == 0)
      searchAtom->axizs().Rotate(A, B, psi);      /* Rotation */
    searchAtom = (atom*)searchAtom->Next();
  }
  searchResidue = (residue*)searchResidue->Next();

  while(searchResidue != NULL){
    searchAtom = searchResidue->first();

    while(searchAtom != NULL){
      searchAtom->axizs().Rotate(A, B, psi);      /* Rotation */
      searchAtom = (atom*)searchAtom->Next();
    }

    searchResidue = (residue*)searchResidue->Next();
  }
}

void BondInit(char chainNo, int aminoNo)
{
  chain* searchChain = chainTable.tops();
  while(searchChain != NULL){
    if(searchChain->chainIDs() != chainNo){
      searchChain = (chain*)searchChain->Next();
      continue;
    }
    residue* searchResidue = searchChain->GetResidue(aminoNo);
    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      while(searchAtom != NULL){
	for(int i=1; i<5 ; ++i) searchAtom->bonds(i, NULL);
	searchAtom = (atom*)searchAtom->Next();
      }
      searchResidue = (residue*)searchResidue->Next();
    }
    break;
  }
}

void RePrint(char chainNo, int aminoNo, int sideChainFlag)
{

  chain* searchChain = chainTable.tops();
  printf(": into :Root:%c;\n", chainNo);
  while(searchChain != NULL){
    if(searchChain->chainIDs() != chainNo){
      searchChain = (chain*)searchChain->Next();
      continue;
    }
    residue* searchResidue = searchChain->GetResidue(aminoNo);

    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      char* aminoID = searchResidue->aminoIDs();
      int resID = searchResidue->resIDs();

      while(searchAtom != NULL){
	double ax = 0.0, ay = 0.0, az = 0.0;
	char symbol = searchAtom->names();
	int remo = searchAtom->remotenesses();
	int bran = searchAtom->branchs();

	if((modeFlag & 0x20) != 0) {     /* main chain only */
	  if(remo > 1 ){
	    searchAtom = (atom*)searchAtom->Next();
	    continue;
	  }
	}

	for(int i=1; i<5 ;++i){
	  if(searchAtom->bonds(i) != NULL){
	    ax = searchAtom->axizs().CoodinateX();
	    ay = searchAtom->axizs().CoodinateY();
	    az = searchAtom->axizs().CoodinateZ();
	    double bx = searchAtom->bonds(i)->axizs().CoodinateX();
	    double by = searchAtom->bonds(i)->axizs().CoodinateY();
	    double bz = searchAtom->bonds(i)->axizs().CoodinateZ();
	    bx = (ax + bx) / 2.0;
	    by = (ay + by) / 2.0;
	    bz = (az + bz) / 2.0;
	    printf(":%s%d:%c%d%d%dl from{%f,%f,%f};\n",
		   aminoID,resID,symbol,remo,bran,i,ax,ay,az);
	    printf(":%s%d:%c%d%d%dl to{%f,%f,%f};\n",
		   aminoID,resID,symbol,remo,bran,i,bx,by,bz);
	  }
	}
	printf(":%s%d:%c%d%ds moves{%f,%f,%f};\n",
		     aminoID,resID,symbol,remo,bran,ax,ay,az);
	searchAtom = (atom*)searchAtom->Next();
      }
      if(sideChainFlag > 1) break;
      searchResidue = (residue*)searchResidue->Next();      
    }
    break;
  }
  printf(":up;\n:up;\n");
}

void ReDisplayRibbonAll(char chainNo, int aminoNo)
{
  chain* searchChain = chainTable.tops();
  printf(": into :Root:%c;\n", chainNo);
  while(searchChain != NULL){
    if(searchChain->chainIDs() != chainNo){
      searchChain = (chain*)searchChain->Next();
      continue;
    }
    residue* searchResidue = searchChain->GetResidue(aminoNo);

    int index = searchChain->nums() - aminoNo - 1;
    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      double xc = 0.0, yc = 0.0, zc = 0.0;
      double xo = 0.0, yo = 0.0, zo = 0.0;

      while(searchAtom != NULL){
	char symbol = searchAtom->names();
	int  remote = searchAtom->remotenesses();
	if(symbol == 'C' && remote == 0){
	  xc = searchAtom->axizs().CoodinateX();
	  yc = searchAtom->axizs().CoodinateY();
	  zc = searchAtom->axizs().CoodinateZ();
	}
	else if(symbol == 'O' && remote==0){
	  xo = searchAtom->axizs().CoodinateX();
	  yo = searchAtom->axizs().CoodinateY();
	  zo = searchAtom->axizs().CoodinateZ();
	}
	searchAtom = (atom*)searchAtom->Next();
      }
      printf(":ELE1 is (:COpoints element %d );\n",index);
      printf(":ELEC is (:ELE1 element 0);\n");
      printf(":ELEO is (:ELE1 element 1);\n");
      printf(":ELEC value{0,%f};\n",xc);
      printf(":ELEC value{1,%f};\n",yc);
      printf(":ELEC value{2,%f};\n",zc);
      printf(":ELEO value{0,%f};\n",xo);
      printf(":ELEO value{1,%f};\n",yo);
      printf(":ELEO value{2,%f};\n",zo);

      searchResidue = (residue*)searchResidue->Next(); 
      index --;
    }
    break;
  }
  printf(":up;\n:up;\n");
}

void LoadMacro(char* fileName)
{
  FILE *fpin;

  if((fpin = fopen(fileName, "r")) == NULL){
    fprintf(stderr,"err : macro file not found.\n");
    return;
  }
  char Buff[80];
  Buff[0] = '\0';
  while(fgets(Buff, 80, fpin) != NULL){
    if(strlen(Buff) < 1) continue;
    int j=0;
    if(strncmp(Buff,"quit",4) == 0){
      break;
    }else if(strncmp(Buff,"RO",2) == 0){
      char *token;
      token = strtok(Buff," ,\t");
      token = strtok(NULL," ,\t");
      int sideChainFlag = 0;
      sideChainFlag = Rotation(token[0],j=atoi(strtok(NULL," ,\t")),
			       strtok(NULL," ,\t"), strtok(NULL," ,\t"),
			       atoi(strtok(NULL," ,\t")));
      
      BondInit(token[0], j);
      BackBone();
      SideChain();
      ConnectBackBone();
      ReConnectHydrogen();
      RePrint(token[0], j, sideChainFlag);
      if((modeFlag & 0x200) != 0) ReDisplayOnlyCA();
      if((modeFlag & 0x01) != 0) ReDisplayRibbonAll(token[0],j);
      printf(":Root moves{%f,%f,%f};\n",gx,gy,gz);
      puts(":sync;");
      continue;
    } else {
      printf("%s\n", Buff);
    }
  }
  fclose(fpin);
}

/* Sphereɽɽ
   chainName : ̾RootꤹƤκͭȤʤ롣
   aminoID   : Ĵֹ桢ꤷĴʾоݤȤ롣
   sphereFlag: 0ɽ1ɽ
*/
void SpherePrint(char* chainNo, int aminoNo, int sphereFlag)
{

  chain* searchChain = chainTable.tops();
  PushGroup("Root");
  PushGroup("Residue");
  while(searchChain != NULL){
    char cname = searchChain->chainIDs();
    if(strncmp(chainNo,"Root",4) != 0 && cname != chainNo[0]){
      searchChain = (chain*)searchChain->Next();
      continue;
    }
    char tmp[20];
    sprintf(tmp,"%c",cname);
    PushGroup(tmp);

    residue* searchResidue = searchChain->GetResidue(aminoNo);

    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      char* aminoID = searchResidue->aminoIDs();
      int resID = searchResidue->resIDs();

//      sprintf(tmp,"%s%d",aminoID,resID);
      sprintf(tmp,"Res%d",resID);
      PushGroup(tmp);

      while(searchAtom != NULL){
	char symbol = searchAtom->names();
	int remo = searchAtom->remotenesses();
	int bran = searchAtom->branchs();

	if((modeFlag & 0x20) != 0) {     // main chain only
	  if(remo > 1 ){
	    searchAtom = (atom*)searchAtom->Next();
	    continue;
	  }
	}
	if(remo == 0 && bran == 0) sprintf(tmp,"%c",symbol);
	else if(remo == 0 && bran != 0) sprintf(tmp,"%c%c",symbol, RBranch(bran));
	else if(remo != 0 && bran == 0) sprintf(tmp,"%c%c",symbol, RRemote(remo));
	else if(remo != 0 && bran != 0) sprintf(tmp,"%c%c%c",symbol, RRemote(remo), RBranch(bran));
//	sprintf(tmp,"%c%d%ds",symbol,remo,bran);
	if(sphereFlag == 1)
	  HideObject(tmp);
	else
	  DrawObject(tmp);
	searchAtom = (atom*)searchAtom->Next();
      }
      PopGroup();
      searchResidue = (residue*)searchResidue->Next();      
    }
    PopGroup();
//    if(sideChainFlag > 1) break;
    searchChain = (chain*)searchChain->Next();
  }
  PopGroup();

  // if group CA is exist
  if(ObjectExistence("CA") > 0) {
    PushGroup("CA");
    chain* searchChain = chainTable.tops();
    while(searchChain != NULL){
      double bx = 0.0, by = 0.0, bz = 0.0;
      char tmp[20];
      sprintf(tmp,"%c",searchChain->chainIDs());
      PushGroup(tmp);

      residue* searchResidue = searchChain->firstResidues();
      
      while(searchResidue != NULL){
	atom* searchAtom = searchResidue->first();
	int resID = searchResidue->resIDs();
	sprintf(tmp,"CA%d",resID);
	HideObject(tmp);
	searchResidue = (residue*)searchResidue->Next();      
      }
      searchChain = (chain*)searchChain->Next();
      PopGroup();
    }
    PopGroup();
  }

  PopGroup();
}

void ChangeSphereDepth(char* chainNo, int aminoNo, int sphereDepth)
{

  chain* searchChain = chainTable.tops();
  PushGroup("Root");
  PushGroup("Residue");

  while(searchChain != NULL){
    char cname = searchChain->chainIDs();
    if(strncmp(chainNo,"Root",4) != 0 && cname != chainNo[0]){
      searchChain = (chain*)searchChain->Next();
      continue;
    }

    char tmp[20];
    sprintf(tmp,"%c",cname);
    MakeGroupObject(tmp);
    PushGroup(tmp);

    residue* searchResidue = searchChain->GetResidue(aminoNo);

    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      char* aminoID = searchResidue->aminoIDs();
      int resID = searchResidue->resIDs();

      sprintf(tmp,"%s%d",aminoID, resID);
      PushGroup(tmp);

      while(searchAtom != NULL){
	char symbol = searchAtom->names();
	int remo = searchAtom->remotenesses();
	int bran = searchAtom->branchs();

	if((modeFlag & 0x20) != 0) {     /* main chain only */
	  if(remo > 1 ){
	    searchAtom = (atom*)searchAtom->Next();
	    continue;
	  }
	}
	printf("%c%d%ds depth is %d;\n",symbol,remo,bran,sphereDepth);
	searchAtom = (atom*)searchAtom->Next();
      }
      PopGroup();
      searchResidue = (residue*)searchResidue->Next();      
    }
    PopGroup();
//    if(sideChainFlag > 1) break;
    break;
  }
  PopGroup();
  PopGroup();
}

/* ܥɤʬѹ */
void ChangeBondDivide(char* chainNo, int aminoNo, int bondDivide)
{

  chain* searchChain = chainTable.tops();
  PushGroup("Root");
  PushGroup("Residue");

  while(searchChain != NULL){
    char cname = searchChain->chainIDs();
    if(strncmp(chainNo,"Root",4) != 0 && cname != chainNo[0]){
      searchChain = (chain*)searchChain->Next();
      continue;
    }
    char tmp[20];
    sprintf(tmp,"%c",cname);
    MakeGroupObject(tmp);
    PushGroup(tmp);

    residue* searchResidue = searchChain->GetResidue(aminoNo);

    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      char* aminoID = searchResidue->aminoIDs();
      int resID = searchResidue->resIDs();

//      sprintf(tmp,"%s%d",aminoID, resID);
      sprintf(tmp,"Res%d", resID);
      PushGroup(tmp);

      while(searchAtom != NULL){
	char symbol = searchAtom->names();
	int remo = searchAtom->remotenesses();
	int bran = searchAtom->branchs();

	if((modeFlag & 0x20) != 0) {     /* main chain only */
	  if(remo > 1 ){
	    searchAtom = (atom*)searchAtom->Next();
	    continue;
	  }
	}
	for(int i=1; i<5 ;++i){
	  if(searchAtom->bonds(i) != NULL){
	    if(remo == 0 && bran == 0) sprintf(tmp,"%c%d",symbol,i);
	    else if(remo == 0 && bran != 0) sprintf(tmp,"%c%c%d",symbol, RBranch(bran),i);
	    else if(remo != 0 && bran == 0) sprintf(tmp,"%c%c%d",symbol, RRemote(remo),i);
	    else if(remo != 0 && bran != 0) sprintf(tmp,"%c%c%c%d",symbol, RRemote(remo), RBranch(bran),i);
	    printf("%s division is %d;\n",tmp, bondDivide);
	  }
	}

	searchAtom = (atom*)searchAtom->Next();
      }
      PopGroup();
      searchResidue = (residue*)searchResidue->Next();      
    }
    PopGroup();
//    if(sideChainFlag > 1) break;
    break;
  }
  PopGroup();
  PopGroup();
}

/* ſΰ֤׻ */
void GravityPoint(double* xg, double* yg, double* zg)
{
  int allCount = 0;
  *xg = *yg = *zg = 0;
  chain* searchChain = chainTable.tops();
  while(searchChain != NULL){
    residue* searchResidue = searchChain->firstResidues();
    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      while(searchAtom != NULL){
	char symbol = searchAtom->names();
	int remo = searchAtom->remotenesses();

	if(symbol == 'C' && remo == 1) {
	  *xg += searchAtom->axizs().CoodinateX();
	  *yg += searchAtom->axizs().CoodinateY();
	  *zg += searchAtom->axizs().CoodinateZ();
	  allCount ++;
	}
	searchAtom = (atom*)searchAtom->Next();
      }
      searchResidue = (residue*)searchResidue->Next();      
    }
    searchChain = (chain*)searchChain->Next();
  }
  *xg = - (*xg / allCount);
  *yg = - (*yg / allCount);
  *zg = - (*zg / allCount);
}

/* Display Residue sequence number */
void ResidueSeqNo(void)
{
  double x,y,z;
  chain* searchChain = chainTable.tops();

  PushGroup("Root");
  if(ObjectExistence("SeqNo") > 0) {
    DrawAll("SeqNo");
    PopGroup();
    return;
  }
  MakeGroupObject("SeqNo");
  PushGroup("SeqNo");

  while(searchChain != NULL){
    char tmp[20];
    sprintf(tmp,"%c",searchChain->chainIDs());
    MakeGroupObject(tmp);
    PushGroup(tmp);

    residue* searchResidue = searchChain->firstResidues();
    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      int resID = searchResidue->resIDs();
      while(searchAtom != NULL){
	char symbol = searchAtom->names();
	int remo = searchAtom->remotenesses();
	if(symbol == 'C' && remo == 1) {
	  x = searchAtom->axizs().CoodinateX();
	  y = searchAtom->axizs().CoodinateY();
	  z = searchAtom->axizs().CoodinateZ();
	  break;
	}
	searchAtom = (atom*)searchAtom->Next();
      }
      x += 0.5;
      y += 0.5;
      z += 0.5;
      sprintf(tmp,"resID%d",resID);
      MakeTextObject(tmp);
      ColorObject(tmp,":SlateGreen");
      MoveObject(tmp, x, y, z);
      char tmp2[6];
      sprintf(tmp2, "%d", resID);
      StringObject(tmp,tmp2);

      searchResidue = (residue*)searchResidue->Next();      
    }
    PopGroup();
    searchChain = (chain*)searchChain->Next();
  }
  PopGroup();
  PopGroup();
}

/* Display C-alpha only */
void DisplayOnlyCA(void)
{
  chain* searchChain = chainTable.tops();

  PushGroup("Root");
  if(ObjectExistence("CA") > 0) {
    DrawAll("CA");
    PopGroup();
    return;
  }
  MakeGroupObject("CA");
  PushGroup("CA");

  while(searchChain != NULL){
    double bx = 0.0, by = 0.0, bz = 0.0;
    char tmp[20];
    sprintf(tmp,"%c",searchChain->chainIDs());
    
    MakeGroupObject(tmp);
    PushGroup(tmp);

    residue* searchResidue = searchChain->firstResidues();

    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      int resID = searchResidue->resIDs();
      
      while(searchAtom != NULL){
	double ax = 0.0, ay = 0.0, az = 0.0;
	char symbol = searchAtom->names();
	int remo = searchAtom->remotenesses();
	int bran = searchAtom->branchs();

	if(symbol == 'C' && remo == 1 && bran == 0){
	  ax = searchAtom->axizs().CoodinateX();
	  ay = searchAtom->axizs().CoodinateY();
	  az = searchAtom->axizs().CoodinateZ();
	  if(bx==0 && by == 0 && bz == 0){
	    sprintf(tmp,"CA%d",resID);
	    MakeSphereObject(tmp);
	    RadiusObject(tmp,"CarbonRadius");
	    ColorObject(tmp, ":Carbon");
	    MoveObject(tmp, ax, ay, az);
	    bx = ax;
	    by = ay;
	    bz = az;
	    searchAtom = (atom*)searchAtom->Next();
	    continue;
	  }
	  sprintf(tmp,"CA%dbond",resID);
	  MakePipe0Object(tmp);
	  TopObject(tmp, ":BondRadius");
	  BottomObject(tmp, ":BondRadius");
	  ColorObject(tmp, ":Carbon");
	  FromObject(tmp, ax, ay, az);
	  ToObject(tmp, bx, by, bz);

	  sprintf(tmp,"CA%d",resID);
	  MakeSphereObject(tmp);
	  RadiusObject(tmp,"CarbonRadius");
	  ColorObject(tmp, ":Carbon");
	  MoveObject(tmp, ax, ay, az);

	  bx = ax;
	  by = ay;
	  bz = az;
	}
	searchAtom = (atom*)searchAtom->Next();
      }
      searchResidue = (residue*)searchResidue->Next();      
    }
    searchChain = (chain*)searchChain->Next();
    PopGroup();
  }
  PopGroup();
  PopGroup();
  modeFlag |= 0x200;
}

void ReDisplayOnlyCA(void)
{

  chain* searchChain = chainTable.tops();
  puts(": into :CA;");
  while(searchChain != NULL){
    double bx = 0.0, by = 0.0, bz = 0.0;
    printf(": into :%c;\n",searchChain->chainIDs());
    residue* searchResidue = searchChain->firstResidues();

    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      int resID = searchResidue->resIDs();

      while(searchAtom != NULL){
	double ax = 0.0, ay = 0.0, az = 0.0;
	char symbol = searchAtom->names();
	int remo = searchAtom->remotenesses();
	int bran = searchAtom->branchs();

	if(symbol == 'C' && remo == 1 && bran == 0){
	  ax = searchAtom->axizs().CoodinateX();
	  ay = searchAtom->axizs().CoodinateY();
	  az = searchAtom->axizs().CoodinateZ();
	  if(bx==0 && by == 0 && bz == 0){
	    printf(":CA%d moves{%f,%f,%f};\n",resID,ax,ay,az);
	    bx = ax;
	    by = ay;
	    bz = az;
	    searchAtom = (atom*)searchAtom->Next();
	    continue;
	  }
	  printf(":CA%dbond from{%f,%f,%f};\n",resID,ax,ay,az);
	  printf(":CA%dbond to{%f,%f,%f};\n",resID,bx,by,bz);

	  printf(":CA%d moves{%f,%f,%f};\n",resID,ax,ay,az);
	  bx = ax;
	  by = ay;
	  bz = az;
	}
	searchAtom = (atom*)searchAtom->Next();
      }
      searchResidue = (residue*)searchResidue->Next();      
    }
    searchChain = (chain*)searchChain->Next();
    puts(":up;");
  }
  puts(":up;\n:up;");
}

void HydrogenConnect(void)
{
  chain* searchChain = chainTable.tops();
  while(searchChain != NULL){
    residue* searchResidue = searchChain->firstResidues();
    atom* tmpAtomN;
    atom* tmpAtomCA;

    puts("push Root;");
//    puts("O2C is [];");

    coodinate axizO[4], axizH[4], axizT[4];
    int i = 0;
    while(searchResidue != NULL){
      atom* searchAtom = searchResidue->first();
      while(searchAtom != NULL){
	char symbol = searchAtom->names();
	int  remote = searchAtom->remotenesses();
	int  branch = searchAtom->branchs();
	  
	if(symbol == 'O' && remote == 0 && branch == 0 ){
	  axizO[i] = searchAtom->axizs();
	}
	else if(symbol == 'H' && remote==0 && branch==2 ){
	  if(axizT[i].CoodinateX() != 0){
	    char tmp[6];
	    sprintf(tmp,"O2C%d",i);
	    MakePipe0Object(tmp);
	    TopObject(tmp, ":O2CRadius");
	    BottomObject(tmp, ":O2CRadius");
	    FromObject(tmp, axizT[i].CoodinateX(), axizT[i].CoodinateY(),
		       axizT[i].CoodinateZ());
	    ToObject(tmp, searchAtom->axizs().CoodinateX(),
		     searchAtom->axizs().CoodinateY(),
		     searchAtom->axizs().CoodinateZ());
	    ColorObject(tmp, ":O2C");
//	    printf("O2C is (O2C insert [{%f,%f,%f},{%f,%f,%f}]);\n",
//		 axizT[i].CoodinateX(),
//		 axizT[i].CoodinateY(),
//		 axizT[i].CoodinateZ(),
//		 searchAtom->axizs().CoodinateX(),
//		 searchAtom->axizs().CoodinateY(),
//		 searchAtom->axizs().CoodinateZ());
	  }
	}
	searchAtom = (atom*)searchAtom->Next();
      }
      searchResidue = (residue*)searchResidue->Next();      
      i++;
      if(i>3) {
	for(i=0; i<4 ; i++) {
	  axizT[i].CoodinateSetX(axizO[i].CoodinateX());
	  axizT[i].CoodinateSetY(axizO[i].CoodinateY());
	  axizT[i].CoodinateSetZ(axizO[i].CoodinateZ());
	}
	i=0;
      }
    }
    searchChain = (chain*)searchChain->Next();
  }
//  puts("Hconect is a polygon;");
//  puts("Hconect color is :RibbonColor;");
//  puts("Hconect points is O2C;");
  puts("pop;");
}
