;;;
;;;   KNOWBEL knowledge representation system
;;;    
;;;    author: Bryan M. Kramer
;;;    
;;;    
;;; Copyright (c) 1990, 1991 University of Toronto, Toronto, ON
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all copies and
;;; supporting documentation.
;;;
;;; The University of Toronto provides this software "as is" without
;;; express or implied warranty.
;;;

;;;    
;;;    


;? user level query functions, integrity constraint prechecking, propositions, untell



(comment defun eval-query (query-or-state &key (vars :all) (belief (std-belief)) (theory *theory*) (horn t) (search :depth) errors)
  (declare (special ac))
  (let (resolvent query state)
    (if (vectorp query-or-state)
      (setf state query-or-state)
      (progn
	(setf query query-or-state)
	(setf state :init)
	))
    (ndprog state (istate)
      (setf ac (make-theory :kb (theory-kb theory)))
      (setf istate (new-state ac theory belief horn))
      (istate-set-search-style istate search)
      (add-wff istate (list 'not query) belief 0 vars errors)
      (if (and errors (car errors))
	(return nil))
      (setf resolvent (backward-inference-loop istate))
      (nd-while resolvent
		(success resolvent)
		(setf resolvent (backward-inference-loop istate))
		)
      )
    )
  )


(defun eval-query (query-or-state &key (vars :all) (belief (std-belief)) (theory *theory*) (horn t) (search :depth) errors context)
  (declare (special ac))
  (let (istate
	(theory (if (not (listp theory)) (list theory) theory)))
    (if (inference-state-p query-or-state)
      (setf istate query-or-state)
      (progn
	(setf ac (make-theory :kb (theory-kb (car theory))))
	(setf istate (new-state ac theory belief horn))
	(istate-set-search-style istate search)
	(setf (istate-context istate) context)
	(add-wff istate (list 'not query-or-state) belief 0 vars errors)
	)
      )
    (if (not (and errors (car errors)))
      (let ((resolvent (backward-inference-loop istate)))
	(if resolvent
	  (cons resolvent istate)
	  nil)
	))
    )
  )



(defun ask-print-answer (resolvent vars)
  (format t "~&True~%")
  (finish-output)
  (doloop (c (ac-collect resolvent)) :vars ((v (if (eq vars :all) nil vars)))
    (format t "~&	~a	~a~%" (car v) (normal-clause c))
    (setf v (cdr v))
    )
  )

#-:ccl
(defun ask (query-or-state &key (vars :all) (belief (std-belief)) (theory *theory*) (horn t) (search :depth) context)
  (let* ((errors (tconc))
	 (result (eval-query query-or-state :vars vars :belief belief :theory theory :horn horn :search search
			     :context context :errors errors)))
    (when (car errors)
      (dbgn-print 'error\: (car errors))
      )
    (when result
      (ask-print-answer (car result) vars)
      (setq *ask-answer* (car result))
      (cdr result)
      ))
  )
#+:ccl
(defun telos-ask (query-or-state &key (vars :all) (belief (std-belief)) (theory *theory*) (horn t) (search :depth))
  (let* ((errors (tconc))
	 (result (eval-query query-or-state :vars vars :belief belief :theory theory :horn horn :search search :errors errors)))
    (when (car errors)
      (dbgn-print 'error\: (car errors))
      )
    (when result
      (ask-print-answer (car result) vars)
      (setq *ask-answer* (car result))
      (cdr result)
      ))
  )


(defun ask1 (query &key (vars :all) (belief (std-belief)) (theory *theory*) (horn t) (search :depth) context)
  (let* ((errors (tconc))
	 (result (eval-query query :vars vars :belief belief :theory theory
			     :context context :horn horn :search search :errors errors)))
    (cond ((car errors) (values nil (car errors)))
	  (result (list (ac-collect (car result))))
	  ))
  )


(defun ask-all (query &key (vars :all) (belief (std-belief)) (theory *theory*) (horn t) (search :depth) context)
  (let* ((errors (tconc)))
    (doloop :iter (result (eval-query query :vars vars :belief belief :theory theory :horn horn
				      :context context :search search :errors errors)
			  (eval-query (cdr result)))
      :while (and (not (car errors)) result)
      :collect (ac-collect (car result))
      ))
  )

(defun forward-inference (antecedant test-theory consequent stash-theory belief)
  ;; questions: 1) does it fire once or on every example?
  ;;		2) is the antecedant built or merely cnf'd?
  )


(defun get-history-interval (h)
  (cond ((time-int-p h) h)
	((clause-var-p h)
	 (get-history-interval (var-type h)))
	(t (let ((i (abstract-prop-history h)))
	     (if i i))))
  )


(defun lookup-inst-prop (src dest history belief state)
  (ndprog state nil
    (nd-loop (inst-prop (tok-inst-of src))
      (nd-if (and (or (clause-wild-card-p dest) (eq (prop-dest inst-prop) dest))
		  (belief-satisfies belief (prop-belief inst-prop))
		  (or (clause-wild-card-p history) (time-test (get-history-interval history):= (prop-history inst-prop) belief)))
	(success inst-prop)
	)
      ))
  )


(defun lookup-is-a-prop (src dest history belief state)
  (ndprog state nil
    (nd-loop (is-a-prop (tok-parents src))
	(nd-if (and (or (clause-wild-card-p dest) (eq (prop-dest is-a-prop) dest))
		    (belief-satisfies belief (prop-belief is-a-prop))
		    (or (clause-wild-card-p history) (time-test (get-history-interval history):= (prop-history is-a-prop) belief)))
	  (success is-a-prop)
	  )
      ))
  )


(defun lookup-attr-prop (src label class-label dest history belief state)
  (ndprog state nil
    (nd-loop (attr-prop (if (clause-wild-card-p class-label)
			  (lookup-index-cdr (tok-attrs src) :all)
			  (lookup-index-cdr (tok-attrs src) class-label)))
	(nd-if (and (or (clause-wild-card-p dest) (eql (attr-value attr-prop) dest))
		    (or (clause-wild-card-p label) (eq (attr-label attr-prop) label))
		    (belief-satisfies belief (attr-belief attr-prop))
		    (or (clause-wild-card-p history) (time-test (get-history-interval history):= (attr-history attr-prop) belief)))
	  (success attr-prop)
	  )
      ))
  )



(defun find-prop (expression theory belief state)
  (labels ((object (x) (or (lookup-type x theory) x))
	 (object* (x) (or (object x) (clause-wild-card))))
    (case (car expression)
      ((:inst inst instance-of :instance-of)
       (lookup-inst-prop (object (cadr expression)) (object (caddr expression)) (or (cadddr expression) (clause-wild-card)) belief state))
      ((:is-a is-a)
       (lookup-is-a-prop (object (cadr expression)) (object (caddr expression)) (or (cadddr expression) (clause-wild-card)) belief state))
      ((:attr attr)
       (lookup-attr-prop (object (cadr expression))
			 (if (clause-wild-card-p (caddr expression)) (caddr expression) (find-string (caddr expression) theory))
			 (if (clause-wild-card-p (cadddr expression)) (cadddr expression) (find-string (cadddr expression) theory))
			 (object* (cadddr (cdr expression)))
			 (or (cadr (cddddr expression)) (clause-wild-card))
			 belief state))
      (t nil))
    )
  )


(defun find-all-props (expression theory belief)
  (let ((val (tconc)))
    (ndprog :init nil
      (nd-for prop prop-cont (find-prop expression theory belief prop-cont)
	      (tconc val prop)
	      )
      )
    (car val))
  )


;; a new ic makes use of any deductive rules that match any of its atoms and
;; recursively from there.

;; a new dr is relevant to some ic if it matches recursivelyany existing dr that is relevant
;; or any ic

;; all this ignores any meta or lisp inferences

(defun match-ic-atoms (atom1 clause1 atom2 clause2 sigma belief)
  (reset-sigma sigma clause1 clause2)
  (cond ((and (eq (car atom1) 'not)
	      (eq (car (cadr atom1)) (car atom2))) 	 ;;; test  for same predicate
	 (unify-work (cadr atom1) atom2 sigma clause1 clause2 belief)
	 )
	((and (eq (car atom2) 'not)
	      (eq (car (cadr atom2)) (car atom1))) 	 ;;; test  for same predicate
	 (unify-work atom1 (cadr atom2) sigma clause1 clause2 belief)
	 )
	)
  )


(defun ic-build-resolvent (atom1 clause1 atom2 clause2 sigma)
  (let* ((new-clause (make-clause))
	 (vars (subst-sigma sigma new-clause clause1 clause2))
	 (resolvent (let ((result nil))
		      (doloop (atom (clause-expression clause1))
			(when (not (eq atom1 atom))
			  (push (apply-sigma atom sigma new-clause clause1 clause2) result)
			  ))
		      (doloop (atom (clause-expression clause2))
			(when (not (eq atom2 atom))
			  (push (apply-sigma atom sigma new-clause clause1 clause2) result)
			  ))
		      result )))
    (setf (clause-expression new-clause) resolvent)
    (setf (clause-n-vars new-clause) (array-dimension vars 0))
    (setf (clause-vars new-clause) vars)
    new-clause)
  )
  


(defun compute-ic-relevance (new-ic theories belief)
  (let ((resolvent-list (list new-ic))
	(relevance (gensym "ic-relevance"))
	clause)
    (setf (clause-get-prop new-ic :ic-relevance) relevance)
    (while resolvent-list
      (setf clause (car resolvent-list))
      (setf resolvent-list (cdr resolvent-list))
      (doloop (atom (clause-expression clause))
       :for (sign nil (eq (car atom) 'not))
       :for (literal nil (if sign (cadr atom) atom))
	(doloop (theory theories)
	  (doloop (match-pair (lookup-theory-index theory sign literal))
	    (let ((match-clause (car match-pair))
		  (match-atom (cadr match-pair)))
	      (when (and (not (eq relevance (clause-get-prop match-clause :ic-relevance)))
			 (belief-satisfies belief (clause-belief-time match-clause)))
		(let ((sigma (make-array (list (+ (clause-n-vars clause) (clause-n-vars match-clause)) 2))))
		  (let ((match (match-ic-atoms atom clause match-atom match-clause sigma belief)))
		    (when match
		      (setf (clause-get-prop match-clause :ic-relevance) relevance)
		      (push (ic-build-resolvent atom clause match-atom match-clause sigma) resolvent-list)
		      ))
		))
	      ))
	  ))
      ))
  )


(defun compute-dr-ic-relevance (clause ded-theories ic-theory belief)
  (if (or (compute-dr-ic-match-ic clause ic-theory belief)
	  (compute-dr-ic-match-ded clause ded-theories belief))
    (setf (clause-get-prop clause :ic-relevance) t)
    )
  )


(defun compute-dr-ic-match-ic (clause ic-theory belief)
  (doloop (atom (clause-expression clause))
   :vars (match)
   :when match :return match
   :for (sign nil (eq (car atom) 'not))
   :for (literal nil (if sign (cadr atom) atom))
    (doloop (match-pair (lookup-theory-index ic-theory sign literal))
     :while (not match)
     :for (match-clause nil (car match-pair))
     :for (match-atom nil (cadr match-pair))
     :when (belief-satisfies belief (clause-belief-time match-clause))
      (let ((sigma (make-array (list (+ (clause-n-vars clause) (clause-n-vars match-clause)) 2))))
	(setf match (match-ic-atoms atom clause match-atom match-clause sigma belief))))
    )
  )


(defun compute-dr-ic-match-ded (clause ded-theories belief)
  (let* ((resolvent-list (tconc nil clause))
	 (ptr (car resolvent-list))
	 (checked nil)
	 (success nil))
    (while (and ptr (not success))
      (setf clause (car ptr))
      (setf ptr (cdr ptr))
      (doloop (atom (clause-expression clause))
       :while (not success)
       :for (sign nil (eq (car atom) 'not))
       :for (literal nil (if sign (cadr atom) atom))
	(doloop (theory ded-theories)
	 :while (not success)
	  (doloop (match-pair (lookup-theory-index theory sign literal))
	   :while (not success)
	    (let ((match-clause (car match-pair))
		  (match-atom (cadr match-pair)))
	      (when (and (not (member match-clause checked))
			 (belief-satisfies belief (clause-belief-time match-clause)))
		(push match-clause checked)
		(let ((sigma (make-array (list (+ (clause-n-vars clause) (clause-n-vars match-clause)) 2))))
		  (let ((match (match-ic-atoms atom clause match-atom match-clause sigma belief)))
		    (when match
		      (if (clause-get-prop match-clause :ic-relevance)
			(setf success t)
			(tconc resolvent-list (ic-build-resolvent atom clause match-atom match-clause sigma))
			)))
		  ))
	      ))
	  ))
      )
    success)
  )




(defun get-time-var (op arg &optional (belief (std-belief)) (theory *theory*) (errors (tconc)))
  (let ((new-var (make-clause-var :number 0
				  :meta-type nil
				  :name 0
				  :type nil
				  :owner nil)))
    (if (clause-var-p arg)
      (if (time-int-p (var-type arg))
	(let ((int (get-time-int nil belief)))
	  (time-assert int op (var-type arg) belief belief belief errors)
	  (setf (var-type new-var) int)
	  )
	(tconc errors `(not time arg ,op ,arg))
	)
      (if (not (consp arg))
	(setf (var-type new-var) (time-create-relative (list op (list arg arg)) theory belief (list op (list arg arg)) errors))
	(setf (var-type new-var) (time-create-relative (cons op arg) theory belief (cons op arg) errors))
	)
      )
    new-var)
  )


(defun simple-time-var (&optional (theory *theory*) (type (theory-all-time *theory*)))
  (make-clause-var :number 0
		   :meta-type nil
		   :name 0
		   :type type
		   :owner nil)
  )



(defun untell (prop)
  (cond ((and (kb-attr-p prop) (eq :+ (belief-time-end (attr-belief prop))))
	 (setf (attr-belief prop) (make-belief-time :start (belief-time-start (attr-belief prop)) :end (get-universal-time))))
	((kb-token-p prop) :error)
	((and (prop-object-p prop) (eq :+ (belief-time-end (prop-belief prop))))
	 (setf (prop-belief prop) (make-belief-time :start (belief-time-start (prop-belief prop)) :end (get-universal-time))))
	(t :error))
  )

(defun undo-untell (prop)
  (cond ((and (kb-attr-p prop) (not (eq :+ (belief-time-end (attr-belief prop)))))
	 (setf (attr-belief prop) (make-belief-time :start (belief-time-start (attr-belief prop)) :end :+)))
	((kb-token-p prop) :error)
	((and (prop-object-p prop) (not (eq :+ (belief-time-end (prop-belief prop)))))
	 (setf (prop-belief prop) (make-belief-time :start (belief-time-start (prop-belief prop)) :end :+)))
	(t :error))
  )

(defmacro parse-history-time (item &optional (object t) (belief (std-belief)) (theory '*theory*))
  `(parse-time ,item (theory-all-time ,theory) ,object ,theory nil ,belief)
  )

(defmacro parse-belief-time (&rest args)
  (cond ((null args) `(make-belief-time))
	((null (cdr args)) `(make-belief-time :start (time-convert-str (time-force-string ,(car args)))))
	((null (cddr args)) `(make-belief-time :start (time-convert-str (time-force-string ,(car args)))
					       :end (time-convert-str (time-force-string ,(cadr args))
								      (get-universal-time)
								      nil)))
	(t `(make-belief-time :start (time-convert-str (time-force-string ,(car args)))
			      :end (time-convert-str (time-force-string ,(cadr args)) (get-universal-time) nil)
			      :constraint (caddr args)))
	))


(defun get-belief-time (start end constraint)
  (cond ((and (null start) (null end) (null constraint)) (make-belief-time))
	((and (null end) (null constraint)) (make-belief-time :start (time-convert-str (time-force-string start))))
	((null constraint) (make-belief-time :start (time-convert-str (time-force-string start))
					      :end (time-convert-str (time-force-string end)
								     (get-universal-time)
								     nil)))
	(t (make-belief-time :start (time-convert-str (time-force-string start))
			     :end (time-convert-str (time-force-string end) (get-universal-time) nil)
			     :constraint constraint))
	))


(defun collect-all-objects (obj objects)
  (ifn (not (member obj (car objects)))
    (tconc objects obj)
    (ifn (kb-token-p obj)
      (doloop (prop (tok-parents obj))
	(collect-all-objects (prop-dest prop) objects)
	(collect-all-objects prop objects)
	)
      (doloop (prop (tok-children obj))
	(collect-all-objects (prop-src prop) objects)
	(collect-all-objects prop objects)
	)
      (doloop (prop (tok-instances obj))
	(collect-all-objects (prop-src prop) objects)
	(collect-all-objects prop objects)
	)
      (doloop (prop (tok-inst-of obj))
	(collect-all-objects (prop-dest prop) objects)
	(collect-all-objects prop objects)
	)
      (doloop (prop (lookup-index-cdr (tok-attrs obj) :all))
	(collect-all-objects (attr-value prop) objects)
	(collect-all-objects prop objects)
	)
      )
    )
  )
    


(defun all-objects (&optional (theory *theory*))
  (let ((objects (tconc)))
    (maphash #'(lambda (key val)
	       (if (kb-token-p val)
		 (collect-all-objects val objects)
		 )
	       )
	   (theory-tokens theory)
	   )
    (car objects)
    )
  )


(defun kb-stats (&optional arg)
  (let ((objects (cond ((null arg) (all-objects))
		       ((theory-p arg) (all-objects arg))
		       ((consp arg) arg)
		       (t (error "bad arg"))))
	(classes 0)
	(tokens 0)
	(isa-links 0)
	(inst-links 0)
	(attrs 0)
	(attr-defs 0)
	(prim 0)
	)
    (doloop (obj objects)
      (cond ((kb-attr-p obj)
	     (if (tok-get-prop obj :class)
	       (++ attr-defs)
	       (++ attrs)))
	    ((kb-token-p obj)
	     (if (tok-get-prop obj :class)
	       (++ classes)
	       (++ tokens))
	     )
	    ((prop-object-p obj :isa) (++ isa-links))
	    ((prop-object-p obj :inst) (++ inst-links))
	    (t (++ prim)))
      )
    (dbgn-print 'classes classes
		'tokens tokens
		'isa-links isa-links
		'inst-links inst-links
		'attribute-links attrs
		'attribute-defs attr-defs
		'prim prim
		'total (length objects)
		'|for hierarchy rooted at token|
		'|depth, frontier size, average in, average out, non-leaf average out| (kb-isa-stats (lookup-type 'token *theory*))
		)
    )
  )


(defun kb-isa-stats-sub (object depth front count in out non-leaf non-leaf-out max-out)
  (++ (car count))
  (++ (car in) (length (tok-parents object)))
  (++ (car out) (length (tok-children object)))
  (ifn (null (tok-children object))
    (++ (car front))
    (+ 1 depth)
    else
    (++ (car non-leaf))
    (++ (car non-leaf-out) (length (tok-children object)))
    (if (> (length (tok-children object)) (car max-out)) (setf (car max-out) (length (tok-children object))))
    (doloop (child (tok-children object))
      :vars ((nd 0 (max nd (kb-isa-stats-sub (prop-src child) (+ 1 depth) front count in out non-leaf non-leaf-out max-out))))
      :return nd)
    )
  )


(defun kb-isa-stats (object)
  (let* ((front (cons 0 nil))
	 (count (cons 0 nil))
	 (in (cons 0 nil))
	 (out (cons 0 nil))
	 (non-leaf (cons 0 nil))
	 (non-leaf-out (cons 0 nil))
	 (max-out (cons 0 nil))
	 (depth (kb-isa-stats-sub object 0 front count in out non-leaf non-leaf-out max-out))
	 )
    (list depth (car front) (/ (car in) (car count)) (/ (car out) (car count)) (/ (car non-leaf-out) (car non-leaf)) (car max-out))
    )
  )
    



(defun normal-class-below (prop test-prop default-history new-belief)
  (let (class test-class h test-h b test-b)
    (cond ((kb-token-p prop)
	   (setf class prop)
	   (setf h default-history)
	   (setf b new-belief))
	  ((listp prop)
	   (setf class (car prop))
	   (setf h (cadr prop))
	   (setq b new-belief))
	  ((prop-object-p prop)
	   (setf class (prop-dest prop))
	   (setf h (prop-history prop))
	   (setf b (prop-belief prop))))
    (cond ((kb-token-p test-prop)
	   (setf test-class test-prop)
	   (setf test-h default-history)
	   (setf test-b new-belief))
	  ((listp test-prop)
	   (setf test-class (car test-prop))
	   (setf test-h (cadr test-prop))
	   (setq test-b new-belief))
	  ((prop-object-p test-prop)
	   (setf test-class (prop-dest test-prop))
	   (setf test-h (prop-history test-prop))
	   (setf test-b (prop-belief test-prop))))
    (if (belief-during test-b b)
      (if (and
	   (or (eq class test-class) (weak-isa-path class test-class))
	   (or (time-test test-h :equals h b) (time-test test-h :during h b)))
	prop)
      (if (and (belief-during b test-b)
	   (or (eq class test-class) (weak-isa-path test-class class))
	   (or (time-test test-h :equals h b) (time-test h :during test-h b))
	   )
	test-prop)
      ))
  )



(defun normal-class-list (new-props old-props default-history new-belief)
 ;; a prop can be one of a kb-prop or a pair (class interval) or simply a class
 ;; assumes old-props is normalized
    (if (null new-props)
      old-props
      (let ((result (tconc))
	    (tested (tconc)))
	(doloop (prop old-props) :vars ((r nil prop))
	  (doloop (other-prop new-props) :vars (below)
	   :when (and (not (eq prop other-prop)) (not (member other-prop (car tested))))
	    (setf below (normal-class-below other-prop r default-history new-belief))
	    (when below
	      (ifn (eq r below)
		(tconc tested other-prop)
		else
		(tconc tested r)
		(setq r below)))
	    )
	 :when (not (member r (car result)))
	  (tconc result r)
	  )
	(doloop :iter (prop-tail new-props (cdr prop-tail))
	 :while prop-tail
	 :vars ((r nil (car prop-tail)))
	  (when (not (member r (car tested)))
	    (doloop (other-prop (cdr prop-tail)) :vars (below)
	     :when (not (member other-prop (car tested)))
	      (setf below (normal-class-below other-prop r default-history new-belief))
	      (when below
		(ifn (eq r below)
		  (tconc tested other-prop)
		  else
		  (tconc tested r)
		  (setq r below)))
	      )
	    (when (not (member r (car result)))
	      (tconc result r))
	    )
	  )
	(car result)))
    )
      


(defun get-class-prop (token property)
  (let ((seen (lconc nil (tok-inst-of token))))
    (doloop (link (car seen))
     :for val := (tok-get-prop (prop-dest link) property)
     :when val :return val
      (doloop (parent-link (tok-parents (prop-dest link)))
       :when (not (member parent-link (car seen)))
	(tconc seen parent-link)
	)
      ))
  )