;;;
;;;   KNOWBEL knowledge representation system
;;;    
;;;    author: Bryan M. Kramer
;;;    
;;;    
;;; Copyright (c) 1990, 1991 University of Toronto, Toronto, ON
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all copies and
;;; supporting documentation.
;;;
;;; The University of Toronto provides this software "as is" without
;;; express or implied warranty.
;;;

;;;    
;;;    


;? theorem prover

 
(eval-when (load compile eval)
  (require 'tools)
  (require 'prove-structs)
  (provide 'prove)
  )


;; next three are macros belonging in inference functions

(defmacro new-state (active-theory theorys belief &optional (horn t) (search-style :depth))
  `(mlet ((theory-list (traverse-theory ,theorys)))
     (make-inference-state :active-theory ,active-theory
			   :belief ,belief
			   :horn ,horn
			   :search-style ,search-style
			   :theory-list theory-list
			   :root-theory (car theory-list))
     )
  )

(defmacro add-cnf-clause (state clause belief name &optional (variables :all) common-vars no-meta errors) ;; cnf but not compiled
  (let ((state-var (gensym "state")))
    `(mlet* ((,state-var ,state)
	     (clause (active-stash (build-clause ,clause
						 (istate-root-theory ,state-var)
						 ,name
						 ,belief
						 ,common-vars
						 ,errors)
				   (istate-active-theory ,state-var))))
	    (goal-build-collect clause ,variables)
	    ,@(when no-meta `((setf (clause-get-prop (ac-clause clause) :no-meta) ,no-meta)))
	    clause)
    )
  )


(defmacro add-wff (state wff belief name &optional (variables :all) errors)
  `(doloop (clause (cnf ,wff)) :vars ((n 0 (+ 1 n))
				      (common-vars (tconc)))
     :collect (add-cnf-clause ,state clause ,belief (list ,name n) ,variables common-vars nil ,errors)
     )
  )




;; a commonly used generator macro



(defmacro gen-status (gen)
  `(mlet ((gen ,gen))
     (cond ((null (sg-clause-ptr gen)) :unmatchable)
	   (t :matched))
     )
  )



(defmacro std-belief ()
  `(make-belief-time)
  )




(defvar *normal-hash* (make-hash-table))
(defvar *normal-number* 0)

(defun normal-hash (object prefix)
  (let ((id (gethash object *normal-hash*)))
    (when (null id)
      (setf id (format nil "~a~d" prefix *normal-number*))
      (setf *normal-number* (+ 1 *normal-number*))
      (setf (gethash object *normal-hash*) id)
      )
    id
    )
  )

(defvar *description-hash* (make-hash-table))
(defvar *description-number* 0)

(defun description-hash (object l reset)
  (if (null *description-number*)
    (format nil "@(~a ~a)" (normal-clause (tok-desc-instance-of object))
	    (normal-clause (tok-desc-attrs object) l reset reset))
    (let ((id (gethash object *description-hash*)))
      (if (null id)
	(progn	
	  (setf id (format nil "/desc~d/"  *description-number*))
	  (setf *description-number* (+ 1 *description-number*))
	  (setf (gethash object *description-hash*) id)
	  (format nil "@~a(~a ~a)" id (normal-clause (tok-desc-instance-of object))
		  (normal-clause (tok-desc-attrs object) l reset reset))
	  )
	id)
      ))
  )

(defun description-hash-reset nil
  (setf *description-hash* (make-hash-table))
  )


(defun normal-belief (belief)
  (cond ((and (consp belief) (eq (car belief) :context))
	 (format nil "c.~a" (normal-belief (caddr belief))))
	((belief-context-p belief)
	 (format nil "/~d.~a/" (time-context-number (belief-context-context belief))
		 (pretty-time (belief-context-belief belief))))
	((integerp belief)
	 (format nil "~a,~a" (pretty-time (belief-time-start belief)) (pretty-time (belief-time-end belief))))
	(belief (format nil ":b ~a" (normal-clause belief)))
	(t "\\\NULL BELIEF///")
    )
  )



(defun normal-clause-number (clause)
  (cond ((active-clause-p clause) (normal-clause (clause-number (ac-clause clause))))
	((clause-p clause) (normal-clause (clause-number clause)))
	((kb-attr-p clause) (normal-clause clause))
	((consp clause) (doloop (s clause) :collect (normal-clause-number s)))
	(t (normal-clause clause)))
  )

(defun normal-time-point (p)
  (or (doloop (same (car (point-same-as p)))
       :vars ((ps nil (if (eq (point-link-source same) p) (point-link-dest same) (point-link-source same))))
       :when (and ps (eq (belief-time-end (point-link-belief same)) :+) (not (time-point-p ps))) :return (pretty-time ps))
      (normal-hash p "tp")
      )
  )

(defun special-token-format (clause level length reset)
  nil)


(defun normal-clause ;; replace variables by atoms
    (clause &optional
	    (level (if *print-level* *print-level* 25)) 
	    (length (if *print-length* *print-length* 25)) 
	    (reset (if *print-length* *print-length* 25)))
  (cond ((< level 1) "#")
	((< length 1) " ...")
	(t 
	 (let ((l (- level 1)))
	   (cond ((null clause) nil)
		 ((special-token-p clause) (special-token-format clause level length reset))
		 ((clause-var-p clause)
		  (if (eq (var-name clause) (var-number clause))
		    (string-downcase (format nil "$~a~a" (var-number clause)
					     (normal-clause (var-type clause) l reset reset)))
		    (string-downcase (format nil "$~a.~d~a" (var-name clause) (var-number clause)
					     (normal-clause (var-type clause) l reset reset)))
		    ))
		 ((kb-attr-p clause) (format nil "[attr ~a ~a ~a ~a]"
					     (normal-clause (attr-label clause) l reset reset)
					     (normal-clause (attr-value clause) l reset reset)
					     (normal-clause (attr-history clause) l reset reset)
					     (normal-belief (attr-belief clause))))
		 ((kb-token-p clause) (string-downcase (format nil "<~a>" (tok-name clause))))
		 ((token-description-p clause) (description-hash clause l reset))
		 ((time-int-p clause)
		  (if (time-not-shared clause)
		    (format nil "(~a)~a,~a"
			    (truncate-string (format nil "~a" (normal-clause (time-not-shared clause) l reset reset)) 6)
			    (if (time-point-p (time-start clause)) (normal-clause (time-start clause) l reset reset) (pretty-time (time-start clause)))
			    (if (time-point-p (time-end clause)) (normal-clause (time-end clause) l reset reset) (pretty-time (time-end clause))))
		    (format nil "~a,~a"
			    (if (time-point-p (time-start clause)) (normal-clause (time-start clause) l reset reset) (pretty-time (time-start clause)))
			    (if (time-point-p (time-end clause)) (normal-clause (time-end clause) l reset reset) (pretty-time (time-end clause))))))
		 ((consp clause) (cons (normal-clause (car clause) l reset reset) (normal-clause (cdr clause) level (- length 1) reset)))
		 ((clause-p clause) (format nil "[~a]~a~a@~a"
					    (normal-clause (clause-number clause) l reset reset)
					    (if (clause-get-prop clause :value)
					      (format nil "== ~a ==" (clause-get-prop clause :value))
					      "")
					    (normal-clause (clause-expression clause) l reset reset)
					    (normal-belief (clause-belief-time clause))
					    ))
		 ((active-clause-p clause)
		  (format nil "<~a(~a)~a~{[~a]~}>"
			  (normal-clause (ac-clause clause) l reset reset)
			  (normal-clause (ac-collect clause) l reset reset)
			  (case (ac-generator clause)
			    (:poison "_poison_")
			    ((nil) "_new_")
			    (:done "_done_")
			    (t "_open_"))
			  (mapcar #'normal-clause-number (ac-justifications clause))))
		 ((prop-object-p clause) (format nil "[~a ~a ~a ~a ~a]"
						 (if (prop-object-p clause :isa) :isa :inst)
						 (normal-clause (prop-src clause) l reset reset)
						 (normal-clause (prop-dest clause) l reset reset)
						 (normal-clause (prop-history clause) l reset reset)
						 (normal-belief (prop-belief clause))))
		 ((meta-type-p clause) (format nil "[m ~a]" (normal-clause (meta-type-type clause) l reset reset)))
		 ((theory-p clause) (format nil "#(theory ~a)" (theory-name clause)))
		 ((kb-p clause) (format nil "#(kb ~d)" (kb-clause-number clause)))
		 ((inference-state-p clause) (format nil "#STATE:~a" (istate-search-style clause)))
		 ((time-point-p clause) (normal-time-point clause))
		 ((stringp clause) (format nil "\"~a\"" clause))
		 ((vectorp clause)
		  (let ((x (car (array-dimensions clause)))
			(s "#("))
		    (doloop :iter (i 0 (+ 1 i)) :while (< i x)
		      (setq s (format nil "~a ~a" s (normal-clause (svref clause i) l (- length 1) reset)))
		      )
		    (format nil "~a)" s)))
		 (t clause)
		 ))))
  )


(defvar *normal-print* t)

(defmacro normal-print (&optional val)
  (if val
    `(setq *normal-print* ,val)
    `(setq *normal-print* (not *normal-print*)))
  )


(defun normal-pp (object &optional (stream *standard-output*))
  (if *print-pretty*
    (if *normal-print*
      (pprint (normal-clause object) stream)
      (pprint object stream)
      )
    (if *normal-print*
      (print (normal-clause object) stream)
      (print object stream)
      )
    )
  )
    


(defvar *dbgn-on* t)

(defmacro dbg (&optional val)
  (if val
    `(setq *dbgn-on* ,val)
    `(setq *dbgn-on* (not *dbgn-on*)))
  )

(defmacro dbgn-print (&rest items)
  `(when *dbgn-on*
     ,@(doloop (item items)
	 :collect (if (and (consp item) (eq (car item) 'quote))
		    `(format t "~%~a" (normal-clause ,item))
		    `(format t " ~a" (normal-clause ,item))
		    ))
     (terpri)
     )
  )


(defmacro dbgl (&rest items)
  `(when *dbgn-on*
     (let ((*description-number* nil))
       ,@(doloop (item items)
	  :collect (if (and (consp item) (eq (car item) 'quote))
		     `(format t "~%~a" (normal-clause ,item))
		     `(lp (normal-clause ,item))
		     ))
       (terpri)
       )
     )
  )


(comment
  `(doloop (item (list ,@items)) (princ (normal-clause item)) (terpri))
  )


(defmacro input-varp (item)
  `(mlet ((item ,item)) (and (symbolp item)(char-equal (char (symbol-name item) 0) #\$)))
  )



(defun traverse-theory1 (th result)
  (when (not (member th (car result)))
    (tconc result th)
    (doloop (inc-th (theory-includes th))
      (traverse-theory1 inc-th result)
      )
    )
  )

(defun traverse-theory (th)
  (let ((result (tconc)))
    (if (listp th)
      (doloop (th th) (traverse-theory1 th result))
      (traverse-theory1 th result)
      )
    (car result)
    )
  )

(defvar *root-type* (make-kb-token :name "proposition"))


(defmacro new-kb (&rest args)
  `(make-kb :tokens (make-hash-table :test 'eq)	    
	    :strings (make-hash-table :test 'equal)
	    :theories (make-hash-table)
	    ,@args)
  )


(defmacro name-theory (th name)
  `(mlet* ((name ,name)(th ,th))
	  (if name (setf (gethash name (kb-theories (theory-kb th))) th))
	  )
  )


(defmacro new-theory (kb name &rest args)
  `(mlet* ((name ,name)
	  (kb ,kb)
	  (theory (make-theory :name name :kb kb ,@args)))
     (if name (setf (gethash name (kb-theories kb)) theory))
     theory)
  )




(defmacro theory-find-theory (theory name)
  `(mlet ((name ,name)
	 (theory ,theory))
     (cond ((null name) theory)
	   ((theory-p name) name)
	   ((stringp name) (gethash (find-symbol (string-upcase name)) (kb-theories (theory-kb theory))))
	   (t (gethash name (kb-theories (theory-kb theory)))))
     )
  )



(defmacro lookup-type1 (name theory)
  `(mlet ((dict (theory-tokens ,theory)))
     (if dict (gethash ,name dict))
     )
  )



(defmacro lookup-string (name theory)
  `(mlet ((dict (theory-strings ,theory)))
     (if dict (gethash ,name dict))
     )
  )


(defmacro type-name (name theory)
  `(mlet ((name ,name))
	 (find-string (if (kb-token-p name)
			(kb-token-name name)
			(if (symbolp name)
			  (string-downcase name)
			 name)
			)
		      ,theory)
	 )
  )

(defmacro new-token (name &optional (theory '*theory*))
  `(mlet* ((name ,name)
	  (tok-name (type-name name theory))
	  (tok (make-kb-token :name name)))
     (setf (gethash tok-name (theory-tokens ,theory)) tok)
     tok
     )
  )

(defmacro new-string (name &optional (theory '*theory*))
  `(mlet* ((name ,name))
     (setf (gethash name (theory-strings ,theory)) name)
     name
     )
  )



(defmacro find-string (name theory)
  `(mlet* ((name ,name)
	  (theory ,theory)
	 (dict (theory-strings theory)))
     (if dict (let ((str (gethash name dict)))
		(if str str (new-string name theory))
		)
       )
     )
  )




(defmacro lookup-type (name &optional (th '*theory*))
  `(mlet ((name ,name)
	  (th ,th))
     (if! (kb-token-p name)
      name
      :elseif (eql #\= (char (string name) 0))
       (or (lookup-type1 (type-name name th) th)
	   (make-member-type name th))
      :else
       (lookup-type1 (type-name name th) th)
       )
     )
  )



(defmacro lookup-type-error (name &optional (th '*theory*))
  `(mlet ((name ,name))
	 (let ((type (lookup-type name ,th)))
	   (if type
	     type
	     (error "~&Unable to identify type of '~s'~%" name)
	     )
	   )
	 )
  )


(defmacro lookup-attr (object name &optional (theory '*theory*))
  `(mlet* ((object ,object)
	   (name (lookup-string ,name ,theory))
	   (token (if (kb-token-p object) object (lookup-type object ,theory))))
	  (lookup-index-cdr (tok-attrs token) name)
	  )
  )


(defmacro now ()
  `(mlet ((p (get-universal-time)))
     (make-time-int :start p
		    :end (+ 1 p))
     )
  )




#-history-alltime
(defmacro default-history ()
  `(theory-all-time *theory*)
  )





#+history-alltime
(defmacro default-history ()
  `(mlet ((h-int (compose-time-int nil nil :-))
	 (b (make-belief-time)))
     (time-assert (now) :during h-int b b b)
     h-int)
  )


(defun lookup-prop (list type1 type2 history belief)
  (dofind (prop list) (and (eq (prop-src prop) type1)
			   (eq (prop-dest prop) type2)
			   (eq (prop-history prop) history)
			   (eq (prop-belief prop) belief)
			   prop)
	  )
  )


(defun normal-reader (stream subchar arg)
  (let ((exp (read stream t nil t)))
    (list 'quote (normal-clause (eval exp)))
    )
  )


(defun type-reader (stream subchar arg)
  (let ((exp (read stream t nil t)))
    (lookup-type exp)
    )
  )


(defun desc-reader (stream subchar arg)
  (let ((exp (read stream t nil t)))
    (list :description exp)
    )
  )

(eval-when (load compile eval)

(defmacro dn ()
  `(progn
     (set-dispatch-macro-character #\# #\l #'type-reader)
     (set-dispatch-macro-character #\# #\n #'normal-reader)
     (set-dispatch-macro-character #\# #\@ #'desc-reader)
     )
  )

)


(defmacro lt (&rest args)
  `(lookup-type ,@args)
  )



(eval-when (load eval)
  (dn)
  )

(defun assert-isa (type1 type2 &optional default-history (belief (std-belief)) prop-list (theory *theory*))
  (let* ((history (or default-history (default-history)))
	 (prop (make-isa-link :src type1 :dest type2 :history-time history :belief-time belief)))
    (tok-make-class type1)
    (tok-make-class type2)
    (when (not (lookup-prop (tok-parents type1) type1 type2 history belief))
      (push prop (tok-parents type1)))
    (when (not (lookup-prop (tok-children type2) type1 type2 history belief))
      (push prop (tok-children type2)))
    (when prop-list (tconc prop-list prop))
    prop
    )
  )


(defun assert-inst (tok type &optional default-history (belief (std-belief)) prop-list (theory *theory*))
  (let* ((history (or default-history (default-history)))
	 (prop (make-inst-link :src tok :dest type :history-time history :belief-time belief)))
    (tok-make-class type)
    (when (not (lookup-prop (tok-inst-of tok) tok type history belief))
      (push prop (tok-inst-of tok)))
    (when (not (lookup-prop (tok-instances type) tok type history belief))
      (push prop (tok-instances type)))
    (when prop-list (tconc prop-list prop))
    prop
    )
  )


(defmacro add-prove-is (tok-name type-name
			&optional (theory '*theory*) history (belief '(std-belief))  prop-list)
  ;; create a token and make it an instance of type, no error check
  `(mlet* ((belief ,belief)
	  (history ,(or history `(default-history)))
	  (tok (new-token ,tok-name ,theory))
	 (type (lookup-type ,type-name ,theory)))
     (assert-inst tok type history belief ,prop-list ,theory)
     tok
     )
  )


(defmacro find-or-create (tok-name type-name theory history belief &optional  prop-list)
  ;; create a token and make it an instance of type, no error check
  `(mlet* ((tok-name ,tok-name)
	  (theory ,theory))
     (or (lookup-type tok-name theory)
	 (add-prove-is (type-name tok-name theory) ,type-name theory ,history ,belief ,prop-list))
     )
  )


(defmacro add-prove-isa (type1 &optional (type2 "proposition") (theory '*theory*)
					 history (belief '(std-belief)) prop-list)
  `(mlet* ((belief ,belief)
	  (history ,(or history `(default-history)))
	  (tok (new-token ,type1 ,theory))
	  (type (lookup-type ,type2 ,theory)))
     (assert-isa tok type history belief ,prop-list ,theory)
     tok
     )
  )


(defun new-description-instance (class time belief theory)
  (let* ((name (gensym (tok-name class)))
	 (tok (make-kb-token :name name)))
    (assert-inst tok class time belief nil theory)
    (setf (tok-description tok) :inst)
    tok
    )
  )
	


(defun type-isa (type1 type2 history belief)
  (cond ((eq type1 type2) history)
	(t
	 (do ((children (tok-children type2) (cdr children))
	      (result nil))
	     ((or result (null children)) result)
	   (when (belief-satisfies belief (prop-belief-time (car children)))
	     (let ((h (if (null history)
			(prop-history-time (car children))
			(time-intersect history (prop-history-time (car children)) belief))))
	       (when h
		 (setq result (type-isa type1 (prop-src (car children)) h belief))
		 ))
	     )
	   )
	 ))
  )



(defun prim-type-instance (token type)
  (declare (special *built-in-type-alist*))
  (if (kb-token-p type)
    (case (cdr (assoc (tok-name type) *built-in-type-alist*))
      (proposition type)
      (string (when (stringp token) type))
      (number (when (numberp token) type))
      (time (when (numberp token) type))
      (description (when (kb-description-p token) type))
      (integer (when (integerp token) type))
      (list (when (listp token) type))
      (real (when (numberp token) type))
      (symbol (when (symbolp token) type))
      (sexpr type)
      (clause (when (clause-p token) type))
      (clause-set (when (not (dofind (clause token) (not (clause-p clause)))) type))
      (time-interval (when (time-int-p token) type))
      (boolean (when (or (eq token t) (null token)) type))
      (theory (when (theory-p token) type))
      (t (let ((fn (tok-prim type)))
					;	 (when fn (dbgn-print 'pti token type fn (funcall fn token) (type-of token)))
	   (if (and fn (funcall fn token))
	     type
	     nil)))
      )
    (if (and (consp type) (eq (car type) :=))
      (member token (cdr type))
      )
    )
  )


(defun type-instance (token type history belief)
  (cond	((eq type *root-type*) history)
	((kb-token-p token)
	 (do ((parents (tok-inst-of token) (cdr parents))
	      (result nil))
	     ((or result (null parents)) result)
	   (when (belief-satisfies belief (prop-belief-time (car parents)))
	     (let ((h (if (null history)
			(prop-history-time (car parents))
			(time-intersect history (prop-history-time (car parents)) belief))))
	       (if h
		   (setq result (type-isa (prop-dest (car parents)) type h belief))
		 ))
	     )))
	((prim-type-instance token type) history)
	((token-description-p token)
	 (doloop (desc-type-prop (tok-desc-instance-of token))
	   :vars ((h nil (type-isa (car desc-type-prop) type (var-type (cadr desc-type-prop)) belief)))
	   :when h :return h))
	(t nil))
  )

					; true if there is some is-a proposition for use in unifier
(defun weak-type-isa (type1 type2 belief)
  (cond ((eq type1 type2)
	 type2)
	(t
	 (do ((parents (tok-parents type1) (cdr parents))
	      (result nil))
	     ((or result (null parents)) result)
	   (when (belief-satisfies belief (prop-belief (car parents)))
	     (setq result (weak-type-isa (prop-dest (car parents)) type2 belief)))
	   )))
  )


(defun weak-isa-path (class ancestor)
  ; true if there is some path regardless of times
  (or (eq class ancestor)
      (doloop (parent (tok-parents class))
	:exists (weak-isa-path (prop-dest parent) ancestor)
	))
  )




(defun weak-type-instance (token type belief)
  (cond	((eq type *root-type*) type)
	((kb-token-p token)
	 (do ((parents (tok-inst-of token) (cdr parents))
	      (result nil))
	     ((or result (null parents)) result)
	   (when (belief-satisfies belief (prop-belief (car parents)))
	     (setq result (weak-type-isa (prop-dest (car parents)) type belief)))
	   ))
	((special-token-p token)
	 (weak-type-isa (special-token-instance-of token) type belief))
	((token-description-p token)
	 (doloop (desc-type-prop (tok-desc-instance-of token))
	  :vars ((h nil (weak-type-isa (car desc-type-prop) type belief)))
	  :when h :return h))
	(t (prim-type-instance token type))
	)
  )


(defun meta-type-instance (token meta-type belief)
  (let ((type (meta-type-type meta-type)))
    (cond ((kb-token-p type)
	   (doloop (prop (tok-inst-of token))
	     :when (and (belief-satisfies belief (prop-belief prop))
			(weak-type-instance (prop-dest prop) type belief))
	     :return t))
	  ((meta-type-p type)
	   (doloop (prop (tok-inst-of token))
	     :when (and (belief-satisfies belief (prop-belief prop))
			(meta-type-instance (prop-dest prop) type belief))
	     :return t))
	  (t (error "bad meta-type")))
    )
  )





(defun meta-compatible-token (token type meta-type belief)
  (let ((match (weak-type-instance token type belief)))
    (cond ((null match) nil)
	  ((null meta-type) match)
	  ((not (kb-token-p token)) match)
	  ((meta-type-instance token meta-type belief) match)
	  (t nil))
    )
  )




(defmacro unify-var-p (clause root1 root2)
 `(and (clause-var-p ,clause) (or (eq (var-owner ,clause) ,root1) (eq (var-owner ,clause) ,root2)))
 )
				   
	 

;;; unifier's weak compatible ---- only belief time taken into accout

(defun compatible (clause1 clause2 sigma root1 root2 belief)
  (cond ((unify-var-p clause1 root1 root2)
	 (let ((type1 (aref sigma (var-num clause1 root1 root2) 1)))
	   (cond ((unify-var-p clause2 root1 root2)
		  (let ((type2 (aref sigma (var-num clause2 root1 root2) 1)))
		    (cond ((time-int-p type1)
			   (and (time-int-p type2) (time-intersect type1 type2 belief)))
			  ((time-int-p type2) nil)
			  (t (if (weak-type-isa type1 type2 belief)
			       type1
			       (if (weak-type-isa type2 type1 belief)
				 type2
				 nil))))))
		 (t (meta-compatible-token clause2 type1 (var-meta-type clause1) belief)))
	   ))
	((unify-var-p clause2 root1 root2)
	 (let ((type2 (aref sigma (var-num clause2 root1 root2) 1)))
	   (cond ((time-int-p type2) nil)
		 (t (meta-compatible-token clause1 type2 (var-meta-type clause2) belief)))))
	(t t))
  )



(defmacro extract-var-name (atom)
  `(let* ((atom (string ,atom))
	  (pos1 (if (eq #\$ (char atom 0)) 1 0))
	  (pos/ (position #\/ atom))
	  (pos (or pos/ (position #\@ atom) (position #\( atom))))
     (if (and pos (> pos pos1))
       (subseq atom pos1 pos)
       (if (null pos) (subseq atom pos1) nil))
     )
  )



(defmacro extract-type-name (atom)
  `(mlet* ((atom (string ,atom))
	  (pos/ (position #\/ atom))
	  (pos@ (position #\@ atom))
	  (pos1 (or pos/ (if pos@ (- pos@ 1)) (position #\( atom)))
	  (len (length atom))
	  (pos (if (and pos1 (>= pos1 (- len 1))) nil pos1))
	  (endpos (position #\[ atom)))
     (if pos
       (progn
	 (when (or (null endpos) (< endpos pos))
	   (if (or pos@ pos/) (setq endpos len) (setq endpos (or (position #\) atom) len))))
	 (if (>= (+ 1 pos) endpos)
	   "proposition"
	   (string-downcase (subseq atom (+ 1 pos) endpos))))
       "proposition"
       )
     )
  )


(defmacro extract-meta-type-name (atom)
  `(mlet* ((atom (string ,atom))
	  (pos1 (position #\[ atom))
	  (len (length atom))
	  (pos (if (and pos1 (>= pos1 (- len 1))) nil pos1))
	  (endpos (position #\] atom)))
     (if pos
       (progn
	 (when (null endpos) (setq endpos (position #\) atom)))
	 (when (or (null endpos) (< endpos pos))
	   (setq endpos len))
	 (if (>= (+ 1 pos) endpos)
	   nil
	   (subseq atom (+ 1 pos) endpos)))
       nil
       )
     )
  )


(defun parse-time-point-hr (time default &optional (start-interval t))
  (multiple-value-bind (sec min hr day month year) (decode-universal-time default)
    (ifn start-interval
      (setq min 0)
      (setq sec 0)
     else
      (setq min 59)
      (setq sec 59)
      )
    (let ((hr-pos (position #\: time)))
      (setq hr (parse-integer time :junk-allowed t))
      (when hr-pos
	(let ((min-pos (position #\: time :start (1+ hr-pos))))
	  (setq min (parse-integer time :junk-allowed t :start (1+ hr-pos)))
	  (if min-pos
	    (setq sec (parse-integer time :junk-allowed t :start (1+ min-pos)))
	    (setq sec 0)
	    ))))
    (encode-universal-time sec min hr day month year)
    )
  )

(defvar *month-ends* (vector 0 31 28 31 30 31 30 31 31 30 31 30 31))

(defun parse-time-point-year (time default &optional (start-interval t))
  (multiple-value-bind (sec min hr day month year) (decode-universal-time default)
    (ifn start-interval
      (setq min 0)
      (setq sec 0)
      (setq day 1)
      (setq month 1)
      (setq hr 0)
     else
      (setq hr 11)
      (setq min 59)
      (setq sec 59)
      (setq day 31)
      (setq month 12)
      )
    (let ((time (string time)))
      (let ((year-pos (position #\_ time)))
	(ifn year-pos
	  (setq year (parse-integer time :junk-allowed t :end year-pos))
	  (incr year-pos)
	  (let ((month-pos (position #\_ time :start year-pos)))
	    (ifn month-pos
	      (setq month (parse-integer time :junk-allowed t :start year-pos))
	      (incr month-pos)
	      (let ((day-pos (position #\_ time :start month-pos)))
		(ifn day-pos 
		  (setq day (parse-integer time :start month-pos :junk-allowed t))
		  (incr day-pos)
		  (let ((hr-pos (position #\_ time :start day-pos)))
		    (setq hr (parse-integer time :start day-pos :junk-allowed t))
		    (when hr-pos
		      (let ((min-pos (position #\_ time :start (1+ hr-pos))))
			(setq min (parse-integer time :junk-allowed t :start (1+ hr-pos)))
			(if min-pos
			  (setq sec (parse-integer time :junk-allowed t :start (1+ min-pos)))
			  (setq sec 0)
			  ))))
		 else
		  (setq day (parse-integer time :junk-allowed t :start month-pos))
		  )
		)
	     else
	      (setq month (parse-integer time :junk-allowed t :start year-pos))
	      )
	    )
	 else
	  (setq year (or (parse-integer time :junk-allowed t) year))
	  )
	)
      )
    (if (> day (svref *month-ends* month)) (setf day (svref *month-ends* month)))
    (if (< year 1900)
      default
      (encode-universal-time sec min hr day month year)
      )
    )
  )






(defun time-convert-str (str &optional (default (get-universal-time))(start-interval t))
  (cond ((null str) default)
	((equal str "-") :-)
	((equal str "+") :+)
	((equal str "now") (get-universal-time))
	((equal str "") (get-universal-time))
        ((position #\_ str) (parse-time-point-year str default start-interval))
	((position #\: str) (parse-time-point-hr str default start-interval))
	(t (parse-time-point-year str default start-interval)))
  )


(defun extract-words1 (str)
  (doloop :iter (pos1 1 (when pos2 (+ 1 pos2))) :iter (pos2 (position #\  str) (when pos2 (position #\  str :start pos1)))
    :while pos1
    :when (not (eq pos1 pos2))
    :collect (subseq str pos1 pos2)
    )
  )


(defun extract-words (str)
  (doloop :iter (pos1 1 (when pos2 (+ pos1 (cadr pos2))))
   :iter (pos2 (multiple-value-list (read-from-string (subseq str pos1) nil :end))
	       (when pos1 (multiple-value-list (read-from-string (subseq str pos1) nil :end))))
    :while (not (eq (car pos2) :end))
    :collect (car pos2))
  )



(defun time-create-relative (expression theory belief owner errors)
  (let ((inter (compose-time-int nil nil belief owner))
	(op (car expression)))
    (if (consp (cadr expression))
      (time-assert inter op (make-time-int :start (time-convert-str (time-force-string (caadr expression)))
					   :end (time-convert-str (time-force-string (cadadr expression)) (get-universal-time) nil)
					   :not-shared (cadr expression))
		   belief belief belief errors)
      (ndprog :init nil
	  (nd-for prop prop-cont (find-prop (cdr expression) theory belief prop-cont)
		  (if (kb-attr-p prop)
		    (time-assert inter op (attr-history prop) belief (attr-belief prop) belief errors)
		    (time-assert inter op (prop-history prop) belief (prop-belief prop) belief errors)
		    )
	    )
	  )
      )
    inter)
  )



(defun extract-type-interval (name belief theory errors)
					; for now, just real intervals, need to get ops in, also time classes
					; recognize intervals by ,
  (if name
    (let* ((expr (string name))
	   (pos-comma (position #\, expr))
	   (pos1 (or pos-comma (position #\^ expr)))
	   (pos2 (position #\@ expr))
	   (len (length expr))
	   )
      (if pos1				; yes is time interval
	(let ((start (time-convert-str (subseq expr (if pos2 (+ pos2 1) 0) pos1)))
	      (end (time-convert-str (subseq expr (+ pos1 1)))))
	  (make-time-int :start start
			 :end end)
	  )
	(if pos2
	  (if (string-equal (subseq expr (+ 1 pos2)) "<>")
	    (let ((int (get-time-int nil belief)))
	      (setf (time-not-shared int) :unconstrained)
	      int)
	    (parse-time-work (extract-words expr) (theory-all-time theory) name theory errors belief))
	  nil))
      )
    )
  )
       


(defun build-meta-type1 (clause theory)
  (let ((type-name (extract-meta-type-name clause)))
    (if type-name
      (make-meta-type :type (build-meta-type1 type-name theory))
      (make-meta-type :type (lookup-type clause theory))
      ))
  )


(defun build-meta-type (clause theory)
  (let ((type-name (extract-meta-type-name clause)))
    (if type-name
      (build-meta-type1 type-name theory)
     nil
      ))
  )


(defun build-description (expression theory vars n owner common-vars default-time belief errors)
  ;;  (dbgn-print 'building expression)
  (make-token-description :instance-of
			  (if (not (consp expression))
			    (lookup-type-error expression theory)
			    (if (not (consp (car expression)))
			      (list (list (lookup-type-error (car expression) theory) default-time))
			      (doloop (type-prop (car expression))
			       :collect
				(if (consp type-prop)
				  (list (lookup-type-error (car type-prop) theory)
					(clause-compile (cadr type-prop) theory vars n owner common-vars belief errors))
				  (list (lookup-type-error type-prop theory) default-time))
				)))
			  :attrs
			  (if (consp (cadr expression))
			    (doloop (attr-group (cadr expression))
			     :collect
			      (cons (find-string (car attr-group) theory)
				    (if (consp attr-group)
				      (doloop (attr-prop (cdr attr-group))
				       :collect
					(list (if (or (symbolp (car attr-prop)) (stringp (car attr-prop)))
						(find-string (car attr-prop) theory)
						(clause-compile (car attr-prop) theory vars n owner common-vars belief errors))
					      (clause-compile (cadr attr-prop) theory vars n owner common-vars belief errors)
					      (if (caddr attr-prop)
						(clause-compile (caddr attr-prop) theory vars n owner common-vars belief errors)
					       default-time)
					      )
					)))
			      ))
			  )
  )



(defun clause-copy-description (desc theory vars n owner common-vars belief errors)
  (make-token-description :instance-of
			  (doloop (pair (tok-desc-instance-of desc))
			   :collect (list (clause-compile (car pair) theory vars n owner common-vars belief errors)
					  (clause-compile (cadr pair) theory vars n owner common-vars belief errors))
			    )
			  :attrs
			  (doloop (attr-group (tok-desc-attrs desc))
			   :collect
			    (cons (car attr-group)
				  (doloop (attr-prop (cdr attr-group))
				   :collect
				    (list (clause-compile (car attr-prop) theory vars n owner common-vars belief errors t)
					  (clause-compile (cadr attr-prop) theory vars n owner common-vars belief errors t)
					  (clause-compile (caddr attr-prop) theory vars n owner common-vars belief errors t)
					  )
				    ))
			    ))
  )
									



(defun clause-compile-clause-var (var vars n owner common-vars errors)
  (let ((var-new (assoc var (car vars)))
	common-var)
    (cond (var-new (cdr var-new))
	  (t (let ((new-var (make-clause-var :number (car n)
					     :meta-type (var-meta-type var)
					     :name (var-name var)
					     :type (var-type var)
					     :owner owner) ))
	       (tconc vars (cons var new-var))
	       (when common-vars (tconc common-vars (cons var new-var)))
	       (setf (car n) (+ 1 (car n)))
	       new-var))))
  )


(defun clause-update-var-type (var spec theory belief errors)
  ;;  (dbgn-print 'uvt var)
  (if (eq (var-type var) (theory-root-object theory))
    (let* ((type (when (or (kb-token-p spec) (time-int-p spec)) spec))
	   (type-str (when (not type) (extract-type-name spec)))
	   (time-type (when type-str (extract-type-interval type-str belief theory errors))))
      (setf (var-type var) (or type time-type (lookup-type type-str theory) (theory-root-object theory)))
      ;; (dbgn-print 'uvt-exit spec type type-str time-type var)
      ))
  var
  )


(defun clause-compile-cnf-var (var theory vars n owner common-vars belief errors)
  (let ((var-new (assoc var (car vars))))    
    (cond (var-new (clause-update-var-type (cdr var-new) (cnf-var-input var) theory belief errors))
	  (t (let* ((spec (cnf-var-input var))
		    (type (when (or (kb-token-p spec) (time-int-p spec)) spec))
		    (type-str (when (not type) (extract-type-name (cnf-var-input var))))
		    (time-type (when type-str (extract-type-interval type-str belief theory errors)))
		    (new-var (make-clause-var :number (car n)
					      :meta-type (when (not type) (build-meta-type spec theory))
					      :name (cnf-var-name var)
					      :type (or type time-type (lookup-type type-str theory) (theory-root-object theory))
					      :owner owner) ))
	       (tconc vars (cons var new-var))
	       (when common-vars (tconc common-vars (cons var new-var)))
	       (setf (car n) (+ 1 (car n)))
	       new-var))))
  )


(defun clause-compile-input-var (clause theory vars n owner common-vars belief errors)
  (let* ((var-str (extract-var-name clause))
	 (var-name (if var-str (string-upcase var-str) (car n))))
    (let ((var (assoc var-name (car vars) :test #'equal))
	  common-var)
      (cond (var (clause-update-var-type (cdr var) clause theory belief errors))
	    ((and common-vars (setq common-var (cdr (assoc var-name (car common-vars) :test #'equal))))
	     (clause-compile-clause-var common-var vars n owner common-vars errors))
	    (t (let* ((type-str (extract-type-name clause))
		      (time-type (extract-type-interval type-str belief theory errors))
		      (new-var (make-clause-var :number (car n)
						:meta-type (build-meta-type clause theory)
						:name var-name
						:type (or time-type (lookup-type type-str theory) (theory-root-object theory))
						:owner owner) ))
		 (tconc vars (cons var-name new-var))
		 (when common-vars (tconc common-vars (cons var-name new-var)))
		 (setf (car n) (+ 1 (car n)))
		 new-var))
	    )))
  )

(defun clause-bquote-compile (clause theory vars n owner common-vars belief errors)
  (cond ((null clause) nil)
	((listp clause)
	 (cond ((eq (car clause) 'comma)
		(clause-compile (cadr clause) theory vars n owner common-vars belief errors))
	       ((consp (car clause))
		(doloop (expr clause)
		 :collect (clause-bquote-compile expr theory vars n owner common-vars belief errors)
		  ))
	       (t (cons (car clause)
			(doloop (expr (cdr clause))
			 :collect (clause-bquote-compile expr theory vars n owner common-vars belief errors)
			  )))))
	(t clause))
  )


(defun clause-compile-object (name theory vars n owner common-vars belief errors)
  (or (lookup-type name theory)
      (let ((obj (add-prove-is (type-name name theory) (theory-root-object theory) theory (theory-all-time theory) belief)))
	(when errors (tconc errors `(warning-object-creating ,name)))
	obj))
  )


(defun clause-compile (clause theory vars n owner common-vars belief errors &optional keep-symbol)
  (cond ((cnf-var-p clause) (clause-compile-cnf-var clause theory vars n owner common-vars belief errors))
	((input-varp clause) (clause-compile-input-var clause theory vars n owner common-vars belief errors))
	((clause-var-p clause)
	 (clause-compile-clause-var clause vars n owner common-vars errors))
	((or (null clause) (eq clause t)) clause)
	((listp clause)
	 (cond ((eq (car clause) 'quote)
		(cadr clause))
	       ((eq (car clause) 'bquote)
		(clause-bquote-compile (cadr clause) theory vars n owner common-vars belief errors))
	       ((eq (car clause) 'object)
		(clause-compile-object (cadr clause) theory vars n owner common-vars belief errors))
	       ((eq (car clause) 'symbol)
		(cadr clause))
	       ((eq (car clause) 'belief) (get-belief-time (cadr clause) (caddr clause) (cadddr clause)))
	       ((eq (car clause) :description)
		(build-description (cadr clause) theory vars n owner common-vars
				   (clause-compile-input-var '|$dft(-,+)| theory vars n owner common-vars belief errors)
				   belief
				   errors))
	       ((consp (car clause))
		(doloop (expr clause)
		 :collect (clause-compile expr theory vars n owner common-vars belief errors keep-symbol)))
	       (t (cons (car clause)
			(doloop (expr (cdr clause))
			 :collect (clause-compile expr theory vars n owner common-vars belief errors keep-symbol))
			))))
	((token-description-p clause) (clause-copy-description clause theory vars n owner common-vars belief errors))
	((numberp clause) clause)
	((stringp clause) (find-string clause theory))
	((clause-wild-card-p clause) clause)
	((keywordp clause) clause)
	((symbolp clause) (if keep-symbol
			    clause
			    (or (lookup-type clause theory) (find-string (string clause) theory))))
	(t clause))
  )


(defun build-clause (clause theory &optional (num 0) (belief (std-belief)) common-vars errors)
  (let ((n (list 0))
	(vars (tconc))
	(owner (make-clause :number num)))
    (let ((compiled (clause-compile clause theory vars n owner common-vars belief errors nil)))
      (setf (clause-expression owner) compiled)
      (setf (clause-belief-time owner) belief)
      (setf (clause-n-vars owner) (car n))
      (setf (clause-vars owner) (apply #'vector (mapcar #'cdr (car vars))))
      owner
      )
    )
  )





(defmacro kb-token (clause sigma root1 root2)
  `(mlet ((clause ,clause))
     (cond ((unify-var-p clause ,root1 ,root2) (aref ,sigma (var-num clause ,root1 ,root2) 1))
	   ((kb-token-p clause) (tok-inst-of clause))
	   ((numberp clause) *number-type*)
	   ((time-int-p clause) clause)
	   ((consp clause)
	    (if (and (symbolp (car clause)) (get (car clause) 'fn-type))
		(get (car clause) 'fn-type)
	      *root-type*))
	   (t clause))
     )
  )




(defun unify-time (interval variable sigma root belief)
  (if (unify-var-p interval root nil)
    (unify-work interval variable sigma root root belief)
    (let* ((vnum (and (unify-var-p variable root nil) (var-num variable root nil)))
	   (type (and vnum (aref sigma vnum 1))))
      (if (and (time-int-p interval) (time-int-p type))
	(let ((inter (time-intersect interval type belief)))
	  (if inter
	    (setf (aref sigma vnum 1) inter)
	    ))
	))
    )
  )






(defun unify-work (clause1 clause2 sigma root1 root2 belief)
  ;(dbgn-print 'unify-work clause1 clause2)
  (cond ((or (clause-wild-card-p clause1) (clause-wild-card-p clause2)) sigma)
	((eql clause1 clause2) sigma)
	(t (let ((unify-type (compatible clause1 clause2 sigma root1 root2 belief)))
	     (cond ((null unify-type) nil)
		   ((unify-var-p clause1 root1 root2) ;; variable
		    (let* ((vnum (var-num clause1 root1 root2))
			   (val (aref sigma vnum 0)))
		      (setf (aref sigma vnum 1) unify-type)	       
		      (cond ((or (eq val '*none*)
				 (unify-var-p val root1 root2))
			     (if (unify-var-p clause2 root1 root2)
			       (let ((vnum2 (var-num clause2 root1 root2)))
				 (if (not (= vnum vnum2))
				   (progn
				     (setf (aref sigma vnum 0) clause2)
				     (setf (aref sigma vnum2 1) unify-type)))
				 )
			       (setf (aref sigma vnum 0) clause2)
			       )
			     sigma)
			    (t (unify-work val clause2 sigma root1 root2 belief)))
		      ))
		   ((unify-var-p clause2 root1 root2) ;; variable
		    (let* ((vnum (var-num clause2 root1 root2))
			   (val (aref sigma vnum 0)))
		      (setf (aref sigma vnum 1) unify-type)
		      (cond ((eq val '*none*)
			     (setf (aref sigma vnum 0) clause1)
			     sigma)
			    (t (unify-work val clause1 sigma root1 root2 belief)))
		      ))
		   ((and (consp clause1) (consp clause2)
			 (= (length clause1) (length clause2)) (eq (car clause1) (car clause2)))
		    (do ((expr1 (cdr clause1) (cdr expr1))
			 (expr2 (cdr clause2) (cdr expr2))
			 (done sigma))
			((or (null done) (null expr1)) done)
		      (setq done (unify-work (car expr1) (car expr2) sigma root1 root2 belief))
		      ))
		   (t nil)))
	   ))
  )

(comment
(defun unify-work (clause1 clause2 sigma root1 root2 belief)
  (cond ((or (clause-wild-card-p clause1) (clause-wild-card-p clause2)) sigma)
	((eql clause1 clause2) sigma)
	(t (let ((unify-type (compatible clause1 clause2 sigma root1 root2 belief)))
	     (cond ((null unify-type)
		    (dbgn-print 'unify-work-failure-type-mismatch clause1 clause2
				'b belief (unify-var-p clause1 root1 root2) (unify-var-p clause2 root1 root2))
		    nil)
		   ((unify-var-p clause1 root1 root2) ;; variable
		    (let* ((vnum (var-num clause1 root1 root2))
			   (val (aref sigma vnum 0)))
		      (setf (aref sigma vnum 1) unify-type)	       
		      (cond ((eq val '*none*)
			     (if (unify-var-p clause2 root1 root2)
			       (let ((vnum2 (var-num clause2 root1 root2)))
				 (if (not (= vnum vnum2))
				   (progn
				     (setf (aref sigma vnum 0) clause2)
				     (setf (aref sigma vnum2 1) unify-type)))
				 )
			       (setf (aref sigma vnum 0) clause2)
			       )
			     sigma)
			    (t (unify-work val clause2 sigma root1 root2 belief)))
		      ))
		   ((unify-var-p clause2 root1 root2) ;; variable
		    (let* ((vnum (var-num clause2 root1 root2))
			   (val (aref sigma vnum 0)))
		      (setf (aref sigma vnum 1) unify-type)
		      (cond ((eq val '*none*)
			     (setf (aref sigma vnum 0) clause1)
			     sigma)
			    (t (unify-work val clause1 sigma root1 root2 belief)))
		      ))
		   ((and (consp clause1) (consp clause2)
			 (= (length clause1) (length clause2)) (eq (car clause1) (car clause2)))
		    (do ((expr1 (cdr clause1) (cdr expr1))
			 (expr2 (cdr clause2) (cdr expr2))
			 (done sigma))
			((or (null done) (null expr1)) done)
		      (setq done (unify-work (car expr1) (car expr2) sigma root1 root2 belief))
		      ))
		   (t (dbgn-print 'unify-work-failure-total-mismatch clause1 clause2) nil)))
	   ))
  )

)

(defun unify (atom1 atom2 clause1 clause2 belief
	      &optional (sigma (make-array (list (+ (clause-n-vars clause1) (clause-n-vars clause2)) 2)
			   :initial-element '*none*)))
  (unify-work atom1 atom2 sigma clause1 clause2 belief)
  )


(defun reset-sigma (sigma clause1 &optional clause2)
  (let ((n1 (clause-n-vars clause1))
	(n2 (if clause2 (clause-n-vars clause2) 0)))
    (do ((j 0 (+ 1 j)))
	((>= j n1))
      (setf (aref sigma j 0) '*none*)
      (setf (aref sigma j 1) (var-type (svref (clause-vars clause1) j)))
      )
    (do ((j 0 (+ 1 j))
	 (i n1 (+ 1 i)))
	((>= j n2))
      (setf (aref sigma i 0) '*none*)
      (setf (aref sigma i 1) (var-type (svref (clause-vars clause2) j)))
      )
    )
  )




(eval-when (load compile eval)

(defun add-to-index-build-alist (keys val)
  (cond ((cdr keys) `(list ,(car keys) ,(add-to-index-build-alist (cdr keys) val)))
	(t `(list ,(car keys) ,val)))
  )



(defun add-to-index-process-keys (place keys val unique)
  (let ((bucket-var (gensym "bucket"))
	(val-var (gensym "VAL")))
    `(let ((,bucket-var (assoc ,(car keys) ,place)))
       (if (null ,bucket-var)
	 (setf ,place (cons ,(add-to-index-build-alist keys val) ,place))
	 ,(if (cdr keys)
	    (add-to-index-process-keys `(cdr ,bucket-var) (cdr keys) val unique)
	    (if unique
	      `(let ((,val-var ,val))
		 (when (not (member ,val-var (cdr ,bucket-var)))
		   (setf (cdr ,bucket-var) (cons ,val-var (cdr ,bucket-var))))
		 )
	      `(setf (cdr ,bucket-var) (cons ,val (cdr ,bucket-var)))
	      )
	    )))
    )
  )
)

;;; never add a level to an existing bucket 


(defmacro add-to-index (place keys val &optional unique)
  (add-to-index-process-keys place keys val unique)
  )

(eval-when (load compile eval)

(defun lookup-index-process (place keys)
  (let ((bucket-var (gensym "bucket")))
    `(let ((,bucket-var (assoc ,(car keys) ,place)))
       (if (null ,bucket-var)
	 nil
	 ,(if (cdr keys)
	    (lookup-index-process `(cdr ,bucket-var) (cdr keys))
	    bucket-var
	    )))
    )
  )


)

(defmacro lookup-index (place &rest keys)
  (lookup-index-process place keys)
  )


(defmacro lookup-index-cdr (place &rest keys)
  `(cdr (lookup-index ,place ,@keys))
  )



;;; search generator functions



;;; generator clause ptr is  (current-clause . generator)
;;; default generator is a list so simply take cdr of clause ptr to advance
;;; otherwise dispatch on generator type to get next clause ptr (currently not imp



(defmacro gen-clause (gen)
  `(car (sg-clause-ptr ,gen))
  )


(defun get-attr-index (index atom)
  (comment dbgn-print 'gai index 'atom atom 'var (clause-var-p (nth 3 atom))
    'all (cdr (lookup-index index (car atom) :all))
    'specific (cdr (lookup-index index (car atom) (nth 2 atom)))
    'vari  (cdr (lookup-index index (car atom) :var))
    )
  (if (clause-var-p (nth 3 atom))
    (cdr (lookup-index index (car atom) :all))
    (append (cdr (lookup-index index (car atom) (nth 3 atom)))
	    (cdr (lookup-index index (car atom) :var)))
    )
  )


(defun get-instance-of-index (index atom)
  (comment dbgn-print 'gioi index 'atom atom 'var (clause-var-p (nth 2 atom))
    'all (cdr (lookup-index index (car atom) :all))
    'specific (cdr (lookup-index index (car atom) (nth 2 atom)))
    'var  (cdr (lookup-index index (car atom) :var))
    )
  (if (clause-var-p (nth 2 atom))
    (cdr (lookup-index index (car atom) :all))
    (append (cdr (lookup-index index (car atom) (nth 2 atom)))
	    (cdr (lookup-index index (car atom) :var)))
    )
  )

(defun lookup-theory-index (theory sign literal)
  (let* ((index (if sign
		  (theory-pos-indices theory)
		  (theory-neg-indices theory)))
	 (indexed
	  (case (car literal)
	    (attr (get-attr-index index literal))
	    (instance-of (get-instance-of-index index literal))
					;  (isa)
	    (t (cdr (assoc (car literal) index))))))
    indexed
    )
  )


(defun gen-next-theory (clause gen t-reset theory-list)
  (let* ((atom (car (clause-expression clause)))
	 (sign (if (eq (car atom) 'not) 'not nil))
	 (pred (if (eq (car atom) 'not) (caadr atom) (car atom))))
    (do ((theories (if t-reset theory-list (cdr (sg-theory-ptr gen))) (cdr theories))
	 (clause-ptr nil))
	((or (null theories) clause-ptr)
	 (progn
	   (setf (sg-theory-ptr gen) theories)
	   (setf (sg-clause-ptr gen) clause-ptr)))
      (let* ((index (if sign
		      (theory-pos-indices (car theories))
		      (theory-neg-indices (car theories))))
	     (literal (if sign
			(cadr atom)
		       atom))
	     (indexed
	      (case pred
		(attr (get-attr-index index literal))
		(instance-of (get-instance-of-index index literal))
		;(isa)
		(t (cdr (assoc pred index))))))
	(setf clause-ptr indexed)
	)
      )
    )
  )


(defmacro gen-next-clause (clause gen theory-list)
  `(mlet ((gen ,gen))
     (if (not (cdr (sg-clause-ptr gen)))
	   (gen-next-theory ,clause gen nil ,theory-list)
       (setf (sg-clause-ptr gen) (cdr (sg-clause-ptr gen)))
       )
     (setf (sg-match-position gen) :init)
     (sg-clause-ptr gen)
     )
  )


(defmacro gen-set-match (gen pos)
  `(setf (sg-match-position ,gen) ,pos)
  )

(defmacro gen-has-clause (gen)
  `(sg-clause-ptr ,gen)
  )


(defmacro gen-active (gen)
  `(mlet ((gen ,gen)) (and gen (sg-clause-ptr gen)))
  )

(defmacro clause-nogood (active-clause)
  `(eq :poison (ac-generator ,active-clause))
  )


(defmacro set-clause-nogood (active-clause &optional (val :poison))
  `(setf (ac-generator ,active-clause) ,val)
  )

  
    
  

(defmacro get-generator (clause theory-list &optional active-theory)
  `(mlet ((clause ,clause)
	 (theory-list ,theory-list)
	 (active-theory ,(if active-theory active-theory nil)))
     (let ((gen (make-search-generator :theory-ptr nil
				       :clause-ptr nil
				       :lisp-state :init
				       :meta-state (if (clause-get-prop clause :no-meta) nil :init)
				       :match-position :init)))
       (gen-next-theory clause gen t (if active-theory (cons active-theory theory-list) theory-list))
       gen
       )
     )
  )




; assumes clauses have verified syntax

(defun match-clauses (clause1 clause2 gen belief state)
  (let ((atom (car (clause-expression clause1)))
	(pos2 (cond ((eq (sg-match-position gen) :init) (clause-expression clause2))
		    (t (sg-match-position gen))))
	(sigma (istate-get-sigma state (+ (clause-n-vars clause1) (clause-n-vars clause2)) 2))
	(done nil))
    (do nil
	((or done (null pos2)) nil)
      (reset-sigma sigma clause1 clause2)
      (cond ((and (eq (car atom) 'not)
		  (eq (car (cadr atom)) (caar pos2))) 	 ;;; test  for same predicate
	     (setq done (unify-work (cadr atom) (car pos2) sigma clause1 clause2 belief))
	     )
	    ((and (eq (caar pos2) 'not)
		  (eq (car (cadar pos2)) (car atom))) 	 ;;; test  for same predicate
	     (setq done (unify-work atom (cadar pos2) sigma clause1 clause2 belief))
	     )
	    )
      (if (null done) (setq pos2 (cdr pos2)))
      )
    (gen-set-match gen pos2)
    done
    )
  )
	    
      
(defun match-all (clause1 clause2 belief &optional (all t))
  (if (or (not all) (= (length (clause-expression clause1)) (length (clause-expression clause2))))
      (let ((sigma (make-array (list (+ (clause-n-vars clause1) (clause-n-vars clause2)) 2)))
	    (all-match t))
	(reset-sigma sigma clause1 clause2)
	(do ((pos1 (clause-expression clause1) (cdr pos1)))
	    ((or (not all-match) (null pos1)) (if all-match sigma))
	  (do ((pos2 (clause-expression clause2) (cdr pos2))
	       (done nil))
	      ((or done (null pos2)) (setq all-match done))
	    (cond ((and (eq (caar pos1) 'not)
			(eq (caar pos2) 'not)
			(eq (car (cadar pos1)) (caadar pos2))) 	 ;;; test  for same predicate
		   (setq done (unify-work (cadar pos1) (cadar pos2) sigma clause1 clause2 belief))
		   )
		  ((eq (caar pos1) (caar pos2)) 	 ;;; test  for same predicate
		   (setq done (unify-work (car pos1) (car pos2) sigma clause1 clause2 belief))
		   )
		  )
	    )
	  )
	)
    nil)
  )
	    
      
(defun apply-sigma-copy-sub (clause sigma new-root root1 root2)
  (cond ((null clause) nil)
	((unify-var-p clause root1 root2)
	 (if (eq (var-owner clause) new-root)
	  clause
	   (let ((vnum (var-num clause root1 root2)))
	     (if (not (aref sigma vnum 1))
	       (progn
		 (setf (aref sigma vnum 1) t)
		 (setf (aref sigma vnum 0)
		   (apply-sigma-copy-sub (aref sigma vnum 0)
					 sigma new-root root1 root2))
		 )
	       )
	     (aref sigma vnum 0)
	     )))
	((consp clause)
	 (doloop (item clause)
	   :collect (apply-sigma-copy-sub item sigma new-root root1 root2)
	  :atom-tail tail (apply-sigma-copy-sub tail sigma new-root root1 root2)))
	((token-description-p clause)
	 (make-token-description :instance-of (apply-sigma-copy-sub (tok-desc-instance-of clause) sigma new-root root1 root2)
				 :attrs (apply-sigma-copy-sub (tok-desc-attrs clause) sigma new-root root1 root2)))
	(t clause)
	)
  )
	    
      
(comment defun apply-sigma (clause sigma new-root root1 root2)
  (cond ((null clause) nil)
	((unify-var-p clause root1 root2)
	 (if (eq (var-owner clause) new-root)
	   clause
	   (let ((v (aref sigma (var-num clause root1 root2) 0)))
	     (if (eq v '*none*) clause v))
	   ))
	((consp clause)
	 (cond ((and (eq (car clause) :sigma) (arrayp (cadr clause)) (caddr clause))
		(let* ((dim (caddr clause))
		       (new (make-array dim)))
		  (doloop :while (< i (car dim)) :iterate (i 0 (+ 1 i))
		    (setf (aref new i 0) (apply-sigma (aref (cadr clause) i 0) sigma new-root root1 root2))
		    (setf (aref new i 1) (aref (cadr clause) i 1))
		    )
		  (list :sigma new)))
	       (t (cons (apply-sigma (car clause) sigma new-root root1 root2)
			(apply-sigma (cdr clause) sigma new-root root1 root2)))))
	((token-description-p clause)
	 (make-token-description :instance-of (apply-sigma (tok-desc-instance-of clause) sigma new-root root1 root2)
				 :attrs (apply-sigma (tok-desc-attrs clause) sigma new-root root1 root2)))
	(t clause)
	)
  )

(defun apply-sigma (clause sigma new-root root1 root2)
  (cond ((null clause) nil)
	((unify-var-p clause root1 root2)
	 (if (eq (var-owner clause) new-root)
	   clause
	   (let ((v (aref sigma (var-num clause root1 root2) 0)))
	     (if (eq v '*none*) clause v))
	   ))
	((consp clause)
	 (cond ((and (eq (car clause) :sigma) (arrayp (cadr clause)) (caddr clause))
		(let* ((dim (caddr clause))
		       (new (make-array dim)))
		  (doloop :while (< i (car dim)) :iterate (i 0 (+ 1 i))
		    (setf (aref new i 0) (apply-sigma (aref (cadr clause) i 0) sigma new-root root1 root2))
		    (setf (aref new i 1) (aref (cadr clause) i 1))
		    )
		  (list :sigma new)))
	       (t (doloop (item clause)
		    :collect (apply-sigma item sigma new-root root1 root2)
		    :atom-tail tail (apply-sigma tail sigma new-root root1 root2)))))
	((token-description-p clause)
	 (make-token-description :instance-of (apply-sigma (tok-desc-instance-of clause) sigma new-root root1 root2)
				 :attrs (apply-sigma (tok-desc-attrs clause) sigma new-root root1 root2)))
	(t clause)
	)
  )



(defun subst-sigma (sigma owner root1 root2)
					; we cheat and use the type field of sigma to indicate copying
  (let ((n 0)
	(k (+ (if root1 (clause-n-vars root1) 0) (if root2 (clause-n-vars root2) 0)))
	vars)
    (do ((i 0 (+ 1 i)))
	((>= i k))
      (if (eq (aref sigma i 0) '*none*)
	(let ((var (make-clause-var :number n
				    :name n
				    :type (aref sigma i 1)
				    :owner owner)))
	  (setf (aref sigma i 0) var)
	  (setf (aref sigma i 1) t)
	  (setf n (+ 1 n)))
	(setf (aref sigma i 1) nil)
	)
      )
    (setq vars (make-array n))
    (do ((i 0 (+ 1 i))
	 (vnum 0))
	((>= i k))
      (when (aref sigma i 1)
	(let ((exp (aref sigma i 0)))
	  (when (and (clause-var-p exp) (eq owner (var-owner exp)))
	    (setf (svref vars vnum) exp)
	    (setq vnum (+ 1 vnum))))
	)      
      )
    (do ((i 0 (+ 1 i)))
	((>= i k))
      (when (not (aref sigma i 1))
	(setf (aref sigma i 1) t)
	(setf (aref sigma i 0) (apply-sigma-copy-sub (aref sigma i 0) sigma owner root1 root2))
	)      
      )
    vars
    )
  )


(defun collect-resolvent (clause1 pos1 clause2 pos2 sigma owner sigma-clause1 sigma-clause2)
  (let ((result (tconc)))
    (do ((tail (if clause2 (clause-expression clause2) nil) (cdr tail)))
	((null tail))
      (if (not (eq tail pos2))
	  (tconc-unique result (apply-sigma (car tail) sigma owner sigma-clause1 sigma-clause2)))
      )
    (do ((tail (clause-expression clause1) (cdr tail)))
	((null tail))
      (if (not (eq tail pos1))
	  (tconc-unique result (apply-sigma (car tail) sigma owner sigma-clause1 sigma-clause2)))
      )
    (car result)
    )
  )



(defun build-resolvent (clause1 pos1 clause2 pos2 sigma time depth-p)
  (let* ((new-clause (make-clause))
	 (vars (subst-sigma sigma new-clause clause1 clause2))
	 (resolvent (if (or depth-p (null clause2))
		      (collect-resolvent clause1 pos1 clause2 pos2 sigma new-clause clause1 clause2)
		      (collect-resolvent clause2 pos2 clause1 pos1 sigma new-clause clause1 clause2))))
    (setf (clause-expression new-clause) resolvent)
    (setf (clause-n-vars new-clause) (array-dimension vars 0))
    (setf (clause-vars new-clause) vars)
    (setf (clause-belief-time new-clause) (if clause2
					    (new-belief-context
					     :includes ((clause-time-context clause1) (clause-time-context clause2))
					     :belief (get-context-belief time))
					    (new-belief-context
					     :includes ((clause-time-context clause1))
					     :belief (get-context-belief time))))
    new-clause)
  )
    
      
	       
(defun resolve-step (clause1 clause2 gen belief state depth-p)
  (let ((sigma (if (belief-satisfies belief (clause-belief-time clause2))
		 (match-clauses clause1 clause2 gen belief state))))
    (if sigma
      (let ((new-clause
	     (build-resolvent
	      clause1 (clause-expression clause1) clause2 (sg-match-position gen) sigma belief depth-p)))
	(values new-clause sigma)
	)
      (values nil nil)
      )
    )
  )
  
(defmacro sign-print-index nil
  `(dbgn-print 'indexing atom sign pred key 'index (if sign (theory-neg-indices theory) (theory-pos-indices theory))
	       ))


(defun doindex-attr (atom sign pred theory clause literalp)
					; (attr object label class-label value prop time)
  (let* ((trial-key (nth 3 (if (eq (car atom) 'not) (cadr atom) atom)))
	 (key (if (clause-var-p trial-key) nil trial-key)))
    (if sign
      (add-to-index (theory-neg-indices theory) (pred :all) (cons clause literalp))
      (add-to-index (theory-pos-indices theory) (pred :all) (cons clause literalp))
      )
    (if key
      (if sign
	(add-to-index (theory-neg-indices theory) (pred key) (cons clause literalp))
	(add-to-index (theory-pos-indices theory) (pred key) (cons clause literalp)))
      (if sign
	(add-to-index (theory-neg-indices theory) (pred :var) (cons clause literalp))
	(add-to-index (theory-pos-indices theory) (pred :var) (cons clause literalp))))
    ;;; also index on arg1
    (let* ((arg1 (nth 1 (if (eq (car atom) 'not) (cadr atom) atom)))
	   (index (if (clause-var-p arg1) :_attr_arg_1_var :_attr_arg_1_token))
	   (val (if (clause-var-p arg1) (var-type arg1) arg1)))
      (if sign
	(add-to-index (theory-neg-indices theory) (pred index val) (cons clause literalp))
	(add-to-index (theory-pos-indices theory) (pred index val) (cons clause literalp))
	)
      )
    )
  )


(defun doindex-instance-of (atom sign pred theory clause literalp)
					; (instance-of instance class time)
  (let* ((trial-key (nth 2 (if (eq (car atom) 'not) (cadr atom) atom))) ;; index on class
	 (key (if (clause-var-p trial-key) nil  trial-key)))
    (if sign
      (add-to-index (theory-neg-indices theory) (pred :all) (cons clause literalp))
      (add-to-index (theory-pos-indices theory) (pred :all) (cons clause literalp)))
    (if key
      (if sign
	(add-to-index (theory-neg-indices theory) (pred key) (cons clause literalp))
	(add-to-index (theory-pos-indices theory) (pred key) (cons clause literalp)))
      (if sign
	(add-to-index (theory-neg-indices theory) (pred :var) (cons clause literalp))
	(add-to-index (theory-pos-indices theory) (pred :var) (cons clause literalp))))
    ;;; also index on arg1
    (let* ((arg1 (nth 1 (if (eq (car atom) 'not) (cadr atom) atom)))
	   (index (if (clause-var-p arg1) :_inst_arg_1_var :_inst_arg_1_token))
	   (val (if (clause-var-p arg1) (var-type arg1) arg1)))
      (if sign
	(add-to-index (theory-neg-indices theory) (pred index val) (cons clause literalp))
	(add-to-index (theory-pos-indices theory) (pred index val) (cons clause literalp))
	)
      )
    )
  )

(defun doindex (expression clause theory &optional horn) ; ignore horn for now
  (declare (ignore horn))
  (do ((literalp expression (cdr literalp)))
      ((null literalp))
    (let* ((atom (car literalp))
	   (sign (if (eq (car atom) 'not) 'not nil))
	   (pred (if (eq (car atom) 'not) (caadr atom) (car atom))))
      (cond ((eq pred 'attr) (doindex-attr atom sign pred theory clause literalp))
	    ((eq pred 'instance-of) (doindex-instance-of atom sign pred theory clause literalp))
	    (sign (add-to-index (theory-neg-indices theory) (pred) (cons clause literalp)))
	    (t (add-to-index (theory-pos-indices theory) (pred) (cons clause literalp))))
      )
    )
  )
    




(defun dounindex (expression clause theory)
  (do ((literalp expression (cdr literalp)))
      ((null literalp))
    (let* ((atom (car literalp))
	   (sign (if (eq (car atom) 'not) 'not nil))
	   (pred (if (eq (car atom) 'not) (caadr atom) (car atom)))
	   (bucket (if sign
		       (assoc pred (theory-neg-indices theory))
		     (assoc pred (theory-pos-indices theory)))))
      (when bucket
	(setf (cdr bucket) (remove clause (cdr bucket))))
      ))
  )
    






(defun stash (expression &optional (theory *theory*) (append nil) (belief (std-belief)) common-vars name props errors)
  (let ((num (theory-clause-number theory))
	nc)
    (if (null num)
	(setq num 0))
    (if (clause-p expression)
      (setq nc expression)
      (setq nc (build-clause expression theory (or name num) belief common-vars errors))
      )
    (setf (clause-props nc) props)
    (if append
	(setf (theory-clauses theory) (append (theory-clauses theory) (list nc)))
      (setf (theory-clauses theory) (cons nc (theory-clauses theory))))
    (doindex (clause-expression nc) nc theory (theory-horn theory))
    (when (not name)
      (setf (theory-clause-number theory) (+ 1 num)))
    nc
    )
  )


(defun stash-b (expression belief &optional (theory *theory*) (append nil) common-vars name props errors)
  (stash expression theory append belief common-vars name props errors)
  )


(defun stash-l (expression theory)
  (stash expression theory t)
  )




(defun active-stash (clause active-theory)
  (let ((new-clause (make-active-clause :clause clause)))
    (doindex (clause-expression clause) new-clause active-theory)
    (if (null (clause-number clause))
      (let ((num (theory-clause-number active-theory)))
	(if (null num)
	  (setq num 0))
	(setf (clause-number clause) num)
	(setf (theory-clause-number active-theory) (+ 1 num))
	))
    (setf (theory-clauses active-theory) (cons new-clause (theory-clauses active-theory)))
    new-clause
    )
  )


(defun active-stash-state (clause state)
  (let ((active-theory (istate-active-theory state))
	(new-clause (make-active-clause :clause clause :generator nil)))
    (if (not (istate-horn state))
      (doindex (clause-expression clause) new-clause active-theory))
    (if (null (clause-number clause))
      (let ((num (theory-clause-number active-theory)))
	(if (null num)
	  (setq num 0))
	(setf (clause-number clause) num)
	(setf (theory-clause-number active-theory) (+ 1 num))
	))
    (if (not (istate-breadth-p state))
      (setf (theory-clauses active-theory) (cons new-clause (theory-clauses active-theory)))
      (let ((tail (list new-clause)))
	(if (null (istate-theory-last state))
	  (progn
	    (setf (theory-clauses active-theory) (append (theory-clauses active-theory) tail))
	    (setf (istate-theory-last state) tail))
	  (progn
	    (setf (cdr (istate-theory-last state)) tail)
	    (setf (istate-theory-last state) tail))
	  ))
      )		 
    new-clause)
  )



; iterate over clauses for a fixed atom

(defun find-inference (active-clause gen state belief-time theory-list)
  (let ((clause (ac-clause active-clause)))
    (do* ((clauses (gen-has-clause gen))
	  (done nil))
	((or done (null clauses)) done)
      (let ((match-clause (if (consp (car clauses))
			    (progn (if (eq (sg-match-position gen) :init)
				     (setf (sg-match-position gen) (cdar clauses))
				     (setf (sg-match-position gen) nil)) ; indexed clauses appear in theory once for each match
				   (caar clauses))
			    (car clauses))))
	(multiple-value-bind (resolvent sigma)
	    (if (active-clause-p match-clause)
	      (when (not (clause-nogood match-clause))
		(resolve-step clause (ac-clause match-clause) gen belief-time state (not (istate-breadth-p state))))
	      (resolve-step clause match-clause gen belief-time state (not (istate-breadth-p state)))
	      )
	  (if resolvent
	    (let ((nc (active-stash-state resolvent state)))
	      (setf (ac-collect nc) (apply-sigma (ac-collect active-clause)
						 sigma (ac-clause nc)
						 clause match-clause))
	      (setf (ac-justifications nc) (list active-clause match-clause))
	      (when (istate-best-p state)
		(setf (clause-get-prop resolvent :value) (clause-get-prop clause :value))
		)
	      (setq done nc)
	      )
	    (setq clauses (gen-next-clause clause gen theory-list))
	    )
	  ))
      )
    )
  )

; if a lisp function on atom, call the lisp function with args (atom sigma pos2) where pos2 is
; generator state information, initialy :init. the lisp function returns 2 values: t or nil for
; success or failure, and a continuation which is stored in pos2 of the clause
; first we have to create the resolvent from sigma, then
; we create a new clause as in find inference above, and modify the current state of the current clause

(defun find-lisp-inference (active-clause state gen belief theory)
  (let ((atom (car (clause-expression (ac-clause active-clause)))))
    (if (sg-lisp-state gen)
	(let* ((pred (if (eq (car atom) 'not) (caadr atom) (car atom)))
	       (fn (if (symbolp pred) (get pred 'lisp-fn))))
	  (if fn
	      (let ((sigma (istate-get-sigma state (clause-n-vars (ac-clause active-clause)) 2)))
		(reset-sigma sigma (ac-clause active-clause))
		(multiple-value-bind (success continuation justifications)
		    (if (consp fn)
		      (funcall (car fn)
			     atom sigma (sg-lisp-state gen)
			     belief active-clause theory state)
		      (funcall fn
			     atom sigma (sg-lisp-state gen)
			     belief active-clause theory))
		  (cond ((eq success t)
			 (let ((nc (active-stash-state (build-resolvent (ac-clause active-clause)
								  (clause-expression (ac-clause active-clause))
								  nil
								  nil
								  sigma
								  belief
								  t)
						 state)))
			   (setf (ac-collect nc) (apply-sigma (ac-collect active-clause) sigma
							      (ac-clause nc)
							      (ac-clause active-clause) nil))
			   (setf (ac-justifications nc) (cons active-clause justifications))
			   (when (and (istate-best-p state) (null (clause-get-prop (ac-clause nc) :value)))
			     (setf (clause-get-prop (ac-clause nc) :value) (clause-get-prop (ac-clause active-clause) :value))
			     )
			   (setf (sg-lisp-state gen) continuation)
			   nc
			   ))
			((eq success :fail)
			 (set-clause-nogood active-clause)
			 nil)
			((eq success :theory) nil)
			(t
			 nil))))
	    ))
      ))
  )





(defun inference-best (state clauses)
  (let ((best nil)
	(best-value -1))
    (doloop (clause clauses)
      :vars ((val 0 (or (get-clause-prop (ac-clause clause) :value) 0)))
      :when (and (not (eq (ac-generator clause) :done))
		 (not (eq (ac-generator clause) :poison))
		 (or (null best) (< val best-value)))		
      (setf best clause)
      (setf best-value val)
      )
    (setf (istate-current-clause state) best)
    best)
  )


; search for an active clause  that succeeds

(defvar *fail-trace* nil)
(defun fail-trace nil
  (setf *fail-trace* (not *fail-trace*))
  *fail-trace*)



(defun backward-inference-step (state)
  (declare (special *fail-trace*))
  (let* ((active-theory (istate-active-theory state))
	 (theory-list (istate-theory-list state))
	 (clauses (theory-clauses active-theory))
	 )
    (doloop :for (active-clause nil (if (istate-best-p state)
				      (inference-best state (theory-clauses active-theory))
				      (car tail-var)))
     :iter (tail-var (when (not (istate-best-p state)) clauses) (when tail-var (cdr tail-var)))
     :while active-clause
     :vars ((done nil)
	    (gen nil (ac-generator active-clause))
	    (new nil nil))
     :for belief := (ac-belief active-clause)
      (when (null gen)
	(setq new t)
	(setq gen (get-generator (ac-clause active-clause) theory-list (if (not (istate-horn state)) active-theory)))
	(setf (ac-generator active-clause) gen)
	)
      (when (and gen (not (eq gen :poison)) (not (eq gen :done)))
	(when (sg-meta-state gen)
	  (setq done (find-meta-inference active-clause state gen))
	  (setq gen (ac-generator active-clause)))
	(when (and (not done) gen (not (eq gen :poison)) (not (eq gen :done)) (sg-lisp-state gen))
	  (setq done (find-lisp-inference active-clause state gen belief (car theory-list)))
	  (setq gen (ac-generator active-clause)))
	(when (and (not done) gen (not (eq gen :poison)) (not (eq gen :done)))
	  (if (not new)
	    (gen-next-clause (ac-clause active-clause) gen theory-list))
	  (when (gen-active gen)
	    (setq done (find-inference active-clause gen state belief theory-list))))
	)
      (when (not done)
	(if new
	  (setf (ac-generator active-clause) :poison)
	  (setf (ac-generator active-clause) :done)
	  )
	(when *fail-trace*
	  (format *trace-output* "~%FAIL~%~a~%" (normal-clause active-clause))
	  ))
     :when done :return (progn 
			  (do-trace (when done (format t "~%~%resolvent: ~a~%~%" (normal-clause done))))
			  done)      
      )
    )
  )
 



(defun goal-build-collect (active-clause variables)
  (let ((collect (tconc))
	(vars (clause-vars (ac-clause active-clause)))
	(n (clause-n-vars (ac-clause active-clause))))
    (cond ((eq variables :all)
	   (do ((i 0 (+ 1 i)))
	       ((>= i n))
	     (tconc collect (svref vars i))))
	  ((listp variables)
	   (do ((tail variables (cdr tail)))
	       ((not tail))
	     (if (input-varp (car tail))
		 (let ((var (find (extract-var-name (car tail)) vars :test #'(lambda (str var)
									 (equal str (var-name var))))))
		   (if var (tconc collect var) (tconc collect (car tail)))
		   ))
	     ))
	  )
    (setf (ac-collect active-clause) (car collect))
    )
  )


  

(defun backward-inference-loop (state)
  (do ((resolvent (backward-inference-step state) (backward-inference-step state)))
      ((or (null resolvent) (null (clause-expression (ac-clause resolvent))))
       resolvent)
    )
  )


(defun backward-inference (query theories &optional (variables :all)
						  (horn t)
						  (active-theory nil)
						  (belief (std-belief)))
  (let* ((root-theory (if (listp theories) (car theories) theories))
	 (active-theory (or active-theory (make-theory :name (gensym "backward-inference") :kb root-theory)))
	 (state (new-state active-theory theories belief horn)))
    (add-wff state query belief 0 variables)
    (values (backward-inference-loop state) state)
    )
  )


(defun backward-theory-inference (active-theory theories belief &optional (horn nil))  ;;; returns clause with empty resolvent if contradiction
  (let* ((root-theory (if (listp theories) (car theories) theories))
	 (state (new-state active-theory theories belief horn)))
    (backward-inference-loop state)
    )
  )


(defun horn-inference (clauses theories belief direction no-meta)
 								   ;;; returns clause with empty resolvent if contradiction; clauses are cnf
  (let* ((horn t)						   ;;; no-meta controls whether the original clauses allow meta-reasoning
	 (root-theory (if (listp theories) (car theories) theories))
	 (active-theory (make-theory :name (gensym "backward-inference") :kb (theory-kb root-theory)))
	 (state (new-state active-theory theories belief horn))
	 result)
    (istate-set-search-style state direction)
    (doloop (clause clauses) :vars ((n 0 (+ 1 n))
				    (common-vars (tconc)))
      (add-cnf-clause state clause belief n nil common-vars no-meta)
     )
    (do-trace
     (format t "~%~%Backward sub proof~%")
     (print-theory active-theory)
     )
    (setq result (backward-inference-loop state))
    (do-trace (format t "~%~%END SUBPROOF result: ~A~%" (not (null result))))
    (values result state)
    )
  )







(defun backward-continue (state)
  (values (backward-inference-loop state) state)
  )


(defmacro query1 (query &optional (belief '(std-belief)) (theory '*theory*))
  `(mlet ((theory ,theory))
     (setq ac (make-theory :kb (theory-kb theory)))
     (backward-all ,query theory :all t ac ,belief)
     t)
  )


(defmacro query (query &optional (vars :all) (belief '(std-belief)) (theory '*theory*))
  `(mlet ((theory ,theory))
     (setq ac (make-theory :kb (theory-kb theory)))
     (backward-inference ,query theory ,vars t ac ,belief))
  )


(defun backward-collect (query theory &optional (variables :all)
						(horn t)
						(active-theory nil)
						(belief (std-belief)))
  (multiple-value-bind (result state) (backward-inference query theory variables horn active-theory belief)
    (doloop
     :while result
     :when (null (clause-expression (ac-clause result)))
     :collect (ac-collect result)	
      (setq result (backward-inference-loop state))
      )
    ))



(defmacro query-collect (query &key (vars :all) (belief '(std-belief)) (theory '*theory*))
  `(mlet ((theory ,theory))
     (setq ac (make-theory :kb (theory-kb theory)))
     (backward-collect ,query theory ,vars t ac ,belief)
     )
  )


(defun backward-all (query theory &optional (variables :all)
					    (horn t)
					    (active-theory nil)
					    (belief (std-belief)))
  (multiple-value-bind (result state) (backward-inference query theory variables horn active-theory belief)
    (do nil
	((null result) state)
      (if (null (clause-expression (ac-clause result)))
	(progn (format t "~%--> ~a" (normal-clause (ac-collect result)))
	       (explain-contradiction result))
	(princ ".")
	)
      (setq result (backward-inference-loop state))
      )
    )
  )



(defun explain-contradiction (clause &optional (suppress t) (stream t))
  (if suppress
    (cond ((clause-p clause) (when (not (numberp (clause-number clause))) (format stream "~&~a~%" (clause-number clause))))
	  ((kb-attr-p clause) (kb-print-attr clause stream))
	  ((listp clause) (doloop (just clause) (explain-contradiction just suppress stream)))
	  ((not (active-clause-p clause)) (format stream "~&~a~%" (normal-clause clause)))
	  (t nil))  
    (format stream "~&~a~%" (normal-clause clause))
    )
  (when (active-clause-p clause)
    (doloop (just (ac-justifications clause))
      (explain-contradiction just suppress stream)
      ))
  )


(defun collect-explanation (clause &optional (suppress t))
  (append
   (if suppress
     (cond ((clause-p clause) (when (not (numberp (clause-number clause))) (list (clause-number clause))))
	   ((kb-attr-p clause) (list clause))
	   ((not (active-clause-p clause)) (list clause))
	   (t nil))  
     (list clause)
     )
   (when (active-clause-p clause)
     (doloop (just (ac-justifications clause))
      :splice (collect-explanation just suppress)
       ))
   )
  )


(defun known (clause &optional (belief (std-belief)) (theory *theory*))
  (let ((goal-clause (build-clause clause theory))
	(theory-list (traverse-theory theory)))
    (do ((theoryp theory-list (cdr theoryp))
	 (result nil))
	((or result (null theoryp)) result)
      (do ((clausep (theory-clauses (car theoryp)) (cdr clausep)))
	  ((or result (null clausep)))
	(let ((sigma (match-all goal-clause (car clausep) belief)))
	  (if sigma (setq result (car clausep)))
	  )
	)
      )
    )
  )


(defun lookup (clause &optional (belief (std-belief)) (theory *theory*))
  (let ((goal-clause (build-clause clause theory))
	(theory-list (traverse-theory theory)))
    (do ((theoryp theory-list (cdr theoryp))
	 (result nil))
	((null theoryp) result)
      (do ((clausep (theory-clauses (car theoryp)) (cdr clausep)))
	  ((null clausep))
	(let ((sigma (match-all goal-clause (car clausep) belief nil)))
	  (if sigma (push (car clausep) result))
	  )
	)
      )
    )
  )


(defun atom-search (atom expr)
  (cond ((null expr) nil)
	((eq atom expr) t)
	((consp expr) (or (atom-search atom (car expr)) (atom-search atom (cdr expr))))
	(t nil)
	)
  )


(defun find-atom (atom &optional (theory *theory*))
  (let ((theory-list (traverse-theory theory)))
    (do ((theoryp theory-list (cdr theoryp))
	 (result nil))
	((null theoryp) result)
      (do ((clausep (theory-clauses (car theoryp)) (cdr clausep)))
	  ((null clausep))
	(let ((sigma (atom-search atom (clause-expression (car clausep)))))
	  (if sigma (push (car clausep) result))
	  )
	)
      )
    )
  )


(defun print-active-clause (active-clause brief)
  (description-hash-reset)
  (if brief
    (let ((clause (ac-clause active-clause)))
      (format t "~&[~a]~a~a~a"
	      (normal-clause (clause-number clause))
	      (if (clause-get-prop clause :value)
		(format nil "= ~a =" (clause-get-prop clause :value))
		"")
	      (normal-clause (car (clause-expression clause)))
	      (case (ac-generator active-clause)
		(:poison "_poison_")
		((nil) "_new_")
		(:done "_done_")
		(t "_open_"))
	      ))
    (progn
      (format t "~&~a~a" (normal-clause (ac-clause active-clause) 10 3 3)
	      (case (ac-generator active-clause)
		(:poison "_poison_")
		((nil) "_new_")
		(:done "_done_")
		(t "_open_")))
      (format t "~%COLLECT:(~a)"	  
	      (normal-clause (ac-collect active-clause)))))
  (format t "~%JUST:  ~A~%"
	  (mapcar #'normal-clause-number (ac-justifications active-clause)))	  
  (force-output)
  )




(defun print-theory (theory &key (reverse nil) (recursive nil) (brief nil) (clauses nil))
  (doloop (clause (if reverse (reverse (theory-clauses theory)) (theory-clauses theory)))
    (if (active-clause-p clause)
      (if (or (null clauses) (member (clause-number (ac-clause clause)) clauses :test #'equall))
	(print-active-clause clause brief))
      (if (or (null clauses) (member (clause-number clause) clauses :test #'equal))
	(format t "~&~a" (normal-clause clause))))
    )
  (terpri)
  (if recursive
      (mapc #'(lambda (theory) (format t "~&-----> ~a~%"
				       (if (theory-name theory) (string (theory-name theory)) " "))
		      (print-theory theory t))
	    (theory-includes theory))
    )
  (terpri)
  )




(defmacro sigma-varp (x) `(clause-var-p ,x))

(defmacro sigma-lookup (var sigma &optional (field 0)) `(aref ,sigma (var-number ,var) ,field))

(defmacro sigma-type (expr sigma)
  `(mlet ((expr ,expr))
     (cond ((sigma-varp expr) (aref ,sigma (var-number expr) 1))
	   ((numberp expr) *number-type*)
	   ((consp expr)
	    (if (and (symbolp (car expr)) (get (car expr) 'fn-type))
		(get (car expr) 'fn-type)
	      *root-type*))
	   (t expr)
	   )
     )
  )

(defmacro sigma-set (var val sigma &optional type)
  `(mlet ((var (var-number ,var)) (val ,val) (sigma ,sigma))
     (setf (aref sigma var 0) val)
     ,@(if type `((setf (aref sigma var 1) ,type)))
     )
  )


(defun no-vars (expr)
  (cond ((null expr) t)
	((clause-var-p expr) nil)
	((consp expr) (and (no-vars (car expr)) (no-vars (cdr expr))))
	(t t))
  )


(defmacro build-arith-ops (&rest ops)
  `(progn
     ,@(doloop (opp ops)
	:when (getd (car opp))
	:collect
	 (dlet* (((op inv inv2) opp))
	   `(setf (get ',op 'lisp-fn)
	      #'(lambda (atom sigma state belief clause theory)
		  (declare (ignore state belief clause theory))
		  (if (eq (car atom) 'not)
		    (dlet* (((o1 o2 r) (cdr (cadr atom))))
		      (cond ((and (numberp o1) (numberp o2))
			     (let ((val (,op o1 o2)))
			       (if (numberp r)
				 (if (= val r)
				   (values t nil)
				   (values nil nil))
				 (if (sigma-varp r)
				   (progn
				     (sigma-set r val sigma)
				     (values t nil))
				   (values nil nil))
				 )
			       ))
			    ((and (numberp o1) (numberp r) (sigma-varp o2))
			     (sigma-set o2 ,(if inv2 `(,inv2 o1 r) `(,inv r o1)) sigma)
			     (values t nil))
			    ((and (numberp o2) (numberp r) (sigma-varp o1))
			     (sigma-set o1 (,inv r o2) sigma)
			     (values t nil))
			    (t (values nil nil)))
		      )
		    (if (dlet* (((o1 o2 r) (cdr atom)))
			  (and (numberp o1) (numberp o2) (numberp r) (not (= (,op o1 o2) r)))
			  )
		      (values t nil (list 'arith))
		      (values nil nil)
		      )
		    )
		  )
	      )
	   )
	 )
     )
  )

(build-arith-ops (+ - nil) (- + -) (* / nil) (/ * /))



(defmacro build-arith-rels (&rest ops)
  `(progn
     ,@(doloop (op ops)
	:when (getd op)
	:collect
	 `(setf (get ',op 'lisp-fn)
	    #'(lambda (atom sigma state belief clause theory)
		(declare (ignore state belief clause theory))
		(if (eq (car atom) 'not)
		  (dlet* (((o1 o2) (cdr (cadr atom))))
		    (cond ((and (numberp o1) (numberp o2))
			   (if (,op o1 o2)
			     (values t nil (list 'relop))
			     (values nil nil nil)
			     ))
			  (t (values nil nil)))
		    )
		  (if (dlet* (((o1 o2) (cdr atom)))
			(and (numberp o1) (numberp o2) (not (,op o1 o2)))
			)
		    (values t nil (list 'relop))
		    (values nil nil)
		    )
		  )
		)
	    )
	 )
     )
  )
  

(build-arith-rels > < >= <=)




(defun resolve-= (atom sigma state belief clause theory)
  (if (eq (car atom) 'not)
    (let* ((args (cdadr atom))
	   (match (unify-work (car args) (cadr args) sigma (ac-clause clause) (ac-clause clause) belief)))
      (if match (values t nil) (values nil nil))
      )
    (if (and (not (sigma-varp (cadr atom))) (not (sigma-varp (caddr atom))))
      (if (not (eq (cadr atom) (caddr atom)))
	(values t nil)
	(values nil nil))
      (values nil nil))
    )
  )


(setf (get '= 'lisp-fn) #'resolve-=)
(setf (get 'is 'lisp-fn) #'resolve-=)




(defresolve fail (atom sigma state belief clause theory)
  (if (eq (car atom) 'not)
    (dlet* (((expr belief-expr theory-expr direction no-meta) (cdadr atom)))
      (let ((use-theory (and (not (clause-var-p theory-expr)) (theory-find-theory theory theory-expr))))
	(if (and use-theory (or (null belief-expr) (not (clause-var-p belief-expr))))
	  (let* ((use-belief (or belief-expr belief))
		 (proved (horn-inference (if (listp expr) expr (list expr)) use-theory use-belief direction no-meta)))
	    (if proved
	      (values nil nil nil)
	      (values t nil (list 'fail))
	      ))
	  (values nil nil nil)
	  )))
    (values nil nil nil)
    )
  )
	    

(defun create-function (symbol lisp-fn result-type &rest arg-types)
  (setf (get symbol :function-signature)
    (cons (list lisp-fn (lookup-type result-type))
	  (do ((argp arg-types (cdr argp))
	       (collect (tconc)))
	      ((null argp) (car collect))
	    (tconc collect (lookup-type (car argp)))
	    )
	  )
    )
  )


(defun normal-type (expr)
  (cond ((null expr) nil)
	((kb-token-p expr) (tok-name expr))
	((consp expr) (cons (normal-type (car expr)) (normal-type (cdr expr))))
	(t expr))
  )

(defun print-token (tok)
  (format t "~a~%~a~%~a~%" (tok-name tok) (normal-clause (tok-parents tok)) (normal-clause (tok-inst-of tok)))
  (do ((ap (lookup-index-cdr (tok-attrs tok) :all) (cdr ap)))
      ((null ap))
    (format t "~a: ~a <~a>~a~%"
	    (normal-clause (attr-label (car ap)))
	    (normal-type (attr-value (car ap)))
	    (normal-type (attr-token (car ap)))
	    (normal-clause (tok-inst-of (attr-token (car ap))))
	    )
    )
  )


(defun prop-check-time (belief history prop)
  (and (belief-satisfies belief (prop-belief prop)) (time-intersect history (prop-history prop) belief))
  )


(defmacro step-bucket (bucket history belief bucketv historyv)
  `(mlet ((sbucket ,bucket)
	 (sbelief ,belief)
	 (shistory ,history)
	 (sresult nil))
     (while (and sbucket (not sresult))
	    (setq sresult (prop-check-time sbelief shistory (car sbucket)))
	    (if (not sresult)
		(setq sbucket (cdr sbucket)))
	    )
     (when sresult
       (setf ,bucketv sbucket)
       (setf ,historyv sresult))
     ))



(defmacro bucket-history (bucket) `(car ,bucket))
(defmacro bucket-current (bucket) `(cadr ,bucket))
(defmacro bucket-rest (bucket) `(cddr ,bucket))


(defmacro build-isa-gen1 (classes)
  `(mlet ((classes ,classes)
	(result nil)
	bucket)
    (while classes
      (step-bucket classes history belief bucket history)
      (if bucket
	(progn (setq classes (tok-children (prop-src (car bucket))))
	       (push (cons history bucket) result))
	(setq classes (cdr classes)))
      )
    result)
  )

(defun build-isa-gen (class history belief)
  (build-isa-gen1 (if (prop-object-p class) (list class) (or (tok-children class) (tok-inst-of class))))
  )


(defun step-isa-gen (gen history belief)
  (cond ((null gen) nil)
	((null (bucket-rest (car gen))) (step-isa-gen (cdr gen) history belief))
	(t (use-struct (car gen) (bucket- rest)
	     (let ((classes rest)
		   (result (cdr gen))
		   bucket)
	       (when (cdr gen)
		 (setq history (bucket-history (cadr gen))))
	       (while classes
		 (step-bucket classes history belief bucket history)
		 (if bucket
		   (progn (setq classes (tok-children (prop-src (car bucket))))
			  (push (cons history bucket) result))
		   (setq classes (cdr classes)))
		 )
	       result)))
	)
  )

(defun step-inst-gen (gen history belief)
  (cond ((null gen) nil)
	(t
	 (let ((new (cdr gen))
	       (instances (bucket-rest (car gen)))
	       (result nil)
	       h)
	   (while (and (null result) new)
	     (step-bucket instances (bucket-history (car new)) belief result h)
	     (when (null result)
	       (setq new (step-isa-gen new history belief))
	       (when new
		 (setq instances (tok-instances (prop-src (bucket-current (car new)))))))
	     )
	   (if (and result new)
	       (cons (cons h result) new))
	   )))
  )


(defun build-inst-gen (class history belief)
  (let ((gen (build-isa-gen1 (tok-inst-of class))))
    (if gen
	(step-inst-gen (cons (list* nil nil (tok-instances (prop-src (bucket-current (car gen)))))
			     gen) history belief)
      )
    )
  )




(defun collect-parents (object &optional belief)
  (let* ((parents (lconc nil (tok-inst-of object)))
	 (new (car parents)))
    (while new
	   (let ((supers (tok-parents (prop-dest (car new)))))
	     (while supers
		    (if (not (member (car supers) (car parents)))
			(tconc parents (car supers)))
		    (setf supers (cdr supers))
		    )
	     )
	   (setq new (cdr new))
	   )
    (car parents)
    )
  )



(defun collect-constrained-parents (object type belief)  
  (let* ((parents (lconc nil (doloop (prop (tok-inst-of object))
			      :when (and (belief-satisfies belief (prop-belief prop))
					 (weak-type-instance (prop-dest prop) type belief))
			      :collect prop)))
	 (new (car parents)))
    (while new
      (doloop (prop (tok-parents (prop-dest (car new))))
       :when (and (belief-satisfies belief (prop-belief prop))
		  (weak-type-instance (prop-dest prop) type belief))
	(tconc parents prop))
      (setq new (cdr new))
      )
    (car parents)
    )
  )




(defun init-gen-parents (prop-list h b)
  (let (new history)
    (step-bucket prop-list h b new history)
    (if new
	(cons history new))
    )
  )


(defun gen-parents (gen h b)
  (let (new history)
    (step-bucket (cddr gen) h b new history)
    (if new
	(cons history new))
    )
  )



(defun build-super-gen (class h b)
  (list (list h class) (cons h (tok-parents class)))
  )


(defun step-super-gen (gen h b)
  (when gen
    (let (new history)
      (step-bucket (cdadr gen) h b new history)
      (if new
	`(,(list history (prop-dest (car new)))
	  ,@(when (cdr new) (list (cons (caadr gen) (cdr new))))
	  ,@(when (tok-parents (prop-dest (car new))) (list (cons history (tok-parents (prop-dest (car new))))))
	  ,@(cddr gen))
	(step-super-gen (cdr gen) h b)
	)
      ))
  )



(def-cond-struct (resolve-isa-gen (:conc-name isa-gen-))
  inst-v isa-v hvn inst-gen parent-gen history type
  )


(def-cond-struct (resolve-inst-gen (:conc-name inst-gen-))
  inst-v hvn inst-gen history
  )


(def-cond-struct (resolve-parents-gen (:conc-name parent-gen-))
  inst-v  hvn parent-gen history
  )


(defun move-isa-inst (state belief)
  (let ((result nil))
    (use-struct state (isa-gen- inst-gen parent-gen type history)
		(while (and (null result) inst-gen)
		       (while (and (null parent-gen) inst-gen)
			      (setf inst-gen (step-inst-gen inst-gen history belief))
			      (when inst-gen
				(setf parent-gen (init-gen-parents (collect-parents (prop-src (bucket-current (car inst-gen))))
								history belief))
				))
		       (while (and (null result) parent-gen)
			      (setq result (prop-dest (bucket-current parent-gen)))
			      (when (not (type-instance result
							type (bucket-history (car inst-gen)) belief))
				(setq result nil)
				(setf parent-gen (gen-parents parent-gen history belief)))
			      )
		       ))
    result)
  )





					; the idea is that we might have functions that can be evaluated
					; if all the arguments are bound
					; if all variables are not bound, we can create generators for the
					; unbound variables at the relation level, so we collect there types
(defun get-term-value (term type sigma clause vars)
  (cond ((null clause) nil)
	((clause-var-p term) (tconc-unique vars term) :none)
	((not (listp term)) term)
	((not (symbolp (car term))) term)
	(t
	 (let ((signature (get (car term) :function-signature)))
	   (if signature
	       (do ((argp (cdr term) (cdr argp))
		    (sigp (cdr signature) (cdr sigp)))
		   ((or (null argp) (null sigp)))
		 )
	     ))))
  )






(defun cnf-compile-input-var (clause)
  (let* ((var-str (extract-var-name clause))
	 (new-var (make-cnf-var :name (if var-str (string-upcase var-str) "")
				:input clause)))
    new-var)
  )




(defun cnf-lookup-var (var bindings)
  (let* ((var-str (extract-var-name var))
	 (new-name (if var-str (string-upcase var-str) ""))
	 (val (assoc new-name bindings :test #'(lambda (x key) (string-equal x (cnf-var-name key))))))
    
    (if val
      (let ((old (cdr val)))
	(if (and
	     (not (string-equal (extract-type-name var) "proposition"))
	     (string-equal (extract-type-name (cnf-var-input old)) "proposition"))
	  (setf (cnf-var-input old) var))
	old)
      (let ((new (make-cnf-var :name new-name
			       :input var))
	    (tail (last bindings)))
	(setf (cdr tail) (list (cons new new)))
	new)
      ))
  )



(defun cnf-subst (expr bindings)
  (cond ((null expr) nil)
	((input-varp expr)
	 (cnf-lookup-var expr bindings))
	((consp expr) (cons (cnf-subst (car expr) bindings) (cnf-subst (cdr expr) bindings)))
	(t expr)
	)
  )



(defun cnf (expr &optional negp (bindings (list (cons (make-cnf-var :name "(") nil))))
  (cond ((not (listp expr)) (list (list (list 'bogus expr))))
	((eq (car expr) '=>) (cnf-=> (cdr expr) negp bindings))
	((eq (car expr) '<=) (cnf-<= (cdr expr) negp bindings))
	((eq (car expr) '<==) (cnf-<= (cdr expr) negp bindings))
	((eq (car expr) 'known) (cnf (cadr expr) negp bindings))
	((member (car expr) '(quote comma bquote)) (list (car expr) (cnf (cadr expr) negp bindings)))
	(negp
	 (cond ((eq (car expr) 'and) (cnf-or (cdr expr) negp bindings))
	       ((eq (car expr) 'or) (cnf-and (cdr expr) negp bindings))
	       ((eq (car expr) 'not) (cnf-not (cadr expr) negp bindings))
	       ((eq (car expr) 'exists) (cnf-forall (cdr expr) negp bindings))
	       ((eq (car expr) 'forall) (cnf-exists (cdr expr) negp bindings))
	       ((eq (car expr) 'fail) (list (list (list 'not (cnf-subst (list* 'fail (cnf (cadr expr) t bindings)
									       (cddr expr))
									bindings)))))
	       ((eq (car expr) 'provable) (list (list (list 'not (cnf-subst (list* 'provable (cnf (cadr expr) t bindings)
										   (cddr expr))
									    bindings)))))
	       ((eq (car expr) 'unprovable) (list (list (list 'not (cnf-subst (list* 'unprovable (cnf (cadr expr) t bindings)
										     (cddr expr))
									      bindings)))))
	       ((eq (car expr) 'compiled) (list (list (list 'not (list (car expr)
								       (cnf-subst (cadr expr) bindings)
								       (cnf (caddr expr) nil bindings)
								       (cnf-subst (cadddr expr) bindings))))))
	       (t (list (list (list 'not (cnf-subst expr bindings)))))))
	(t
	 (cond ((eq (car expr) 'and) (cnf-and (cdr expr) negp bindings))
	       ((eq (car expr) 'or) (cnf-or (cdr expr) negp bindings))
	       ((eq (car expr) 'not) (cnf-not (cadr expr) negp bindings))
	       ((eq (car expr) 'exists) (cnf-exists (cdr expr) negp bindings))
	       ((eq (car expr) 'forall) (cnf-forall (cdr expr) negp bindings))
	       ((eq (car expr) 'fail) (list (list (cnf-subst (list* 'fail (cnf (cadr expr) nil bindings)
								    (cddr expr)) bindings))))
	       ((eq (car expr) 'provable) (list (list (cnf-subst (list* 'provable (cnf (cadr expr) t bindings)
									(cddr expr))
								 bindings))))
	       ((eq (car expr) 'unprovable) (list (list (cnf-subst (list* 'unprovable (cnf (cadr expr) t bindings)
									  (cddr expr))
								   bindings))))
	       ((eq (car expr) 'compiled) (list (list (list (car expr)
							    (cnf-subst (cadr expr) bindings)
							    (cnf (caddr expr) nil bindings)
							    (cnf-subst (cadddr expr) bindings)))))
	       (t (list (list (cnf-subst expr bindings))))))
	)
  )


(defun cnf-not (expr negp bindings)
  (cnf expr (not negp) bindings)
  )



(defun cnf-and (expr negp bindings)
  (if (cdr expr)
      (let ((left (cnf (car expr) negp bindings))
	    (right (cnf-and (cdr expr) negp bindings)))
	(append left right)
	)
    (cnf (car expr) negp bindings)
    )
  )



(defun cnf-or (expr negp bindings)
  (if (cdr expr)
      (let ((left (cnf (car expr) negp bindings))
	    (right (cnf-or (cdr expr) negp bindings))
	    (result nil))
	(do ((left-ptr left (cdr left-ptr)))
	    ((null left-ptr))
	  (do ((right-ptr right (cdr right-ptr)))
	      ((null right-ptr))
	    (push (append (car left-ptr) (car right-ptr)) result)
	    )
	  )
	result
	)
    (cnf (car expr) negp bindings)
    )
  )










(defun cnf-build-free (bindings)
  (doloop (pair bindings)
    :when (eq (car pair) (cdr pair))
    :collect (car pair)
    )
  )




(defun cnf-exists (expr negp bindings)
  (do ((expr-ptr expr (cdr expr-ptr))
       (new-bindings bindings)
       (free (cnf-build-free bindings))
       (result nil))
      ((or result (null expr-ptr)) result)
    (cond ((input-varp (car expr-ptr))
	   (push (cons (cnf-compile-input-var (car expr-ptr)) (cons (gensym "sk") free)) new-bindings))
	  ((and (listp (car expr-ptr)) (input-varp (caar expr-ptr)))
	   (mapc #'(lambda (var) (push (cons (cnf-compile-input-var var) (cons (gensym "sk") free)) new-bindings))
		 (car expr-ptr)))
	  (t (if negp
	       (setq result (cnf-or expr-ptr negp new-bindings))
	       (setq result (cnf-and expr-ptr negp new-bindings)))))
    )
  )



(defun cnf-forall (expr negp bindings)
  (do ((expr-ptr expr (cdr expr-ptr))
       (new-free bindings)
       (result nil))
      ((or result (null expr-ptr)) result)
    (cond ((input-varp (car expr-ptr))
	   (let ((nvar (cnf-compile-input-var (car expr-ptr)))) (push (cons nvar nvar) new-free)))
	  ((and (listp (car expr-ptr)) (input-varp (caar expr-ptr)))
	   (doloop (var (car expr-ptr))
	    :vars ((nvar nil (cnf-compile-input-var var)))
	     (push (cons nvar nvar) new-free)))
	  (t (if negp
	       (setq result (cnf-or expr-ptr negp new-free))
	       (setq result (cnf-and expr-ptr negp new-free)))))
    )
  )


(defun cnf-=> (expr negp bindings)
  (do ((tail expr (cdr tail))
       (collect (tconc)))
      ((null tail) (cnf (cons 'or (car collect)) negp bindings))
    (if (null (cdr tail))
	(tconc collect (car tail))
      (tconc collect (list 'not (car tail)))
      )
    )
  )


(defun cnf-<= (expr negp bindings)
  (cnf (cons 'or (cons (car expr) (mapcar #'(lambda (x) (list 'not x)) (cdr expr)))) negp bindings)
  )
  
  
    
(defun cnf-print (x)
  (mapc #'print x)
  )



(defun cnf-stash (expr &optional (theory *theory*) (belief (std-belief)) (common-vars (tconc nil)))
  (mapcar #'(lambda (cnf-expr) (stash-b cnf-expr belief theory t common-vars))
	(cnf expr)
	)
  )




(defun includes (th1 th2)
  (let ((inc (theory-includes th1)))
    (setf (theory-includes th1) (append inc (list th2)))
    )
  )



(defun unincludes (th1 th2)
  (setf (theory-includes th1) (remove th2 th1))
  )



(defun empty (th)
  (setf (theory-clause-number th) 0)
  (setf (theory-clauses th) nil)
  )


(defun drop (clause &optional (theory *theory*))
  (if (numberp clause)
      (setf (theory-clauses theory)
	(remove-if #'(lambda (x) (= (clause-number x) clause)) (theory-clauses theory)))
    (setf (theory-clauses theory) (remove clause (theory-clauses theory)))
    )
  )



(defun kill (atom &optional (theory *theory*))
  (mapc #'(lambda (clause) (drop clause theory))
	(find-atom atom theory)
	)
  )



(defun read-kb (name &optional (theory *theory*))
  (if (open name :direction :probe)
      (with-open-file (input-stream name :direction :input)
	(do ((x nil))
	    ((eq x 'done))
	  (setq x (read input-stream nil 'done))
	  (if (listp x)
	      (cnf-stash x theory)
	    )))
    )
  )




(defun update-kb-inst (name1 name2 theory history belief)
  (let ((t1 (lookup-type name1 theory))
	(t2 (lookup-type name2 theory)))
    (if (not t2)
	(setq t2 (add-prove-isa name2 "proposition" theory history belief)))
    (if (not t1)
	(setq t1 (add-prove-is name1 t2 theory history belief))
      (assert-inst t1 t2 history belief nil theory))
    )
  )



(defun update-kb-isa (name1 name2 theory history belief)
  (let ((t1 (lookup-type name1 theory))
	(t2 (lookup-type name2 theory)))
    (if (not t2)
	(setq t2 (add-prove-isa name2 "proposition" theory history belief)))
    (if (not t1)
	(setq t1 (add-prove-isa name1 t2 theory history belief))
      (assert-isa t1 t2 history belief nil theory))
    )
  )





(defun kb-parse-tokens (x theory history belief)
  (cond ((null x) nil)
	((listp x) (do ((p (cdr x) (cdr p)))
		       ((null p))
		     (kb-parse-tokens (car p) theory history belief)))
	((symbolp x) (update-kb-inst x "proposition" theory history belief))
	(t nil))
  )




(defun attr-step-instance (state history belief)
  (use-struct state (attr-gen-  instance)
    (if (consp instance)
      (setf instance (step-inst-gen instance history belief))
      (setf instance nil))
    )
  )




(defun attr-step-parents (state history belief)
  (use-struct state (attr-gen- parents instance)
    (setf parents (step-super-gen parents history belief))
    (while (and instance (null parents))
      (attr-step-instance state history belief)
      (when instance
	(setf parents (build-super-gen (prop-src (bucket-current (car instance))) (bucket-history (car instance)) belief))
	)
      )
    )
  )


;; we aren't doing a meta-check here on prop-type

(defun attr-check-time-label (prop-class attr-prop history belief value value-type label class-label prop-type uprop)
  (when
      (and (or (not label) (eq label (attr-label attr-prop)))
	   (or (not class-label) (eq class-label (attr-token-class-label (prop-dest prop-class))))
	   (or (not value) (eql value (attr-value attr-prop)))
	   (or (not value-type) (type-instance (attr-value attr-prop) value-type history belief))
	   (or (not uprop) (eq (attr-token attr-prop) uprop))
	   (or (not prop-type) (type-instance (attr-token attr-prop) prop-type history belief))	   
	   (belief-satisfies belief (attr-belief attr-prop)))
    (time-intersect history (attr-history attr-prop) belief)
    )
  )
       


(defun attr-generate (state history belief value value-type label class-label propt uprop new theory)
  (let ((result nil))
    (use-struct state (attr-gen- prop-class prop parents instance)
      
      (while (and instance (not result))
	
	(if new
	  (setq new nil)
	  (when prop-class
	    (setf prop-class (gen-parents prop-class history belief))))
	(while (and prop (null prop-class))
	  (setf prop (cdr prop))
	  (when prop
	    (setf prop-class
	      (init-gen-parents (collect-constrained-parents (car prop) (theory-attribute-class theory) belief)
				(bucket-history (car parents)) belief))
	    ))
	(while (and instance (null prop-class))
	  (attr-step-parents state history belief)
	  
	  (when parents
	    (setf prop (if class-label
			 (lookup-index-cdr (tok-attrs (bucket-current (car parents))) class-label)
			 (lookup-index-cdr (tok-attrs (bucket-current (car parents))) :all)))
	    
	    (while (and prop (null prop-class))
	      (setf prop-class
		(init-gen-parents (collect-constrained-parents (car prop) (theory-attribute-class theory) belief)
				  (bucket-history (car parents)) belief))
	      (when (null prop-class)
		(setf prop (cdr prop)))
	      )
	    )
	  )
	(when prop
	  (let ((h (attr-check-time-label
		    (bucket-current prop-class) (car prop) history  belief value
		    value-type label class-label propt uprop)))
	    (if h
	      (setf result (list (prop-src (bucket-current (car instance))) (car prop) h (bucket-current prop-class)))))
	  ))
      )
    result
    )
  )




(defun attr-build-state (class-label object objectv label value history belief sigma propt theory)
  (let ((gen (make-attr-gen)))
    (use-struct gen (attr-gen- prop-class prop parents instance)
      (cond ((sigma-varp object)
	     (setf instance (build-inst-gen (aref sigma objectv 1) history belief)))
	    ((kb-token-p object)
	     (let (new h)
	       (step-bucket (tok-inst-of object) history belief new h)
	       (if new
		 (setf instance (list (cons h new))))
	       )
	     ))
      (while (and instance (null parents))
	(setf parents (build-super-gen (prop-src (bucket-current (car instance))) (bucket-history (car instance)) belief))
	(when (null parents)
	  (attr-step-instance gen history belief)
	  )
	)
      (when parents
	(setf prop (if class-label
		     (lookup-index-cdr (tok-attrs (bucket-current (car parents))) class-label)
		     (lookup-index-cdr (tok-attrs (bucket-current (car parents))) :all)))
	(when prop
	  
	  (setf prop-class (init-gen-parents (collect-constrained-parents (car prop) (theory-attribute-class theory) belief)
					     (bucket-history (car parents)) belief))
	  ))
      )
    gen)
  )
		  



;; (attr object label class-label value prop history)

(comment


(defun resolve-attr (atom sigma state belief clause theory)
  (cond
   ((or (null state) (not (eq (car atom) 'not))) nil)
   (t
    (dlet* ((new (eql state :init))
	    (args (cdadr atom))
	    ((object label class-label value prop history-var) args)
	    (objectv (when (clause-var-p object) (var-number object)))
	    (labelv (when (clause-var-p label) (var-number label)))
	    (class-labelv (when (clause-var-p class-label) (var-number class-label)))
	    (valuev (when (clause-var-p value) (var-number value)))
	    (propv (when (clause-var-p prop) (var-number prop)))
	    (historyv (when (clause-var-p history-var) (var-number history-var)))
	    (history (if historyv (aref sigma historyv 1) *all-time*)))
      (when (eq history *root-type*) (setq history *all-time*))
      (when new
	(setf state (attr-build-state (when (not class-labelv) class-label)
				      object objectv label value history belief sigma (if propv (aref sigma propv 1) (theory-root-object theory)) theory))
	)
      (let ((r (attr-generate state history belief
			      (when (not valuev) value)
			      (when valuev (aref sigma valuev 1))
			      (when (not labelv) label)
			      (when (not class-labelv) class-label)
			      (if propv (aref sigma propv 1) (theory-root-object theory))
			      (when (not propv) prop)
			      new
			      theory)))
	(when r
	  (if objectv (setf (aref sigma objectv 0) (car r)))
	  (if labelv (setf (aref sigma labelv 0) (attr-label (cadr r))))
	  (if class-labelv (setf (aref sigma class-labelv 0) (attr-token-class-label (prop-dest (cadddr r)))))
	  (if valuev (setf (aref sigma valuev 0) (attr-value (cadr r))))
	  (if propv (setf (aref sigma propv 0) (attr-token (cadr r))))
	  (if historyv (setf (aref sigma historyv 1) (caddr r)))
	  (values t state (list (cadr r)))
	  )
	)
      )
    ))
  )



  (setf (get 'attr 'lisp-fn) #'resolve-attr)

)



(defun make-member-type (name theory)
  (let* ((belief (std-belief))
	 (atom (intern (string-upcase (subseq (string name) 1))))
	 (new (add-prove-is (type-name name theory) (theory-root-object theory) theory (theory-all-time theory) belief)))
    (assert-isa new (theory-root-object theory) (theory-all-time theory)  belief nil theory)
    (setf (tok-prim new) #'(lambda (x) (eq atom x)))
    new)
  )