;*******************************************************************
; BAY.LISP
;

(defclass bay-disp ()
  ((id-string :accessor id-string :initarg :id-string)
   (fill-direction :accessor fill-direction :initarg :fill-direction)
   (internal-window :accessor internal-window)
   (max-icons :accessor max-icons)
   (icon-vector :accessor icon-vector))) 


(defun make-bay-disp (id-string win fill-direction)
  (if (< (height win) (+ *small-icon-height* 32))
      (cerror "To continue" "Window not high enough for a bay"))
  (let ((iww (- (width win) 10))
        (iwh (+ *small-icon-height* 10))
        (new-thing (make-instance 'bay-disp
                                  :id-string id-string
                                  :fill-direction fill-direction)))
    (setf (internal-window new-thing) 
		  (make-window (+ (x-coord win) 10) 
					   (+ (y-coord win) 17) 
					   iww 
					   iwh))
    (setf (max-icons new-thing) 
          (truncate (- iww 3) (+ *small-icon-width* 3)))
    (setf (icon-vector new-thing) 
		  (fill (make-array (max-icons new-thing)) nil))
	(bay-draw-outline new-thing)
    new-thing))

;  Tire and weapons bay (these have a single slot)

(defun make-single-bay-disp (id-string pos)
  (make-bay-disp id-string
                 (make-window (x-coord pos)
                              (y-coord pos)
                              (+ *small-icon-width* 17)
                              (+ *small-icon-height* 35))
                 'right))

;*************************************************************************

(defmethod dm-refresh ((self bay-disp)) 
  (bay-draw-outline self)
  (map 'list  
	   #'(lambda (icon-thing) (if (not (null icon-thing)) 
								  (dm-redraw self nil icon-thing)))
	   (icon-vector self)))

(defmethod dm-locate-obj ((self bay-disp) sym)
  (let ((vecpos (position sym (icon-vector self) 
                          :test #'(lambda (sym vec-elt)
                                    (and vec-elt
                                         (eq sym (unique-id vec-elt)))))))
    (cond (vecpos (let ((win (bay-vloc->icon-window vecpos)))
                    (make-window (+ (x-coord (internal-window self)) (x-coord win))
                                 (+ (y-coord (internal-window self)) (y-coord win))
                                 *small-icon-width*
                                 *small-icon-height*)))
          (t nil))))

(defmethod next-empty-space ((self bay-disp) x)
  (declare (ignore x))
  (let ((vloc (bay-first-empty-vector-pos self)))
    (if (null vloc) (cerror "To continue" "No room in this bay"))
    (let ((win (bay-vloc->icon-window vloc)))
      (make-window (+ (x-coord (internal-window self)) (x-coord win))
                   (+ (y-coord (internal-window self)) (y-coord win))
                   *small-icon-width*
                   *small-icon-height*))))

(defmethod display-window ((self bay-disp))
  (internal-window self))

(defmethod dm-add ((self bay-disp) x obj)
  (declare (ignore x))
  (let ((vloc (bay-first-empty-vector-pos self)))
    (if (null vloc)
      (cerror "To continue" "No room to add ~a in bay ~a" obj self))
    (disp-in-small-icon obj
                        (add-position-offset (internal-window self)
                                             (bay-vloc->icon-position vloc)))
    (setf (svref (icon-vector self) vloc) obj)))

(defmethod dm-del ((self bay-disp) x obj)
  (declare (ignore x))
  (let ((vloc (position obj (icon-vector self))))
    (if (null vloc)
      (cerror "To continue" "Can't delete ~a from bay ~a" obj self))
    (disp.with-clip-window (internal-window self)
                           (disp.clear-rectangle (bay-vloc->icon-window vloc)))
    (setf (svref (icon-vector self) vloc) '())))

(defmethod dm-redraw ((self bay-disp) x obj)
  (declare (ignore x))
  (let ((vloc (position obj (icon-vector self))))
    (if (null vloc)
      (cerror "To continue" "Can't redraw ~a in bay ~a" obj self))
    (disp.with-clip-window (internal-window self)
                           (disp.clear-rectangle (bay-vloc->icon-window vloc)))
    (disp-in-small-icon obj
                        (add-position-offset (internal-window self)
                                             (bay-vloc->icon-position vloc)))))

(defmethod dm-empty ((self bay-disp) x)  
  (declare (ignore x))
  (do ((l (coerce (icon-vector self) 'list) (cdr l)))
      ((null l) T)
    (if (not (null (car l))) (dm-del self (car l)))))

;************************************************************************

(defun bay-first-empty-vector-pos (self)
  (bay-firstpos (icon-vector self) (fill-direction self)))

(defun bay-firstpos (vec direction)
  (let ((lastpos (- (length vec) 1)))
    (bay-firstpos-aux vec
                      direction
                      (if (eq direction 'right) 
                        0 
                        lastpos)
                      lastpos)))

(defun bay-firstpos-aux (vec direction cntpos lastpos)
  (cond ((or (< cntpos 0) (> cntpos lastpos))
         nil)
        ((null (svref vec cntpos)) 
         cntpos)
        (t (bay-firstpos-aux vec
                             direction
                             (if (eq direction 'left)
                                 (- cntpos 1)
                                 (+ 1 cntpos))
                             lastpos))))

(defun bay-vloc->icon-position (vloc)
  (make-position (+ 3 (* vloc (+ *small-icon-width* 3))) 5))

(defun bay-vloc->icon-window (vloc)
  (let ((pos (bay-vloc->icon-position vloc)))
    (make-window (x-coord pos)
                 (y-coord pos)
                 (+ *small-icon-width* 1)
                 (+ *small-icon-height* 1))))

;*******************************************************************

(defun bay-draw-outline (bay-disp)
  (disp.draw-rectangle (internal-window bay-disp))
  (disp.with-font (disp.medium-font)
    (disp.with-clip-window (internal-window bay-disp)
	  (disp.text-at (id-string bay-disp) (make-position 5 -10)))))


