(in-package 'user)

(require 'pcl)
(use-package 'pcl)

(defmacro string-append (&rest strings)
  `(funcall #'concatenate 'string ,@strings))

; (Note that since this is a defvar, you can define this directory
;   ahead of time and this declaration will not override.)

;(defvar *simulator-dir* "~/ai/simulator")
(defvar *simulator-dir* "~/ai/truckworld")

(defconstant *real-simulator-dir* 
  (string-append *simulator-dir* "/simulator"))
(defconstant *displayer-dir* 
  (string-append *simulator-dir* "/displayer"))
(defconstant *graphics-dir* 
  (string-append *simulator-dir* "/displayer/graphics"))
(defconstant *command-processor-dir* 
  (string-append *simulator-dir* "/command-processor"))

;********************************************************************
;  The entry point is LOAD-SIMULATOR (bottom of file)

;********************************************************************

(defmacro define-simulator-system (displayer-type cp-type)
  `(mk:defsystem simulator
    :source-extension "lisp"
    :binary-pathname nil
    :binary-extension "fasl"
    :package user
    :components
    ((:module basics 
      :source-pathname *simulator-dir*
      :components ("lisp-req"))
     (:module simulator
      :depends-on ("basics")
      :source-pathname *real-simulator-dir*
      :components ("global-defs"  "lowlevel"  "messages"  "operations"
		   "object-methods"  "environment-defs"  "thingoid-defs"
		   "thingoids" "parcel-defs" "container-defs"
		   "containers" "liquids" "vessel-defs"
		   "vessels" "arm-defs" "arms"
		   "truck-defs" "trucks" "truck-support-defs"
		   "truck-support" "misc-obj-defs" "misc-objs"
		   "map-defs" "map-nodes" "map-links"
		   "enemy-unit-defs" "enemy-units"
		   "world-defs" "worlds" "manual"
		   "interface"))
     ,@(displayer-module displayer-type)
     ,@(command-processor-module cp-type))))

;********************************************************************

(defun displayer-module (displayer-type)
  (cond
    ((eq displayer-type 'graphic)
     '((:module graphics
		  :depends-on ("basics")
		  :source-pathname *graphics-dir*
		  :components ("syntax"  "primitives" "interface"))
       (:module displayer-support
		  :source-pathname *displayer-dir*
		  :depends-on ("basics")
		  :components ("constants" "arm-support" "loc-support" 
			       "heading-support" "map-methods"))
       (:module displayer
		  :depends-on ("basics" "graphics" "displayer-support"
			       "simulator")
		  :source-pathname *displayer-dir*
		  :components ("methods" "icon" "arm" "bay" 
			       "fuel" "heading" "speed" "status"
			       "truck" "map" "loc"
			       "graphic-displayer-methods"
			       "graphic-displayer"))))
    ((eq displayer-type 'simple)
     '((:module displayer
		  :depends-on ("basics")
		  :source-pathname *displayer-dir*
		  :components ("displayer-methods" "simple-displayer"))))
    (t (error "Don't know how to build a displayer for ~a" displayer-type))))

;********************************************************************

(defun command-processor-module (cp-type)
  (cond
	((eq cp-type 'simple)
	 '((:module command-processor
	      :depends-on ("basics")
	      :source-pathname *command-processor-dir*
	      :components ("simple-command-processor"))))
	((eq cp-type 'local)
	 '((:module command-processor
	      :depends-on ("basics")
	      :source-pathname *command-processor-dir*
	      :components ("local-command-processor"))))
	((eq cp-type 'remote)
	 '((:module command-processor
	      :depends-on ("basics")
	      :source-pathname *command-processor-dir*
	      :components ("remote-command-processor"))))))


;*****************************************************************************
;  Here's the entry point:  default is to build and load a simulator 
;  with a simple displayer and local command processor.
;

(defmacro load-simulator (&key (displayer ''simple) 
			       (command-processor ''simple) 
			       (operation ''load))
  `(progn
    (define-simulator-system ,(cadr displayer) ,(cadr command-processor))
    (operate-on-system 'simulator 
     ,operation
     :verbose t 
	 :force :new-source
     :load-source-if-no-binary t)))


(defun load-standard-simulator ()
  (load-simulator :displayer 'graphic))