;************************************************************************
;
; top-level external interface for running the simulator
;
; the input function is expected to return an sexpr that
; will be accepted by execute-command.
;
; a list consisting of (command-status return-value sensor-data)
; is passed to the output function.
;
;************************************************************************

(let ((update-number 0))
  (defun run-simulator (truck world)
    (let* ((input-output (make-command-processor))
	   (input (car input-output))
	   (output (cadr input-output)))
      
      (loop
	  (let ((command (funcall input)))
	    (when (null command) (return))
	    
	    (if (legal-command? command)
		;
		; execute command, returning result and elapsed time
		;
		(let* ((cmd-return
			(multiple-value-list (do-command command truck world)))
		       (time-to-update
			(if (cdr cmd-return)
			    (cadr cmd-return)
			    (nominal-time (car command))))
		       (result (car cmd-return)))
		  
		;
		; update the world if we took any time
		;
		  (when (and result (> time-to-update 0))
		    (setf update-number
			  (move-world-ahead world time-to-update))
		    (update (environment truck)
			    update-number
			    (actual-time world)))
		  
		;
		; process return results to hand to output function
		;
		  (let ((hard-error (read-hardware-error)))
		    
		    (when (eq 'truck-mishap hard-error)
		      (post-sensor-datum 'truck-mishap
					 (truck-status truck)
					 nil nil nil))
		    (funcall output 
			     (list
			      (cond ((and (not result) (null hard-error))
				     'command-error)
				    ((null hard-error) 'okay)
				    (t hard-error))
			      result
			      (read-sensor-data)))))
		
		;
		; else not a legal command
		;
		(funcall output '(invalid-command nil nil))))))))

;************************************************************************
; for whatever arcane reasons, the exported names of functions
; do not always match the internal names of the same functions.
; so the table below does the mapping.

(let ((real-name (make-hash-table)))
  (setf (gethash 'arm-grasp real-name) 'truck-grasp)
  (setf (gethash 'arm-ungrasp real-name) 'truck-ungrasp)
  (setf (gethash 'arm-pour real-name) 'truck-pour)
  (setf (gethash 'arm-ladle real-name) 'truck-ladle)
  (setf (gethash 'arm-move real-name) 'truck-arm-move)
  (setf (gethash 'arm-examine real-name) 'truck-examine)
  (setf (gethash 'arm-toggle real-name) 'truck-toggle)
  (setf (gethash 'truck-move real-name) 'truck-move)
  (setf (gethash 'eye-examine real-name) 'eye-examine)
  (setf (gethash 'eye-scan real-name) 'eye-scan)
  (setf (gethash 'eye-monitor real-name) 'eye-monitor)
  (setf (gethash 'eye-unmonitor real-name) 'eye-unmonitor)
  (setf (gethash 'truck-time real-name) 'truck-time)
  (setf (gethash 'truck-status real-name) 'truck-status)
  (setf (gethash 'truck-heading real-name) 'truck-heading)
  (setf (gethash 'truck-fuel real-name) 'truck-fuel)
  (setf (gethash 'truck-speed real-name) 'truck-speed)
  
(defun do-command (command truck world)
  (case (car command)
	;
	; special cases: functions that are implemented "in place"
	;
    (truck-time (actual-time world))
    (truck-turn (setf (truck-heading truck) (second command)) t)
    (truck-set-speed (setf (truck-speed truck) (second command)) t)
    (wait (let ((amount (if (numberp (second command))
			    (second command)
			    (time->number (second command)))))
	    (values t (if (> amount 0) amount 0))))
    (speak (let ((truck-displayer (displayer truck)))
	     (if truck-displayer
		 (dm-speak truck-displayer (second command)))
	     t))
	;
	; the normal case: look up function in hash table, and
	; call it, with truck inserted as first argument
	;
    (t (apply (symbol-function (gethash (car command) real-name))
	      (cons truck (cdr command)))))))

;************************************************************************
; check that it is a command we've heard of, and that it has
; the right number of arguments

(defun legal-command? (command)
  (case (car command)
    ((arm-grasp arm-ungrasp arm-toggle 
	  arm-examine eye-examine
		arm-pour arm-ladle arm-move)
     (eql (length command) 3))
    ((truck-turn truck-set-speed
		 eye-scan eye-monitor eye-unmonitor wait)
     (eql (length command) 2))
    ((truck-move truck-time truck-status truck-heading
		 truck-fuel truck-speed)
     (eql (length command) 1))
    (speak t)
    (otherwise nil)))

;************************************************************************
; the nominal time to do any command.
; commands which take variable time (e.g. truck-move, wait)
; return the time that they acrually took

(let ((time-table (make-hash-table)))
  (setf (gethash 'arm-grasp time-table) 1)
  (setf (gethash 'arm-ungrasp time-table) 1)
  (setf (gethash 'arm-pour time-table) 1)
  (setf (gethash 'arm-ladle time-table) 1)
  (setf (gethash 'arm-move time-table) 2)
  (setf (gethash 'arm-examine time-table) 2)
  (setf (gethash 'arm-toggle time-table) 1)
  (setf (gethash 'truck-turn time-table) 2)
  (setf (gethash 'truck-set-speed time-table) 1)
  (setf (gethash 'truck-move time-table) 0)
  (setf (gethash 'eye-examine time-table) 2)
  (setf (gethash 'eye-scan time-table) 5)
  (setf (gethash 'eye-monitor time-table) 4)
  (setf (gethash 'eye-unmonitor time-table) 1)
  (setf (gethash 'wait time-table) 0)
  (setf (gethash 'speak time-table) 0)
  (setf (gethash 'truck-time time-table) 0)
  (setf (gethash 'truck-status time-table) 0)
  (setf (gethash 'truck-heading time-table) 0)
  (setf (gethash 'truck-fuel time-table) 0)
  (setf (gethash 'truck-speed time-table) 0)
  
  (defun nominal-time (command)
    (gethash command time-table)))

