;;; file containing generic function definitions

;;; Note that where "operations" are referred to in the documention
;;; they are implemented as generic functions with methods.

; --------------------------------------------------------------------
; ** Simulator operations of general interest
; -------------------------------------------
;
; Things defined in this file:
;
; Operations on ALL simulator objects:
;  (UNIQUE-ID self)    => identifing symbol
;  (KIND-ID self)      => major class symbol
;  (CLASS? self class) => boolean     --> omitted.  this has become a DEFUN in 
;                                         object-defs.lisp
;  (RECYCLE self)      => void                 
;  (UPDATE self number time)  => void
;  (EXAMINE self)      => () of sexp
;  (TOGGLE self)       => boolean
;
; Operations on ENVIRONMENT type objects:
;  (VISIBLE? self class) => boolean
;  (BUMPINESS self)      => bumpiness since last UPDATE
;  (WEATHER self)        => current weather
;  (ENEMY-ACTIVITY self) => current enemy activity
;  (DAY-TIME self)       => 'DAY or 'NIGHT
;  (TIME self)           => '(day hours minutes)
;
; Operations on THINGOID type objects
;
; Operations on parcel type objects:
;  (SPACE-FULL self)  => integer
;  (EMPTY? self)      => boolean
;  (GROSS-BIGNESS self)            => bigness of self and contents
;
; Operations on CONTAINER type objects: 
;  (WILL-HOLD? self thing)         => boolean
;  (PUT-IN self thing)             => T or ()
;  (TAKE-OUT self thing)           => T or ()
;  (GET-IF-HOLDS-ID self thing-id) => thing of ()
;
; Operations on VESSEL type objects:
;  (WILL-HOLD? self amount) => boolean
;  (POUR-IN self amount)    => T or ()
;  (POUR-OUT self amount)   => T or ()
;
; Operations on ARM type objects:
;  (ARM-MOVE self pos-whatsit)  => T or ()
;  (ARM-GRASP self thing-id)    => T or ()
;  (ARM-UNGRASP self thing-id)  => T or ()
;  (ARM-POUR self thing-id)     => T or ()
;  (ARM-LADLE self thing-id)    => T or ()
;  (ARM-EXAMINE self thing-id)  => () or a list of facts
;  (ARM-TOGGLE self thing-id)   => T or ()
;  (ARM-FOLDED? self)           => boolean
;
; Operations on objects that are used to form the MAP:
;  (MAP-CONNECT-START self map-node connection-dir)  \  Used for
;  (MAP-CONNECT-FINISH self map-node connection-dir)  |  building
;  (MAP-CONNECT self connection-dir link direction)   |  and showing
;                                                    /   maps.
;
; Operations on TRUCK type objects:
;  (TRUCK-ARM-MOVE self arm-id location)  => T or ()
;  (TRUCK-GRASP self arm-id thing-id) 
;  (TRUCK-UNGRASP self arm-id thing-id)
;  (TRUCK-POUR self arm-id thing-id)
;  (TRUCK-LADLE self arm-id thing-id)
;  (TRUCK-EXAMINE self arm-id thing-id)
;  (TRUCK-TOGGLE self arm-id thing-id)
;  (TRUCK-MOVE self)
;  (TRUCK-TRAVEL self link distance)
;  (TRUCK-TIRES self)   =>
;  (TRUCK-FUEL self)
;  (EYE-EXAMINE self location thing-id)
;  (EYE-SCAN self location)
;  (EYE-MONITOR self class)
;  (EYE-UNMONITOR self class)
;
; Operations to make objects display nicely
;  (PRINT-object self stream)         => #{id}
;  (SHOW self)                 => detailed description of self
;  (SHOW-CONTENTS self)        => (#{thing1} #{thing2} ...)
;  (SHOW-INSIDE self thing-id) => (SHOW thing) if contained in self
;  (DISPLAYER self)            => graphics displayer of self (settable)
;  (DISPLAY-INFO self)         => info for displayer
; --------------------------------------------------------------------
; * Operations handled by ALL types of object
; -------------------------------------------
; Operations on ALL simulator objects:
;  (UNIQUE-ID self)    => identifing symbol
;  (KIND-ID self)      => major class symbol
;  (CLASS? self class) => boolean            ->  No.  has become a DEFUN 
;                                                 in object-defs.lisp
;  (RECYCLE self)      => void                   
;  (UPDATE self number time)  => void
;  (EXAMINE self)      => boolean
;  (TOGGLE self)       => boolean

(defmacro define-generically (name args)
  `(progn
	 (defgeneric ,name ,args)
	 (defmethod ,name ,args (declare (ignore ,@args)) nil)))

(defmacro define-with-method (name args &body body)
  `(progn
	 (defgeneric ,name ,args)
	 (defmethod ,name ,args ,@body)))

(define-generically unique-id (self))
(define-generically kind-id (self))

;  class hierarchy --- class? and my-classes are now functions;
;  objects define extra-classes

(define-with-method extra-classes (self) '())

(define-generically recycle (self))  
(define-generically update (self number time))
(define-generically toggle (self))
(define-generically examine (self))

; --------------------------------------------------------------------
; * Special ENVIRONMENT type operations
; -------------------------------------
; Operations on ENVIRONMENT type objects:
;  (VISIBLE? self class) => boolean
;  (BUMPINESS self)      => bumpiness since last UPDATE
;  (WEATHER self)        => current weather
;  (ENEMY-ACTIVITY self) => current enemy activity
;  (DAY-TIME self)       => 'DAY or 'NIGHT
;  (TIME self)           => '(day hours minutes)

(define-generically visible? (self class))

(define-with-method bumpiness (self)
  (cond ((class? (environment self)
                 'environment)
         (bumpiness (environment self)))
        (t '())))

(define-generically actual-time (self))
        
; --------------------------------------------------------------------
; * Special THINGOID type operations
; ----------------------------------
; Operations on THINGOID type objects




; --------------------------------------------------------------------
; * Special parcel type operations
; ---------------------------------
; Operations on parcel type objects:
;  (CAPACITY self)          => integer
;  (SPACE-EMPTY self)       => integer
;  (SPACE-FULL self)        => integer
;  (EMPTY? self)            => boolean
;  (WILL-HOLD? self amount) => boolean
;  (GROSS-BIGNESS self)     => bigness of self and contents


(define-generically will-hold? (self thing))

(define-with-method space-full (self)
  (declare (ignore self))
  0)

(define-generically empty? (self))

(define-with-method gross-bigness (Self)
  (bigness self))


; --------------------------------------------------------------------
; * Special CONTAINER type operations
; -----------------------------------
; Operations on CONTAINER type objects:
;  (PUT-IN self thing)             => T or ()
;  (TAKE-OUT self thing)           => T or ()
;  (GET-IF-HOLDS-ID self thing-id) => thing


(define-generically put-in (self thing))
(define-generically take-out (self thing))
(define-generically get-if-holds-id (self thing-id))

; --------------------------------------------------------------------
; * Special VESSEL type operations
; --------------------------------
; Operations on VESSEL type objects:
;  (POUR-IN self amount)    => T or ()
;  (POUR-OUT self amount)   => T or ()

(define-generically pour-in (self amount))
(define-generically pour-out (self amount))


; --------------------------------------------------------------------
; * Special ARM type operations
; -----------------------------
; Operations on ARM type objects:
;  (ARM-MOVE self pos-whatsit)  => T or ()
;  (ARM-GRASP self thing-id)    => T or ()
;  (ARM-UNGRASP self thing-id)  => T or ()
;  (ARM-POUR self thing-id)     => T or ()
;  (ARM-LADLE self thing-id)    => T or ()
;  (ARM-EXAMINE self thing-id)  => () or a list of facts
;  (ARM-TOGGLE self thing-id)   => boolean
;  (ARM-FOLDED? self)           => boolean

(define-generically arm-move (self pos-whatsit))
(define-generically arm-grasp (self thing-id))
(define-generically arm-ungrasp (self thing-id))
(define-generically arm-pour (self thing-id))
(define-generically arm-ladle (self thing-id))
(define-generically arm-examine (self thing-id))
(define-generically arm-toggle (self thing-id))
(define-generically arm-folded? (self))


; --------------------------------------------------------------------
; * Special MAP type operations (MAP-LINK, MAP-NODE, MAP-LINK-SUBNODE)
; --------------------------------------------------------------------
; Operations on objects that are used to form the MAP:
;  (MAP-CONNECT-START self map-node connection-dir)  \  Used for
;  (MAP-CONNECT-FINISH self map-node connection-dir)  |  building
;  (MAP-CONNECT self connection-dir link direction)   |  and showing
;                                                    /   maps.
;  (MAP-SECTOR self) => the map sector containing an object
;
;  Why these were EVER methods I have no idea,  but they aren't
;  any more!!!!

;(define-generically map-connect-start (self map-node connection-dir))
;(define-generically map-connect-finish (self map-node connection-dir))
;(define-generically map-connect (self connection-dir link direction))

(define-with-method map-sector (Self)
  (let ((holder (environment self)))
    (cond ((null holder) '())
          ((symbolp holder) '())
          (t (map-sector holder)))))

;  (ROAD-TRAVERSE self truck direction pattern) => [distance time]
;  (ROAD-DIRECTION self)                        => direction

(define-generically road-traverse (self truck direction pattern))
(define-generically road-direction (self))

; --------------------------------------------------------------------
; * Special TRUCK type operations
; -------------------------------
;  (TRUCK-ARM-MOVE self arm-id location)  => T or ()
;  (TRUCK-GRASP self arm-id thing-id) 
;  (TRUCK-UNGRASP self arm-id thing-id)
;  (TRUCK-POUR self arm-id thing-id)
;  (TRUCK-LADLE self arm-id thing-id)
;  (TRUCK-EXAMINE self arm-id thing-id)
;  (TRUCK-TOGGLE self arm-id thing-id)
;  (TRUCK-TRAVEL self link distance)
;  (TRUCK-TIRES self)   =>
;  (TRUCK-FUEL self)    =>
;  (EYE-EXAMINE self thing-id)
;  (EYE-SCAN self location)


(Define-Generically truck-arm-move (self arm-id location))
(define-generically truck-grasp (self arm-id thing-id))
(define-generically truck-ungrasp (self arm-id thing-id))
(define-generically truck-pour (self arm-id thing-id))
(define-generically truck-ladle (self arm-id thing-id))
(define-generically truck-examine (self arm-id thing-id))
(define-generically truck-toggle (self arm-id thing-id))

(define-with-method  truck-move (self)
  (declare (ignore self))
  (values 0 0))

(define-with-method truck-travel (self link distance)
  (declare (ignore self link distance))
  (values 0 0))

(define-generically eye-examine (self location thing-id))
(define-generically eye-scan (self location))
(define-generically eye-monitor (self class))
(define-generically eye-unmonitor (self class))
(define-generically truck-tires (self))
(define-generically truck-fuel (self))

; --------------------------------------------------------------------
; * Operations to make objects display nicely - printed
; -----------------------------------------------------
; Operations to make objects display nicely
;  (print-object self stream)         => #{id} <defined by default>
;  (SHOW self)                 => detailed description of self
;  (SHOW-CONTENTS self)        => (#{thing1} #{thing2} ...)
;  (SHOW-INSIDE self thing-id) => (SHOW thing) if contained in self

(define-with-method show (self)
  (display self))

(define-generically show-contents (self))
(define-generically show-inside (self thing-id))

