/*
    error.c -- Error handling.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

/******************************* EXPORTS ******************************/

object Kwrong_type_argument;
object Ktoo_few_arguments;
object Ktoo_many_arguments;
object Kunexpected_keyword;
object Kinvalid_form;
object Kunbound_variable;
object Kinvalid_variable;
object Kundefined_function;
object Kinvalid_function;
object Kcatch;
object Kcatchall;
object Kprotect;

/******************************* ------- ******************************/


void bds_overflow()
{
	--bds_top;
	if (bds_limit > bds_org + BDSSIZE + 2*BDSGETA)
		error("bind stack overflow.");
	bds_limit += BDSGETA;
	FEerror("Bind stack overflow.", 0);
}

int frs_overflow()		/* used as condition in list.d */
{
	--frs_top;
	if (frs_limit > frs_org + FRSSIZE + 2*FRSGETA)
		error("frame stack overflow.");
	frs_limit += FRSGETA;
	FEerror("Frame stack overflow.", 0);
}

void ihs_overflow()
{
	--ihs_top;
	if (ihs_limit > ihs_org + IHSSIZE + 2*IHSGETA)
		error("invocation history stack overflow.");
	ihs_limit += IHSGETA;
	FEerror("Invocation history stack overflow.", 0);
}

void cs_overflow()
{
#ifdef DOWN_STACK
	if (cs_limit < cs_org - cssize)
	  cs_limit -= CSGETA;
#else
	if (cs_limit > cs_org + cssize)
	  cs_limit += CSGETA;
#endif
	FEerror("Control stack overflow.", 0);
}

void end_of_file()
{
	FEerror("end of file.", 0);
}

void error(char *s)
{
	printf("\nUnrecoverable error: %s\n", s);
	fflush(stdout);
#ifdef BSD
	signal(SIGIOT, SIG_DFL); /* avoid getting into a loop with abort */
#endif BSD
	abort();
}

/*****************************************************************************/
/*		Support for Lisp Error Handler				     */
/*****************************************************************************/

object siSuniversal_error_handler;

static object null_string;

object siSterminal_interrupt;

terminal_interrupt(bool correctable)
{
	funcall(2, siSterminal_interrupt, correctable? Ct : Cnil);
}

object
ihs_function_name(object x)
{
	object y;

	switch (type_of(x)) {
	case t_symbol:
		return(x);

	case t_cons:
		y = CAR(x);
		if (y == Slambda)
			return(Slambda);
		if (y == Slambda_closure)
			return(Slambda_closure);
		if (y == Slambda_block) {
			x = CDR(x);
			if (type_of(x) != t_cons)
				return(Slambda_block);
			return(CAR(x));
		}
		if (y == Slambda_block_closure) {
			x = CDR(x);
			if (type_of(x) != t_cons)
				return(Slambda_block_closure);
			x = CDR(x);
			if (type_of(x) != t_cons)
				return(Slambda_block_closure);
			x = CDR(x);
			if (type_of(x) != t_cons)
				return(Slambda_block_closure);
			x = CDR(x);
			if (type_of(x) != t_cons)
				return(Slambda_block_closure);
			return(CAR(x));
		}
		return(Cnil);

	case t_cfun:
		return(x->cf.cf_name);

	default:
		return(Cnil);
	}
}

object
ihs_top_function_name()
{
	object x;
	ihs_ptr h = ihs_top;

	while (h >= ihs_org) {
		x = ihs_function_name(h->ihs_function);
		if (x != Cnil)
			return(x);
		h--;
	}
	return(Cnil);
}

FEerror(char *s, int narg, ...)
{
	va_list args;
	object rest = Cnil, *r = &rest;

	va_start(args, narg);
	while (narg--)
	  r = &CDR(*r = CONS(va_arg(args, object), Cnil));
	return(funcall(7, siSuniversal_error_handler,
		       Kerror,                  /*  :ERROR  */
		       Cnil,                    /*  not correctable  */
		       ihs_top_function_name(), /*  function  */
		       null_string,             /*  continue-format-string  */
		       make_simple_string(s),   /*  error-format-string  */
		       rest));
      }

FEwrong_type_argument(object type, object value)
{       funcall(7, siSuniversal_error_handler, Kwrong_type_argument,
		Cnil, ihs_top_function_name(), null_string,
		make_simple_string("~S is not of type ~S."),
		list(2, value, type));
}

FEtoo_few_arguments(int *nargp)
{       object fname = ihs_top_function_name();
	funcall(7, siSuniversal_error_handler, Ktoo_few_arguments,
		Cnil, fname,
		null_string,
		make_simple_string("~S requires more than ~R argument~:p."),
		list(2, fname, MAKE_FIXNUM(*nargp)));
}

FEtoo_few_argumentsF(object args)
{       object fname = ihs_top_function_name();
	funcall(7, siSuniversal_error_handler, Ktoo_few_arguments,
		Cnil, fname, null_string,
		make_simple_string("Too few arguments."),
		list(2, fname, args));
}

FEtoo_many_arguments(int *nargp)
{       object fname = ihs_top_function_name();
	funcall(7, siSuniversal_error_handler, Ktoo_many_arguments,
		Cnil, fname,
		null_string, make_simple_string("~S requires less than ~R argument~:p."),
		list(2, fname, MAKE_FIXNUM(*nargp)));
}

FEtoo_many_argumentsF(object args)
{       object fname = ihs_top_function_name();
	funcall(7, siSuniversal_error_handler, Ktoo_many_arguments, Cnil,
		fname, null_string,
		make_simple_string("Too many arguments."),
		list(2, fname, args));
}

FEinvalid_macro_call(object name)
{
	funcall(7, siSuniversal_error_handler, Kinvalid_form, Cnil,
		name, null_string,
		make_simple_string("Invalid macro call to ~S."),
		CONS(name, Cnil));
}

FEunexpected_keyword(object key)
{
	if (!keywordp(key))
	  FEerror("~S is not a keyword.", 1, key);
	funcall(7, siSuniversal_error_handler, Kunexpected_keyword, Cnil,
		ihs_top_function_name(), null_string,
		make_simple_string("~S does not allow the keyword ~S."),
		list(2, ihs_top_function_name(), key));
}

FEinvalid_form(char *s, object form)
{
	funcall(7, siSuniversal_error_handler, Kinvalid_form, Cnil,
		ihs_top_function_name(), null_string, make_simple_string(s),
		CONS(form, Cnil));
}

FEunbound_variable(object sym)
{
	funcall(7, siSuniversal_error_handler, Kunbound_variable, Cnil,
		ihs_top_function_name(), null_string,
		make_simple_string("The variable ~S is unbound."),
		CONS(sym, Cnil));
}

FEinvalid_variable(char *s, object obj)
{
	funcall(7, siSuniversal_error_handler, Kinvalid_variable, Cnil,
		ihs_top_function_name(), null_string, make_simple_string(s),
		CONS(obj, Cnil));
}

FEundefined_function(object fname)
{
	funcall(7, siSuniversal_error_handler, Kundefined_function, Cnil,
		ihs_top_function_name(), null_string,
		make_simple_string("The function ~S is undefined."),
		CONS(fname, Cnil));
}

FEinvalid_function(object obj)
{
	funcall(7, siSuniversal_error_handler, Kinvalid_function, Cnil,
		ihs_top_function_name(), null_string,
		make_simple_string("~S is invalid as a function."),
		CONS(obj, Cnil));
}

CEerror(char *err_str, char *cont_str, int narg, ...)
{
	int i = narg;
	va_list args;
	object rest = Cnil, *r = &rest;

	va_start(args, narg);
	while (i--)
	  r = &CDR(*r = CONS(va_arg(args, object), Cnil));
	return(funcall(7, siSuniversal_error_handler,
		       Kerror,                  /*  :ERROR  */
		       Ct,                      /*  correctable  */
		       ihs_top_function_name(), /*  function  */
		       make_simple_string(cont_str),/*  continue-format-string  */
		       make_simple_string(err_str),/*  error-format-string  */
		       rest));
      }

/*
	Lisp interface to IHS
*/

ihs_ptr get_ihs_ptr(object x)
{
	ihs_ptr p;

	if (!FIXNUMP(x))
		goto ILLEGAL;
	p = ihs_org + fix(x);
	if (ihs_org <= p && p <= ihs_top)
		return(p);
ILLEGAL:
	FEerror("~S is an illegal ihs index.", 1, x);
}

siLihs_top(int narg)
{
	check_arg(0);
	VALUES(0) = MAKE_FIXNUM(ihs_top - ihs_org);
	RETURN(1);
}

siLihs_fun(int narg, object arg)
{
	check_arg(1);
	VALUES(0) = get_ihs_ptr(arg)->ihs_function;
	RETURN(1);
}

siLihs_env(int narg, object arg)
{       object *lex;
	check_arg(1);
	lex = get_ihs_ptr(arg)->ihs_base;
	VALUES(0) = list(3, lex[0], lex[1], lex[2]);
	RETURN(1);
}

frame_ptr get_frame_ptr(object x)
{
	frame_ptr p;

	if (!FIXNUMP(x))
		goto ILLEGAL;
	p = frs_org + fix(x);
	if (frs_org <= p && p <= frs_top)
		return(p);
ILLEGAL:
	FEerror("~S is an illegal frs index.", 1, x);
}

siLfrs_top(int narg)
{
	check_arg(0);
	VALUES(0) = MAKE_FIXNUM(frs_top - frs_org);
	RETURN(1);
}

siLfrs_bds(int narg, object arg)
{
	check_arg(1);
	VALUES(0) = MAKE_FIXNUM(get_frame_ptr(arg)->frs_bds_top - bds_org);
	RETURN(1);
}

siLfrs_class(int narg, object arg)
{
	enum fr_class c;

	check_arg(1);

	c = get_frame_ptr(arg)->frs_class;
	if (c == FRS_CATCH) VALUES(0) = Kcatch;
	else if (c == FRS_PROTECT) VALUES(0) = Kprotect;
	else if (c == FRS_CATCHALL) VALUES(0) = Kcatchall;
	else FEerror("Unknown frs class was detected.", 0);
	RETURN(1);
}

siLfrs_tag(int narg, object arg)
{
	check_arg(1);
	VALUES(0) = get_frame_ptr(arg)->frs_val;
	RETURN(1);
}

siLfrs_ihs(int narg, object arg)
{
	check_arg(1);
	VALUES(0) = MAKE_FIXNUM(get_frame_ptr(arg)->frs_ihs - ihs_org);
	RETURN(1);
}

bds_ptr get_bds_ptr(object x)
{
	bds_ptr p;

	if (!FIXNUMP(x))
		goto ILLEGAL;
	p = bds_org + fix(x);
	if (bds_org <= p && p <= bds_top)
		return(p);
ILLEGAL:
	FEerror("~S is an illegal bds index.", 1, x);
}

siLbds_top(int narg)
{
	check_arg(0);
	VALUES(0) = MAKE_FIXNUM(bds_top - bds_org);
	RETURN(1);
}

siLbds_var(int narg, object arg)
{
	check_arg(1);
	VALUES(0) = get_bds_ptr(arg)->bds_sym;
	RETURN(1);
}

siLbds_val(int narg, object arg)
{
	check_arg(1);
	VALUES(0) = get_bds_ptr(arg)->bds_val;
	RETURN(1);
}

siLsch_frs_base(int narg, object fr, object ihs)
{
	frame_ptr x;
	ihs_ptr y;

	check_arg(2);
	y = get_ihs_ptr(ihs);
	for (x = get_frame_ptr(fr); x <= frs_top && x->frs_ihs < y; x++);
	VALUES(0) = (x > frs_top) ? Cnil : MAKE_FIXNUM(x - frs_org);
	RETURN(1);
}

siLinternal_super_go(int narg, object tag, object arg2, object arg3)
{
	frame_ptr fr;

	check_arg(3);
	fr = frs_sch(tag);
	if (fr == NULL)
		FEerror("The tag ~S is missing.", 1, tag);
	VALUES(0) = (Null(arg3)) ? arg2 : CONS(tag, arg2);
	unwind(fr, tag, 1);
}

/*      bootstrap version                */
siLuniversal_error_handler(int narg,
			   object k, object c, object f, object cs, object es)
{

	printf("\nLisp initialization error: %s\n", es->st.st_self);
#ifndef ALFA
	exit(0);
#endif
}

check_arg_failed(int narg, int req)
{       object fname = ihs_top_function_name();
	funcall(7, siSuniversal_error_handler,
		(narg < req) ? Ktoo_few_arguments : Ktoo_many_arguments,
		Cnil, fname, null_string,
		(narg < req)
		? make_simple_string("~S requires ~R argument~:p,~%\
but only ~R ~:*~[were~;was~:;were~] supplied.")
		: make_simple_string("~S requires only ~R argument~:p,~%\
but ~R ~:*~[were~;was~:;were~] supplied."),
		list(3, fname, MAKE_FIXNUM(req), MAKE_FIXNUM(narg)));
}

illegal_declare(object form)
{
	FEinvalid_form("~S is an illegal declaration form.", form);
}

illegal_index(object x, object i)
{
	FEerror("~S is an illegal index to ~S.", 2, i, x);
}

not_a_symbol(object obj)
{
	FEinvalid_variable("~S is not a symbol.", obj);
}

not_a_variable(object obj)
{
	FEinvalid_variable("~S is not a variable.", obj);
}

object
wrong_type_argument(object typ, object obj)
{
	FEwrong_type_argument(typ, obj);
	return(VALUES(0));
}

Lerror(int narg, object eformat, ...)
{
	int i = narg;
	va_list args;
	object rest = Cnil, *r = &rest;

	if (narg == 0) FEtoo_few_arguments(&narg);

	va_start(args, eformat);
	while (i--)
	  r = &CDR(*r = CONS(va_arg(args, object), Cnil));
	return(funcall(7, siSuniversal_error_handler,
		       Kerror,
		       Cnil,
		       ihs_function_name((ihs_top - 1)->ihs_function),
		       null_string,
		       eformat,
		       rest));
}

Lcerror(int narg, object cformat, object eformat, ...)
{
	int i = narg;
	va_list args;
	object rest = Cnil, *r = &rest;
	
	if (narg < 2) FEtoo_few_arguments(&narg);
	va_start(args, eformat);
	while (i--)
	  r = &CDR(*r = CONS(va_arg(args, object), Cnil));
	funcall(7, siSuniversal_error_handler,
		Kerror,
		Ct,
		ihs_function_name((ihs_top - 1)->ihs_function),
		cformat,
		eformat,
		rest);
	VALUES(0) = Cnil;
	RETURN(1);
}

void init_error()
{
	make_function("ERROR", Lerror);
	make_function("CERROR", Lcerror);

	Kerror = make_keyword("ERROR");
	Kwrong_type_argument = make_keyword("WRONG-TYPE-ARGUMENT");
	Ktoo_few_arguments = make_keyword("TOO-FEW-ARGUMENTS");
	Ktoo_many_arguments = make_keyword("TOO-MANY-ARGUMENTS");
	Kunexpected_keyword = make_keyword("UNEXPECTED-KEYWORD");
	Kinvalid_form = make_keyword("INVALID-FORM");
	Kunbound_variable = make_keyword("UNBOUND-VARIABLE");
	Kinvalid_variable = make_keyword("INVALID-VARIABLE");
	Kundefined_function = make_keyword("UNDEFINED-FUNCTION");
	Kinvalid_function = make_keyword("INVALID-FUNCTION");

	make_si_function("IHS-TOP", siLihs_top);
	make_si_function("IHS-FUN", siLihs_fun);
	make_si_function("IHS-ENV", siLihs_env);

	Kcatch = make_keyword("CATCH");
	Kprotect = make_keyword("PROTECT");
	Kcatchall = make_keyword("CATCHALL");

	make_si_function("FRS-TOP", siLfrs_top);
	make_si_function("FRS-BDS", siLfrs_bds);
	make_si_function("FRS-CLASS", siLfrs_class);
	make_si_function("FRS-TAG", siLfrs_tag);
	make_si_function("FRS-IHS", siLfrs_ihs);

	make_si_function("BDS-TOP", siLbds_top);
	make_si_function("BDS-VAR", siLbds_var);
	make_si_function("BDS-VAL", siLbds_val);

	make_si_function("SCH-FRS-BASE", siLsch_frs_base);

	make_si_function("INTERNAL-SUPER-GO", siLinternal_super_go);

	siSuniversal_error_handler =
	make_si_function("UNIVERSAL-ERROR-HANDLER",
			 siLuniversal_error_handler);

	null_string = make_simple_string("");
	enter_mark_origin(&null_string);

	siSterminal_interrupt = make_si_ordinary("TERMINAL-INTERRUPT");
	enter_mark_origin(&siSterminal_interrupt);
}
