/*
    file.d -- File interface.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/

/*
	IMPLEMENTATION-DEPENDENT

	The file contains code to reclaim the I/O buffer
	by accessing the FILE structure of C.
*/

#include "config.h"

#if defined(BSD) && !defined(MSDOS)
#include <sys/ioctl.h>
#endif

/******************************* EXPORTS ******************************/
object Vstandard_input;
object Vstandard_output;
object Verror_output;
object Vquery_io;
object Vdebug_io;
object Vterminal_io;
object Vtrace_output;

object Kabort;
object Kdirection;
object Kinput;
object Koutput;
object Kio;
object Kprobe;
object Kelement_type;
object Kdefault;
object Kif_exists;
object Kerror;
object Knew_version;
object Krename;
object Krename_and_delete;
object Koverwrite;
object Kappend;
object Ksupersede;
object Kerror;
object Kcreate;
object Kprint;
object Kif_does_not_exist;
object Kset_default_pathname;
object Kstart;
object Kend;

/******************************* IMPORTS ******************************/

extern object readc();

/******************************* ------- ******************************/

static object terminal_io;

object siVignore_eof_on_terminal_io;

bool
feof1(FILE *fp)
{
	if (!feof(fp))
		return(FALSE);
	if (fp == terminal_io->sm.sm_object0->sm.sm_fp) {
		if (Null(symbol_value(siVignore_eof_on_terminal_io)))
			return(TRUE);
#ifdef unix
		fp = freopen("/dev/tty", "r", fp);
#endif
		if (fp == NULL)
			error("can't reopen the console");
		return(FALSE);
	}
	return(TRUE);
}

#undef  feof
#define feof    feof1

void end_of_stream(object strm)
{
	FEerror("Unexpected end of ~S.", 1, strm);
}

/*
	Input_stream_p(strm) answers
	if stream strm is an input stream or not.
	It does not check if it really is possible to read
	from the stream,
	but only checks the mode of the stream (sm_mode).
*/
bool
input_stream_p(object strm)
{
BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {
	case smm_input:
		return(TRUE);

	case smm_output:
		return(FALSE);

	case smm_io:
		return(TRUE);

	case smm_probe:
		return(FALSE);

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_broadcast:
		return(FALSE);

	case smm_concatenated:
		return(TRUE);

	case smm_two_way:
		return(TRUE);

	case smm_echo:
		return(TRUE);

	case smm_string_input:
		return(TRUE);

	case smm_string_output:
		return(FALSE);

	default:
		error("illegal stream mode");
	}
}

/*
	Output_stream_p(strm) answers
	if stream strm is an output stream.
	It does not check if it really is possible to write
	to the stream,
	but only checks the mode of the stream (sm_mode).
*/
bool
output_stream_p(object strm)
{
BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {
	case smm_input:
		return(FALSE);

	case smm_output:
		return(TRUE);

	case smm_io:
		return(TRUE);

	case smm_probe:
		return(FALSE);

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_broadcast:
		return(TRUE);

	case smm_concatenated:
		return(FALSE);

	case smm_two_way:
		return(TRUE);

	case smm_echo:
		return(TRUE);

	case smm_string_input:
		return(FALSE);

	case smm_string_output:
		return(TRUE);

	default:
		error("illegal stream mode");
	}
}

object
stream_element_type(object strm)
{
	object x;

BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {
	case smm_input:
	case smm_output:
	case smm_io:
	case smm_probe:
		return(strm->sm.sm_object0);

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_broadcast:
		x = strm->sm.sm_object0;
		if (endp(x))
			return(Ct);
		return(stream_element_type(CAR(x)));

	case smm_concatenated:
		x = strm->sm.sm_object0;
		if (endp(x))
			return(Ct);
		return(stream_element_type(CAR(x)));

	case smm_two_way:
		return(stream_element_type(strm->sm.sm_object0));

	case smm_echo:
		return(stream_element_type(strm->sm.sm_object0));

	case smm_string_input:
		return(Sstring_char);

	case smm_string_output:
		return(Sstring_char);

	default:
		error("illegal stream mode");
	}
}

/*
	Open_stream(fn, smm, if_exists, if_does_not_exist)
	opens file fn with mode smm.
	Fn is a namestring.
*/
object
open_stream(object fn, enum smmode smm, object if_exists,
	    object if_does_not_exist)
{
	object x;
	FILE *fp;
	char *fname = fn->st.st_self;

	if (smm == smm_input || smm == smm_probe) {
		fp = fopen(fname, "r");
		if (fp == NULL) {
			if (if_does_not_exist == Kerror)
				cannot_open(fn);
			else if (if_does_not_exist == Kcreate) {
				fp = fopen(fname, "w");
				if (fp == NULL)
					cannot_create(fn);
				fclose(fp);
				fp = fopen(fname, "r");
				if (fp == NULL)
					cannot_open(fn);
			} else if (Null(if_does_not_exist))
				return(Cnil);
			else
			 FEerror("~S is an illegal IF-DOES-NOT-EXIST option.",
				 1, if_does_not_exist);
		}
	} else if (smm == smm_output || smm == smm_io) {
		if (if_exists == Knew_version && if_does_not_exist == Kcreate)
			goto CREATE;
		fp = fopen(fname, "r");
		if (fp != NULL) {
			fclose(fp);
			if (if_exists == Kerror)
				FEerror("The file ~A already exists.", 1, fn);
			else if (if_exists == Krename) {
				if (smm == smm_output)
					fp = backup_fopen(fname, "w");
				else
					fp = backup_fopen(fname, "w+");
				if (fp == NULL)
					cannot_create(fn);
			} else if (if_exists == Krename_and_delete ||
				   if_exists == Knew_version ||
				   if_exists == Ksupersede) {
				if (smm == smm_output)
					fp = fopen(fname, "w");
				else
					fp = fopen(fname, "w+");
				if (fp == NULL)
					cannot_create(fn);
			} else if (if_exists == Koverwrite) {
				fp = fopen(fname, "r+");
				if (fp == NULL)
					cannot_open(fn);
			} else if (if_exists == Kappend) {
				if (smm == smm_output)
					fp = fopen(fname, "a");
				else
					fp = fopen(fname, "a+");
				if (fp == NULL)
				FEerror("Cannot append to the file ~A.",1,fn);
			} else if (Null(if_exists))
				return(Cnil);
			else
				FEerror("~S is an illegal IF-EXISTS option.",
					1, if_exists);
		} else {
			if (if_does_not_exist == Kerror)
				FEerror("The file ~A does not exist.", 1, fn);
			else if (if_does_not_exist == Kcreate) {
			CREATE:
				if (smm == smm_output)
					fp = fopen(fname, "w");
				else
					fp = fopen(fname, "w+");
				if (fp == NULL)
					cannot_create(fn);
			} else if (Null(if_does_not_exist))
				return(Cnil);
			else
			 FEerror("~S is an illegal IF-DOES-NOT-EXIST option.",
				 1, if_does_not_exist);
		}
	} else
		error("illegal stream mode");
	x = alloc_object(t_stream);
	x->sm.sm_mode = (short)smm;
	x->sm.sm_fp = fp;
	fp->_BASE = BASEFF;
	x->sm.sm_object0 = Sstring_char;
	x->sm.sm_object1 = fn;
	x->sm.sm_int0 = x->sm.sm_int1 = 0;
	setbuf(fp, alloc_contblock(BUFSIZ));
	return(x);
}

/*
	Close_stream(strm, abort_flag) closes stream strm.
	The abort_flag is not used now.
*/
void close_stream(object strm, bool abort_flag)        /*  Not used now!  */
{
	object x;

BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {
	case smm_output:
		if (strm->sm.sm_fp == stdout)
			FEerror("Cannot close the standard output.", 0);
		if (strm->sm.sm_fp == NULL)
			closed_stream(strm);
		fflush(strm->sm.sm_fp);
		insert_contblock((char *)(strm->sm.sm_fp->_BASE), BUFSIZ);
		strm->sm.sm_fp->_BASE = NULL;
		fclose(strm->sm.sm_fp);
		strm->sm.sm_fp = NULL;
		break;

	case smm_input:
		if (strm->sm.sm_fp == stdin)
			FEerror("Cannot close the standard input.", 0);

	case smm_io:
	case smm_probe:
		if (strm->sm.sm_fp == NULL)
			closed_stream(strm);
		insert_contblock((char *)(strm->sm.sm_fp->_BASE), BUFSIZ);
		strm->sm.sm_fp->_BASE = NULL;
		fclose(strm->sm.sm_fp);
		strm->sm.sm_fp = NULL;
		break;

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_broadcast:
		for (x = strm->sm.sm_object0; !endp(x); x = CDR(x))
			close_stream(CAR(x), abort_flag);
		break;

	case smm_concatenated:
		for (x = strm->sm.sm_object0; !endp(x); x = CDR(x))
			close_stream(CAR(x), abort_flag);
		break;

	case smm_two_way:
		close_stream(strm->sm.sm_object0, abort_flag);
		close_stream(strm->sm.sm_object1, abort_flag);
		break;

	case smm_echo:
		close_stream(strm->sm.sm_object0, abort_flag);
		close_stream(strm->sm.sm_object1, abort_flag);
		break;

	case smm_string_input:
		break;          /*  There is nothing to do.  */

	case smm_string_output:
		break;          /*  There is nothing to do.  */

	default:
		error("illegal stream mode");
	}
}

object
make_two_way_stream(object istrm, object ostrm)
{
	object strm;

	strm = alloc_object(t_stream);
	strm->sm.sm_mode = (short)smm_two_way;
	strm->sm.sm_fp = NULL;
	strm->sm.sm_object0 = istrm;
	strm->sm.sm_object1 = ostrm;
	strm->sm.sm_int0 = strm->sm.sm_int1 = 0;
	return(strm);
}

object
make_echo_stream(object istrm, object ostrm)
{
	object strm;

	strm = make_two_way_stream(istrm, ostrm);
	strm->sm.sm_mode = (short)smm_echo;
	return(strm);
}

object
make_string_input_stream(object strng, int istart, int iend)
{
	object strm;

	strm = alloc_object(t_stream);
	strm->sm.sm_mode = (short)smm_string_input;
	strm->sm.sm_fp = NULL;
	strm->sm.sm_object0 = strng;
	strm->sm.sm_object1 = OBJNULL;
	strm->sm.sm_int0 = istart;
	strm->sm.sm_int1 = iend;
	return(strm);
}

object
make_string_output_stream(int line_length)
{
	object strng, strm;

	line_length++;
	strng = alloc_object(t_string);
	strng->st.st_hasfillp = TRUE;
	strng->st.st_adjustable = TRUE;
	strng->st.st_displaced = Cnil;
	strng->st.st_dim = line_length;
	strng->st.st_fillp = 0;
	strng->st.st_self = NULL; /*  For GC sake  */
	strng->st.st_self = alloc_relblock(line_length, sizeof(char));
	strng->st.st_self[0] = '\0';
	strm = alloc_object(t_stream);
	strm->sm.sm_mode = (short)smm_string_output;
	strm->sm.sm_fp = NULL;
	strm->sm.sm_object0 = strng;
	strm->sm.sm_object1 = OBJNULL;
	strm->sm.sm_int0 = strm->sm.sm_int1 = 0;
	return(strm);
}

object
get_output_stream_string(object strm)
{
	object strng;

	strng = copy_simple_string(strm->sm.sm_object0);
	strm->sm.sm_object0->st.st_fillp = 0;
	return(strng);
}

int
readc_stream(object strm)
{
	int c;

BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {
	case smm_input:
	case smm_io:
		if (strm->sm.sm_fp == NULL)
			closed_stream(strm);
		c = getc(strm->sm.sm_fp);
		c &= 0377;
		if (feof(strm->sm.sm_fp))
			end_of_stream(strm);
		strm->sm.sm_int0++;
		return(c);

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_concatenated:
	CONCATENATED:
		if (endp(strm->sm.sm_object0)) {
			end_of_stream(strm);
		}
		if (stream_at_end(CAR(strm->sm.sm_object0))) {
			strm->sm.sm_object0
			= CDR(strm->sm.sm_object0);
			goto CONCATENATED;
		}
		c = readc_stream(CAR(strm->sm.sm_object0));
		return(c);

	case smm_two_way:
#ifdef unix
		if (strm == terminal_io)                                /**/
			flush_stream(terminal_io->sm.sm_object1);       /**/
#endif
		strm->sm.sm_int1 = 0;
		strm = strm->sm.sm_object0;
		goto BEGIN;

	case smm_echo:
		c = readc_stream(strm->sm.sm_object0);
		if (strm->sm.sm_int0 == 0)
			writec_stream(c, strm->sm.sm_object1);
		else
			--(strm->sm.sm_int0);
		return(c);

	case smm_string_input:
		if (strm->sm.sm_int0 >= strm->sm.sm_int1)
			end_of_stream(strm);
		return(strm->sm.sm_object0->st.st_self
		       [strm->sm.sm_int0++]);

	case smm_output:
	case smm_probe:
	case smm_broadcast:
	case smm_string_output:
		cannot_read(strm);

	default:
		error("illegal stream mode");
	}
}

void unreadc_stream(int c, object strm)
{
BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {
	case smm_input:
	case smm_io:
		if (strm->sm.sm_fp == NULL)
			closed_stream(strm);
		ungetc(c, strm->sm.sm_fp);
		--strm->sm.sm_int0;
		break;

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_concatenated:
		if (endp(strm->sm.sm_object0))
			goto UNREAD_ERROR;
		strm = CAR(strm->sm.sm_object0);
		goto BEGIN;

	case smm_two_way:
		strm = strm->sm.sm_object0;
		goto BEGIN;

	case smm_echo:
		unreadc_stream(c, strm->sm.sm_object0);
		(strm->sm.sm_int0)++;
		break;

	case smm_string_input:
		if (strm->sm.sm_int0 <= 0)
			goto UNREAD_ERROR;
		--strm->sm.sm_int0;
		break;

	case smm_output:
	case smm_probe:
	case smm_broadcast:
	case smm_string_output:
		goto UNREAD_ERROR;

	default:
		error("illegal stream mode");
	}
	return;

UNREAD_ERROR:
	FEerror("Cannot unread the stream ~S.", 1, strm);
}

writec_stream(int c, object strm)
{
	object x;
	char *p;
	int i;

BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {
	case smm_output:
	case smm_io:
		strm->sm.sm_int0++;
		if (c == '\n')
			strm->sm.sm_int1 = 0;
		else if (c == '\t')
			strm->sm.sm_int1 = (strm->sm.sm_int1&~07) + 8;
		else
			strm->sm.sm_int1++;
		if (strm->sm.sm_fp == NULL)
			closed_stream(strm);
		putc(c, strm->sm.sm_fp);
		break;

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_broadcast:
		for (x = strm->sm.sm_object0; !endp(x); x = CDR(x))
			writec_stream(c, CAR(x));
		break;

	case smm_two_way:
		strm->sm.sm_int0++;
		if (c == '\n')
			strm->sm.sm_int1 = 0;
		else if (c == '\t')
			strm->sm.sm_int1 = (strm->sm.sm_int1&~07) + 8;
		else
			strm->sm.sm_int1++;
		strm = strm->sm.sm_object1;
		goto BEGIN;

	case smm_echo:
		strm = strm->sm.sm_object1;
		goto BEGIN;

	case smm_string_output:
		strm->sm.sm_int0++;
		if (c == '\n')
			strm->sm.sm_int1 = 0;
		else if (c == '\t')
			strm->sm.sm_int1 = (strm->sm.sm_int1&~07) + 8;
		else
			strm->sm.sm_int1++;
		x = strm->sm.sm_object0;
		if (x->st.st_fillp >= x->st.st_dim) {
			if (!x->st.st_adjustable)
				FEerror("The string ~S is not adjustable.",
					1, x);
#ifdef THREADS
			start_critical_section(); /* avoid losing p */
#endif THREADS
			p = alloc_relblock(x->st.st_dim * 2 + 16,
					   sizeof(char));
			for (i = 0;  i < x->st.st_dim;  i++)
				p[i] = x->st.st_self[i];
			i = x->st.st_dim * 2 + 16;
#define ADIMLIM         16*1024*1024
			if (i >= ADIMLIM)
				FEerror("Can't extend the string.", 0);
			x->st.st_dim = i;
			adjust_displaced(x, p - x->st.st_self);
#ifdef THREADS
			end_critical_section();
#endif THREADS
		}
		x->st.st_self[x->st.st_fillp++] = c;
		break;

	case smm_input:
	case smm_probe:
	case smm_concatenated:
	case smm_string_input:
		cannot_write(strm);

	default:
		error("illegal stream mode");
	}
	return(c);
}

void writestr_stream(char *s, object strm)
{
	while (*s != '\0')
		writec_stream(*s++, strm);
}

void flush_stream(object strm)
{
	object x;

BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {
	case smm_output:
	case smm_io:
		if (strm->sm.sm_fp == NULL)
			closed_stream(strm);
		fflush(strm->sm.sm_fp);
		break;

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_broadcast:
		for (x = strm->sm.sm_object0; !endp(x); x = CDR(x))
			flush_stream(CAR(x));
		break;

	case smm_two_way:
		strm = strm->sm.sm_object1;
		goto BEGIN;

	case smm_echo:
		strm = strm->sm.sm_object1;
		goto BEGIN;

	case smm_string_output: {
	  	object strng = strm->sm.sm_object0;
		strng->st.st_self[strng->st.st_fillp] = '\0';
		break;
	      }
	case smm_input:
	case smm_probe:
	case smm_concatenated:
	case smm_string_input:
		FEerror("Cannot flush the stream ~S.", 1, strm);

	default:
		error("illegal stream mode");
	}
}

clear_input_stream(object strm)
{
	object x;

BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {

	case smm_input:
	  if (strm->sm.sm_fp == NULL)
	    closed_stream(strm);
	  fseek(strm->sm.sm_fp, 0L, 2);
	  break;

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_broadcast:
		for (x = strm->sm.sm_object0; !endp(x); x = CDR(x))
			flush_stream(CAR(x));
		break;

	case smm_two_way:
		strm = strm->sm.sm_object0;
		goto BEGIN;

	case smm_echo:
		strm = strm->sm.sm_object0;
		goto BEGIN;

	case smm_string_output:
	  break;

	case smm_io:
	case smm_output:
	case smm_probe:
	case smm_concatenated:
	case smm_string_input:
	  FEerror("Cannot clear the input of the stream ~S.", 1, strm);
	  break;

	default:
		error("illegal stream mode");
	}
}

clear_output_stream(object strm)
{
	object x;

BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {

	case smm_output:
	  if (strm->sm.sm_fp == NULL)
	    closed_stream(strm);
	  fseek(strm->sm.sm_fp, 0L, 2);
	  break;

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_broadcast:
		for (x = strm->sm.sm_object0; !endp(x); x = CDR(x))
			flush_stream(CAR(x));
		break;

	case smm_two_way:
		strm = strm->sm.sm_object1;
		goto BEGIN;

	case smm_echo:
		strm = strm->sm.sm_object1;
		goto BEGIN;

	case smm_string_output:
	  break;

	case smm_io:
	case smm_input:
	case smm_probe:
	case smm_concatenated:
	case smm_string_input:
	  FEerror("Cannot clear the output of the stream ~S.", 1, strm);
	  break;

	default:
		error("illegal stream mode");
	}
}

bool
stream_at_end(object strm)
{
	int c;

#ifdef CLOS
	if (type_of(strm) == t_instance)
	  return(FALSE);
#endif

BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {
	case smm_io:
	case smm_input:
		if (strm->sm.sm_fp == NULL)
			closed_stream(strm);
		c = getc(strm->sm.sm_fp);
		if (feof(strm->sm.sm_fp))
			return(TRUE);
		else {
			ungetc(c, strm->sm.sm_fp);
			return(FALSE);
		}

	case smm_output:
		return(FALSE);

/*      case smm_io:
		return(FALSE);
 */
	case smm_probe:
		return(FALSE);

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_broadcast:
		return(FALSE);

	case smm_concatenated:
	CONCATENATED:
		if (endp(strm->sm.sm_object0))
			return(TRUE);
		if (stream_at_end(CAR(strm->sm.sm_object0))) {
			strm->sm.sm_object0
			= CDR(strm->sm.sm_object0);
			goto CONCATENATED;
		} else
			return(FALSE);

	case smm_two_way:
#ifdef unix
		if (strm == terminal_io)                                /**/
			flush_stream(terminal_io->sm.sm_object1);       /**/
#endif
		strm = strm->sm.sm_object0;
		goto BEGIN;

	case smm_echo:
		strm = strm->sm.sm_object0;
		goto BEGIN;

	case smm_string_input:
		if (strm->sm.sm_int0 >= strm->sm.sm_int1)
			return(TRUE);
		else
			return(FALSE);

	case smm_string_output:
		return(FALSE);

	default:
		error("illegal stream mode");
	}
}

bool
listen_stream(object strm)
{
	int c;

BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {
	case smm_input:
	case smm_io:

		if (strm->sm.sm_fp == NULL)
			closed_stream(strm);
		if (feof(strm->sm.sm_fp))
				return(FALSE);
#ifdef FIONREAD
#ifdef __linux__
		if (strm->sm.sm_fp->_IO_read_ptr !=
		    strm->sm.sm_fp->_IO_read_end)
#else
		if (strm->sm.sm_fp->_cnt > 0)
#endif
			return(TRUE);
		c = 0;
		ioctl(strm->sm.sm_fp->_FILE, FIONREAD, &c);
		if (c <= 0)
		  return(FALSE);
#endif FIONREAD
		return(TRUE);

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_concatenated:
	CONCATENATED:
		if (endp(strm->sm.sm_object0))
			return(FALSE);
		strm = CAR(strm->sm.sm_object0);        /* Incomplete! */
		goto BEGIN;

	case smm_two_way:
	case smm_echo:
		strm = strm->sm.sm_object0;
		goto BEGIN;

	case smm_string_input:
		if (strm->sm.sm_int0 < strm->sm.sm_int1)
			return(TRUE);
		else
			return(FALSE);

	case smm_output:
	case smm_probe:
	case smm_broadcast:
	case smm_string_output:
		FEerror("Can't listen to ~S.", 1, strm);

	default:
		error("illegal stream mode");
	}
}

int
file_position(object strm)
{
BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {
	case smm_input:
	case smm_output:
	case smm_io:
		/*  return(strm->sm.sm_int0);  */
		if (strm->sm.sm_fp == NULL)
			closed_stream(strm);
		return(ftell(strm->sm.sm_fp));

	case smm_string_output:
		return(strm->sm.sm_object0->st.st_fillp);

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_probe:
	case smm_broadcast:
	case smm_concatenated:
	case smm_two_way:
	case smm_echo:
	case smm_string_input:
		return(-1);

	default:
		error("illegal stream mode");
	}
}

int
file_position_set(object strm, int disp)
{
BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {
	case smm_input:
	case smm_output:
	case smm_io:
		if (strm->sm.sm_fp == NULL)
			closed_stream(strm);
		if (fseek(strm->sm.sm_fp, disp, 0) < 0)
			return(-1);
		strm->sm.sm_int0 = disp;
		return(0);

	case smm_string_output:
		if (disp < strm->sm.sm_object0->st.st_fillp) {
			strm->sm.sm_object0->st.st_fillp = disp;
			strm->sm.sm_int0 = disp;
		} else {
			disp -= strm->sm.sm_object0->st.st_fillp;
			while (disp-- > 0)
				writec_stream(' ', strm);
		}
		return(0);

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_probe:
	case smm_broadcast:
	case smm_concatenated:
	case smm_two_way:
	case smm_echo:
	case smm_string_input:
		return(-1);

	default:
		error("illegal stream mode");
	}
}

int
file_length(object strm)
{
BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {
	case smm_input:
	case smm_output:
	case smm_io:
		if (strm->sm.sm_fp == NULL)
			closed_stream(strm);
		return(file_len(strm->sm.sm_fp));

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_probe:
	case smm_broadcast:
	case smm_concatenated:
	case smm_two_way:
	case smm_echo:
	case smm_string_input:
	case smm_string_output:
		return(-1);

	default:
		error("illegal stream mode");
	}
}

int
file_column(object strm)
{
	int i;
	object x;

BEGIN:
	switch ((enum smmode)strm->sm.sm_mode) {
	case smm_output:
	case smm_io:
	case smm_two_way:
	case smm_string_output:
		return(strm->sm.sm_int1);

	case smm_synonym:
		strm = symbol_value(strm->sm.sm_object0);
		if (type_of(strm) != t_stream)
			FEwrong_type_argument(Sstream, strm);
		goto BEGIN;

	case smm_echo:
		strm = strm->sm.sm_object1;
		goto BEGIN;

	case smm_input:
	case smm_probe:
	case smm_string_input:
		return(-1);

	case smm_concatenated:
		if (endp(strm->sm.sm_object0))
			return(-1);
		strm = CAR(strm->sm.sm_object0);
		goto BEGIN;

	case smm_broadcast:
		for (x = strm->sm.sm_object0; !endp(x); x = CDR(x)) {
			i = file_column(CAR(x));
			if (i >= 0)
				return(i);
		}
		return(-1);

	default:
		error("illegal stream mode");
	}
}

Lmake_synonym_stream(int narg, object sym)
{
	object x;

	check_arg(1);
	check_type_symbol(&sym);
	x = alloc_object(t_stream);
	x->sm.sm_mode = (short)smm_synonym;
	x->sm.sm_fp = NULL;
	x->sm.sm_object0 = sym;
	x->sm.sm_object1 = OBJNULL;
	x->sm.sm_int0 = x->sm.sm_int1 = 0;
	VALUES(0) = x;
	RETURN(1);
}


Lmake_broadcast_stream(int narg, ...)
{
	object x, streams;
	int i;
	va_list ap;

	va_start(ap, narg);
	for (i = 0;  i < narg;  i++) {
		x = va_arg(ap, object);
		if (type_of(x) != t_stream || !output_stream_p(x))
			cannot_write(x);
		}
	va_start(ap, narg);
	if (narg == 0)
		streams = Cnil;
		else x = streams = CONS(va_arg(ap, object), Cnil);
	for (i = 1;  i < narg; x = CDR(x), i++)
		CDR(x) = CONS(va_arg(ap, object), Cnil);
	x = alloc_object(t_stream);
	x->sm.sm_mode = (short)smm_broadcast;
	x->sm.sm_fp = NULL;
	x->sm.sm_object0 = streams;
	x->sm.sm_object1 = OBJNULL;
	x->sm.sm_int0 = x->sm.sm_int1 = 0;
	VALUES(0) = x;
	RETURN(1);
}

Lmake_concatenated_stream(int narg, ...)
{
	object x, streams;
	int i;
	va_list ap;

	va_start(ap, narg);
	for (i = 0;  i < narg;  i++) {
		x = va_arg(ap, object);
		if (type_of(x) != t_stream || !input_stream_p(x))
			cannot_read(x);
		}
	va_start(ap, narg);
	if (narg == 0)
		streams = Cnil;
		else x = streams = CONS(va_arg(ap, object), Cnil);
	for (i = 1;  i < narg; x = CDR(x), i++)
		CDR(x) = CONS(va_arg(ap, object), Cnil);
	x = alloc_object(t_stream);
	x->sm.sm_mode = (short)smm_concatenated;
	x->sm.sm_fp = NULL;
	x->sm.sm_object0 = streams;
	x->sm.sm_object1 = OBJNULL;
	x->sm.sm_int0 = x->sm.sm_int1 = 0;
	VALUES(0) = x;
	RETURN(1);
}

Lmake_two_way_stream(int narg, object strm1, object strm2)
{
	check_arg(2);

	if (type_of(strm1) != t_stream || !input_stream_p(strm1))
		cannot_read(strm1);
	if (type_of(strm2) != t_stream || !output_stream_p(strm2))
		cannot_write(strm2);
	VALUES(0) = make_two_way_stream(strm1, strm2);
	RETURN(1);
}

Lmake_echo_stream(int narg, object strm1, object strm2)
{
	check_arg(2);

	if (type_of(strm1) != t_stream || !input_stream_p(strm1))
		cannot_read(strm1);
	if (type_of(strm2) != t_stream || !output_stream_p(strm2))
		cannot_write(strm2);
	VALUES(0) = make_echo_stream(strm1, strm2);
	RETURN(1);
}

@(defun make_string_input_stream (strng &o istart iend)
	int s, e;
@
	check_type_string(&strng);
	if (Null(istart))
		s = 0;
	else if (!FIXNUMP(istart))
		goto E;
	else
		s = fix(istart);
	if (Null(iend))
		e = strng->st.st_fillp;
	else if (!FIXNUMP(iend))
		goto E;
	else
		e = fix(iend);
	if (s < 0 || e > strng->st.st_fillp || s > e)
		goto E;
	@(return `make_string_input_stream(strng, s, e)`)

E:
	FEerror("~S and ~S are illegal as :START and :END~%\
for the string ~S.",
		3, istart, iend, strng);
@)

Lmake_string_output_stream(int narg)
{
	check_arg(0);
	VALUES(0) = make_string_output_stream(64);
	RETURN(1);
}

Lget_output_stream_string(int narg, object strm)
{
	check_arg(1);

	if (type_of(strm) != t_stream ||
	    (enum smmode)strm->sm.sm_mode != smm_string_output)
		FEerror("~S is not a string-output stream.", 1, strm);
	VALUES(0) = get_output_stream_string(strm);
	RETURN(1);
}

/*
	(SI:OUTPUT-STREAM-STRING string-output-stream)

		extracts the string associated with the given
		string-output-stream.
*/
siLoutput_stream_string(int narg, object strm)
{
	check_arg(1);
	if (type_of(strm) != t_stream ||
	    (enum smmode)strm->sm.sm_mode != smm_string_output)
		FEerror("~S is not a string-output stream.", 1, strm);
	VALUES(0) = strm->sm.sm_object0;
	RETURN(1);
}

Lstreamp(int narg, object strm)
{
	check_arg(1);

	VALUES(0) = ((type_of(strm) == t_stream) ? Ct : Cnil);
	RETURN(1);
}

Linput_stream_p(int narg, object strm)
{
	check_arg(1);

	check_type_stream(&strm);
	VALUES(0) = (input_stream_p(strm) ? Ct : Cnil);
	RETURN(1);
}

Loutput_stream_p(int narg, object strm)
{
	check_arg(1);

	check_type_stream(&strm);
	VALUES(0) = (output_stream_p(strm) ? Ct : Cnil);
	RETURN(1);
}

Lstream_element_type(int narg, object strm)
{
	check_arg(1);

	check_type_stream(&strm);
	VALUES(0) = stream_element_type(strm);
	RETURN(1);
}

@(defun close (strm &key abort)
@
	check_type_stream(&strm);
	close_stream(strm, abort != Cnil);
	@(return Ct)
@)

@(defun open (filename
	      &key (direction Kinput)
		   (element_type Sstring_char)
		   (if_exists Cnil iesp)
		   (if_does_not_exist Cnil idnesp)
	      &aux strm)
	enum smmode smm;
@
	check_type_or_pathname_string_symbol_stream(&filename);
	filename = coerce_to_namestring(filename);
	if (direction == Kinput) {
		smm = smm_input;
		if (!idnesp)
			if_does_not_exist = Kerror;
	} else if (direction == Koutput) {
		smm = smm_output;
		if (!iesp)
			if_exists = Knew_version;
		if (!idnesp) {
			if (if_exists == Koverwrite ||
			    if_exists == Kappend)
				if_does_not_exist = Kerror;
			else
				if_does_not_exist = Kcreate;
		}
	} else if (direction == Kio) {
		smm = smm_io;
		if (!iesp)
			if_exists = Knew_version;
		if (!idnesp) {
			if (if_exists == Koverwrite ||
			    if_exists == Kappend)
				if_does_not_exist = Kerror;
			else
				if_does_not_exist = Kcreate;
		}
	} else if (direction == Kprobe) {
		smm = smm_probe;
		if (!idnesp)
			if_does_not_exist = Cnil;
	} else
		FEerror("~S is an illegal DIRECTION for OPEN.",
			1, direction);
	strm = open_stream(filename, smm, if_exists, if_does_not_exist);
	@(return strm)
@)

@(defun file_position (file_stream &o position)
	int i;
@
	check_type_stream(&file_stream);
	if (Null(position)) {
		i = file_position(file_stream);
		if (i < 0)
			@(return Cnil)
		@(return `MAKE_FIXNUM(i)`)
	} else {
		if (position == Kstart)
			i = 0;
		else if (position == Kend)
			i = file_length(file_stream);
		else if (!FIXNUMP(position) ||
		    (i = fix((position))) < 0)
			FEerror("~S is an illegal file position~%\
for the file-stream ~S.",
				2, position, file_stream);
		if (file_position_set(file_stream, i) < 0)
			@(return Cnil)
		@(return Ct)
	}       
@)

Lfile_length(int narg, object strm)
{
	int i;

	check_arg(1);
	check_type_stream(&strm);
	i = file_length(strm);
	VALUES(0) = (i < 0) ? Cnil : MAKE_FIXNUM(i);
	RETURN(1);
}

siLget_string_input_stream_index(int narg, object strm)
{
	check_arg(1);
	check_type_stream(&strm);
	if ((enum smmode)strm->sm.sm_mode != smm_string_input)
		FEerror("~S is not a string-input stream.", 1, strm);
	VALUES(0) = MAKE_FIXNUM(strm->sm.sm_int0);
	RETURN(1);
}

siLmake_string_output_stream_from_string(int narg, object strng)
{
	object strm;

	check_arg(1);
	if (type_of(strng) != t_string || !strng->st.st_hasfillp)
		FEerror("~S is not a string with a fill-pointer.", 1, strng);
	strm = alloc_object(t_stream);
	strm->sm.sm_mode = (short)smm_string_output;
	strm->sm.sm_fp = NULL;
	strm->sm.sm_object0 = strng;
	strm->sm.sm_object1 = OBJNULL;
	strm->sm.sm_int0 = strng->st.st_fillp;
	strm->sm.sm_int1 = 0;
	VALUES(0) = strm;
	RETURN(1);
}

siLcopy_stream(int narg, object in, object out)
{
	check_arg(2);
	check_type_stream(&in);
	check_type_stream(&out);
	while (!stream_at_end(in))
		writec_stream(readc_stream(in), out);
	flush_stream(out);
	VALUES(0) = Ct;
	RETURN(1);
}

too_long_file_name(object fn)
{
	FEerror("~S is a too long file name.", 1, fn);
}

cannot_open(object fn)
{
	FEerror("Cannot open the file ~A.", 1, fn);
}

cannot_create(object fn)
{
	FEerror("Cannot create the file ~A.", 1, fn);
}

cannot_read(object strm)
{
	FEerror("Cannot read the stream ~S.", 1, strm);
}

cannot_write(object strm)
{
	FEerror("Cannot write to the stream ~S.", 1, strm);
}

closed_stream(object strm)
{
	FEerror("The stream ~S is already closed.", 1, strm);
}


init_file()
{
	object standard_input;
	object standard_output;
	object standard;
	object x;

	standard_input = alloc_object(t_stream);
	standard_input->sm.sm_mode = (short)smm_input;
	standard_input->sm.sm_fp = stdin;
	standard_input->sm.sm_object0 = Sstring_char;
	standard_input->sm.sm_object1
#ifdef unix
	= make_simple_string("stdin");
#endif
	standard_input->sm.sm_int0 = 0;
	standard_input->sm.sm_int1 = 0;

	standard_output = alloc_object(t_stream);
	standard_output->sm.sm_mode = (short)smm_output;
	standard_output->sm.sm_fp = stdout;
	standard_output->sm.sm_object0 = Sstring_char;
	standard_output->sm.sm_object1
#ifdef unix
	= make_simple_string("stdout");
#endif
	standard_output->sm.sm_int0 = 0;
	standard_output->sm.sm_int1 = 0;

	terminal_io = standard
	= make_two_way_stream(standard_input, standard_output);
	enter_mark_origin(&terminal_io);

	Vterminal_io
	= make_special("*TERMINAL-IO*", standard);

	x = alloc_object(t_stream);
	x->sm.sm_mode = (short)smm_synonym;
	x->sm.sm_fp = NULL;
	x->sm.sm_object0 = Vterminal_io;
	x->sm.sm_object1 = OBJNULL;
	x->sm.sm_int0 = x->sm.sm_int1 = 0;
	standard = x;

	Vstandard_input
	= make_special("*STANDARD-INPUT*", standard);
	Vstandard_output
	= make_special("*STANDARD-OUTPUT*", standard);
	Verror_output
	= make_special("*ERROR-OUTPUT*", standard);

	Vquery_io
	= make_special("*QUERY-IO*", standard);
	Vdebug_io
	= make_special("*DEBUG-IO*", standard);
	Vtrace_output
	= make_special("*TRACE-OUTPUT*", standard);

}

init_file_function()
{
	Kabort = make_keyword("ABORT");

	Kdirection = make_keyword("DIRECTION");
	Kinput = make_keyword("INPUT");
	Koutput = make_keyword("OUTPUT");
	Kio = make_keyword("IO");
	Kprobe = make_keyword("PROBE");
	Kelement_type = make_keyword("ELEMENT-TYPE");
	Kdefault = make_keyword("DEFAULT");
	Kif_exists = make_keyword("IF-EXISTS");
	Kerror = make_keyword("ERROR");
	Knew_version = make_keyword("NEW-VERSION");
	Krename = make_keyword("RENAME");
	Krename_and_delete = make_keyword("RENAME-AND-DELETE");
	Koverwrite = make_keyword("OVERWRITE");
	Kappend = make_keyword("APPEND");
	Ksupersede = make_keyword("SUPERSEDE");
	/*  Kerror = make_keyword("ERROR");  */
	Kcreate = make_keyword("CREATE");
	Kif_does_not_exist = make_keyword("IF-DOES-NOT-EXIST");
	Kset_default_pathname = make_keyword("SET-DEFAULT-PATHNAME");

	make_function("MAKE-SYNONYM-STREAM", Lmake_synonym_stream);
	make_function("MAKE-BROADCAST-STREAM", Lmake_broadcast_stream);
	make_function("MAKE-CONCATENATED-STREAM",
		      Lmake_concatenated_stream);
	make_function("MAKE-TWO-WAY-STREAM", Lmake_two_way_stream);
	make_function("MAKE-ECHO-STREAM", Lmake_echo_stream);
	make_function("MAKE-STRING-INPUT-STREAM",
		      Lmake_string_input_stream);
	make_function("MAKE-STRING-OUTPUT-STREAM",
		      Lmake_string_output_stream);
	make_function("GET-OUTPUT-STREAM-STRING",
		      Lget_output_stream_string);

	make_si_function("OUTPUT-STREAM-STRING", siLoutput_stream_string);

	make_function("STREAMP", Lstreamp);
	make_function("INPUT-STREAM-P", Linput_stream_p);
	make_function("OUTPUT-STREAM-P", Loutput_stream_p);
	make_function("STREAM-ELEMENT-TYPE", Lstream_element_type);
	make_function("CLOSE", Lclose);
	make_function("OPEN", Lopen);
	make_function("FILE-POSITION", Lfile_position);
	make_function("FILE-LENGTH", Lfile_length);

	make_si_function("GET-STRING-INPUT-STREAM-INDEX",
			 siLget_string_input_stream_index);
	make_si_function("MAKE-STRING-OUTPUT-STREAM-FROM-STRING",
			 siLmake_string_output_stream_from_string);
	make_si_function("COPY-STREAM", siLcopy_stream);

	siVignore_eof_on_terminal_io
	= make_si_special("*IGNORE-EOF-ON-TERMINAL-IO*", Cnil);
}
