/*
    num_arith.c  -- Arithmetic operations
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/

#include "config.h"
#include "num_include.h"

object
bignum1(int val)
{
	object z;

	z = alloc_object(t_bignum);
	z->big.big_car = val;
#ifndef THREADS
	z->big.big_cdr = NULL; /* done by alloc_object */
#endif
	return(z);
      }

object
bignum2(int most, int least)
{
	object z, z1;

	z = alloc_object(t_bignum);
	z->big.big_car = least;
#ifndef THREADS
	z->big.big_cdr = NULL;
#endif
	z1 = (object)(z->big.big_cdr
	= (struct bignum *)alloc_object(t_bignum));
	z1->big.big_car = most;
#ifndef THREADS
	z1->big.big_cdr = NULL;
#endif
	return(z);
}	

object
bignum3(int most, int middle, int least)
{
	object z, z1;

	z = alloc_object(t_bignum);
	z->big.big_car = least;
#ifndef THREADS
	z->big.big_cdr = NULL;
#endif
	z1 = (object)(z->big.big_cdr
	= (struct bignum *)alloc_object(t_bignum));
	z1->big.big_car = middle;
#ifndef THREADS
	z1->big.big_cdr = NULL;
#endif
	z1 = (object)(z1->big.big_cdr
	= (struct bignum *)alloc_object(t_bignum));
	z1->big.big_car = most;
#ifndef THREADS
	z1->big.big_cdr = NULL;
#endif
	return(z);
}	

object
fixnum_times(int i, int j)
{

	int s, h, l;
	object z;

	if (i == 0 || j == 0)
		return(MAKE_FIXNUM(0));
	if (i < 0) {
	  if (i == MOST_NEGATIVE_FIX && j > -4 && j <= 4)
	      return(bignum1(i*j));
	  i = -i;
	  s = -1;
	} else
	  s = 1;
	if (j < 0) {
	  if (j == MOST_NEGATIVE_FIX && i <= 4)
	      return(bignum1(i*j*s));
	  j = -j;
	  s = -s;
	}
	extended_mul(i, j, 0, &h, &l);
	if (h != 0) {
	  if (s < 0) {
	    if (l == 0)
	      return(bignum2(-h, 0));
	    else
	      return(bignum2(~h, (-l) & MASK));
	  } else
	    return(bignum2(h, l & MASK));
	} else
	  if (l <= MOST_POSITIVE_FIX)
	    return(MAKE_FIXNUM(s*l));
	  else
	    return(bignum1(s*l));
}

object
fix_big_times(int i, object b)
{
	int j, s;
	struct bignum *x;

	if (i == 1)
		return(b);
	if (i == -1)
		return(number_negate(b));
	x = copy_big(b);
	if ((s = big_sign(x)) < 0)
		complement_big(x);
	if (i < 0) {
		i = -i;
		s = -s;
	}
	mul_int_big(i, x);
L:
	if (s < 0)
		complement_big(x);
	x = (struct bignum *)normalize_big_to_object(x);
	return((object)x);
}

object
big_big_times(object x, object y)
{
	int	i, j;
	struct bignum *z;

	if ((i = big_sign(x)) < 0)
		x = (object)big_minus(x);
	if ((j = big_sign(y)) < 0)
		y = (object)big_minus(y);
	z = big_times(x, y);
	if (i > 0 && j < 0 || i < 0 && j > 0)
		complement_big(z);
	z = (struct bignum *)normalize_big_to_object(z);
	return((object)z);
}

object
number_to_complex(object x)
{
	object z;

	if (FIXNUMP(x)) {	/* Immediate fixnums */
		z = alloc_object(t_complex);
		z->cmp.cmp_real = x;
		z->cmp.cmp_imag = MAKE_FIXNUM(0);
		return(z);
	      }

	switch (type_of(x)) {

	case t_fixnum:
	case t_bignum:
	case t_ratio:
	case t_shortfloat:
	case t_longfloat:
		z = alloc_object(t_complex);
		z->cmp.cmp_real = x;
		z->cmp.cmp_imag = MAKE_FIXNUM(0);
		return(z);

	case t_complex:
		return(x);

	default:
		FEwrong_type_argument(Snumber, x);
	}
}

object
number_plus(object x, object y)
{
	int i, j, k;
	double dx, dy;
	object z, z1;
	
	switch (type_of(x)) {

	case t_fixnum:
	        switch (type_of(y)) {
		case t_fixnum:
			if((i = fix(x)) == 0)
				return(y);
			if((j = fix(y)) == 0)
				return(x);
			if ((k = i + j) >= MOST_NEGATIVE_FIX &&
			    k <= MOST_POSITIVE_FIX)
			  return(MAKE_FIXNUM(k));
			else
			  return(bignum1(k));

		case t_bignum:
			if ((i = fix(x)) == 0)
				return(y);
			z = (object)copy_big(y);
			if(i > 0)
				add_int_big(i, z);
			else
				sub_int_big(-i, z);
		        z = normalize_big_to_object(z);
			return(z);
		case t_ratio:
			z = number_times(x, y->rat.rat_den);
			z = number_plus(z, y->rat.rat_num);
			z = make_ratio(z, y->rat.rat_den);
			return(z);
		case t_shortfloat:
			(shortfloat)dx = fix(x);
			(shortfloat)dy = sf(y);
			goto SHORTFLOAT;
		case t_longfloat:
			dx = (double)(fix(x));
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			FEwrong_type_argument(Snumber, y);
		}

	case t_bignum:
		switch (type_of(y)) {
		case t_fixnum:
			if((j = fix(y)) == 0)
				return(x);
			z = (object)copy_big(x);
			if(j > 0)
				add_int_big(j, z);
			else
				sub_int_big(-j, z);
			z = normalize_big_to_object(z);
			return(z);
		case t_bignum:
			z = (object)big_plus(x, y);
			z = normalize_big_to_object(z);
			return(z);
		case t_ratio:
			z = number_times(x, y->rat.rat_den);
			z = number_plus(z, y->rat.rat_num);
			z = make_ratio(z, y->rat.rat_den);
			return(z);
		case t_shortfloat:
			(shortfloat)dx = (shortfloat)number_to_double(x);
			(shortfloat)dy = sf(y);
			goto SHORTFLOAT;
		case t_longfloat:
			dx = number_to_double(x);
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			FEwrong_type_argument(Snumber, y);
		}

	case t_ratio:
		switch (type_of(y)) {
		case t_fixnum:
		case t_bignum:
			z = number_times(x->rat.rat_den, y);
			z = number_plus(x->rat.rat_num, z);
			z = make_ratio(z, x->rat.rat_den);
			return(z);
		case t_ratio:
			z1 = number_times(x->rat.rat_num,y->rat.rat_den);
			z = number_times(x->rat.rat_den,y->rat.rat_num);
			z = number_plus(z1, z);
			z1 = number_times(x->rat.rat_den,y->rat.rat_den);
			z = make_ratio(z, z1);
			return(z);
		case t_shortfloat:
			(shortfloat)dx = (shortfloat)number_to_double(x);
			(shortfloat)dy = sf(y);
			goto SHORTFLOAT;
		case t_longfloat:
			dx = number_to_double(x);
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			FEwrong_type_argument(Snumber, y);
		}

	case t_shortfloat:
		switch (type_of(y)) {
		case t_fixnum:
			(shortfloat)dx = sf(x);
			(shortfloat)dy = fix(y);
			goto SHORTFLOAT;
		case t_shortfloat:
			(shortfloat)dx = sf(x);
			(shortfloat)dy = sf(y);
			goto SHORTFLOAT;
		case t_longfloat:
			dx = (double)(sf(x));
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			dx = (double)(sf(x));
			dy = number_to_double(y);
			goto SHORTFLOAT;
		}
	SHORTFLOAT:
		z = alloc_object(t_shortfloat);
		sf(z) = (shortfloat)(dx + dy);
		return(z);

	case t_longfloat:
		dx = lf(x);
		switch (type_of(y)) {
		case t_fixnum:
			dy = (double)(fix(y));
			goto LONGFLOAT;
		case t_shortfloat:
			dy = (double)(sf(y));
			goto LONGFLOAT;
		case t_longfloat:
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			dy = number_to_double(y);
			goto LONGFLOAT;
		}
	LONGFLOAT:
		z = alloc_object(t_longfloat);
		lf(z) = dx + dy;
		return(z);

	COMPLEX:
		x = number_to_complex(x);
	case t_complex:
		y = number_to_complex(y);
		z = number_plus(x->cmp.cmp_real, y->cmp.cmp_real);
		z1 = number_plus(x->cmp.cmp_imag, y->cmp.cmp_imag);
		z = make_complex(z, z1);
		return(z);

	default:
		FEwrong_type_argument(Snumber, x);
	}
}

object
one_plus(object x)
{
	int i;
	double dx;
	object z, z1;
	
	switch (type_of(x)) {

	case t_fixnum:
		i = fix(x) + 1;
 		if (MOST_NEGATIVE_FIX <= i && i <= MOST_POSITIVE_FIX)
 			return(MAKE_FIXNUM(i));
 		return(bignum1(i));

	case t_bignum:
		return(number_plus(x, MAKE_FIXNUM(1)));

	case t_ratio:
		z = number_plus(x->rat.rat_num, x->rat.rat_den);
		z = make_ratio(z, x->rat.rat_den);
		return(z);

	case t_shortfloat:
		z = alloc_object(t_shortfloat);
		sf(z) = (shortfloat)(sf(x) + 1.0);
		return(z);

	case t_longfloat:
		dx = lf(x);
		z = alloc_object(t_longfloat);
		lf(z) = dx + 1.0;
		return(z);

	case t_complex:
	COMPLEX:
		z = one_plus(x->cmp.cmp_real);
		z = make_complex(z, x->cmp.cmp_imag);
		return(z);

	default:
		FEwrong_type_argument(Snumber, x);
	}
}

object
number_minus(object x, object y)
{
	int i, j, k;
	double dx, dy;
	object z, z1;
	
	switch (type_of(x)) {

	case t_fixnum:
		switch(type_of(y)) {
		case t_fixnum:
			if ((j = fix(y)) == 0)
				return(x);
			if ((k = fix(x) - j) >= MOST_NEGATIVE_FIX &&
			    k <= MOST_POSITIVE_FIX)
			  return(MAKE_FIXNUM(k));
			else
			  return(bignum1(k));
		case t_bignum:
			z = (object)big_minus(y);
			if ((i = fix(x)) == 0)
				;
			else if(i > 0)
				add_int_big(i, z);
			else
				sub_int_big(-i, z);
		        z = normalize_big_to_object(z);
			return(z);
		case t_ratio:
			z = number_times(x, y->rat.rat_den);
			z = number_minus(z, y->rat.rat_num);
			z = make_ratio(z, y->rat.rat_den);
			return(z);
		case t_shortfloat:
			(shortfloat)dx = (shortfloat)fix(x);
			(shortfloat)dy = sf(y);
			goto SHORTFLOAT;
		case t_longfloat:
			dx = (double)(fix(x));
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			FEwrong_type_argument(Snumber, y);
		}

	case t_bignum:
		switch (type_of(y)) {
		case t_fixnum:
			if((j = fix(y)) == 0)
				return(x);
			z = (object)copy_big(x);
			if (j > 0)
				sub_int_big(j, z);
			else
				add_int_big(-j, z);
			z = normalize_big_to_object(z);
			return(z);
		case t_bignum:
			y = (object)big_minus(y);
			z = (object)big_plus(x, y);
			z = normalize_big_to_object(z);
			return(z);
		case t_ratio:
			z = number_times(x, y->rat.rat_den);
			z = number_minus(z, y->rat.rat_num);
			z = make_ratio(z, y->rat.rat_den);
			return(z);
		case t_shortfloat:
			(shortfloat)dx = (shortfloat)number_to_double(x);
			(shortfloat)dy = sf(y);
			goto SHORTFLOAT;
		case t_longfloat:
			dx = number_to_double(x);
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			FEwrong_type_argument(Snumber, y);
		}

	case t_ratio:
		switch (type_of(y)) {
		case t_fixnum:
		case t_bignum:
			z = number_times(x->rat.rat_den, y);
			z = number_minus(x->rat.rat_num, z);
			z = make_ratio(z, x->rat.rat_den);
			return(z);
		case t_ratio:
			z = number_times(x->rat.rat_num,y->rat.rat_den);
			z1 = number_times(x->rat.rat_den,y->rat.rat_num);
			z = number_minus(z, z1);
			z1 = number_times(x->rat.rat_den,y->rat.rat_den);
			z = make_ratio(z, z1);
			return(z);
		case t_shortfloat:
			(shortfloat)dx = (shortfloat)number_to_double(x);
			(shortfloat)dy = sf(y);
			goto SHORTFLOAT;
		case t_longfloat:
			dx = number_to_double(x);
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			FEwrong_type_argument(Snumber, y);
		}

	case t_shortfloat:
		switch (type_of(y)) {
		case t_fixnum:
			(shortfloat)dx = sf(x);
			(shortfloat)dy = (shortfloat)(fix(y));
			goto SHORTFLOAT;
		case t_shortfloat:
			(shortfloat)dx = sf(x);
			(shortfloat)dy = sf(y);
			goto SHORTFLOAT;
		case t_longfloat:
			dx = (double)(sf(x));
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			dx = (double)(sf(x));
			dy = number_to_double(y);
			goto SHORTFLOAT;
		}
	SHORTFLOAT:
		z = alloc_object(t_shortfloat);
		sf(z) = (shortfloat)(dx - dy);
		return(z);

	case t_longfloat:
		dx = lf(x);
		switch (type_of(y)) {
		case t_fixnum:
			dy = (double)(fix(y));
			goto LONGFLOAT;
		case t_shortfloat:
			dy = (double)(sf(y));
			goto LONGFLOAT;
		case t_longfloat:
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			dy = number_to_double(y);
		}
	LONGFLOAT:
		z = alloc_object(t_longfloat);
		lf(z) = dx - dy;
		return(z);

	COMPLEX:
		x = number_to_complex(x);
	case t_complex:
		y = number_to_complex(y);
		z = number_minus(x->cmp.cmp_real, y->cmp.cmp_real);
		z1 = number_minus(x->cmp.cmp_imag, y->cmp.cmp_imag);
		z = make_complex(z, z1);
		return(z);

	default:
		FEwrong_type_argument(Snumber, x);
	}
}

object
one_minus(object x)
{
	int i;
	double dx;
	object z, z1;
	
	switch (type_of(x)) {

	case t_fixnum:
		i = fix(x) - 1;
 		if (MOST_NEGATIVE_FIX <= i && i <= MOST_POSITIVE_FIX)
 			return(MAKE_FIXNUM(i));
		else
		  return(bignum1(i));

	case t_bignum:
		return(number_minus(x, MAKE_FIXNUM(1)));

	case t_ratio:
		z = number_minus(x->rat.rat_num, x->rat.rat_den);
		z = make_ratio(z, x->rat.rat_den);
		return(z);

	case t_shortfloat:
		z = alloc_object(t_shortfloat);
		sf(z) = (shortfloat)(sf(x) - 1.0);
		return(z);

	case t_longfloat:
		dx = lf(x);
		z = alloc_object(t_longfloat);
		lf(z) = dx - 1.0;
		return(z);

	case t_complex:
	COMPLEX:
		z = one_minus(x->cmp.cmp_real);
		z = make_complex(z, x->cmp.cmp_imag);
		return(z);

	default:
		FEwrong_type_argument(Snumber, x);
	}
}

object
number_negate(object x)
{
	object	z, z1;

	switch (type_of(x)) {

	case t_fixnum:
		if (fix(x) == MOST_NEGATIVE_FIX)
			return(bignum1(- MOST_NEGATIVE_FIX));
		else
			return(MAKE_FIXNUM(-fix(x)));

	case t_bignum:
		z = (object)big_minus(x);
		z = normalize_big_to_object(z);
		return(z);

	case t_ratio:
		z1 = number_negate(x->rat.rat_num);
		z = alloc_object(t_ratio);
		z->rat.rat_num = z1;
		z->rat.rat_den = x->rat.rat_den;
		return(z);

	case t_shortfloat:
		z = alloc_object(t_shortfloat);
		sf(z) = -sf(x);
		return(z);

	case t_longfloat:
		z = alloc_object(t_longfloat);
		lf(z) = -lf(x);
		return(z);

	case t_complex:
		z = number_negate(x->cmp.cmp_real);
		z1 = number_negate(x->cmp.cmp_imag);
		z = make_complex(z, z1);
		return(z);

	default:
		FEwrong_type_argument(Snumber, x);
	}
}

object
number_times(object x, object y)
{
	object z, z1;
	double dx, dy;

	switch (type_of(x)) {

	case t_fixnum:
		switch (type_of(y)) {
		case t_fixnum:
			return(fixnum_times(fix(x), fix(y)));
		case t_bignum:
			return(fix_big_times(fix(x), y));
		case t_ratio:
			z = number_times(x, y->rat.rat_num);
			z = make_ratio(z, y->rat.rat_den);
			return(z);
		case t_shortfloat:
			(shortfloat)dx = (shortfloat)fix(x);
			(shortfloat)dy = sf(y);
			goto SHORTFLOAT;
		case t_longfloat:
			dx = (double)(fix(x));
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			FEwrong_type_argument(Snumber, y);
		}

	case t_bignum:
		switch (type_of(y)) {
		case t_fixnum:
			return(fix_big_times(fix(y), x));
		case t_bignum:
			return(big_big_times(x, y));
		case t_ratio:
			z = number_times(x, y->rat.rat_num);
			z = make_ratio(z, y->rat.rat_den);
			return(z);
		case t_shortfloat:
			(shortfloat)dx = (shortfloat)number_to_double(x);
			(shortfloat)dy = sf(y);
			goto SHORTFLOAT;
		case t_longfloat:
			dx = number_to_double(x);
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			FEwrong_type_argument(Snumber, y);
		}

	case t_ratio:
		switch (type_of(y)) {
		case t_fixnum:
		case t_bignum:
			z = number_times(x->rat.rat_num, y);
			z = make_ratio(z, x->rat.rat_den);
			return(z);
		case t_ratio:
			z = number_times(x->rat.rat_num,y->rat.rat_num);
			z1 = number_times(x->rat.rat_den,y->rat.rat_den);
			z = make_ratio(z, z1);
			return(z);
		case t_shortfloat:
			(shortfloat)dx = (shortfloat)number_to_double(x);
			(shortfloat)dy = sf(y);
			goto SHORTFLOAT;
		case t_longfloat:
			dx = number_to_double(x);
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			FEwrong_type_argument(Snumber, y);
		}

	case t_shortfloat:
		switch (type_of(y)) {
		case t_fixnum:
			(shortfloat)dx = sf(x);
			(shortfloat)dy = (shortfloat)(fix(y));
			goto SHORTFLOAT;
		case t_shortfloat:
			(shortfloat)dx = sf(x);
			(shortfloat)dy = sf(y);
			goto SHORTFLOAT;
		case t_longfloat:
			dx = (double)(sf(x));
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			dx = (double)(sf(x));
			dy = number_to_double(y);
			break;
		}
	SHORTFLOAT:
		z = alloc_object(t_shortfloat);
		sf(z) = (shortfloat)(dx * dy);
		return(z);

	case t_longfloat:
		dx = lf(x);
		switch (type_of(y)) {
		case t_fixnum:
			dy = (double)(fix(y));
			goto LONGFLOAT;
		case t_shortfloat:
			dy = (double)(sf(y));
			goto LONGFLOAT;
		case t_longfloat:
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			dy = number_to_double(y);
		}
	LONGFLOAT:
		z = alloc_object(t_longfloat);
		lf(z) = dx * dy;
		return(z);

	COMPLEX:
		x = number_to_complex(x);
	case t_complex:
	{
		object z11, z12, z21, z22;

		y = number_to_complex(y);
		z11 = number_times(x->cmp.cmp_real, y->cmp.cmp_real);
		z12 = number_times(x->cmp.cmp_imag, y->cmp.cmp_imag);
		z21 = number_times(x->cmp.cmp_imag, y->cmp.cmp_real);
		z22 = number_times(x->cmp.cmp_real, y->cmp.cmp_imag);
		return(make_complex(number_minus(z11, z12),
				    number_plus(z21, z22)));
	}

	default:
		FEwrong_type_argument(Snumber, x);
	}
}

object
number_divide(object x, object y)
{
	object z, z1;
	double dx, dy;

	switch (type_of(x)) {

	case t_fixnum:
	case t_bignum:
		switch (type_of(y)) {
		case t_fixnum:
		case t_bignum:
			if(number_zerop(y) == TRUE)
				zero_divisor();
			if (number_minusp(y) == TRUE) {
				x = number_negate(x);
				y = number_negate(y);
			}
			z = make_ratio(x, y);
			return(z);
		case t_ratio:
			if(number_zerop(y->rat.rat_num))
				zero_divisor();
			z = number_times(x, y->rat.rat_den);
			z = make_ratio(z, y->rat.rat_num);
			return(z);
		case t_shortfloat:
			(shortfloat)dx = (shortfloat)number_to_double(x);
			(shortfloat)dy = sf(y);
			goto SHORTFLOAT;
		case t_longfloat:
			dx = number_to_double(x);
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			FEwrong_type_argument(Snumber, y);
		}

	case t_ratio:
		switch (type_of(y)) {
		case t_fixnum:
		case t_bignum:
			if (number_zerop(y))
				zero_divisor();
			z = number_times(x->rat.rat_den, y);
			z = make_ratio(x->rat.rat_num, z);
			return(z);
		case t_ratio:
			z = number_times(x->rat.rat_num,y->rat.rat_den);
			z1 = number_times(x->rat.rat_den,y->rat.rat_num);
			z = make_ratio(z, z1);
			return(z);
		case t_shortfloat:
			(shortfloat)dx = (shortfloat)number_to_double(x);
			(shortfloat)dy = sf(y);
			goto SHORTFLOAT;
		case t_longfloat:
			dx = number_to_double(x);
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			FEwrong_type_argument(Snumber, y);
		}

	case t_shortfloat:
		switch (type_of(y)) {
		case t_fixnum:
			(shortfloat)dx = sf(x);
			(shortfloat)dy = (shortfloat)(fix(y));
			goto SHORTFLOAT;
		case t_shortfloat:
			(shortfloat)dx = sf(x);
			(shortfloat)dy = sf(y);
			goto SHORTFLOAT;
		case t_longfloat:
			dx = (double)(sf(x));
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			dx = (double)(sf(x));
			dy = number_to_double(y);
			goto LONGFLOAT;
		}
	SHORTFLOAT:
		z = alloc_object(t_shortfloat);
		if (dy == 0.0)
			zero_divisor();
		sf(z) = (shortfloat)(dx / dy);
		return(z);


	case t_longfloat:
		dx = lf(x);
		switch (type_of(y)) {
		case t_fixnum:
			dy = (double)(fix(y));
			goto LONGFLOAT;
		case t_shortfloat:
			dy = (double)(sf(y));
			goto LONGFLOAT;
		case t_longfloat:
			dy = lf(y);
			goto LONGFLOAT;
		case t_complex:
			goto COMPLEX;
		default:
			dy = number_to_double(y);
		}
	LONGFLOAT:
		z = alloc_object(t_longfloat);
		if (dy == 0.0)
			zero_divisor();
		lf(z) = dx / dy;
		return(z);

	COMPLEX:
		x = number_to_complex(x);
	case t_complex:
	{
		object z1, z2, z3;

		y = number_to_complex(y);
		z1 = number_times(y->cmp.cmp_real, y->cmp.cmp_real);
		z2 = number_times(y->cmp.cmp_imag, y->cmp.cmp_imag);
		if (number_zerop(z3 = number_plus(z1, z2)))
			zero_divisor();
		z1 = number_times(x->cmp.cmp_real, y->cmp.cmp_real);
		z2 = number_times(x->cmp.cmp_imag, y->cmp.cmp_imag);
		z1 = number_plus(z1, z2);
		z = number_times(x->cmp.cmp_imag, y->cmp.cmp_real);
		z2 = number_times(x->cmp.cmp_real, y->cmp.cmp_imag);
		z2 = number_minus(z, z2);
		z1 = number_divide(z1, z3);
		z2 = number_divide(z2, z3);
		z = make_complex(z1, z2);
		return(z);
	}

	default:
		FEwrong_type_argument(Snumber, x);
	}
}

integer_quotient_remainder_1(object x, object y, object *qp, object *rp)
{
	enum type tx, ty;
	int i, j, q, r;
		
	tx = type_of(x);
	ty = type_of(y);
	if (tx == t_fixnum) {
 		if (ty == t_fixnum) {
			if (fix(y) == 0)
				zero_divisor();
			if (fix(y) == MOST_NEGATIVE_FIX)
				if (fix(x) == MOST_NEGATIVE_FIX) {
					*qp = MAKE_FIXNUM(1);
					*rp = MAKE_FIXNUM(0);
					return;
				} else {
					*qp = MAKE_FIXNUM(0);
					*rp = x;
					return;
				}
			if (fix(x) == MOST_NEGATIVE_FIX && fix(y) == 1) {
			  *qp = x;
			  *rp = MAKE_FIXNUM(0);
			  return;
			}
			*qp = MAKE_FIXNUM(fix(x) / fix(y));
			*rp = MAKE_FIXNUM(fix(x) % fix(y));
			return;
		}
		if (ty == t_bignum) {
			*qp = MAKE_FIXNUM(0);
			*rp = x;
			return;
		} else
			FEwrong_type_argument(Sinteger, y);
	}
	if (tx == t_bignum) {
		if (ty == t_fixnum) {
			if (fix(y) == 0)
				zero_divisor();
			x = (object)copy_big(x);
			if ((i = big_sign(x)) < 0) {
				complement_big(x);
			}
			if (fix(y) < 0) {
				q = -fix(y);
				j = -i;
			} else {
				q = fix(y);
				j = i;
			}
             		r = div_int_big(q, x);
			if (j < 0) {
				complement_big(x);
			}
			*qp = normalize_big_to_object(x);
			*rp = MAKE_FIXNUM(i < 0 ? -r : r);
			return;
		}
		else if (ty == t_bignum) {
			if ((i = big_sign(x)) < 0)
				x = (object)big_minus(x);
			if (big_sign(y) < 0) {
				y = (object)big_minus(y);
				j = -i;
			} else
				j = i;
			big_quotient_remainder(x, y, qp, rp);
			if (j < 0) {
				complement_big(*qp);
			}
			if (i < 0) {
				complement_big(*rp);
			}
			*qp = normalize_big_to_object(*qp);
			*rp = normalize_big_to_object(*rp);
			return;
		}
		else
			FEwrong_type_argument(Sinteger, y);
	}
	FEwrong_type_argument(Sinteger, y);
}

object
integer_divide1(object x, object y)
{
	object q, r;

	integer_quotient_remainder_1(x, y, &q, &r);
	return(q);
}

object
get_gcd(object x, object y)
{
	int	i, j, k;
	object	q, r;

	if (number_minusp(x))
		x = number_negate(x);
	if (number_minusp(y))
		y = number_negate(y);

L:
	if (FIXNUMP(x) && FIXNUMP(y)) {
		i = fix(x);
		j = fix(y);
LL:
		if (i < j) {
			k = i;
			i = j;
			j = k;
		}
		if (j == 0)
			return(MAKE_FIXNUM(i));
		k = i % j;
		i = j;
		j = k;
		goto LL;
	}

	if (number_compare(x, y) < 0) {
		r = x;
		x = y;
		y = r;
	}
	if (FIXNUMP(y) && fix(y) == 0)
		return(x);
	integer_quotient_remainder_1(x, y, &q, &r);
	x = y;
	y = r;
	goto L;
}

/* (+          )   */
Lplus(int narg, ...)
{
        int i; va_list nums;
	object numi, sum = MAKE_FIXNUM(0);

	va_start(nums, narg);
	for (i = 0;  i < narg;  i++) {
	  numi = va_arg(nums, object);
	  check_type_number(&numi);
	  sum = number_plus(sum, numi);
	}
	VALUES(0) = sum;
	RETURN(1);
}

/*  (-		)  */
Lminus(int narg, object num, ...)
{
	int i;
	va_list nums;
	object diff;

	if (narg == 0)
		FEtoo_few_arguments(&narg);
	check_type_number(&num);
	if (narg == 1) {
		VALUES(0) = number_negate(num);
		RETURN(1);
	}
	va_start(nums, num);
	for (i = 1, diff = num;  i < narg;  i++) {
	  num = va_arg(nums, object);
	  check_type_number(&num);
	  diff = number_minus(diff, num);
	}
	VALUES(0) = diff;
	RETURN(1);
}

/*  (*		)  */
Ltimes(int narg, ...)
{
	int i; va_list nums;
	object numi, prod = MAKE_FIXNUM(1);

	va_start(nums, narg);
	for (i = 0;  i < narg;  i++) {
	  numi = va_arg(nums, object);
	  check_type_number(&numi);
	  prod = number_times(prod, numi);
	}
	VALUES(0) = prod;
	RETURN(1);
}

/*  (/		)  */
Ldivide(int narg, object num, ...)
{
	int i;
	va_list nums;
	object quot;

	if (narg == 0)
		FEtoo_few_arguments(&narg);
	check_type_number(&num);
	if (narg == 1)
	  quot = number_divide(MAKE_FIXNUM(1), num);
	else {
	  va_start(nums, num);
	  quot = num;
	  narg--;
	  for (i = 0;  i < narg;  i++) {
	    num = va_arg(nums, object);
	    check_type_number(&num);
	    quot = number_divide(quot, num);
	  }
	}
	VALUES(0) = quot;
	RETURN(1);
}

/*  (1+ x)  */
Lone_plus(int narg, object x)
{
	check_arg(1);
	check_type_number(&x);
	VALUES(0) = one_plus(x);
	RETURN(1);
}

/*  (1-	x)  */
Lone_minus(int narg, object x)
{
	check_arg(1);
	check_type_number(&x);
	VALUES(0) = one_minus(x);
	RETURN(1);
}

Lconjugate(int narg, object c)
{
	object i;

	check_arg(1);
	check_type_number(&c);
	if (type_of(c) == t_complex) {
		i = number_negate(c->cmp.cmp_imag);
		VALUES(0) = make_complex(c->cmp.cmp_real, i);
	} else
	  VALUES(0) = c;
	RETURN(1);
}

Lgcd(int narg, ...)
{
	int i;
	va_list nums; object gcd, numi;

	if (narg == 0) {
		VALUES(0) = MAKE_FIXNUM(0);
		RETURN(1);
	      }
	va_start(nums, narg);
	gcd = va_arg(nums, object);
	check_type_integer(&gcd);
	if (narg == 1) {
		VALUES(0) = number_minusp(gcd) ? number_negate(gcd) : gcd;
		RETURN(1);
	      }
	for (i = 1;  i < narg;  i++) {
	  numi = va_arg(nums, object);
	  check_type_integer(&numi);
	  gcd = get_gcd(gcd, numi);
	}
	VALUES(0) = gcd;
	RETURN(1);
}

Llcm(int narg, object lcm, ...)
{
	object t, g;
	int i;
	va_list nums; object numi;

	if (narg == 0)
		FEtoo_few_arguments(&narg);
	check_type_integer(&lcm);
	va_start(nums, lcm);
	for (i = 1;  i < narg;  i++) {
	  numi = va_arg(nums, object);
	  check_type_integer(&numi);
	  t = number_times(lcm, numi);
	  g = get_gcd(lcm, numi);
	  lcm = number_divide(t, g);
	}
	VALUES(0) =  number_minusp(lcm) ? number_negate(lcm) : lcm;
	RETURN(1);
}

zero_divisor()
{
	FEerror("Zero divisor.", 0);
}

init_num_arith()
{
	make_function("+", Lplus);
	make_function("-", Lminus);
	make_function("*", Ltimes);
	make_function("/", Ldivide);
	make_function("1+", Lone_plus);
	make_function("1-", Lone_minus);
	make_function("CONJUGATE", Lconjugate);
	make_function("GCD", Lgcd);
	make_function("LCM", Llcm);
}
