;*              Copyright 1992 Digital Equipment Corporation
;*                         All Rights Reserved
;*
;* Permission to use, copy, and modify this software and its documentation is
;* hereby granted only under the following terms and conditions.  Both the
;* above copyright notice and this permission notice must appear in all copies
;* of the software, derivative works or modified versions, and any portions
;* thereof, and both notices must appear in supporting documentation.
;*
;* Users of this software agree to the terms and conditions set forth herein,
;* and hereby grant back to Digital a non-exclusive, unrestricted, royalty-free
;* right and license under any changes, enhancements or extensions made to the
;* core functions of the software, including but not limited to those affording
;* compatibility with other hardware or software environments, but excluding
;* applications which incorporate this software.  Users further agree to use
;* their best efforts to return to Digital any such changes, enhancements or
;* extensions that they make and inform Digital of noteworthy uses of this
;* software.  Correspondence should be provided to Digital at:
;* 
;*			Director, Cambridge Research Lab
;*			Digital Equipment Corp
;*			One Kendall Square, Bldg 700
;*			Cambridge MA 02139
;* 
;* This software may be distributed (but not offered for sale or transferred
;* for compensation) to third parties, provided such third parties agree to
;* abide by the terms and conditions of this notice.
;* 
;* THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
;* WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
;* MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
;* CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
;* DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;* PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
;* ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
;* SOFTWARE.

; $Id: bobsyn.scm,v 1.25 1993/01/09 15:41:15 jmiller Exp $

(define bob-binary-operators
  `((ADD 100 100)
    (AS 30 30)
    (ASSIGN 10 10)
    (BITWISE-AND 60 60)
    (BITWISE-OR 40 40)
    (DIVIDE 110 110)
    (GREATER-THAN 80 80)
    (GREATER-THAN-OR-EQUAL 80 80)
    (LEFT-SHIFT 90 90)
    (LESS-THAN 80 80)
    (LESS-THAN-OR-EQUAL 80 80)
    (LOGICAL-AND 30 30)
    (LOGICAL-EQUAL 70 70)
    (LOGICAL-OR 20 20)
    (MINUS 100 100)
    (MULTIPLY 110 110)
    (REMAINDER 110 110)
    (RIGHT-SHIFT 90 90)
    (UNEQUAL 70 70)
    (XOR 50 50)))

(define bob-unary-operators
  `((BITWISE-NEGATE 140)
    (LOGICAL-NEGATE 130)
    (MINUS 120)))

(define *lex-watch* #f)
(define lex/start 'initialized-later)
(define lex/read 'initialized-later)
(define lex/peek 'initialized-later)
(define abort 'initialized-later)
(let ((lexer #F)
      (saved-lexeme? #F)
      (saved-lexeme #F)
      (saved-offset 0))
  (set! lex/start
	(lambda (string abort-operation)
	  (set! lexer (make-bob-lexer string))
	  (set! saved-lexeme? #F)
	  (set! saved-offset 0)
	  (set! abort
		(lambda args
		  (apply abort-operation string saved-offset args)))
	  'OK))
  (set! lex/peek
	(lambda ()
	  (if saved-lexeme?
	      saved-lexeme
	      (lexer (lambda (lexeme offset error-args)
		       (if error-args
			   (apply abort 'error 'reading 'input error-args)
			   (begin
			     (set! saved-lexeme? true)
			     (set! saved-lexeme lexeme)
			     (set! saved-offset offset)
			     lexeme)))))))
  (set! lex/read
	(lambda ()
	  (let ((result (lex/peek)))
	    (set! saved-lexeme? #F)
	    (if *lex-watch* (write-line (list 'lex result)))
	    result))))

(define (bob string error-handler)
  (call-with-current-continuation
   (lambda (error-exit)
     (lex/start string (lambda args (error-exit (apply error-handler args))))
     (let ((result (bob/unit)))
       (if (token? 'end-of-file)
	   result
	   (error-exit (error-handler 'not-at-end-of-file result)))))))

(define does-not-match '(no match))
(define (match? x) (not (eq? x does-not-match)))

(define (is-token? obj)
  (and (pair? obj)
       (eq? (car obj) 'token)))

(define (token-check? name)
  ;; Never gobbles the current token
  (let ((next (lex/peek)))
    (and (is-token? next)
	 (eq? (cadr next) name))))

(define (token? name)
  ;; Gobbles the current token if it is name, else leaves it
  (and (token-check? name)
       (lex/read)))

(define (expect parser)
  (let ((result (parser)))
    (if (match? result)
	result
	(abort 'expected 'syntax parser 'got (lex/peek)))))

(define (expect-token name)
  (if (token? name)
      'OK
      (abort 'expected 'token name 'got (lex/peek))))

(define (bob/unit)
  (optionally-semicolon-separated bob/compilation-declaration)
  (let ((decls (optionally-semicolon-separated bob/declaration)))
    `(UNIT ,decls ,(semicolon-separated-exprs))))

(define (make-separated separator)
  (lambda (parser)
    (let loop ((result '()))
      (let ((next (parser)))
	(cond ((not (match? next))
	       (reverse result))
	      ((not (token? separator))
	       (reverse (cons next result)))
	      (else (loop (cons next result))))))))

(define (make-optionally-separated separator)
  (lambda (parser)
    (let loop ((result '()))
      (let ((next (parser)))
	(if (not (match? next))
	    (reverse result)
	    (begin
	      (token? separator)
	      (loop (cons next result))))))))

(define optionally-semicolon-separated
  (make-optionally-separated 'semicolon))
(define optionally-comma-separated
  (make-optionally-separated 'comma))
(define semicolon-separated (make-separated 'semicolon))
(define comma-separated (make-separated 'comma))
(define (semicolon-separated-exprs)
  (semicolon-separated (lambda () (bob/expression 0))))
(define (comma-separated-exprs)
  (comma-separated (lambda () (bob/expression 0))))

(define (bob/compilation-declaration)
  (cond ((token? 'unary)
	 (write-line (list 'unary-declarations
			   (comma-separated bob/symbol))))
	((token? 'binary)
	 (let ((names (comma-separated bob/symbol)))
	   (token? 'bind)
	   (if (token? 'like)
	       (write-line (list 'binary-declarations names 'just 'like
				 (expect bob/binop)))
	       (if (token? 'on)
		   (begin
		     (token? 'the)
		     (expect-token 'left)
		     (token? 'like)
		     (let ((left (expect bob/binop)))
		       (token? 'and)
		       (token? 'on)
		       (token? 'the)
		       (expect-token 'right)
		       (token? 'like)
		       (write-line (list 'binary-declarations names
					 'left left
					 'right (expect bob/binop)))))
		   (write-line (list 'binary-declarations names
				     'just 'like (expect bob/binop)))))))
	(else does-not-match)))

(define (bob/op? table)
  (lambda ()
    ;; Never devours the input token
    (let ((next (lex/peek)))
      (if (and (is-token? next)
	       (assq (cadr next) table))
	  (cadr next)
	  does-not-match))))

(define bob/binop? (bob/op? bob-binary-operators))
(define bob/unop? (bob/op? bob-unary-operators))

(define (bob/op op-handler)
  (lambda ()
    (let ((result (op-handler)))
      (if (match? result)
	  (begin (lex/read) result)
	  does-not-match))))
(define bob/binop (bob/op bob/binop?))
(define bob/unop (bob/op bob/unop?))

(define (unop/precedence operator)
  (cadr (assq operator bob-unary-operators)))

(define (binop/left-precedence operator)
  (cadr (assq operator bob-binary-operators)))

(define (binop/right-precedence operator)
  (caddr (assq operator bob-binary-operators)))

(define (keyword-node? obj)
  (and (pair? obj)
       (eq? (car obj) 'KEYWORD)))

(define (keyword->symbol obj)
  (let ((string (symbol->string (cadr obj))))
    (string->symbol (substring string 0 (- (string-length string) 1)))))

(define (name? obj) (symbol? obj))

(define (bob/symbol)
  (let ((name (lex/peek)))
    (if (symbol? name)
	(lex/read)
	does-not-match)))

(define (bob/name)
  (cond ((token? 'setter)
	 (expect-token 'of)
	 `(SETTER ,(expect bob/symbol)))
	((name? (lex/peek)) (lex/read))
	(else does-not-match)))

(define (bob/keyword)
  (let ((name (lex/peek)))
    (if (keyword-node? name)
	(cadr (lex/read))
	does-not-match)))

(define (bob/var)
  (define (simple)
    (let ((next (lex/peek)))
      (cond ((symbol? next)
	     `(VARIABLE ,(lex/read) <OBJECT>))
	    ((keyword-node? next)
	     (lex/read)
	     `(VARIABLE ,(keyword->symbol next)
			,(expression)))
	    (else does-not-match))))
  (let ((try (simple)))
    (cond ((match? try) try)
	  ((token? 'setter)
	   (expect-token 'of)
	   (let ((name (expect simple)))
	     `(VARIABLE (SETTER ,(list-ref name 1)) ,(list-ref name 2))))
	  (else does-not-match))))

(define (bob/simple-literal)
  (let ((next (lex/peek)))
    (cond ((or (number? next)
	       (string? next)
	       (char? next)
	       (eq? next #T)
	       (eq? next #F))
	   `(LITERAL ,(lex/read)))
	  ((keyword-node? next) (lex/read))
	  ((token? 'false) `(LITERAL #F))
	  ((token? 'true) `(LITERAL #T))
	  (else does-not-match))))

(define (bob/compound-literal)
  (let ((name (bob/symbol)))
    (cond ((match? name) `(SYMBOL ,name))
	  ((token? 'open-paren)
	   (let ((result `(CONSTANT-LIST ,@(comma-separated bob/constant))))
	     (expect-token 'close-paren)
	     result))
	  ((token? 'open-brace)
	   (let ((result `(CONSTANT-VECTOR ,@(comma-separated bob/constant))))
	     (expect-token 'close-brace)
	     result))
	  ((or (token-check? 'close-paren)
	       (token-check? 'close-brace))
	   does-not-match)
	  ((is-token? (lex/peek))
	   `(SYMBOL ,(list-ref (lex/read) 2)))
	  (else does-not-match))))

(define (bob/literal)
  (let ((simple? (bob/simple-literal)))
    (cond ((match? simple?) simple?)
	  ((token? 'quote) (bob/compound-literal))
	  (else does-not-match))))

(define (bob/constant)
  (let ((simple? (bob/simple-literal)))
    (if (match? simple?)
	simple?
	(bob/compound-literal))))

(define (bob/block)
  (define (exceptions)
    (define (exception)
      (define (get-name/expr)
	(let ((expr (expression)))
	  (if (token? 'called)
	      `(,(expect bob/name) ,expr)
	      `(#F ,expr))))
      (define (get-if)
	(if (token? 'if)
	    (expression)
	    #F))
      (token? 'and)			; Ignore if present
      (if (not (token? 'on))
	  #F
	  (let* ((name-and-expr (get-name/expr))
		 (if-clause (get-if)))
	    (expect-token 'do)
	    (let ((expr (expression+reraise)))
	      (if (token? 'described)
		  (begin
		    (expect-token 'by)
		    `(EXCEPTION ,name-and-expr ,if-clause ,expr ,(expression)))
		  `(EXCEPTION ,name-and-expr ,if-clause ,expr #F))))))
    (let loop ((result '()))
      (let ((next (exception)))
	(cond (next (loop (cons next result)))
	      ((null? result)
	       (abort 'required 'an 'exception (lex/peek)))
	      (else (reverse result))))))	    		   
  (if (not (token? 'open-brace))
      does-not-match
      (let ((declarations
	     (optionally-semicolon-separated bob/declaration-or-exit)))
	(let ((expressions (semicolon-separated-exprs)))
	  (expect-token 'close-brace)
	  (let ((exceptions (if (token? 'except) (exceptions) '())))
	    `(DECLARE ,declarations ,expressions ,exceptions))))))

(define (bob/full-declaration exit?)
  (define (separator)
    (cond ((token? 'as))
	  ((and (expect-token 'to)
		(expect-token 'be)))))
  (define (end-decl var-names other-name)
    (separator)
    `(BIND ,var-names ,other-name ,(expression)))
  (define (name->variable name) `(VARIABLE ,name <OBJECT>))
  (cond ((token? 'declare)
	 (cond ((token? 'others)
		(end-decl '() (expect bob/name)))
	       ((token? 'method)
		(let ((var (name->variable (expect bob/name))))
		  `(BIND (,var) #F ,(method-body))))
	       ((token? 'generic)
		(token? 'function)
		(let ((var (name->variable (expect bob/name))))
		  `(BIND (,var) #F ,(generic-function-body))))
	       (else
		(let ((vars (comma-separated bob/var)))
		  (end-decl vars
			    (if (token? 'others)
				(expect bob/name)
				#F))))))
	((and exit? (token? 'exit)) `(BIND-EXIT ,(expect bob/name)))
	(else does-not-match)))

(define (bob/declaration) (bob/full-declaration #F))
(define (bob/declaration-or-exit) (bob/full-declaration #T))

(define (bob/method)
  (if (not (token? 'method))
      does-not-match
      (method-body)))

(define (method-body)
  (let ((next-method #F)
	(default #F)
	(keywords '()))
    (define (bob/keyword-decl)
      (let ((internal-name #F)
	    (default-value #F))
	(if (keyword-node? (lex/peek))
	    (let ((keyword (lex/read)))
	      (if (token? 'called) (set! internal-name (expect bob/name)))
	      (if (token? 'assign) (set! default-value (expression)))
	      `(KEYWORD ,keyword ,internal-name ,default-value))
	    does-not-match)))
    (expect-token 'open-paren)
    (let ((bound-vars (comma-separated bob/var)))
      (expect-token 'close-paren)
      (if (token? 'next)
	  (begin
	    (expect-token 'method)
	    (set! next-method (expect bob/name))))
      (if (token? 'remaining)
	  (cond ((token? 'keywords)
		 (set! default (expect bob/name))
		 (set! keywords #T))
		((token? 'arguments)
		 (set! default (expect bob/name)))
		(else (abort 'expected "arguments" 'or "keywords"
			     'got (lex/peek)))))
      (if (token? 'keywords)
	  (let ((keyword-decls (comma-separated bob/keyword-decl)))
	    (set! keywords
		  (if (null? keyword-decls)
		      #T
		      keyword-decls))))
      `(METHOD ,bound-vars ,next-method ,default
	       ,keywords ,(expect bob/block)))))

(define (generic-function-body)
  (define (get-methods)
    (let loop ((result '()))
      (if (token? 'with)
	  (loop (cons (expression) result))
	  (reverse result))))
  (expect-token 'open-paren)
  (let ((bound-vars (comma-separated bob/name)))
    (expect-token 'close-paren)
    (let ((default #F)
	  (keywords '()))
      (if (token? 'remaining)
	  (cond ((token? 'keywords)
		 (set! default (expect bob/name))
		 (set! keywords #T))
		((token? 'arguments)
		 (set! default (expect bob/name)))
		(else (abort 'expected "arguments" 'or "keywords"
			     'got (lex/peek)))))
      (if (token? 'keywords)
	  (let ((keyword-list (comma-separated bob/keyword)))
	    (set! keywords
		  (if (null? keyword-list)
		      #T
		      keyword-list))))
      `(GENERIC-FUNCTION ,bound-vars ,default
			 ,keywords ,(get-methods)))))

(define (bob/generic-function)
  (if (not (token? 'generic))
      does-not-match
      (begin
	(token? 'function)		; Optional
	(generic-function-body))))

(define (bob/case)
  (define (bob/case-clause)
    (let ((cases (comma-separated-exprs)))
      (if (null? cases)
	  does-not-match
	  (begin
	    (expect-token 'case-terminator)
	    `(CASE-CLAUSE ,cases ,(expression))))))
  (if (not (token? 'case))
      does-not-match
      (let ((main-expr (expression))
	    (predicate-expr 'id?))
	(if (token? 'using)
	    (set! predicate-expr (expression)))
	(expect-token 'open-brace)
	(let* ((clauses (semicolon-separated bob/case-clause))
	       (result
		(if (token? 'others)
		    (begin
		      (expect-token 'case-terminator)
		      (let ((result
			     `(CASE ,main-expr ,predicate-expr
				    ,clauses ,(expression))))
			(token? 'semicolon)
			result))
		    `(CASE ,main-expr ,predicate-expr ,clauses #F))))
	  (expect-token 'close-brace)
	  result))))

(define (bob/do)
  (if (not (token? 'do))
      does-not-match
      (let ((iterator (bob/for-iterator)))
	(if (match? iterator)
	    `(FOR () (TEST-FIRST ,iterator ,(for-expression)))
	    (let* ((expr (for-expression))
		   (iterator (bob/for-iterator)))
	      (if (match? iterator)
		  `(FOR () (TEST-LAST ,iterator ,expr))
		  `(FOR () (FOREVER ,expr))))))))

(define (bob/for)
  (define (bob/foreach-name)
    (let* ((index-name (expect bob/name))
	   (collection-expr (begin (expect-token 'in) (expression))))
      `(FOREACH-NAME ,index-name ,collection-expr)))
  (define (bob/for-name)
    (let* ((index-name (expect bob/name))
	   (initial-expr (begin (expect-token 'assign) (expression)))
	   (by-expr (if (token? 'then) (expression) #F)))
      `(FOR-NAME ,index-name ,initial-expr ,by-expr)))
  (cond ((not (token? 'for)) does-not-match)
	((token? 'each) 
	 (let ((names (comma-separated bob/foreach-name)))
	   `(FOREACH ,names ,(for-body))))
	(else (let ((names (comma-separated bob/for-name)))
		`(FOR ,names ,(for-body))))))

(define (for-body)
  (if (token? 'do)
      (let* ((expr (for-expression))
	     (iterator (bob/for-iterator)))
	(if (match? iterator)
	    `(TEST-LAST ,iterator ,expr)
	    `(FOREVER ,expr)))
      (let ((iterator (bob/for-iterator)))
	(if (match? iterator)
	    `(TEST-FIRST ,iterator ,(for-expression))
	    (abort 'expected 'iterator 'got (lex/peek))))))

(define (for-expression)
  (let ((expr (expression)))
    (if (token? 'yielding)
	`(YIELDING ,(expression) ,expr)
	`(YIELDING (literal #F) ,expr))))

(define (bob/for-iterator)
  (cond ((token? 'while)
	 `(WHILE ,(expression)))
	((token? 'until)
	 `(UNTIL ,(expression)))
	(else does-not-match)))
	 
(define (bob/if)
  (define (then)
    (expect-token 'then)
    (let ((consequent (expression)))
      (cond ((token? 'else) `(,consequent ,(expression)))
	    ((token? 'elseif)
	     (let ((predicate (expression)))
	       `(,consequent (IF ,predicate ,@(then)))))
	    (else `(,consequent (LITERAL #F))))))
  (if (not (token? 'if))
      does-not-match
      (let ((predicate (expression)))
	`(IF ,predicate ,@(then)))))

(define (read-slot)
  (let ((gen-fn (expression))
	(setter-fn #F)
	(initialization-fn #F)
	(initial-value #F)
	(init-keyword #F)
	(required-init-keyword #F)
	(allocation #F))
    (if (token? 'set)
	(begin
	  (expect-token 'by)
	  (set! setter-fn (expression))))
    (cond ((token? 'initially)
	   (if (token? 'call)
	       (set! initialization-fn (expression))
	       (set! initial-value (expression))))
	  ((token? 'must)
	   (expect-token 'be)
	   (expect-token 'initialized)
	   (expect-token 'by)
	   (set! required-init-keyword (expect bob/keyword)))
	  ((token? 'initialized)
	   (expect-token 'by)
	   (set! init-keyword (expect bob/keyword))))
    (if (token? 'stored)
	(begin
	  (expect-token 'in)
	  (cond ((token? 'object)
		 (if (token? 'class)
		     (set! allocation 'subclass)))
		((token? 'class) (set! allocation 'class))
		((token? 'nowhere) (set! allocation 'virtual))
		(else (abort 'expected 'storage 'class 'got (lex/peek))))))
    `(SLOT ,gen-fn ,setter-fn ,initialization-fn ,initial-value
	   ,init-keyword ,required-init-keyword ,allocation)))

(define (bob/subclass)
  (define (bob/class-slot)
    (if (token? 'slot)
	(read-slot)
	does-not-match))
  (if (not (token? 'subclass))
      does-not-match
      (begin
	(expect-token 'of)
	(let ((superclasses (comma-separated-exprs)))
	  (if (token? 'with)
	      `(CLASS ,superclasses
		      ,(optionally-semicolon-separated bob/class-slot))
	      `(CLASS ,superclasses ()))))))

(define (bob/add)
  (if (not (token? 'add))
      does-not-match
      (if (token? 'slot)
	  (begin
	    (expect-token 'to)
	    (let ((class (expression)))
	      (expect-token 'read)
	      (expect-token 'by)
	      `(ADD-SLOT ,class ,(read-slot))))
	  (let ((method (expression)))
	    (expect-token 'to)
	    (if (token? 'generic) (expect-token 'function))
	    `(ADD-METHOD ,method ,(expression))))))

(define (bob/values)
  (if (not (token? 'values))
      does-not-match
      `(VALUES ,(comma-separated-exprs))))

(define (bob/reraise)
  (if (not (token? 'reraise))
      does-not-match
      `(RERAISE)))

;;;; These procedures are called after reading an expression and
;;;; detecting a special operator following it

(define (collection-selector collection)
  ;; Called after a "[" has been read in expression context.
  (let ((result `(ELEMENT ,collection ,(comma-separated-exprs))))
    (expect-token 'close-bracket)
    result))

(define (function-call function)
  ;; Called after a "(" has been read in expression context.
  (define (arg-list)
    (if (token-check? 'close-paren)
	does-not-match
	(let ((expr (expression)))
	  (cond ((or (token-check? 'comma)
		     (token-check? 'close-paren))
		 expr)
		((keyword-node? expr)
		 `(KEYWORD-ARG ,(cadr expr) ,(expression)))
		(else `(TEST-KEYWORD-ARG ,expr ,(expression)))))))
  (let ((result `(CALL ,function ,(comma-separated arg-list))))
    (expect-token 'close-paren)
    result))

(define (bob/try)
  (define (catcher)
    (let ((class (expression))
	  (condition #F)
	  (description #F)
	  (condition-name #F))
      (if (token? 'if) (set! condition (expression)))
      (if (token? 'described)
	  (begin
	    (expect-token 'by)
	    (set! description (expression))))
      (expect-token 'case-terminator)
      (if (or (token? 'using) (token? 'binding) (token? 'with))
	  (set! condition-name (expect bob/name)))
      `(CATCH ,class ,condition ,description ,condition-name
	      ,(expression))))
  (if (not (token? 'try))
      does-not-match
      (let ((body (expression)))
	(define (loop)
	  (cond ((token? 'and)
		 (expect-token 'catch)
		 (let ((this (expect catcher)))
		   (cons this (loop))))
		((token? 'catch)
		 (let ((this (expect catcher)))
		   (cons this (loop))))
		(else '())))
	`(TRY ,body ,(loop)))))

(define (bob/protect)
  (if (not (token? 'protect))
      does-not-match
      (let ((body (expression)))
	(or (token? 'by) (token? 'with) (expect-token 'unwind))
	`(PROTECT ,body ,(expression)))))

;;;; The main expression parser

(define (bob/expression precedence)
  (define (try-matches ops)
    (let loop ((ops ops))
      (if (null? ops)
	  does-not-match
	  (let ((result ((car ops))))
	    (if (not (match? result))
		(loop (cdr ops))
		result)))))
  (let loop
      ((left-expr
	(cond ((match? (bob/unop?))
	       (let ((operator (bob/unop)))
		 `(UNOP ,operator
			,(bob/require-expression
			  (unop/precedence operator)))))
	      ((token? 'open-paren)
	       (discard-char)
	       (let ((result (expression)))
		 (expect-token 'close-paren)
		 result))
	      (else (try-matches *special-forms*)))))
    (if (match? left-expr)
	(let ((operator (bob/binop?)))
	  (cond ((match? operator)
		 (let ((left (binop/left-precedence operator)))
		   (if (> precedence left)
		       left-expr
		       (let ((right-expr
			      (begin (lex/read)
				     (bob/require-expression
				      (binop/right-precedence operator)))))
			 (loop
			  (cond ((not (eq? 'assign operator))
				 `(BINOP ,operator ,left-expr
					 ,right-expr))
				((name? left-expr)
				 `(ASSIGN ,left-expr ,right-expr))
				((and (pair? left-expr)
				      (eq? (car left-expr) 'CALL)
				      (pair? (cdr left-expr))
				      (name? (list-ref left-expr 1)))
				 `(SETTER-CALL ,(list-ref left-expr 1)
					       ,(list-ref left-expr 2)
					       ,right-expr))
				((and (pair? left-expr)
				      (eq? (car left-expr) 'ELEMENT))
				 `(SETTER-CALL ELEMENT
					       ,(cons (list-ref left-expr 1)
						      (list-ref left-expr 2))
					       ,right-expr))
				(else (abort 'bad 'left 'side 'of
					     'assignment left-expr))))))))
		((token? 'open-paren)
		 (loop (function-call left-expr)))
		((token? 'open-bracket)
		 (loop (collection-selector left-expr)))
		(else left-expr)))
	left-expr)))

(define standard-special-forms
  (list bob/literal bob/name bob/block bob/method bob/case bob/do
	bob/for bob/if bob/add bob/subclass bob/values
	bob/generic-function bob/try bob/protect))

(define (bob/require-expression precedence)
  (let ((result (bob/expression precedence)))
    (if (match? result)
	result
	(abort 'required 'expression 'got (lex/peek)))))

(define (expression) (bob/require-expression 0))
 
(define (expression+reraise)
  (let ((old-forms *special-forms*))
    (if (memq bob/reraise old-forms)
	(expression)
	(begin
	  (set! *special-forms* (cons bob/reraise *special-forms*))
	  (let ((result (expression)))
	    (set! *special-forms* old-forms)
	    result)))))

(define *special-forms* standard-special-forms)

