%****************************************************************************%
%                                                                            %
%                                                                            %
%                        Environment Manipulator                             %
%                                                                            %
%                                                                            %
%                   Copyright (C) 1988,1989 Afzal Ballim                     %
%                                                                            %
%                                                                            %
%                                                                            %
%****************************************************************************%

:- dynamic obj_syn_rule/3.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This file contains code to load syntax rules into the system. The syntax
% rules are subsequently used when reading objects of the type specified
% by the rules. E.g., for loading environments and procedure rules.
% This file requires predicates defined in the file 'io.pl' which resides 
% at the top level 'EM' directory.

% Bugs: this method reads as it processes, which is nice in some ways, but
% a total disaster if back-tracking is required, because it causes Prolog
% (C Prolog anyway) to crash. Hence, any rules in the file which are not
% well-formed will cause a crash. IF PROLOG CRASHES WHILE LOADING CHECK
% THE SYNTAX DEFINITION FILES!!

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% LOADSYNTAX: Consults a file containing the syntax rules for objects that
% can be read by the loader.
%
% Args: File = name of the file to consult. By convention, this file should 
%              have a ".syn" extender. It should be arranged according to 
%              the syntax for such files. This syntax can be found in 
%              the file "README"
%
loadsyntax(File)
    :-
    file_2_buf(File,Buffer),           % fill the buffer from the file
    synload(Buffer),                   % load the syntax rules from buffer
    !,outp([nl,'Object syntax file: ',File,', consulted.',nl]).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SYNLOAD: load syntactic rules from a buffer.
% Args:  InpIn = buffer to load from
%
synload([]).
synload(InpIn)
:-
    load1rule(InpIn,InpOut),
    synload(InpOut).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% LOAD1RULE: loads one rule for object definitions, and asserts that rule
%            at the end of the data base.
% Args: InpI = buffer going in
%       InpO = buffer when finished
%
load1rule([],[]).
load1rule(InpI,InpO)
:-
    load1rname(Name,InpI,InpO1),!,
    load1rbody(Body,InpO1,InpO2),!,
    load1ract(Act,InpO2,InpO),!,
    assertrule(Name,Body,Act).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% LOAD1RNAME: reads in the name of a rule. A rule name is of the form
%             "<" name ">" where "name" is a string of letters, numbers, 
%             and underscores.
% Args: Name = the rule name
%       InpI = buffered words when commence
%       InpO = buffered words when done
%
load1rname(null,[],[]).
load1rname(Name,['<',Name,'>'|InpO],InpO).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% LOAD1RBODY: reads in the body of a rule. A rule body is a "=" followed by
%             the definition, and terminated by the start of the action list
% Args: Body = the rule body
%       InpI = buffered words when commence
%       InpO = buffered words when done
%
load1rbody(null,[],[]).
load1rbody(Body,['='|InpI],InpO)
:-
    !,buildrbody(Body,InpI,InpO,['-','-','>'|_]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% LOAD1RACT: reads in the action list for a rule. An action list is "-->"
%            followed by a sequential list of actions to be performed on the
%            local queue, terminated by a full stop
%
% Args: Act  = action list
%       InpI = buffered words when commence
%       InpO = buffered words when done
%
load1ract(null,[],[]).
load1ract(Act,['-','-','>'|InpI],InpO)
:-
    !,buildract(Act,InpI,InpO,['.'|_]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% BUILDRBODY: does the actual work of building the structure for the body
%             of a rule.
%             The components of a rule body can be...
%              1) quoted word, e.g., ',' '''' (single quoted quote), 'abc'
%              2) rule name, e.g., <word>
%              3) optional things, i.e., {...}
%              4) choices, i.e., (..|..|..|..)
%              5) basics, i.e., *STRING*, *WORD*, *SEPARATOR*, *NUMBER*
%             These components are built into a list, as follows...
%              1) is appended to the list as "literal(Word)"
%              2) appends as obj_syn_rule(NAME)
%              3) appends as opt(Option-list)
%              4) appends as choice(Choice-list)
%              5) appends as basic(Type)
%
% Args: Body  = the actual body
%       InpI  = buffered input
%       InpO  = buffered output
%       TermL = list of characters which indicate termination (+tail),
%               useful for embedded bodies, e.g., options.
%

% have reached termination indicator
buildrbody([],TermL,TermL,TermL).
buildrbody(null,[],[],_).

% quoted word
buildrbody([literal(Word)|Body],['''',Word,''''|InpI],InpO,TermL)
:-
    buildrbody(Body,InpI,InpO,TermL).

% choices within a choice section
buildrbody(Cbody,['|'|InpI],InpO,[')'|_])
:-
    buildrbody(Cbody,InpI,InpO,[')'|_]).

% syntax rule
buildrbody([obj_syn_rule(Rule)|Body],['<',Rule,'>'|InpI],InpO,TermL)
:-
    buildrbody(Body,InpI,InpO,TermL).

% optional
buildrbody([opt(OBody)|Body],['{'|InpI],InpO,TermL)
:-
    buildrbody(OBody,InpI,InpIt,['}'|_]),
    InpIt = ['}'|InpIn],!,
    buildrbody(Body,InpIn,InpO,TermL).

% choice
buildrbody([choice(CBody)|Body],['('|InpI],InpO,TermL)
:-
    buildrbody(CBody,InpI,InpIt,[')'|_]),
    InpIt = [')'|InpIn],!,
    buildrbody(Body,InpIn,InpO,TermL).

% basic
buildrbody([basic(Type)|Body],['*',Type,'*'|InpI],InpO,TermL)
:-
    buildrbody(Body,InpI,InpO,TermL).

% other, problem with this syntax
buildrbody([],InpI,InPI,_)
:-
    outp([nl,'Problem reading in body of syntax rules']),
    outp([nl,'Context: ']),outp5(InpI),outp([nl]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% BUILDRACT: does the actual work of building the structure for the act
%             of a rule.
%             The components of a rule act can be...
%              1) ^     -- return the queue, should be the last act
%              2) [n]   -- make a list of the first "n" things on the queue
%              3) [*]   -- turn the entire queue into a list
%              4) !p,n  -- form a predicate called "p" with the first n 
%                          things on the queue as arguments (if n=* then all)
%              5) !+,n  -- as !p,n except use the first thing on the queue
%                          as the predicate name, and n is from there on
%              6) @w    -- add w to the front of the queue
%              7) x<>y  -- switch the xth and yth objects in the queue
%             These components are built into a list, as follows...
%              1) is simply appended as "return"
%              2) appends as list(n)
%              3) appends as list(all)
%              4) appends as predicate(p,n)
%              5) appends as predicatehead(n)
%              6) appends as add(w)
%              7) appends as switch(x,y)
%
% Args: Act   = the actual act
%       InpI  = buffered input
%       InpO  = buffered output
%       TermL = list of characters which indicate termination (+tail),
%

% have reached termination indicator
buildract([],[TermL|L],L,[TermL|L]).
buildract(null,[],[],_).

% ^
buildract([return|Act],['^'|InpI],InpO,TermL)
:-
    buildract(Act,InpI,InpO,TermL).

% [*]
buildract([list(all)|Act],['[',*,']'|InpI],InpO,TermL)
:-
    buildract(Act,InpI,InpO,TermL).

% [n]
buildract([list(N)|Act],['[',N,']'|InpI],InpO,TermL)
:-
    buildract(Act,InpI,InpO,TermL).

% !+,* 
buildract([predicatehead(all)|Act],['!','+',',','*'|InpI],InpO,TermL)
:-
    buildract(Act,InpI,InpO,TermL).

% !+,n
buildract([predicatehead(N)|Act],['!','+',',',N|InpI],InpO,TermL)
:-
    buildract(Act,InpI,InpO,TermL).


% !p,* predicate from them all
buildract([predicate(P,all)|Act],['!',P,',','*'|InpI],InpO,TermL)
:-
    buildract(Act,InpI,InpO,TermL).

% !p,n
buildract([predicate(P,N)|Act],['!',P,',',N|InpI],InpO,TermL)
:-
    buildract(Act,InpI,InpO,TermL).

% @w
buildract([add(W)|Act],['@',W|InpI],InpO,TermL)
:-
    buildract(Act,InpI,InpO,TermL).

% x<>y
buildract([switch(X,Y)|Act],[X,'<','>',Y|InpI],InpO,TermL)
:-
    buildract(Act,InpI,InpO,TermL).


% other, problem with this syntax
buildract([],InpI,InPI,_)
:-
    outp([nl,'Problem reading in act of syntax rules']),
    outp([nl,'Context: ']),outp(InpI),outp([nl]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSERTRULE: asserts one rule for reading an object.
% Args: EOIi = last stream indicator
%       EOIo = returned stream indicator
%       Name = name of rule
%       Body = syntactic definition of the object
%       Act  = action(s) to be taken on rule satisfaction
%

% error condition
assertrule(null,_,_).

% already defined
assertrule(Name,Body,Act)
:-
    obj_syn_rule(Name,Body,Act).

% otherwise...
assertrule(Name,Body,Act)
:-
    write('-'),
    assertz(obj_syn_rule(Name,Body,Act)).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This file contains the code to load files of objects (each file must 
% contain objects of all the same type). The objects loaded from the file
% are returned as a list. Objects are loaded according to their syntax rules

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CLEAR_SYNTAX: retracts all object syntax rules
%
clear_syntax
:-
    retractall(obj_syn_rule(_,_,_)).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SYN_OBJS: lists the objects that have a defined syntax
%
syn_objs
:-
    setof(X,(B,A)^obj_syn_rule(X,B,A),L),
    outp([nl,'Defined objects are:',nl]),
    map(outp([tab(10),Y,nl]),Y,L).

syn_objs
:-
    outp([nl,'no objects have been defined',nl]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% LOAD_OBJ_FILE: consults a file containing objects of a particular type,
%                and returns the internal representation of those objects.
%
% Args: File  = the file to be read
%       ObjT  = type of objects the file contains
%       ObjL  = the list of objects read from the file
%
load_obj_file(File,ObjT,ObjL)
:-
  if(obj_syn_rule(ObjT,Body,Act),           % if the object is defined
     (file_2_buf(File,Bufi),                 % fill a buffer from the file
      load_objs(Body,Act,ObjL,Bufi,Bufo),   % load the objects
      outp([File,' consulted',nl])),
     (outp([nl,'Unknown object type: ',ObjT,nl]),
      ObjL=[])).

% for testing
load_obj_tty(ObjT,ObjL)
:-
  if(obj_syn_rule(ObjT,Body,Act),           % if the object is defined
     (inp(Bufi),
      load_objs(Body,Act,ObjL,Bufi,Bufo)),   % load the objects
     (outp([nl,'Unknown object type: ',ObjT,nl]),
      ObjL=[])).

% load from a buffer instead
load_obj_buf(ObjT,ObjL,Buf)
:-
  if(obj_syn_rule(ObjT,Body,Act),           % if the object is defined
     load_objs(Body,Act,ObjL,Buf,Bufo),   % load the objects
     (outp([nl,'Unknown object type: ',ObjT,nl]),
      ObjL=[])).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% LOAD_OBJS: loads objects of a particular type from a buffer. The objects
%            are assembled into their internal form according to the syntax
%            rules for the objects. The internal form of the loaded objects
%            is returned to a list.
%
% Args: Body  = the rule body for the object type being loaded
%       Act   = the rule action...
%       ObjL  = the returned internal forms
%       Bufi  = the buffer when starting
%       Bufo  = the buffer after wards
%

% empty buffer, so done
load_objs(_,_,[],[],[]).

load_objs(Body,Act,[Obj|ObjL],Bufi,Bufo)
:-
    load_1_obj(Body,Act,Obj,Bufi,Bufot),
    load_objs(Body,Act,ObjL,Bufot,Bufo).


% failure
%
load_objs(_,_,[],Bufi,[])
:-
    outp([nl,'Cannot load object',nl,'context is: ']),
    outp5(Bufi).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% LOAD_1_OBJ: reads 1 object of a given type from a buffer, and assembles
%             its internal form.
%
% Args: Body  = rule body
%       Act   = the rule action...
%       Obj   = the internal form of the one object read (note:a list!!!)
%       Bufi  = the buffer when starting
%       Bufo  = the buffer after wards

load_1_obj(Body,Act,Obj,Bufi,Bufo)
:-
    read_obj_body(Body,Queue,Bufi,Bufo),
    act_on_queue(Act,Queue,Obj).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% INT_LOAD1O: a variant on load_1_obj which does not report that a failure
%             to read an object has occurred. Used internally by 
%             read_obj_body to surpress spurious reporting of error messages.
% 
% Args: Body  = rule body
%       Act   = the rule action...
%       Obj   = the internal form of the one object read (note:a list!!!)
%       Bufi  = the buffer when starting
%       Bufo  = the buffer after wards

int_load1o(Body,Act,Obj,Bufi,Bufo)
:-
    read_obj_body(Body,Queue,Bufi,Bufo),
    act_on_queue(Act,Queue,Obj).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% READ_OBJ_BODY: uses the rule body to process the buffer, reading things
%                and assembling items on a queue.
%
% Args: Body  = rule body  (possibly partial rule body)
%       Queue = the used for building up the internal representation
%       Bufi  = buffer when starting
%       Bufo  = buffer when finished

% finished processing
read_obj_body([],[],Bufi,Bufi).

% is the next object the literal Word? 
% (Word is some separator, word or number)
%
read_obj_body([literal(Word)|Body],Queue,[PWord|Bufi],Bufo)
:-
    Word=PWord,
    read_obj_body(Body,Queue,Bufi,Bufo).

% is the next object a string -- a non-separator symbol
%
read_obj_body([basic('STRING')|Body],[String|Queue],[String|Bufi],Bufo)
:-
    \+is_separator(String),                     %% only non-separators
    read_obj_body(Body,Queue,Bufi,Bufo).

% a word? i.e., some non-separator/non-number symbol
%
read_obj_body([basic('WORD')|Body],[String|Queue],[String|Bufi],Bufo)
:-
    \+is_separator(String),                     %% only non-separators
    \+number(String),                           %% non-numbers
    read_obj_body(Body,Queue,Bufi,Bufo).

% a separator?
%
read_obj_body([basic('SEPARATOR')|Body],[String|Queue],[String|Bufi],Bufo)
:-
    is_separator(String),                     %% only non-separators
    read_obj_body(Body,Queue,Bufi,Bufo).

% do we have an integer number next?
%
read_obj_body([basic('NUMBER')|Body],[Number|Queue],[Number|Bufi],Bufo)
:-
    number(Number),
    read_obj_body(Body,Queue,Bufi,Bufo).

% expand an embedded rule, and see if the buffer satisfies it
%
read_obj_body([obj_syn_rule(ObjT)|Body],Queue,Bufi,Bufo)
:-
  if(obj_syn_rule(ObjT,RBody,RAct),           % if the object is defined
     (int_load1o(RBody,RAct,ObjI,Bufi,Bufot),
      read_obj_body(Body,Queue2,Bufot,Bufo),
      append(ObjI,Queue2,Queue)),
     (outp([nl,'Unknown object type: ',ObjT,nl]),
      !,fail)).

% we have reached an optional point in the rule
% can we continue without invoking the optionality?
%
read_obj_body([opt(L)|Body],Queue,Bufi,Bufo)
:-
    read_obj_body(Body,Queue,Bufi,Bufo).

% can we satisfy the option?
%
read_obj_body([opt(L)|Body],Queue,Bufi,Bufo)
:-
    read_obj_body(L,Opt,Bufi,Bufot),
    read_obj_body(Body,RQueue,Bufot,Bufo),
    append(Opt,RQueue,Queue).


% choice point, one of a number of things can be satisfied
%

% will the first choice work?
%
read_obj_body([choice([C|_])|Body],Queue,Bufi,Bufo)
:-
    read_obj_body([C],Queue1,Bufi,Bufot),
    read_obj_body(Body,Queue2,Bufot,Bufo),
    append(Queue1,Queue2,Queue).


% try the remaining  choices
%
read_obj_body([choice([_|R])|Body],Queue,Bufi,Bufo)
:-
    read_obj_body([choice(R)|Body],Queue,Bufi,Bufo).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ACT_ON_QUEUE: performs the various actions on the input queue that are 
%               dictated by an action list. Produces an  object at
%               the end, by returning the queue.
%
% Args: Act    = a list of actions to perform
%       Queue  = queue which we start with
%       Obj    = returned object

act_on_queue([],_,_).
act_on_queue([return],Queue,Queue).

act_on_queue([list(all)|Act],Queue,Obj)
:-
    act_on_queue(Act,[Queue],Obj).

act_on_queue([list(N)|Act],Queue,Obj)
:-
    number(N),
    make_list(N,Queue,QueueO),
    act_on_queue(Act,QueueO,Obj).

act_on_queue([predicate(P,all)|Act],Queue,Obj)
:-
    length(Queue,N),
    make_predicate(P,N,Queue,QueueO),
    act_on_queue(Act,QueueO,Obj).

act_on_queue([predicate(P,N)|Act],Queue,Obj)
:-
    make_predicate(P,N,Queue,QueueO),
    act_on_queue(Act,QueueO,Obj).

act_on_queue([predicatehead(all)|Act],[P|Queue],Obj)
:-
    length(Queue,N),
    make_predicate(P,N,Queue,QueueO),
    act_on_queue(Act,QueueO,Obj).

act_on_queue([predicatehead(N)|Act],[P|Queue],Obj)
:-
    make_predicate(P,N,Queue,QueueO),
    act_on_queue(Act,QueueO,Obj).

act_on_queue([switch(X,Y)|Act],Queue,Obj)
:-
    switch_posn(X,Y,Queue,QueueO),
    act_on_queue(Act,QueueO,Obj).

act_on_queue([add(W)|Act],Queue,Obj)
:-
    act_on_queue(Act,[W|Queue],Obj).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MAKE_LIST: takes a queue of objects, and makes the first N objects into a
%            list, which becomes the new head of the queue replacing those
%            N objects.
%
% Args: N  = the number of objects to make into a list
%       Li = the list to begin with
%       Lo = the list to end with
%
make_list(N,Li,Lo)
:-
    strip_list(N,Li,H,Lr),
    Lo = [H|Lr].

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% STRIP_LIST: takes of the first N objects from a list and returns them as
%             a seperate list.
%
% Args:  N  = Number of items to strip of the list (from the head onwards).
%        Li = list to begin with
%        It = items that have been stripped off
%        Lo = list left over having stripped off the rest

strip_list(0,Li,[],Li).

strip_list(N,[H|Li],[H|It],Lo)
:-
    M is N-1,
    strip_list(M,Li,It,Lo).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MAKE_PREDICATE: makes a new predicate from a list. The predicate is 
%                 returned as the head of a list, whose remainder is that 
%                 part of the original list which did not go into the making
%                 of the predicate.
%
% Args:  P  = the name of the predicate
%        N  = the number of arguments it has
%        Qi = the list to begin with
%        Qo = the predicate and remainder of the original list

make_predicate(P,N,Qi,[Pred|Qt])
:-
    make_list(N,Qi,[Args|Qt]),
    Pred =.. [P|Args].
%****************************************************************************%
%                                                                            %
%                                                                            %
%                        Environment Manipulator                             %
%                                                                            %
%                                                                            %
%                   Copyright (C) 1988,1989 Afzal Ballim                     %
%                                                                            %
%                                                                            %
%                                                                            %
%****************************************************************************%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This file contains the basic input/output facilities, to write message
% lists, and read free text from the current input stream.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% OUTP: output a message to the top level
% Args: Mlist = a list of things to be written out. They can be terms, the
%               special functions "nl" or "tab(X)", or strings.
%
% E.g., outp([nl,tab(5),'Text contains a term --> ',Term,' and a var ',V])
% where Term=ball, and where V is uninstantiated will produce something like
% 
%     Text contains a term --> ball and a var _0


outp(Mlist) :- 
	map(outp_term(Term),Term,Mlist),!.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% OUTP5: output a message to the top level, but only the first few items
% Args: Mlist = a list of things to be written out. They can be terms, the
%               special functions "nl" or "tab(X)", or strings.
outp5(Buf)
:-
    length(Buf,Len),
    if(Len < 6,
       outp(Buf),
       (Buf=[B1,B2,B3,B4,B5|_],
        outpSL([B1,B2,B3,B4,B5,'...'],' '))).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% OUTPSL: print out a list of things, separated by a particular separator. 
%
% Args: L = a list of terms
%       S = the separator

outpSL([H|T],S)
:-
    write(H),
    map((write(S),outp_term(X)),X,T),!.

outpSL([],_).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% OUTP_TERM: write out one term
% Args: Term = a variable, "nl", "tab(X)", or a term
%
outp_term(Var):- var(Var),!,write(Var).
outp_term(tab(X)):-!,tab(X).
outp_term(nl) :- !,nl.
outp_term([]) :-!.
outp_term([H|T]) :- !,write(H),write(' '),outp_term(T).
outp_term(T) :- write(T),!.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% INPUT predicates.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FILE_2_BUF: reads everything from a file into a buffer
%
% Args: File   = the name of the file to read
%       Buffer = buffer to be filled

file_2_buf(File,Buffer)
:-
    seeing(Cstream),
    see(File),
    fill_buffer(eol,Buffer),!,
    seen.

% error
file_2_buf(File,[])
:-
    outp([nl,'Error accessing the file: ',File,nl]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FILL_BUFFER: reads everything from a buffer until reach an "eof" indicator
%
% Args: Eoi    = last stream indicator read
%       Buffer = returned buffered words

fill_buffer(eof,[]).
fill_buffer(eoe,[]).
fill_buffer(_,Buffer)
:-
    skip_comments(Eoi,Line),
    fill_buffer(Eoi,BufRest),
    append(Line,BufRest,Buffer).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% INP: input a line from current stream
% Args: List = list of words read
inp(W) :- inpret(W,_).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% INPRet: input a line from the current input stream, return end code
% Args:  Word_list = a list of words and seperators read from input
%        Eoi       = indicator to show how input terminated
%
inpret(Word_list,Eoi) :-
        get0(Char1),                       % first character from stream
        skip_fluff(Char1,Char2),           % skip spaces and tabs
        inp_words(Char2,Word_list,Eoi).    % get the words

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% INP_WORDS: does the actual work of getting words from the input stream
% Args: Char1 = last character read from the stream
%       Wlist = list of words read from the stream
%       EOI   = signal to show how stream terminated

inp_words(26,[],eof) :- !.         % terminate on end of file
inp_words(10,[],eol) :- !.         % or line feed
inp_words(-1,[],eof):-!.           % or another end of file
inp_words(Char1,[Word|Words],Eoi)  % otherwise
    :- 
      inp_word(Char1,Str,C_sep),   % get a word in string form
      name(Word,Str),              % turn it into an atom
      skip_fluff(C_sep,Char2),     % skip spaces
      inp_words(Char2,Words,Eoi).  % get the rest of the words

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% INP_WORD: get one from the stream
% Args: Char1 = last non-space character read from stream
%       Str   = a string of characters
%       Char2 = next character after end of current word
%
inp_word(Char1,[],Char1)       % if Char1 is space or EOI, then terminate
 :- 
    (space_or_tab(Char1) ; end_input(Char1)),!.   

inp_word(Char1,[Char1],Char2) % terminate if Char1 is a separator
 :-
	separator(Char1),get0(Char2),!.

% otherwise, keep reading as long as we don't have a separator
inp_word(Char1,[Char1|Chars],Char2) :-
	get0(CharN),
	if(separator(CharN),
	   (Chars = [],Char2 = CharN),
           inp_word(CharN,Chars,Char2)).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SKIP_FLUFF: skips over spaces and tabs in the input stream
% Args: Char1 = last separator read from the stream
%       Char2 = returned non-space separator (=Char1 if Char1 is non-space)

skip_fluff(Char1,Char2) 
:-
    space_or_tab(Char1),!,
    get0(Nchar),
    skip_fluff(Nchar,Char2).

skip_fluff(Char1,Char1).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SPACE_OR_TAB: succeeds if given a space or tab character
space_or_tab(32) :- !. %  space 
space_or_tab(9)  :- !. %  tab 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% END_INPUT: succeeds if given a line feed, end of file, or -1
%
end_input(26) :- !.  %  end of file = ^Z into which Unix's ^D is converted 
end_input(10) :- !.  %  end of line = newline = line feed 
end_input(-1):-!.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% IS_SEPARATOR: takes a char and sees if it is a separator.
% Args: Char = a character
%
is_separator(C)
:-
    atomic(C),
    name(C,[Code]),!,	% only succeed if it is a single character
    separator(Code).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SEPARATOR: defines the valid separator characters according to their ascii
%            codes.
%
separator(123).  % {
separator(125).  % }
separator(47).  % /
separator(92).  % \
separator(126).  % ~
separator(44).   % ,
separator(59).   % ;
separator(124).  % |
separator(60).   % <
separator(62).   % >
separator(45).   % -
separator(33).   % !
separator(42).   % *
separator(61).   % =
separator(63).   % ?
separator(36).   % $
separator(58).   % :
separator(91).   % [
separator(93).   % ]
separator(64).   % @
separator(35).   % #
separator(37).   % %
separator(94).   % ^
separator(38).   % &
separator(43).   % +
separator(61).   % =
separator(40).   % (
separator(41).   % )
separator(46).   % .
separator(39).   % '
separator(34).   % "


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SKIP_COMMENTS: read over comment lines until encounter first non-comment
% Args: EOI = stream indicator to return
%       CL  = current line, i.e., non-comment
%
skip_comments(EOI,CL)
    :-
    inpret(L,Eoi),!,		% read a line
    cond([
          [ (Eoi = eof,EOI=Eoi),       % if have reached eof
	    if(L=['%'|_],	       %   and it's a comment
               CL=[],                  %   then return blank input 
               CL=L)],                 %   else return L 
          [ L= [],                     % if it's a blank line 
            skip_comments(EOI,CL)],    % skip it
          [ L=['%'|_],                 % if it begins in a comment 
            skip_comments(EOI,CL)],    % skip it
          [ true, (EOI=Eoi,CL=L)]]).   % else return that line


