/*******************************************************************
**                                                                **
**                    Genetic Library                             **
**                                                                **
**   A library of functions to give an environment for Genetic    **
**                      Algorithms.                               **
**                                                                **
**  While these functions were designed to be used in the field   **
**  of neural networks, they could easily (hopefully) be used on  **
**   other types of optimisations.  The library implements both   **
**   a 1-stage GA and a 2-stage GA. A one stage GA is a GA where  **
**   you only have one population, in a 2-stage GA you may have   **
**  several populations, and a "master" population wich controls  **
**                 the "sub" poulations.                          **
**                                                                **
**        This library has been implemented and tested on:        **
**                                                                **
**           IBM RS/6000 & AIX 3.2 & GCC 2.5.7                    **
**                                                                **
**                 PC & DOS 6.2 % MSC 7.0                         **
**                                                                **
**   This software is part of my Diplomarbeit at the University   **
**  Kassel, Researchgroup Neural Networks.  The software may be   **
**   used for academic and educational purposes.  The use in a    **
**   corporate environment depends on the written permission of   **
**   the author.  I give no warranty for the functions in this    **
**        library, although all have been tested.                 **
**                                                                **
**      You may alter the files in this library, but are not      **
**    permitted to redistribute the files under their original    **
**  name.  If you write additions to this library and would like  **
**    them to be included in future releases, contact me, I'll    **
**           include them and give you credit.                    **
**                                                                **
**      Send questions, bugs, requests for new features and       **
**                      additions to                              **
**                                                                **
**         jochenr@neuro.informatik.uni-kassel.de                 **
**                   University Kassel                            **
**                 FG Neuronale Netzwerke                         **
**                     Jochen Ruhland                             **
**                Heinrich-Plett-Str.  40                         **
**                     D-34132 Kassel                             **
**                        Germany                                 **
**                                                                **
*******************************************************************/
/*
    Output - calculate the output, based on a popelem
    error - calculate the error, based on popelem, in and out

*/

/*#ifdef GENLIB_DEBUG
#undef GENLIB_DEGUB
#endif
#define GENLIB_DEBUG 0*/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>

#include <support.h>
#include <portrnd.h>
#include <debug.h>
#include <mgenops.h>
#include <rlist.h>
#include <g0.h>

#include "vartop.h"


#if (IN_LAYER==3)&(H_LAYER==2)&(OUT_LAYER==1)
static struct n_list pre_node[NUM_OF_CON] =
{	{0,3}, {0,4}, {0,5}, {0,6}, {0,7},
    {1,3}, {1,4}, {1,5}, {1,6}, {1,7},
    {2,3}, {2,4}, {2,5}, {2,6}, {2,7},
    {3,5}, {3,6}, {3,7},
    {4,5}, {4,6}, {4,7},
    {5,7},
    {6,7}
};

#endif

/*
**  Funktionsname : cleanup_nodelist
**
**  Beschreibung  : Korrigiert die Nodeliste, wirft unbenutzte Connections
**					raus, korrigiert Genom
**  Input         : node - Zeiger auf eine Nodeliste
**					pop Zeiger auf das bestimmende genom
**  Output        : int Anzahl der Synapsen
*/
int cleanup_nodelist( struct node_list *node, popelem_f *pop )
{
    int i,conn;
    struct node_list new_node;
    popelem_f new_pop,mask;
    int nn_reach[IN_LAYER+H_LAYER+H_LAYER+OUT_LAYER];

    for( i=0; i<IN_LAYER+H_LAYER+H_LAYER; i++ )
    	nn_reach[i] = 0;
    for( ; i<IN_LAYER+H_LAYER+H_LAYER+OUT_LAYER; i++)
    	nn_reach[i] = 1;
    /* Prinzipiell sind zunaechst nur die Outputs wichtig */

    for( i=node->max_num_of_con-1; i>=0; i-- )
    {
    	int from, to;
    	from = node->nodes[i].from;
    	to = node->nodes[i].to;
    	if( nn_reach[to] )
    		nn_reach[from] = 1;
    }
    /* Wir wissen nun, welche Neuronen wichtig sind */
    for( conn=0, i=0; i<node->max_num_of_con; i++ )
    {
    	int from, to;
    	from = node->nodes[i].from;
    	to = node->nodes[i].to;
    	if( nn_reach[to] )
    	{
    		/* Die Synapse ist wichtig */
    		new_node.nodes[conn].from = from;
    		new_node.nodes[conn].to = to;
    		conn++;
    	}
    }
    new_node.max_num_of_con = conn;
    /* Diese Liste enthaelt nur noch die wichtigen Synapsen */
    for( conn=0, i=0, mask=1, new_pop=0;
    				i<NUM_OF_CON; i++,mask<<=1 )
    {
    	if( i>=new_node.max_num_of_con )
    		break; /* Keine weiteren Synapsen da */
    	if( pre_node[i].from==new_node.nodes[conn].from &&
    	 pre_node[i].to==new_node.nodes[conn].to )
    	{
    		/* Wir haben eine Synapse gefunden */
    		conn++; /* Die naechste testen */
    		new_pop |= mask; /*Im Genom verewigen */
    	}
    }
    /* In new_pop steht nun das korrigierte Genom */
    *node = new_node; /* Die Nodelist uebertragen */
    *pop = new_pop;
    return new_node.max_num_of_con;
}

/*
**  Funktionsname : make_node_list
**
**  Beschreibung  : Erstellt aus der Genom-Information die Liste der
                	zu dieser Information gehrigen Node_liste
**  Input         : node Zeiger auf den Zeiger auf die Nodeliste
                	pop Zeiger auf die Populationsinformation
**  Output        : Wieviel Bytes werden fr das Genom bentigt ?
*/
int make_node_list(struct node_list **node, popelem_f *pop )
{
    popelem_f mask;
    float p_of_more_conns, error_now;
    struct node_list *my_node;
    int i, max,extraconns;
    DEBUG_call(make_node_list);
    assert( node != NULL );
    assert( pop != NULL );
    my_node = (struct node_list *)malloc( sizeof(struct node_list) );
    assert( my_node != NULL );

    /* Bei einem zu grossem Fehler werden zustzliche connections
       erzeugt. */
    /* die Anzahl von Synapsen die hinzukommen ist immer
       NUM_OF_CON * p_of_more_con */
    error_now = get_rlist_meta()[0].pop_error;
    if( error_now < 0.5 )
    {
        p_of_more_conns = 0.0;
    }
    else if( error_now < 1.5 )
    {
        /* Drei mehr hinzu */
        p_of_more_conns = 3.0 / (float)NUM_OF_CON;
    }
    else if (error_now < 2.5 )
    {
        /* Sechs hinzu */
        p_of_more_conns = 6.0 / (float)NUM_OF_CON;
    }
    else
    {
        /* Acht hinzu */
        p_of_more_conns = 8.0 / (float)NUM_OF_CON;
    }

    do
    {
        for( extraconns=i=max=0, mask=1; i<NUM_OF_CON; i++,mask<<=1 )
    	    if( (*pop)&mask )
    	    {
    		    /* In der Population ist die Synapse da */
    		    my_node->nodes[max]=pre_node[i];
    		    max++;
    	    }
    	    else if( p_of_more_conns > frand16() )
    	    {
    		    /* Die Synapse kommt hinzu */
    		    (*pop) |= mask;
    		    my_node->nodes[max]=pre_node[i];
    		    max++;
    		    extraconns ++;
    	    }
    	my_node->max_num_of_con = max;
/*		printf( "Orig : %d ",max );*/
    	max = cleanup_nodelist( my_node,pop );
/*		printf( " Clean :%d\n",max );*/

        if( max < 2 )
        {
    		/* Problemspezifisch wissen wir, dass es zwei relevante
    		   Eingaenge gibt, wir brauchen also mindestens zwei
    		   Synapsen */
    	    /* Das Genom nochmals mit Zufaelligen Werten besetzen */
/*			printf( "*" );*/
    	    __randomize_popelem( (USHORT *)pop, sizeof(*pop) );
        }
    } while (max<2);
    *node = my_node;
    /* Wir bentigen eine genom fr max Synapsen, fr jede Synapse */
    /* zwei Bytes, also Rckgabe 2*max */
    max*=2;
/*    DEBUGr_int(max);
    DEBUGr_int(extraconns);*/
    return max;
}

/*
**  Funktionsname : destroy_node_list
**
**  Beschreibung  : Lst die Nodelist-Struktur wieder auf
**  Input         : node Zeiger auf diese Nodeliste
**  Output        : node
*/
void destroy_node_list( struct node_list *node )
{
    DEBUG_call(destroy_node_list);
    assert(node!=NULL);
    free( node );
}


/*
**  Funktionsname : output
**
**  Beschreibung  : Errechnet die Outputs fr ein Genom bei gegebenem
    	    Input und Node Liste
**  Input	  : p Zeiger auf Gewichte
    	    in Zeiger auf EIngaben
    	    out Zeiger auf Ausgaben
    	    n Zeiger auf Nodeliste
**  Output	  : nope
*/
void output( popelem *p, float *in, float *out, struct node_list *n )
{
    float nn[IN_LAYER+H_LAYER+H_LAYER+OUT_LAYER];
    int i,j;
    int l_ready; /* Das letzte schon fertige Neuron */
/*		DEBUG_call(output);*/
    DEBUG_pointer(p);
    DEBUG_pointer(in);
    DEBUG_pointer(out);
    DEBUG_pointer(n);
    assert(p!=NULL);
    assert(in!=NULL);
    assert(out!=NULL);
    assert(n!=NULL);

    for( i=0; i<IN_LAYER; i++ )
    	nn[i] = in[i];
    for( ; i<IN_LAYER+H_LAYER+H_LAYER+OUT_LAYER; i++ )
    	nn[i] = 0.0F;
    l_ready = IN_LAYER;
    /* l_ready ist der Index des ersten Neurons, da noch nicht mit
       s() behandelt wurde */
    /* Zunchst die einzelnen Synapsen durchgehen */
    for( j=0; j < n->max_num_of_con; j++ )
    {
        /* Die einzelnen Synapsen durchgehen, nach Liste */
        #define from (n->nodes[j].from)
        #define to (n->nodes[j].to)
        if( from>=l_ready )
        {
    		/* Wir brauchen den Wert des From Neurons */
    		/* und gerade dieses mu mit s() behandelt werden, evtl auch
    		   noch Neuronen davor */
    		for( ; l_ready <= from; l_ready++ )
    			nn[l_ready] = s( nn[l_ready] );
        }
        nn[to]+=(((float)p[j]/32768.0F*(float)MAX_SYN_VALUE))
    		*nn[from];
        #undef from
        #undef to
    }
    /* Nun noch die Ausgabewerte rausfischen */
    for( i=0; i < OUT_LAYER; i++ )
    	out[i]	= s(nn[i + IN_LAYER+H_LAYER+H_LAYER]);
    /* Die Output Neuronen knnen niemals in der from-Liste auftauchen */
}

/*
**  Funktionsname : error
**
**  Beschreibung  : Fehler fr ein Target im 2.GA
**  Input	  : p Zeiger auf Populationsinfo
    	    in Zeiger auf Eingabedaten
    	    out Zeiger auf erwarteten Output
    	    n Zeiger auf Verbindungsliste
**  Output	  : der Fehler
*/
float error(popelem *p, float *in, float *out, struct node_list	*n)
{
    double err,ze;
    int i;
    float nn_out[OUT_LAYER];
/*	DEBUG_call(error);*/
    DEBUG_pointer(p);
    DEBUG_pointer(in);
    DEBUG_pointer(out);
    DEBUG_pointer(n);
    assert(p!=NULL);
    assert(in!=NULL);
    assert(out!=NULL);
    assert(n!=NULL);
    /* Wird von output mit den einzelnen Out's gefllt */
    output( p, in, nn_out, n );

    for( err=0.0F, i=0; i < OUT_LAYER; i++ )
    {
    	if( (ze=fabs( (double)out[i] - (double)nn_out[i] )) > 0.5 )
    		ze += 0.5;
    	err += ze;
    }
    return (float)err;
}

/*
**  Funktionsname : ges_error
**
**  Beschreibung  : Gesamter Fehler fr alle Targets im 2.GA
**  Input	  : p Zeiger auf Populationsinfo
    	    t Zeiger auf Targets
    	    n Zeiger auf Spareinfo - hier Nodeliste
**  Output	  : der aufsummerte Fehler
*/
extern float ges_error(popelem *p, struct target *t, struct node_list *n )
{
    float ges;
    int i;
/*	DEBUG_call(ges_error);*/
    DEBUG_pointer(p);
    DEBUG_pointer(t);
    DEBUG_pointer(n);
    assert(p!=NULL);
    assert(t!=NULL);
    assert(n!=NULL);
    for( ges=0.0F,i=0; i < N_TARGET; i++ )
    	ges += error( p, t[i].input, t[i].output, n );
    return ges;
}

