/* ---------------------------------------------------------- 
%   (C)1993 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <klic/config.h>
#include "klic.h"
#ifdef USELOCKF
#include <fcntl.h>
#endif

extern void *malloc();

char *dbdir = 0;
char *initdbdir = KLICLIB;
int nocfiles = 0;

struct atomrec {
  struct atomrec *next;
  int id;
  char *name;
} *atomroot;
int nextatom;

struct functrec {
  struct functrec *next;
  int id;
  struct atomrec *principal_functor;
  int arity;
} *functroot;
int nextfunct;

struct predrec {
  struct predrec *next;
  char *name;
} *predroot;
int nextpred;

int changes;

static void error_exit(format, arg1, arg2)
     char *format, *arg1, *arg2;
{
  (void)fprintf(stderr, format, arg1, arg2);
  putc('\n', stderr);
  exit(-1);
}

usage_error(command)
     char *command;
{
  error_exit("Usage: %s [-n] [-x dbdir] [-X initdbdir] file ...", command, 0);
}

static struct atomrec *enter_atom(name)
     char *name;
{
  struct atomrec *a;
  int len = strlen(name);
  for (a=atomroot; a!=0; a=a->next) {
    if (strcmp(name,a->name) == 0) return a;
  }
  a = (struct atomrec *)malloc(sizeof(struct atomrec));
  a->name = strcpy((char *)malloc(len+1),name);
  a->next = atomroot;
  a->id = nextatom++;
  atomroot = a;
  changes = 1;
  return a;
}

static void enter_functor(name)
     char *name;
{
  struct functrec *f;
  char *p;
  struct atomrec *functor;
  int arity;

  for (p=name+strlen(name)-1; *p!='_'; p--)
    ;
  *p = 0;
  arity = atoi(p+1);
  functor = enter_atom(name);
  for (f=functroot; f!=0; f=f->next) {
    if (f->principal_functor == functor &&
	f->arity == arity)
      return;
  }
  f = (struct functrec *)malloc(sizeof(struct functrec));
  f->principal_functor = functor;
  f->arity = arity;
  f->next = functroot;
  f->id = nextfunct++;
  functroot = f;
  changes = 1;
}

static void enter_predicate(name)
     char *name;
{
  struct predrec *p;
  int len = strlen(name);

  for (p=predroot; p!=0; p=p->next) {
    if (strcmp(name,p->name)==0) return;
  }

  p = (struct predrec *)malloc(sizeof(struct predrec));
  p->name = strcpy((char *)malloc(len+1),name);
  p->next = predroot;
  nextpred++;
  predroot = p;
  changes = 1;
}

static void enter_one_line(buf, pathname)
     char *buf;
     char *pathname;
{
  if (buf[0] == '#') {
    return;			/* comment line */
  } else if (strncmp(buf,"atom_", 5) == 0) {
    (void) enter_atom(buf+5);
  } else if (strncmp(buf,"functor_",8)==0) {
    enter_functor(buf+8);
  } else if (strncmp(buf,"predicate_",10)==0) {
    enter_predicate(buf+10);
  } else if (strncmp(buf,"module_",7)==0) {
    /* Currently module entries are not looked into */
  } else {
    error_exit("Unrecognized line in file %s:\n%s",
	       pathname, buf);
  }
}

static void read_db_file(file, pathname)
     FILE *file;
     char *pathname;
{
  char buf[BUFSIZE];
  while (fgets(buf,BUFSIZE,file) != 0) {
    *strchr(buf,'\n') = 0;
    enter_one_line(buf, pathname);
  }
  (void)fclose(file);
}

static void read_ext_file(pathname)
     char *pathname;
{
  FILE *file;

  file = fopen(pathname, "r");
  if (file==0) error_exit("Cannot open file %s", pathname, 0);
  read_db_file(file, pathname);
}

static char *make_path(dir, name)
     char *dir;
     char *name;
{
  char *path;
  if (dir != 0) {
    path = (char *)malloc(strlen(dir)+strlen(name)+2);
    (void)sprintf(path, "%s/%s", dir, name);
  } else {
    path = name;
  }
  return path;
}

static FILE *open_cwd_out(name)
     char *name;
{
  FILE *file;
  if ((file=fopen(name, "w")) == 0) {
    error_exit("Cannot open file %s", name, 0);
  }
  return file;
}

static FILE *open_db_out(name)
     char *name;
{
  FILE *file;
  char *path = make_path(dbdir, name);
  if ((file=fopen(path, "w")) == 0) {
    error_exit("Cannot open file %s", path, 0);
  }
  return file;
}

static void reverse_atomrecs()
{
  struct atomrec
    *next = atomroot,
    *x = 0;
  while (next != 0) {
    struct atomrec *last = x;
    x = next;
    next = x->next;
    x->next = last;
  }
  atomroot = x;
}

static void reverse_functrecs()
{
  struct functrec
    *next = functroot,
    *x = 0;
  while (next != 0) {
    struct functrec *last = x;
    x = next;
    next = x->next;
    x->next = last;
  }
  functroot = x;
}

static void reverse_predrecs()
{
  struct predrec
    *next = predroot,
    *x = 0;
  while (next != 0) {
    struct predrec *last = x;
    x = next;
    next = x->next;
    x->next = last;
  }
  predroot = x;
}

static int hexvalue(c, name)
     int c;
     char *name;
{
  if ('0' <= c && c <= '9') return c - '0';
  if ('A' <= c && c <= 'F') return c - 'A' + 10;
  if ('a' <= c && c <= 'f') return c - 'a' + 10;
  error_exit("Invalid atom name: %s", name, 0);
}

static void real_atom_name(name,q)
     char *name, *q;
{
  char *p = name;
  while (*p != 0) {
    if (*p == '_') {
      int c = *++p;
      if (c == '_') {
	*q = '_';
      } else {
	int n = (hexvalue(c,name) << 4);
	c = *++p;
	*q = n+hexvalue(c,name);
      }
    } else {
      *q = *p;
    }
    p++; q++;
  }
  *q = 0;
}

static void print_c_string(file, str)
     FILE *file;
     char *str;
{
  int c;
  (void)putc('\"', file);
  while ((c=*str) != 0) {
    if (c=='\\' || c=='"') (void)putc('\\', file);
    (void)putc(c, file);
    str++;
  }
  (void)putc('\"', file);
}

static write_db_files(inittime)
     time_t inittime;
{
  FILE *atomh, *atomc, *functh, *functc, *predc, *klicdb;
  struct atomrec *a;
  struct functrec *f;
  struct predrec *p;

  klicdb = open_db_out("klic.db");
  (void)fprintf(klicdb, "#KLIC DB for system @%ld\n", inittime);

  reverse_atomrecs();
  if (!nocfiles) {
    atomh = open_cwd_out("atom.h");
    atomc = open_db_out("atom.c");
    (void)fprintf(atomh, "#include <klic/atomstuffs.h>\n");
    (void)fprintf(atomc, "char *init_atomname[] = {\n");
  }
  for (a=atomroot; a!=0; a=a->next) {
    char realname[BUFSIZE];
    real_atom_name(a->name,realname);
    if (!nocfiles) {
      (void)fprintf(atomh, "#define atom_%s %dL+ATOMNUMBERBASE\n",
		    a->name, a->id);
      (void)putc('\t', atomc);
      print_c_string(atomc, realname);
      (void)putc(',', atomc);
      (void)putc('\n', atomc);
    }
    (void)fprintf(klicdb, "atom_%s\n", a->name);
  }
  if (!nocfiles) {
    (void)fprintf(atomc, "};\n");
    (void)fprintf(atomc, "unsigned long initial_atoms = %d;\n", nextatom);
    (void)fprintf(atomc, "char** atomname = init_atomname;\n");
    (void)fclose(atomh);
    (void)fclose(atomc);
  }

  reverse_functrecs();
  if (!nocfiles) {
    functh = open_cwd_out("funct.h");
    functc = open_db_out("funct.c");
    (void)fprintf(functh, "#include <klic/functorstuffs.h>\n\n");
    (void)fprintf(functc, "#include <klic/atomstuffs.h>\n\n");
    (void)fprintf(functc, "unsigned long init_functors[] = {\n");
  }
  for (f=functroot; f!=0; f=f->next) {
    if (!nocfiles) {
      (void)fprintf(functh, "#define functor_%s_%d\t%dL+FUNCTORNUMBERBASE\n",
		    f->principal_functor->name, f->arity,
		    f->id);
      (void)fprintf(functc, "\t%dL+ATOMNUMBERBASE,\n",
		    f->principal_functor->id);
    }
    (void)fprintf(klicdb, "functor_%s_%d\n",
		  f->principal_functor->name, f->arity);
  }
  if (!nocfiles) {
    (void)fprintf(functc, "};\n\nunsigned long init_arities[] = {\n");
    for (f=functroot; f!=0; f=f->next) {
      (void)fprintf(functc, "\t%dL,\n", f->arity);
    }
    (void)fprintf(functc, "};\n");
    (void)fprintf(functc, "unsigned long initial_functors = %d;\n", nextfunct);
    (void)fprintf(functc, "unsigned long *functors = init_functors;\n");
    (void)fprintf(functc, "unsigned long *arities = init_arities;\n");
    (void)fclose(functh);
    (void)fclose(functc);
  }

  reverse_predrecs();
  if (!nocfiles) {
    predc = open_db_out("predicates.c");
    (void)fprintf(predc, "char *init_predicates[] = {\n");
  }
  for (p=predroot; p!=0; p=p->next) {
    if (!nocfiles) {
      (void)fprintf(predc, "\t\"predicate_%s\",\n", p->name);
    }
    (void)fprintf(klicdb, "predicate_%s\n", p->name);
  }
  if (!nocfiles) {
    (void)fprintf(predc, "};\n");
    (void)fprintf(predc, "unsigned long initial_predicates = %d;\n", nextpred);
    (void)fprintf(predc, "char **predicates = init_predicates;\n");
    (void)fclose(predc);
  }
}

from_same_installation(klicdb, inittime)
     FILE *klicdb;
     time_t inittime;
{
  int c;
  time_t dbtime;
  if (fgetc(klicdb) != '#') return 0;
  do {
    c = fgetc(klicdb);
    if (c == '\n' || c == EOF) {
      return 0;
    }
  } while (c != '@');
  if (fscanf(klicdb, "%ld", &dbtime) != 1) return 0;
  if (dbtime != inittime) return 0;
  rewind(klicdb);
  return 1;
}

#define Optarg() \
( argv[optind][charind+1] != 0 ? \
  argv[optind]+charind+1 : \
  (optind++, argv[optind] ))

main(argc,argv)
     int argc;
     char **argv;
{
  int optind;
  char *dbpath, *initpath;
  FILE *klicdb;
  int n;
  int c;
  struct stat initstat;
#ifdef USELOCKF
  int fd;
  char fdbuf[BUFSIZE];
#endif
  for (optind = 1;
       optind<argc && argv[optind][0]=='-';
       optind++) {
    int charind;
    for (charind = 1;
	 argv[optind][charind] != 0;
	 charind++) {
      switch (c = argv[optind][charind]) {
      case 'X': initdbdir = Optarg(); goto nextarg;
      case 'x': dbdir = Optarg(); goto nextarg;
      case 'n': nocfiles = 1; break;
      default: usage_error(argv[0]);
      }
    }
  nextarg:;
  }
  atomroot = 0; nextatom = 0;
  functroot = 0; nextfunct = 0;
  predroot = 0; nextpred = 0;

  initpath = make_path(initdbdir, "klicdb.init");
  dbpath = make_path(dbdir, DBFILENAME);

  if (stat(initpath, &initstat) != 0) {
    error_exit("Can't access file: %s", initpath);
  }

#ifdef USELOCKF
  strcpy(fdbuf, dbpath);
  strcat(fdbuf, ".lock");
  lockf(fd=open(fdbuf,O_RDWR), F_LOCK, 1);
#endif
  klicdb = fopen(dbpath, "r");
  if (klicdb != 0 &&
      from_same_installation(klicdb, initstat.st_mtime)) {
    read_db_file(klicdb, dbpath);
    changes = 0;
  } else {
    FILE *klicinit = fopen(initpath, "r");
    if (klicinit == 0) {
      error_exit("Can't open initial database file: %s", initpath, 0);
    }
    read_db_file(klicinit, initpath);
    changes = 1;
  }

  for (n=optind; n<argc; n++) {
    read_ext_file(argv[n]);
  }
  if (!changes && !nocfiles) {
    struct stat dbst, atomhst, atomcst, functhst, functcst, predcst;
    if (stat(dbpath, &dbst) != 0 ||
	stat("atom.h", &atomhst) != 0 ||
	dbst.st_mtime > atomhst.st_mtime ||
	stat("funct.h", &functhst) != 0 ||
	dbst.st_mtime > functhst.st_mtime ||
	stat(make_path(dbdir, "atom.c"), &atomcst) != 0 ||
	dbst.st_mtime > atomcst.st_mtime ||
	stat(make_path(dbdir, "funct.c"), &functcst) != 0 ||
	dbst.st_mtime > functcst.st_mtime ||
	stat(make_path(dbdir, "predicates.c"), &predcst) != 0 ||
	dbst.st_mtime > predcst.st_mtime) {
      changes = 1;
    }
  }
  if (changes) {
    write_db_files(initstat.st_mtime);
  }
#ifdef USELOCKF
  close(fd);
  lockf(fd,F_ULOCK,1);
#endif
  return 0;
}
