function [h,a] = musicsp(rx,N,arg3,arg4,arg5)
%MUSICSP MUltiple SIgnal Classification spectrum estimation.
%       [H,A]=MUSICSP(Rx,n) computes the transfer function denominator
%	polynomial of the spectral estimate of the signal whose
%       correlation matrix is given in Rx.  Rx must be a square
%       matrix.  Functions assumes the number of complex sinusoids 
%	contained in the signal is one less than the rank of 
%	correlation matrix Rx.  Return argument A is the "root"
%	MUSIC polynomial.
%	
%	Note: A real sinusoidal signal contains two complex
%	sinusoidals.
%
%       [H,A]=MUSICSP(Rx,N,n) forces algorithm to assume there
%       are N complex sinusoidal signals.  N must be at least one 
%	less than the rank correlation matrix Rx.
%
%       [H,A]=MUSICSP(X,P,N,n) computes the transfer function denominator
%	polynomial of the spectral estimate of the signal X.  The
%	function computes a PxP correlation matrix using the 
%	'rxxcovar' function and assumes P-1 complex sinusoids.
%
%       [H,A]=MUSICSP(X,P,N,n,'rxxmethod') computes the PxP 
%       correlation matrix using the method prescribed by 
%       'RXXMETHOD'.  Valid options are:
%           'rxxcorr'  - AR auto-correllation estimate method.
%           'rxxcovar' - AR covariance estimate method.
%           'rxxmdcov' - AR modified-covariance estimate method.
%       The 'rxxcovar' option is the default if no method is 
%       given.
%
%       See also RXXTRUE, RXXCORR, RXXCOVAR, RXXMDCOV

%       Dennis W. Brown 4-27-94
%       Naval Postgraduate School, Monterey, CA
%       May be freely distributed.
%       Not for use in commercial products.

% check args
if nargin < 1 | nargin > 5,
    error('musicsp: Invalid number of input arguments...');
end;

% if input is a vector, estimate the correlation matrix.
if min(size(rx)) == 1,

	if nargin < 4,
		error('musicsp: Invalid number of input arguments...');
	end;

	% figure out if we have a vector
	if max(size(rx)) == 1,
		error('ar_corr: Input arg "x" must be a 1xN or Nx1 vector.');
	end;

	% work with Nx1 vectors
	rx = rx(:);
	method = 'rxxcovar';

	% got to have at least 3 args for this use
	if nargin == 1,
		error('musicsp: Invalid number of input arguments.');
	end;

	% second arg is P
	P = N;

	% third arg is N 
	N = arg3;

	% fourth is number of points
	n = arg4;

	if nargin == 5,
		if ~isstr(arg5),
			error('musicsp: Correlation method must be a string...');
		else,
			method = arg5;
		end;
	end;
	
	% estimate correlation matrix
	if strcmp(method,'rxxcorr'),
		rx = rxxcorr(rx,P);
	elseif strcmp(method,'rxxcovar'),
		rx = rxxcovar(rx,P);
	elseif strcmp(method,'rxxmdcov'),
		rx = rxxmdcov(rx,P);
	else,
		error('musicsp: Invalid correlation estimation method specified...');
	end;

else,

	% more arg checking
	[P,nn] = size(rx);
	if P ~= nn,
		error('musicsp: Rx must be a square matrix...');
	end;

	% m equals rank of Rx (Rx assumed to be singular)

	if nargin == 2,
		if N > m-1,
			error('musicsp: N must be less the rank of Rx...');
		end;
	end;
end;

% compute eigenvalues and vectors and sort in descending order
[u,s,v] = svd(rx);

% noise subspace
E = v(:,N+1:P);

% form projection matrix
p = E * E';


[h,a] = freqeig(p,n);
