;;; -*- Scheme -*-

;;;; These are the definitions mentioned in the text of the Problem Set 2
;;;; handout:

(define (expt b n)
  (if (= n 0)
      1
      (* b (expt b (- n 1)))))

;;; Similarly, we could also make a procedure that multiplies a number
;;; by an integer by repeatedly adding:

(define (mult b n)
  (if (= n 0)
      0
      (+ b (mult b (- n 1)))))

;;; These procedures are both instances of the same plan; let's give it
;;; a name:

(define (n-times b n f id)
  (if (= n 0)
      id
      (f b (n-times b (- n 1) f id))))

;;; Unary functions have many of the properties of numbers.  For
;;; example, one can combine functions by composition.  The following
;;; procedure, named compose, will produce a procedure that computes the
;;; composition of the functions computed by its arguments, f and g.

(define (compose f g)
  (lambda (x)
    (f (g x))))

;;; We can also define an identity among the functions -- the function
;;; whose value is its argument.  The procedure below implements the
;;; identity function.

(define (identity x) x)

;;; Repeated composition of functions is such a common usage that it is
;;; helpful to introduce a specific procedure for expressing it,
;;; analogous to exponentiation and multiplication:

(define (repeatedly fn n)
  (n-times fn n compose identity))

;;; Section 1.2.4 of the text also describes a technique for speeding up
;;; the operation of exponentiation by using a ``divide and conquer''
;;; algorithm that cuts the problem size in half when possible:

(define (fast-exp b n)
  (cond ((= n 0) 1)
        ((even? n) (square (fast-exp b (/ n 2))))
        (else (* b (fast-exp b (- n 1))))))

;;; We can try to apply the same idea more generally to n-times:

(define (fast-n-times b n f id)
  (cond ((= n 0) id)
        ((even? n)
         ((lambda (x) (f x x))
          (fast-n-times b (/ n 2) f id)))
        (else
         (f b (fast-n-times b (- n 1) f id)))))

;;; Now we can use this procedural abstraction to define fast-exp:

(define (fast-exp b n)
  (fast-n-times b n * 1))

;;; Just as before, we may also apply the same idea to function
;;; composition:

(define (fast-repeatedly fn n)
  (fast-n-times fn n compose identity))

;;; Then, we may revisit the idea of defining expt in terms of this
;;; fast-repeatedly:

(define (fast-rep-exp b n)
  ((fast-repeatedly (lambda (x) (* b x))
                    n)
   1))

;;; To see how often the fundamental multiply step is performed by
;;; fast-exp and fast-rep-exp, we change them slightly, introducing a
;;; version of multiply that prints an asterisk every time it is used.
;;; In the spirit of higher-order procedures, we may define the procedure
;;; marking to yield a procedure that applies marking's argument to two
;;; arguments and also prints the asterisk.

(define (marking f)
  (lambda (a b)
    (princ "*")
    (f a b)))

;;; To instrument our procedures, we then re-define fast-exp and
;;; fast-reg-exp to use (marking *) instead of *.

(define (fast-exp b n)
  (fast-n-times b n (marking *) 1))

(define (fast-rep-exp b n)
  ((fast-repeatedly (lambda (x) ((marking *) b x))
                    n)
   1))


;;; Turtle Geometry

(define (line rect)
  (draw-line rect -1 0 1 0))

(define (half-line rect)
  (draw-line rect -.5 0 .5 0))

(define (rotate picture angle)
  (lambda (rect)
    (picture (rotated rect angle))))

(define (translate picture x y)
  (lambda (rect)
    (picture (translated rect x y))))

(define (scale picture factor)
  (lambda (rect)
    (picture (scaled rect factor))))

(define (superpose p1 p2)
  (lambda (rect)
    (p1 rect)
    (p2 rect)))

(define (turtle-gosper length n)
  (clear-graphics)
  (reset-turtle! -0.5 0 0)  ;; start turtle at (-.5,0) heading right
  (tosper length n))

(define (tosper length n)
  (cond ((zero? n) (forward length))
        (else
         (left +pi/4)
         (tosper (* length sqrt2/2) (- n 1))
         (right +pi/2)
         (tosper (* length sqrt2/2) (- n 1))
         (left +pi/4))))

(define (gosper-improve picture)
  (let ((p (scale picture sqrt2/2)))
    (superpose (translate (rotate p +pi/4)
                          -.25
                          .25)
               (translate (rotate p -pi/4)
                          .25
                          .25))))

(define (turtle-gosper3 length n)
  (clear-graphics)
  (reset-turtle! -0.5 0 0)  ;; start turtle at (-.5,0) heading right
  (tosper3 length n))

(define (tosper3 length n)
  (cond ((zero? n) (forward length))
        (else
         (left (- +pi/2 +pi/6))
         (tosper3 (* length 0.5) (- n 1))
         (right +pi/3)
         (tosper3 (* length 0.5) (- n 1))
         (right +pi/3)
         (tosper3 (* length 0.5) (- n 1))
         (left (- +pi/2 +pi/6)))))

(define (generalized-turtle-gosper length sides levels)
  (clear-graphics)
  (reset-turtle! -.5 0 0)
  (tc length sides levels))