(define (try example the-rules)
  (derive (cadr example) (car example) the-rules))

(define (derive goal given-facts the-rules)
  (define (loop new-facts old-facts)
    (let ((result (take-a-step new-facts old-facts the-rules)))
      (let ((nnfacts (car result))
            (nofacts (cadr result)))
        (cond ((null? nnfacts)
               (append nofacts '(stumped)))
              ((member goal nnfacts)
               (append nofacts (list goal) '(qed)))
              (else 
                (loop nnfacts nofacts))))))
  (if (member goal given-facts)
      '(trivial)
      (loop given-facts '())))

;;; The strategy we apply is to keep trying to take a step until
;;; either no new facts are derived, or we find the goal.  Taking a
;;; step means trying to apply each rule by matching its antecedents
;;; against only those known facts that were derived in the last step.
;;; This prevents one of the more obvious ways of deriving redundant
;;; information.

(define (take-a-step new-facts old-facts the-rules)
  (let ((all-facts (append old-facts new-facts)))
    (let ((nnfacts (mapa (lambda (fact)
			   (try-new-fact fact all-facts the-rules))
                        new-facts)))
      (let ((unfacts (uniquefy nnfacts)))
	(list unfacts all-facts)))))

(define (try-new-fact fact all-facts the-rules)
  (mapa (lambda (rule) (try-rule-against-fact rule fact all-facts))
        the-rules))

(define (try-rule-against-fact rule fact all-facts)
  (mapa (lambda (clause) 
          (try-rest-of-rule rule
                            clause
                            (match clause fact the-empty-dictionary)
                            all-facts))
        (antecedents rule)))

(define (try-rest-of-rule rule clause dict all-facts)
  (if (failed? dict)
      '()
      (mapa (lambda (ndict) (derive-rule-conclusion rule ndict all-facts))
            (match-rest (delete clause (antecedents rule))
                        dict
                        all-facts))))

(define (derive-rule-conclusion rule dict facts)
  (let ((new (instantiate (consequent rule) dict)))
    (if (member new facts)
        '()
        (list new))))
 
(define (match-rest as dict facts)
  (if (null? as)
      (list dict)
      (mapa (lambda (fact)
              (let ((ndict (match (car as) fact dict)))
                (if (failed? ndict)
                    '()
                    (match-rest (cdr as) ndict facts))))
            facts)))

(define (uniquefy set)
  (cond ((null? set) '())
        ((member (car set) (cdr set))
         (uniquefy (cdr set)))
        (else 
          (cons (car set)
                (uniquefy (cdr set))))))

(define (mapa f l)
  (if (null? l)
      '()
      (append (f (car l))
              (mapa f (cdr l)))))

(define (delete x l)
  (cond ((null? l) '())
        ((eq? x (car l))
         (delete x (cdr l)))
        (else
          (cons (car l)
                (delete x (cdr l))))))

(define (failed? x)
  (eq? x 'failed))

(define the-empty-dictionary '())
  

;;; Rule syntax -- antecedents, consequent...

(define (make-rule antecedents consequent rule-name) 
  (list antecedents consequent rule-name))

(define (antecedents rule) (car rule))
(define (consequent rule) (cadr rule))
(define (rule-name rule) (caddr rule))



