(declare (usual-integrations))

(define (try-rules expression the-rules)
  (define (scan rules)
    (if (null? rules)
	(make-failure)
	(match (rule-pattern (car rules))
	       expression
	       (make-empty-dictionary)
	       (lambda () (scan (cdr rules)))
	       (lambda (dictionary fail)
		 (instantiate (rule-skeleton (car rules)) dictionary)))))
  (scan the-rules))

;;; the matcher takes a pattern, data, dictionary, and success and
;;; failure continutations
;;; dictionary is triples (var value info)
;;; fail is (lambda () ...)
;;; succeed is (lambda (dict fail) ....)
;;; The idea is that the succeed continuation passes along the place
;;; to fail back to, if we need to backtrack later

(define (match pat dat dict fail succeed)
  (cond ((eq? dict 'failed?) (error "testing"))
	((eq? pat dat) (succeed dict fail))
        ((arbitrary-element? pat)
         (element-match pat dat dict fail succeed))
        ((constant? pat)
         (if (same-constant? pat dat) (succeed dict fail) (fail)))
        ((start-arbitrary-segment? pat)
	 (if (or (pair? dat) (null? dat))
	     (segment-match (car pat) 
			    dat 
			    dict 
			    fail 
			    (lambda (rest dict fail)
			      (match (cdr pat) rest dict fail succeed)))
	     (fail)))
	;;this must come after clause above, to permit matching an
	;;arbitary segment against null data
        ((constant? dat) (fail))
        (else
         (match
	  (car pat) (car dat) dict fail
	  (lambda (dict fail)
	    (match (cdr pat) (cdr dat) dict fail succeed))))))


(define (element-match arb-var dat dict fail succeed)
  (let ((vname (var-name arb-var))
	(p (var-restriction arb-var)))
    (let ((e (lookup vname dict)))
      (if e
	  (let ((val (entry-value e)))
	    (if (equal? val dat)
		(if (not-falure (restrict-element p val))
		    (succeed dict fail)
		    (fail))
		(fail)))
	  (let ((res (restrict-element p dat)))
	    (if (not-failure res)
		(succeed (extend-dictionary vname dat res dict) fail)
		(fail)))))))


;;; Try to match some initial segment of the data.  The success
;;; continuation for this is
;;; (lambda (rest-of-data dictionary fail) ...)
;;; Thus, if the match succeeds, we go on to try to match the rest of
;;; the data against the rest of the pattern, while keeping the
;;; failure point as a place to backtrack to.

(define (segment-match seg-var dat dict fail succeed-with-rest)
  (let ((vname (var-name seg-var))
	(p (var-restriction seg-var)))
    (let ((e (lookup vname dict)))
      (if e
	  ;;if there is already a binding for that segment variable,
	  ;;see if this matches the segment at the head of the data,
	  ;;and either fail or succeed with the rest of the data.
          (let ((seg (entry-value e)))
	    (let ((rest-list (compare-segment-to-list-head seg dat)))
	      (if (and (not (eq? rest-list 'failed))
		       (not-failure (restrict-segment p seg)))
		  (succeed-with-rest rest-list dict fail)
		  (fail))))
	  ;;otherwise, try matching the segment against successviely
	  ;;longer initial segments of the data.  TRY-SEGMENT tries an
	  ;;initial segment.  If it fails, it tries a longer segment.
	  ;;If it succeeds, it calls SUCCEED-WITH-REST, but with a
	  ;;failure continuation that will try a longer segment if we
	  ;;need to backtrack to this point.
	  (let ()
	    (define (try-segment rest)
	      (define (try-longer-segment)
		(if (null? rest) (fail) (try-segment (cdr rest))))
	      (let ((res (restrict-segment p (make-segment dat rest))))
		(if (not-failure res)
		    ;;if match succeeds
		    (succeed-with-rest
		     rest 
		     (extend-dictionary vname (make-segment dat rest) res dict)
		     ;;pass along this failure continuation
		     try-longer-segment)
		    ;;if match fails
		    (try-longer-segment))))
	    (try-segment dat))))))

;;; compare segment with initial segment of data.  If they are equal,
;;; return with the rest of the data.  Otherwise return 'failed.

(define (compare-segment-to-list-head segment list)
  (let ((endseg (segment-end segment)))
    (define (scan seg-ptr rest-list)
      (cond ((eq? seg-ptr endseg) rest-list)
            ((null? rest-list) 'fail)
	    ((equal? (car seg-ptr) (car rest-list))
             (scan (cdr seg-ptr) (cdr rest-list)))
	    (else 'fail)))
    (scan (segment-start segment) list)))


;;; Syntax of the expressions being manipulated:

(define (constant? exp) (not (pair? exp)))
(define same-constant? equal?)

;;; Failure

(define (make-failure) 'failed)
(define (not-failure exp) (not (eq? exp 'failed)))

;;; Rule syntax

(define (rule-pattern rule) (car rule))
(define (rule-skeleton rule) (cadr rule))

;;; Pattern syntax.

(define (arbitrary-element? pat)
  (and (pair? pat) (eq? (car pat) '?)))

(define (start-arbitrary-segment? pat)
  (and (pair? pat) (pair? (car pat)) (eq? (caar pat) '??)))

(define (var-name pat)
  (cadr pat))

(define (var-restriction pat)
  (if (cddr pat)
      (evaluate-in-global-env (caddr pat))
      (lambda (x) true)))

(define (restrict-element proc elt)
  (proc elt))

;;; Segments

(define (make-segment start end)
  (list '*segment* start end))

(define (segment? x)
  (and (pair? x)
       (eq? (car x) '*segment*)))

(define segment-start cadr)
(define segment-end caddr)

(define (segment->list segment)
  (let ((end (segment-end segment)))
    (define (collect-segment pointer)
      (if (eq? pointer end)
          '()
          (cons (car pointer)
                (collect-segment (cdr pointer)))))
    (collect-segment (segment-start segment))))

(define (restrict-segment proc seg)
  (proc (segment->list seg)))
  

(define (convert-matcher-entry val)
  (if (segment? val)
      (segment->list val)
      val))

;;; Dictionaries

(define (make-empty-dictionary) '())

(define (make-entry name data info)
  (list name data info))

(define entry-name car)
(define entry-value cadr)
(define entry-info caddr)

(define lookup assq)

(define (extend-dictionary name data info dict)
  (cons (make-entry name data info) dict))

(define (dictionary-info name dict)
  (let ((e (lookup name dict)))
    (if e
	(entry-info e)
	(error "name not in dictionary" name))))

(define (dictionary-value name dict)
  (let ((e (lookup name dict)))
    (if e
	(entry-value e)
	(error "name not in dictionary" name))))

;;;evaluation

(define global-environment (the-environment))

(define (evaluate-in-global-env exp)
  (eval exp global-environment))

;;; instantiation


(define (instantiate skeleton dictionary)
  (cond ((not (pair? skeleton)) skeleton)
        ((element-value? skeleton)
         (dictionary-info (element-expression skeleton) dictionary))
        (else
         (cons (instantiate (car skeleton) dictionary)
               (instantiate (cdr skeleton) dictionary)))))

#|
This alternate version of instantiate allows one to evaluate arbitrary
expressions after substituting the dictionary values, e.g., (: (+ x y))

(define (instantiate skeleton dictionary)
  (cond ((not (pair? skeleton)) skeleton)
        ((element-value? skeleton)
         (evaluate-element (element-expression skeleton) dictionary))
	((segment-value? (car skeleton))
         (evaluate-segment (segment-expression (car skeleton))
                           dictionary
                           (instantiate (cdr skeleton) dictionary)))
        (else
         (cons (instantiate (car skeleton) dictionary)
               (instantiate (cdr skeleton) dictionary)))))


(define (evaluate-element expression dictionary)
  (evaluate-in-global-env (substitute-in expression dictionary)))

(define (substitute-in expression dictionary)
  (let walk ((e expression))
    (if (pair? e)
	(cons (walk (car e)) (walk (cdr e)))
	(let ((v (lookup e dictionary)))
	  (if v
	      (list 'quote (entry-info v))
	      e)))))

|#

;;; Skeleton instantation syntax.

(define (element-value? skel)
  (and (pair? skel) (eq? (car skel) ':)))

(define (element-expression skel) (cadr skel))

