/*---------------------------------------------------------------------*/
/*    Copyright (c) 1994 by Manuel Serrano. All rights reserved.       */
/*                                                                     */
/*                                     ,--^,                           */
/*                               _ ___/ /|/                            */
/*                           ,;'( )__, ) '                             */
/*                          ;;  //   L__.                              */
/*                          '   \    /  '                              */
/*                               ^   ^                                 */
/*                                                                     */
/*                                                                     */
/*    This program is distributed in the hope that it will be useful.  */
/*    Use and copying of this software and preparation of derivative   */
/*    works based upon this software are permitted, so long as the     */
/*    following conditions are met:                                    */
/*           o credit to the authors is acknowledged following         */
/*             current academic behaviour                              */
/*           o no fees or compensation are charged for use, copies,    */
/*             or access to this software                              */
/*           o this copyright notice is included intact.               */
/*      This software is made available AS IS, and no warranty is made */
/*      about the software or its performance.                         */
/*                                                                     */
/*      Bug descriptions, use reports, comments or suggestions are     */
/*      welcome Send them to                                           */
/*        <Manuel.Serrano@inria.fr>                                    */
/*        Manuel Serrano                                               */
/*        INRIA -- Rocquencourt                                        */
/*        Domaine de Voluceau, BP 105                                  */
/*        78153 Le Chesnay Cedex                                       */
/*        France                                                       */
/*---------------------------------------------------------------------*/


/*---------------------------------------------------------------------*/
/*    serrano/prgm/project/bigloo/runtime1.6/Clib/string.c ...         */
/*                                                                     */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Fri Jul 17 11:33:47 1992                          */
/*    Last change :  Tue Feb  8 09:53:56 1994 (serrano)                */
/*                                                                     */
/*    La gestion C des chaines de caracteres                           */
/*---------------------------------------------------------------------*/
#include <ctype.h>
#include <stdio.h>
#include <bigloo.h>

/*---------------------------------------------------------------------*/
/*    make_string ...                                                  */
/*---------------------------------------------------------------------*/
obj_t
make_string( k, c )
obj_t  k;
obj_t  c;
{
   obj_t string , aux;
   int len = CINT( k );

   string = MAKE_ATOMIC_OBJECT( STRING_SIZE + len + 1, HEADER_STRING, aux );

   string->string_t.length = k;

   memset( (char *)string + STRING_SIZE, CCHAR( c ), len );
   ((char *)string + STRING_SIZE)[ len ] = '\0';

   return BREF( string );
}

/*---------------------------------------------------------------------*/
/*    c_constant_string_to_string ...                                  */
/*---------------------------------------------------------------------*/
obj_t
c_constant_string_to_string( c_string )
char *c_string;
{
   return c_string_to_string( c_string );
}

/*---------------------------------------------------------------------*/
/*    c_string_to_string ...                                           */
/*---------------------------------------------------------------------*/
obj_t
c_string_to_string( c_string )
char *c_string;
{
   obj_t string, aux;
   int i;
   int len;

   if( !c_string )
      c_string = "";

   len = strlen( c_string );
	
   string = MAKE_ATOMIC_OBJECT( STRING_SIZE + len + 1, HEADER_STRING, aux );

   string->string_t.length = BINT( len );

   strcpy( (char *)string + STRING_SIZE, c_string );

   return BREF( string );
}

/*---------------------------------------------------------------------*/
/*    string_append ...                                                */
/*---------------------------------------------------------------------*/
obj_t
string_append( s1, s2 )
obj_t s1, s2;
{
   int l1, l2;
   obj_t string;
   obj_t a;
   
   l1 = CINT( STRING( s1 ).length );
   l2 = CINT( STRING( s2 ).length );

   string = MAKE_ATOMIC_OBJECT( STRING_SIZE + l1 + l2 + 1, HEADER_STRING, a );

   string->string_t.length = BINT( l1 + l2 );

	memcpy( (char *)string + STRING_SIZE, &STRING_REF( s1, BINT( 0 ) ), l1 );
	memcpy( (char *)string + STRING_SIZE + l1,
			  &STRING_REF( s2, BINT( 0 ) ),
			  l2 );
	((char *)string + STRING_SIZE)[ l1 + l2 ] = '\0';
	
   return BREF( string );
}
 
/*---------------------------------------------------------------------*/
/*    substring ...                                                    */
/*---------------------------------------------------------------------*/
obj_t
c_substring( src_string, min, max )
obj_t src_string, min, max;
{
   int len;
   obj_t dst_string, a;
   
   len = CINT( SUB_I( max, min ) );

   dst_string = MAKE_ATOMIC_OBJECT( STRING_SIZE + len + 1, HEADER_STRING, a );

   dst_string->string_t.length = BINT( len );

   memcpy( (char *)dst_string + STRING_SIZE,
			   &STRING_REF( src_string, min ),
            len );
   ((char *)dst_string + STRING_SIZE)[ len ] = '\0';

   return BREF( dst_string );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    blit_string ...                                                  */
/*---------------------------------------------------------------------*/
obj_t
blit_string( s1, offset1, s2, offset2, len )
obj_t s1, offset1, s2, offset2, len;
{
	bcopy( &STRING_REF( s1, offset1 ),
			 &STRING_REF( s2, offset2 ),
			 CINT( len ) );
	return BUNSPEC;
}
	
/*---------------------------------------------------------------------*/
/*    integer_to_string ...                                            */
/*---------------------------------------------------------------------*/
char *
integer_to_string( x, radix )
int x, radix;
{
   char *aux;

   aux = (char *)ALLOCATE_ATOMIC( 1024 );

   switch( radix )
   {
      case 8 :
         sprintf( aux, "#o%o", x );
         break;
         
      case 10 :
         sprintf( aux, "%d", x );
         break;
         
      case 16 :
         sprintf( aux, "#x%x", x );
         break;
   }

   return aux;
}

/*---------------------------------------------------------------------*/
/*    real_to_string ...                                               */
/*---------------------------------------------------------------------*/
char *
real_to_string( x )
double x;
{
   char *aux;

   aux = (char *)ALLOCATE_ATOMIC( 1024 );

   sprintf( aux, "%f", x );
   
   return aux;
}

/*---------------------------------------------------------------------*/
/*    unsigned char                                                    */
/*    bigloo_strcmp ...                                                */
/*---------------------------------------------------------------------*/
unsigned char
bigloo_strcmp( o1, o2 )
obj_t o1, o2;
{
	int   l1,  l2;
	char *s1, *s2;
	int   i;
	
	l1 = CINT( STRING_LENGTH( o1 ) );
	l2 = CINT( STRING_LENGTH( o2 ) );

	s1 = BSTRING_TO_CSTRING( o1 );
	s2 = BSTRING_TO_CSTRING( o2 );

	if( l1 == l2 )
	{
		for( i = 0; (*s1 == *s2) && (i < l1); s1++, s2++, i++ );

		return i == l1;
	}
	else
	   return 0;
}

/*---------------------------------------------------------------------*/
/*    strcicmp ...                                                     */
/*---------------------------------------------------------------------*/
unsigned char
strcicmp( st1, st2 )
char *st1, *st2;
{
   for( ; tolower( *st1 ) == tolower( *st2 ); st1++, st2++ )
      if( *st1 == '\0' )
         return 1;

   return 0;
}

/*---------------------------------------------------------------------*/
/*    string_lt ...                                                    */
/*---------------------------------------------------------------------*/
unsigned char
string_lt( st1, st2 )
char *st1, *st2;
{
   for( ; *st1 == *st2; st1++, st2++ )
      if( *st1 == '\0' )
         return 0;

   return *st1 < *st2;
}

/*---------------------------------------------------------------------*/
/*    string_le ...                                                    */
/*---------------------------------------------------------------------*/
unsigned char
string_le( st1, st2 )
char *st1, *st2;
{
   for( ; *st1 == *st2; st1++, st2++ )
      if( *st1 == '\0' )
         return 1;

   return *st1 <= *st2;
}

/*---------------------------------------------------------------------*/
/*    string_gt ...                                                    */
/*---------------------------------------------------------------------*/
unsigned char
string_gt( st1, st2 )
char *st1, *st2;
{
   for( ; *st1 == *st2; st1++, st2++ )
      if( *st1 == '\0' )
         return 0;

   return *st1 > *st2;
}

/*---------------------------------------------------------------------*/
/*    string_ge ...                                                    */
/*---------------------------------------------------------------------*/
unsigned char
string_ge( st1, st2 )
char *st1, *st2;
{
   for( ; *st1 == *st2; st1++, st2++ )
      if( *st1 == '\0' )
         return 1;

   return *st1 >= *st2;
}

/*---------------------------------------------------------------------*/
/*    string_cilt ...                                                  */
/*---------------------------------------------------------------------*/
unsigned char
string_cilt( st1, st2 )
char *st1, *st2;
{
   for( ; tolower( *st1 ) == tolower( *st2 ); st1++, st2++ )
      if( *st1 == '\0' )
         return 0;

   return tolower( *st1 ) < tolower( *st2 );
}

/*---------------------------------------------------------------------*/
/*    string_cile ...                                                  */
/*---------------------------------------------------------------------*/
unsigned char
string_cile( st1, st2 )
char *st1, *st2;
{
   for( ; tolower( *st1 ) == tolower( *st2 ); st1++, st2++ )
      if( *st1 == '\0' )
         return 1;

   return tolower( *st1 ) <= tolower( *st2 );
}

/*---------------------------------------------------------------------*/
/*    string_cigt ...                                                  */
/*---------------------------------------------------------------------*/
unsigned char
string_cigt( st1, st2 )
char *st1, *st2;
{
   for( ; tolower( *st1 ) == tolower( *st2 ); st1++, st2++ )
      if( *st1 == '\0' )
         return 0;

   return tolower( *st1 ) > tolower( *st2 );
}

/*---------------------------------------------------------------------*/
/*    string_cige ...                                                  */
/*---------------------------------------------------------------------*/
unsigned char
string_cige( st1, st2 )
char *st1, *st2;
{
   for( ; tolower( *st1 ) == tolower( *st2 ); st1++, st2++ )
      if( *st1 == '\0' )
         return 1;

   return tolower( *st1 ) >= tolower( *st2 );
}

/*---------------------------------------------------------------------*/
/*    escape_C_string ...                                              */
/*    -------------------------------------------------------------    */
/*    Cette fonction construit une chaine ou la representation des     */
/*    caracteres de controles a ete remplacee par ces caracteres.      */
/*    ex:     +---+---+---+---+          +---+---+---+                 */
/*            | \ | n | a | 0 |    ==>   | \n| a | 0 |                 */
/*            +---+---+---+---+          +---+---+---+                 */
/*    Cette conversion est utile pour l'interprete car les chaines     */
/*    lues ne sont pas parsees. On donne donc la possibilite de le     */
/*    faire avec cette fonction.                                       */
/*---------------------------------------------------------------------*/
obj_t
escape_C_string( src )
unsigned char *src;
{
   /* on supprime un caractere de cette chaine car elle est rendue par le */
   /* lecteur comme etant `"tototo'. Ceci est dut au fait qu'on utilise   */
   /* la fonction `the-small-string' qui supprime le premier et le        */
   /* dernier caractere de la chaine lu. Comme les chaines etrangeres     */
   /* commence par 2 caracteres, on en supprime 1 autre maintenant.       */

   int len = strlen( ++src );
   unsigned char *dst;
   obj_t string, aux;

   string = MAKE_ATOMIC_OBJECT( STRING_SIZE + len + 1, HEADER_STRING, aux );

   dst = ((unsigned char *)string + STRING_SIZE);

   while( *src )
   {
      if( *src != '\\' )
         *dst++ = *src++;
      else
      {
			len--;
	 
         switch( *++src ) 
         {
            case '\0' : *dst++ = '\\';
                        break;

            case 'n'  : *dst++ = '\n';
                        break;
                        
            case 't'  : *dst++ = '\t';
                        break;
                        
            case 'b'  : *dst++ = '\b';
                        break;
                        
            case 'r'  : *dst++ = '\r';
                        break;
                        
            case 'f'  : *dst++ = '\f';
                        break;
                        
            case 'v'  : *dst++ = '\v';
                        break;
                        
            case '\\' : *dst++ = '\\';
                        break;
                        
            case '\'' : *dst++ = '\'';
                        break;
                        
            case '"'  : *dst++ = '\"';
                        break;

#if( defined( __STDC___ ) )                          
            case 'a'  : *dst++ = '\a';
                        break;

            case '?'  : *dst++ = '\?';
                        break;
#endif                        

            default   : if( isdigit( *(src) ) &&
									 isdigit( *(src+1) ) &&
									 isdigit( *(src+2) ) )
				            /* C'est une representation alpha-numerique `\???' */
				            {
									unsigned char aux;

									aux = (*src     - '0')*64 +
									      (*(src+1) - '0')*8 +
											(*(src+2) - '0');
									*dst++ = aux;
									src+=2;
									
									len -= 2;
								}
					         else
				               *dst++ = *src;
					  
                        break;
         }
         src++;
      }
   }
   *dst = '\0';
   
   string->string_t.length = BINT( len );

   return BREF( string );
}
       
/*---------------------------------------------------------------------*/
/*    escape_scheme_string ...                                         */
/*    -------------------------------------------------------------    */
/*    Cette fonction ressemble a la precedente mais elle filtre moins  */
/*    de caracteres                                                    */
/*---------------------------------------------------------------------*/
obj_t
escape_scheme_string( src )
char *src;
{
   int   len = strlen( src );
   char *dst;
   obj_t string, aux;

   string = MAKE_ATOMIC_OBJECT( STRING_SIZE + len + 1, HEADER_STRING, aux );

   dst = ((char *)string + STRING_SIZE);

   while( *src )
   {
      if( *src != '\\' )
         *dst++ = *src++;
      else
      {
			len--;
	 
         switch( *++src )  
         {
            case '\\' : *dst++ = '\\';
                        break;
                        
            case '"'  : *dst++ = '"';
                        break;

            default   : *dst++ = *src;
                        break;
         }
         src++;
      }
   }
   *dst = '\0';
   
   string->string_t.length = BINT( len );

   return BREF( string );
}
       
/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    string_for_read ...                                              */
/*---------------------------------------------------------------------*/
obj_t
string_for_read( bstring )
obj_t bstring;
{
	unsigned char *dst;
	unsigned char *src = BSTRING_TO_CSTRING( bstring );
	int   r, w, len = CINT( STRING_LENGTH( bstring ) );
	obj_t res;

	dst = malloc( len * 4 );

	for( r = 0, w = 0; r < len; r++ )
	   switch( src[ r ] )
		{
	      case '\n' : dst[ w++ ] = '\\';
			            dst[ w++ ] = 'n';
				         break;

	      case '\t' : dst[ w++ ] = '\\';
				         dst[ w++ ] = 't';
				         break;

	      case '\b' : dst[ w++ ] = '\\';
				         dst[ w++ ] = 'b';
				         break;
				  
	      case '\r' : dst[ w++ ] = '\\';
				         dst[ w++ ] = 'r';
				         break;
				  
	      case '\f' : dst[ w++ ] = '\\';
			            dst[ w++ ] = 'f';
				         break;

	      case '\v' : dst[ w++ ] = '\\';
				         dst[ w++ ] = 'v';
				         break;

	      case '"'  : dst[ w++ ] = '\\';
				         dst[ w++ ] = '"';
				         break;
				  
	      case '\\' : dst[ w++ ] = '\\';
				         dst[ w++ ] = '\\';
				         break;

			default :   if( (src[r] < 128) && (isprint( src[ r ] )) )
			               dst[ w++ ] = src[ r ];
				         else
							{
   				            sprintf( &dst[ w ], "\\%03o",
										  ((unsigned char *)src)[ r ] );
								w += 4;
							}
	}

	dst[ w ] = '\0';
	
   res = c_string_to_string( dst );

	free( dst );

	return res;
}

					  
