;;;======================================================================
;;; -*- Mode:Lisp; Syntax:Common-Lisp; ; Base:10; -*-
;;;
;;; ******************************
;;; *  PORTABLE AI LAB -  IDSIA  *
;;; ******************************
;;;
;;; Filename:    choose.cl
;;; Short Desc:  implementatios of ATP staregies
;;
;;; Version:    1.0
;;; Status:     Experimental
;;; Last Mod:   Jun 1991
;;; Author:     Fabio Baj
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;


(in-package :atp)

;;=======================================================	
;; The theorem prover is independent from the strategy
;; used for choosing two clauses to make infernces.
;; A strategy is an object containing all the information
;; needed to define it. Every strategy is defined by the 
;; following properties / methods.
;;
;; init  : {} --> {}
;;         the procedure to be used to initialize the sets
;;         of clauses-ids, used by the strategy.
;; ch-ins: CLAUSE-ID x LIST-OF(CLAUSE-ID) -->  LIST-OF(CLAUSE-ID)
;;         the procedure to be used to add new clauses to
;;         the sets used by the strategy.
;; ins-sets: the sets in which new clauses have to be added
;; del-sets: the sets from which  the id of adeleted clause
;;         : have to be removed.
;; choose  : {} --> (CLAUSE-ID CLAUSE-ID)
;;            the procedure to be used for choosing two ids
;;           from sets.
;; NOTE that the only the ch-ins field can be modified by the
;; user, setting the flag " insert-procedure"
;;
;; This structure is named chooseinfo.
;;
;; The interface with the rest of the program consists 
;; in a call in the main ATP loop (file en-strat.cl)to
;;
;;     (funcall (choose-clauses-function (eval $strategy$)))
;;
;; to obtain the next pair of rules to superpose
;; where $strategy$ is the strategy selected by the user
;; (standard, set-of-support, manual, prolog)
;;
;; In the file update.cl there is the call
;;
;;     (remove-from-sets (car anc) (delete-sets (eval $strategy$)))
;;
;; To ensure that a deleted clause is also removed from the 
;; set used by a particular strategy.
;;
;; and the call
;;    (funcall (insert-function $strategy$) id (eval s))))
;; for each set in (insert-sets $strategy$)
;; when a new rule is generated.
;;
;; in the file preproc.cl there is the call to 
;;  (init-function $strategy$)
;;
;; Defining new strategies according to this structure
;; allows to have not to change anything in the rest of
;; the program.
;; Notice that if the strategy need no sets, they can simply
;; set to nil. See , for example the manual strategy.
;;

;;=======================================================
;; The standard startegy picks a rule-id from $todo$
;; and every time the choose method is called by the
;; main algorithm, it returns a clause-ids pair.
;;
;; standard-choose
;; LOOP
;;   pick a rule-id RID1 from $todo$
;;   for each rule-id RID2 in $done$
;;        return <RID1,RID2>
;;   add    RID1 to $done$
;;   remove RID1 from $todo$
;; END LOOP
;;
;; The default procedure to insert newly generated ids
;; in $todo$ is fifo-ins, so that the resulting startegy
;; is a breadth-first one. If the user selects a "lifo"
;; insert pocedure, we have a depth first strategy. If
;; the smallest-ins procedure is choosen, we obtain a
;; smallest first strategy.
;;

(setq standard
  (make-instance 'strategy
		    :init-function  'init-standard
		    :insert-function  'smallest-ins
		    :insert-sets '($todo$)
		    :delete-sets '($todo$ $b-done$ $done$)
		    :choose-clauses-function 'standard-choose))  

(defun init-standard() 
  (setq $using$ nil)
  (setq $done$ nil)
  (setq $b-done$ nil)
  (setq $prolog-mode$ nil))

(defun standard-choose()
  (cond ((null $b-done$) 
	 (setq $done$ (cons (car $todo$) $done$ ))
	 (setq $using$ (pop $todo$))
	 (setq $b-done$ $done$))
	(t))
  (list $using$ (pop $b-done$)))

  

;;====================================================
;; Here are the procedures to insert new rule-ids in sets.
;; there is a First in First Out procedure,
;; a Last in First Out procedure, whose implementation are
;; straightforward.
;; The smallest-ins strategy keeps the set sorted with
;; respect to the rule-size-comp criterion, that simply
;; compares the weight associated to the rules. 
;; Notice that the user can modify the comparing criterion
;; setting the parameters that are used to compute rule weights.
;; see the file weights.cl

(defun fifo-ins (id set)
  (append set (list id)))

(defun lifo-ins (id set)
  (cons id set))

(defun smallest-ins (id set)
 (insert-ord id set  #'rule-size-comp nil))

(defun insert-ord (id set test res )
  (cond ((null set) (reverse (cons  id  res)))
	((funcall test  (car set) id)
	 (append  (reverse (cons  id  res))  set))
	(t (insert-ord id (cdr set) test (cons  (car  set) res)))))

(defun rule-size-comp (id1 id2)
  (>= (size (get id1 'info))
      (size (get id2 'info))))

;;=======================================================	
;;------ A Prolog-like Strategy ------------	  


(setq  prolog-strat 
  (make-instance 'strategy
		    :init-function   'prolog-init
		    :insert-function 'prolog-ins
		    :insert-sets     '()
		    :delete-sets     '()
		    :choose-clauses-function 'prolog-choose))

(defun prolog-init ()
  (let ((sosrest (collect-sos-rest)))
    (setq c0 (identity-clause))
    (setq $prolog-mode$ t)
    (setq $exist-check$ nil)
    (setq $semantic-simplification$ t)
    (setf (get 'c0 'info) (classify (identity-clause) 'axiom))
    (setq $program$  (cons 'c0 (sort (cdr sosrest) 'clause-id<)))
    (setq $goal-stack$ (list (list (caar sosrest) (car $program$))))
    (if (not (null (caar sosrest))) 
	(progn
	  (let* 
	      ((goal-id (caar sosrest)))
	    (remake-clause     'nil
			       (append (negative-atoms (eval goal-id))  
					(list (cons '$ans
					       (collect-integer-vars
						     (negative-atoms (eval goal-id))))))
			        (eval goal-id))
	    (setq $last-clause$ (clause-id-1- goal-id))))
      (setq  $last-clause$ (atom-conc  'c(1- $rule-id))))))


(defun the-user-provided-a-subgoal (sosrest)
  (not (null (caar sosrest))))

(defun identity-clause () (make-clause '((= -1 -1)) 'nil))
 

(defun prolog-ins (id )
  (setq $goal-stack$ (cons (list id (car $program$)) $goal-stack$)))

(defun prolog-choose ()
  (let ((pair (pop $goal-stack$)))
   (cond 
    ((eq 'cut pair) (prolog-choose))
    ((null pair) '(nil nil))
    (t (cond ((eq $last-clause$ (cadr pair)) pair)
	     (t (push (list (car pair) (clause-id-1+ (cadr pair)))  $goal-stack$)
		pair))))))
  
(defun put-a-cut-marker-in-the-stack ()
  (setq  $goal-stack$ (cons (car $goal-stack$)
			    (cons 'cut (cdr $goal-stack$)))))

(defmethod contains-the-cut ((clause clause-class))
  (member '! (negative-atoms clause)))

(defun cut-the-stack ()
 (cond  ((eq 'cut (pop $goal-stack$)))	 
	(t (cut-the-stack))))

(defun clause-id-1+ (a1)
  (atom-conc 'c
	     (1+
	      (read-from-string
	       (string-left-trim '(#\c #\C) a1)))))
(defun clause-id-1- (a1)
  (atom-conc 'c
	     (1-
	      (read-from-string
	       (string-left-trim '(#\c #\C) a1)))))

;;=======================================================
;;------- A Set of Support Strategy ------------------
;;
;; this strategy chooses id pairs in which at leats one
;; rule-id must belong to the set of support.
;; New rule-ids are added to the est of support.
;; initially the set of support consists of the rules
;; corresponding to the formulas appearing in
;; list(sos) ..... end-of-list, in the user file.
;;


(setq *sos*
  (make-instance 'strategy
		    :init-function    'sos-init
		    :insert-function  'smallest-ins
		    :insert-sets      '($todo-sos$)
		    :delete-sets      '($todo$ $b-done$ $done-sos$)
		    :choose-clauses-function 'ch-sos))

(defun sos-init() 
  (setq $b-done$ nil)
  (setq $using$ nil)
  (setq $todo-sos$ nil)
  (setq $done-sos$ nil)
  (setq $done-rest$ nil)
  (setq $prolog-mode$ nil)
  (let ((sosrest (collect-sos-rest)))
    (setq $sos$ (car sosrest))
    (setq $rest$ (cdr sosrest))))
	

	

(defun ch-sos ()
  (if (or (not (null $sos$)) (not (null $b-done$))   )
      (progn
	(if (null $b-done$)
	    (progn
	      (setq $using$ (pop $sos$))
	      (setq $done-sos$ (append $done-sos$ (list $using$)))
	      (setq $b-done$ $done-sos$)))
	(list $using$ (pop $b-done$)))
    (if (or (not (null $rest$)) (not (null $b-done$))   )
	(progn
	  (if (null $b-done$)
	      (progn
		(setq $using$ (pop $rest$))
		(setq $done-rest$ (append $done-rest$ (list $using$)))
		(setq $b-done$ $done-sos$)))
	  (list $using$ (pop $b-done$)))
      (progn
		(setq $done-sos$ (append $done-sos$ $done-rest$))
	(setq $done-rest$ nil)
	(if (null $b-done$)
	    (progn
	      (setq $using$ (pop $todo-sos$))
	      (setq $done-sos$ (append $done-sos$ (list $using$)))
	      (setq $b-done$ $done-sos$)))
	(list $using$ (pop $b-done$))))))     
	 
    
 
(defun collect-sos-rest ()
  (let ((all (copy-list(append $simps$ $rules$)))
	(sos) (rest))
    (dolist ( x all (cons sos rest))
      (cond (( eq 'sos (car(ancestors (get x 'info))))
	     (setq sos (cons x sos)))

	    (t (setq rest (cons x rest)))))))



;;=======================================================	
;;       ----- A Manual Strategy ------
;; The user can interactively choose the next rules to be
;; superposed


(setq  *man*
  (make-instance 'strategy
		    :init-function    '(lambda ()(setq $prolog-mode$ nil))
		    :insert-function  '(lambda ()())
		    :insert-sets      '()
		    :delete-sets      '()
		    :choose-clauses-function 'choose-manual))

(defun choose-manual ()
  (if *interface*   (setf (title *dialog-disp*) 
		      "Please enter two clause numbers ('end' to end)"))
  (format-display  (if *interface*  *out-disp* t )
		   "~%~%------------ Current Database State ---------------~%" )
  (list-rules)
  (let (( x nil)(y nil))
    (if *interface*
	(progn 
	  (clear-display  *dialog-disp* )
	  (write-display *dialog-disp* "X? " 5 30)
	  (write-display *dialog-disp* "Y? " 5 5)
	  (setq x (read-display *dialog-disp*  25 30))
	  (setq y (read-display *dialog-disp*  25 5)))
      (progn
	(princ "X?   :") (setq x  (read-line))
	(princ "Y?   :") (setq y  (read-line))))
    (check x y)))

(defun check (x y) 
  (if (or (string= "end"  x) (string= "end" y))
	(list nil nil)
    (if (eq 'man 
	    (catch 'man-error
	      (and (check1 x )(check1 y))))
	(choose-manual)
      (list  (right-case (intern (concatenate 'string "c"
				  (without-blank(string-left-trim '(#\c  #\C) x))) :atp))
	     (right-case(intern  (concatenate 'string "c"
				   (without-blank (string-left-trim '(#\c  #\C)y))) :atp))))))
     
    
(defun check1 (n-string)
  (let ((number (clause-number n-string)))
    (cond ((and (integerp number) (<=  number $rule-id)) number)
	  (t  (man-error (format nil "~A:  Unknown clause" n-string))))))
   
		    
      
(defun man-error (string)
  (if (not *interface*)
      (format t "~%** ATP **: ~A~%" string)
    (acknowledge-dialog string :title "ATP: Error"))
  (throw 'man-error 'man))


(defun clause-number (cl-id)
  (if (string= "" cl-id)
      (man-error (format nil "Enter a clause id !!"))
  (read-from-string (string-left-trim '(#\c #\C) cl-id))))
