;;; -*- Mode: LISP; Package: atp; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   skolem.cl
;;; Short Desc:  This module transforms a Well Formed Formula, in prefix notation,
;;;              into a set  of clauses, introducing Skolem functions and constants
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   8.9.91 - FB
;;; Author:     Fabio Baj
;;;
;;; Copyright (c) 1992 Istituto D|all|e Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in |all|
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :atp)





(defvar *sk-verbose* nil)

;;-------------------------------------------------------
;; clausify-main : SEXPR  --->  LIST-OF-CLAUSES
;; Behavior      : Tranforms 'fmla' into a list of clauses
;;               : duplicate literals in clauses and duplicate clauses
;;               : are discarded. Clauses containing a literal and its negation
;;               : are also descarded.
;; Side effects  : None
;; Example       : clausify-main(<=> A B))  = (((~ A) B)  (A (~B)))

(defun clausify-main (fmla)
  (rem-dup-clauses  
   (make-clauses
    (conjuctions-in
     (universal-out
      (skolemize
       (negations-in
	(implications-out
	 (ren-vars 
	  (bi-implications-out fmla))))))))))


(defun bi-implications-out (fmla)
  (if *sk-verbose*  (print-formula  fmla  ))
  (format  *sk-verbose* "~%*Removing bi-implications*~%~%") 
   (bi-impl-out fmla))
     
  
  
(defun bi-impl-out (fmla)
  (cond 
   ((atomic-pref fmla) fmla)
   ((eq (car fmla) '<=> )
    (list '&
	  (list '=> (bi-impl-out (cadr fmla))
		(bi-impl-out (caddr fmla)))
	  (list '=> (bi-impl-out (caddr fmla))
		(bi-impl-out (cadr fmla)))))
   (t
    (cons (car fmla) (mapcar 'bi-impl-out (cdr fmla))))))



(defun implications-out (fmla)
  (if *sk-verbose*  (print-formula  fmla  ))
  (format  *sk-verbose* "~%*Removing implications*~%~%") 
  (impl-out fmla))
      
  
(defun impl-out (fmla)
     (cond 
          ((atomic-pref fmla) fmla)
          ((eq (car fmla) '=> )
               (list '+
                     (list '~ (impl-out (cadr fmla)))
                     (impl-out (caddr fmla))))
          (t   (cons (car fmla) (mapcar 'impl-out (cdr fmla))))))


(defun negations-in (fmla)
   (if *sk-verbose*  (print-formula  fmla  ))
  (format  *sk-verbose* "~%*Bringing negations in*~%~%") 
 (negin fmla))
     
  
(defun negin (f)
  (let ((arg))
    (cond 
     ((atomic-pref f) f)
     ((eq '~ (car f))
      (setq arg (cadr f))
      (cond ((atomic-pref arg) f)
	    ((eq '~ (car arg)) (negin (cadr arg)))
	    ((eq '|exists| (car arg))
	     (list '|all| (cadr arg) (negin (cons '~ (cddr arg)))))
	    ((eq '|all| (car arg))
	     (list '|exists| (cadr arg) (negin (cons '~ (cddr arg)))))
	    ((eq `& (car arg))
	     (list '+ 
		   (negin (list '~ (cadr arg)))
		   (negin (list '~ (caddr arg)))))
	    ((eq `+ (car arg))
	     (list '& 
		   (negin (list '~ (cadr arg)))
		   (negin (list '~ (caddr arg)))))))
     (t (list (car f) (negin (cadr f)) (negin (caddr f)))))))
 

(defun conjuctions-in (fmla)
   (if *sk-verbose*  (print-formula  fmla  ))
  (format  *sk-verbose* "~%*Bringing conjuctions in*~%~%") 
 (conj-in fmla))
                        
(defun conj-in (f)
  (cond 
   ((or-pref f) 
    (conj-1 (list '+ (conj-in (cadr f)) (conj-in (caddr f)))))
   ((and-pref f) 
    (list '& (conj-in (cadr f)) (conj-in (caddr f))))
   (t f))) 


(defun conj-1 (f)
  (let ((arg1 (cadr  f))
	(arg2 (caddr f)))
    (cond ((and-pref arg2) (list '&
				 (conj-in (list '+ arg1 (cadr  arg2)))
				 (conj-in (list '+ arg1 (caddr arg2)))))
	  ((and-pref arg1) (list '&
				 (conj-in (list '+ (cadr  arg1) arg2))
				 (conj-in (list '+ (caddr arg1) arg2))))
	  (t f))))

(defun make-clauses (fmla)
	      (if *sk-verbose*  (print-formula  fmla  ))
	      (format  *sk-verbose* "~%*Making clauses ... *~%~%") 
	      ( mk-clauses fmla))
 
(defun mk-clauses (f)
  (cond
   ((and-pref f)
    (append (mk-clauses (cadr f)) (mk-clauses (caddr f))))
   ((or-pref f)
    (let ((cl (inclause f)))
      (cond ((member '$taut cl) (list))
	    (t (list cl)))))
   (t (list (list f)))))

(defun inclause (f) (inclause-aux f nil))
(defun inclause-aux (f clause)
  (cond ((literal f) (adj-lit f clause))
	(t (inclause-aux (caddr f) (inclause-aux (cadr f) clause)))))

(defun adj-lit (l cl)
     (cond
        ((member l cl :test #'equal) cl)
        ((member (list '~ l) cl :test #'equal) (list '$taut))
        ((and (not (atomic-pref l))
              (member (cadr l) cl)) (list '$taut))
        (t (cons l cl))))

(defun ren-vars (f)(car (rename-vars f nil)))
(defun rename-vars (f varlist)
  (let ((res1)(res2)(x))
    (cond
     ((atomic-pref f) (cons f varlist))
     ((member (car f) '(& + =>) )
      (setq res1  (rename-vars (cadr f ) varlist))
      (setq res2  (rename-vars (caddr f) (cdr res1)))
      (cons (list (car f) (car res1)(car res2))
	    (cdr res2)))
     ((eq (car f) '~)
      (setq res1 (rename-vars (cadr f) varlist))
      (cons (list '~ (car res1))  (cdr res1)))
     ((member (car f) '(|all| |exists|))
      (cond 
       ((member (cadr f) varlist)
	(setq x (gsym 'x-)))
       (t  (setq x (cadr f))))
      (setq res1  (rename-vars  (subst x (cadr f) (caddr f))
				(cons x varlist)))
      (cons (list (car f) x (car res1))
	    (cdr res1))))))  
                

(defun universal-out (fmla)
   (if *sk-verbose*  (print-formula  fmla  ))
  (format  *sk-verbose* "~%*Removing universal quantifiers*~%~%") 
  (univ-out fmla))       
        
  
(defun univ-out (f)
     (cond 
        ((literal f) f)
        ((eq '& (car f))    (list '& (univ-out (cadr f)) (univ-out (caddr f))))
        ((eq '+ (car f))    (list '+ (univ-out (cadr f)) (univ-out (caddr f))))
        ((eq '|all| (car f))  (univ-out (caddr f)))))            
   


(defun  skolemize (fmla)
  (if *sk-verbose*  (print-formula  fmla  ))
  (format  *sk-verbose* "~%*Introducing Skolem functions*~%~%") 
  (skolem fmla nil))       
         
(defun skolem (f bind-list)
     (cond 
        ((literal f) f)
        ((eq '& (car f))   (list '& (skolem (cadr  f) bind-list)
                                    (skolem (caddr f) bind-list)))
        ((eq '+ (car f))   (list '+ (skolem (cadr  f) bind-list)
                                   (skolem (caddr f) bind-list)))
        ((eq '|all| (car f)) (list '|all|
                                 (cadr f)                          
                                 (skolem (caddr f) (cons (cadr f) bind-list))))
                  (t (skolem (subst (add-var (gsym 'F) bind-list)
                                    (cadr f)
                                    (caddr f))
                              bind-list))))
(defun add-var (s vl)
  (cond ((null vl) s)
	(t (cons s vl))))
(defun rem-dup-clauses (l-cl)
     (rem-aux l-cl nil))
(defun rem-aux (l-cl temp)
     (cond ((null l-cl) (reverse temp))
           (t (rem-aux (rem1-dup (car l-cl) (cdr l-cl))
                       (cons (car l-cl) temp)))))
(defun rem1-dup (cl l-cl)
    (cond  ((null l-cl) nil)
           ((p-equal cl (car l-cl)) (rem1-dup cl (cdr l-cl)))
           (t (cons (car l-cl) (rem1-dup cl  (cdr l-cl))))))


(defun and-pref (f)
     (cond ((literal f) nil)
           (t (eq '& (car f)))))

(defun or-pref (f)
     (cond ((literal f) nil)
           (t (eq '+ (car f)))))

(defun literal (l)
     (or (atomic-pref l)
         (eq '~ (car l))))

(defun atomic-pref (l)
     (or (atom l)
         (not (member (car l) '(~ & + <=> => |all| |exists|)))))
       

(defun print-clauses (cl-l)
  (mapcar #'out-a-clause cl-l))
(defun out-a-clause (cl)
  (mapcar #'(lambda (x)(print-atm x)) cl)(nl))
		      
         
           
 
