;; -*- Mode: LISP; Package: atp; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   atn-interface.cl
;;; Short Desc: general user interface for the  ATN tool
;;;            
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   18.9.91 - FB
;;; Author(s):  Mike Lenz, Fabio Baj
;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;; Mike Lenz wrote the character oriented version of the interface
;;; Fabio Baj wrote the graphic user interface
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :atn)

(defvar *net* nil)                  
(defvar *show-diagnostics* nil)    
(defvar *single-step* nil)  
(defvar *show-status* nil)
(defvar *show-graphics* nil)

(defun init-sizes()
  (setq *atn-tool-width* (scalew 575))  
  (setq *atn-output-width* (scalew 670)) 
  (setq *atn-tool-height* (scaleh 400)) 
  (setq *atn-tool-left* (scalew  380))
  (setq *atn-tool-bottom* (scaleh 470))

(setq *nodefont* (my-findfont  12 ))
(setq *labelfont* (my-findfont 11 ))
(setq *nodesize* (scalew 20))
(setq *margin* (scalew 40))
(setq *selfarrowlen* (scalew 70))

(setq *defwidth*  (scalew 300))
(setq *defheight* (scaleh 300)))


(setq *gin* t)
(defvar *last-sentence* "")

(defvar *got-step-command* nil)
(defvar *lexs-menu* nil)             (defvar *nets-menu* nil)
(defvar *atn-tool-window* nil)       (defvar *step-exit-button* nil)  
(defvar *atn-output-window* nil)     (defvar *sentence-window* nil)
(defvar *atn-buttons-list* nil)      (defvar *radio-buttons-list* nil)
(defvar *lexicon-button* nil)        (defvar *networks-button* nil)
(defvar *regs-button* nil)           (defvar *hold-button* nil)
(defvar *stack-button* nil)           (defvar *lookup-button* nil)
(defvar *new-sentence-button* nil)   (defvar *parse-sentence-button* nil)
(defvar *help-tool-button* nil)      (defvar *exit-tool-button* nil)
(defvar *graph-button* nil)          (defvar *step-button* nil)
(defvar *diag-button* nil)           (defvar *trace-button* nil)
(defvar *step-step-button* nil)      (defvar *step-go-button* nil)
(defvar *atn-tool-width*)   
(defvar *atn-output-width*)
(defvar *atn-tool-height*) 
(defvar *atn-tool-left*)
(defvar *atn-tool-bottom*) 
(defvar *net-name* "")                (defvar *lex-name* "")
(defvar *lexicon-filename* nil)
(defvar *network-filename* nil)




(defun atn-tool()
  (setq *net-name* "None")
  (setq *lex-name* "None")
  (clear-lexicon)
  (setq *last-sentence* "")
  
  (setq *show-status* nil)
  (setq *show-diagnostics* nil)
  (setq *show-graphics* nil)
  (setq *single-step* nil)
  (setq *net* nil)
  
  
  (setq *subnet* (ignore-errors (caar *net*)))
  (setq *got-step-command* nil)
  
  (setf  *atn-output-window* (make-instance 'scroll-display :title "ATN: Output"
					   ;;   :parent *atn-tool-window*
					    :font (my-findfont 10)
                                           :width *atn-output-width* 
					   :left   *atn-tool-left* 
					   :height  (- *atn-tool-height* (scaleh 200))
					   :bottom (- *atn-tool-bottom* 26 (- *atn-tool-height* (scaleh 200)))))
  (setf  *atn-tool-window* (make-instance 'display :title "Augmented Transition Networks Tool"
					 :left   *atn-tool-left*
					 :width  *atn-tool-width*
					 :height *atn-tool-height* :font (my-findfont 13)
					 :bottom *atn-tool-bottom*)) 
  (setf  *sentence-window* (make-instance 'scroll-display :title "ATN: Current Sentence"
					 :parent *atn-tool-window*
					 :width  (- *atn-tool-width* (scalew 20))
					 :left    (scalew 10)    :font (my-findfont 10)
					 :height  (scaleh 50)
					 :bottom  (scaleh  50)))
  (format-display  *sentence-window* "~A" *last-sentence*)
  (show-step-window)	
  (disable-step-window)	
  (setq *nets-files* (make-directory "net-"))
  (if *nets-files*
      (setf *nets-menu*   
	(make-instance 'menu
	  :items (mapcar '(lambda (filename)
			   (list filename (list 'lambda  nil (list 'set-net filename))))
			 *nets-files*)
			 :query "Networks")))
   (setq *lexs-files* (make-directory "lex-"))
   (if *lexs-files*
       (setf *lexs-menu* 
	 (make-instance 'menu
	   :items (mapcar '(lambda (filename)
			    (list filename (list 'lambda  nil (list 'set-lex filename))))
			 *lexs-files*)
	   :query "Lexicons")))
  
   (setf  *exit-tool-button*    (make-instance 'push-button :font (my-findfont 13) :label "Exit" :width  (scalew 100)))
  (setf *help-tool-button*  (make-instance 'help-button :subject "Augmented Transition Networks Tool" 
					   :width  (scalew 100)   :label "Help"  :font (my-findfont 13)
					   :technical (add-path "atn-tool-tec.asc" *atn-path*)
					   :general (add-path "atn-tool-desc.asc" *atn-path*)))
				 
  (setf  *new-sentence-button* (make-instance 'push-button  :font (my-findfont 13)
				 :label  "New Sentence" :width (scalew 180)))
  
  (setf  *lookup-button*       (make-instance 'push-button :label "Lookup" :font (my-findfont 13)
					      :width (scalew  100)))
  (setf  *graph-button* (make-instance 'radio-button  :font (my-findfont 13) :label "Graphics"  :status *show-graphics* :value nil))
  (setf  *step-button* (make-instance 'radio-button    :font (my-findfont 13)  :label "Step"  :status *single-step*  :value nil ))
  (setf  *diag-button*  (make-instance   'radio-button  :font (my-findfont 13) :label "Debug" :status *show-diagnostics* :value nil))
  (setf  *trace-button*  (make-instance 'radio-button  :font (my-findfont 13)  :label "Trace" :status *show-status*  :value nil ))
  (setf  *hold-button*  (make-instance 'push-button  :font (my-findfont 13) :label "Hold" :width (scalew 100) )) 
  (setf  *stack-button*  (make-instance 'push-button :font (my-findfont 13)  :label "Stack" :width  (scalew 100) ))
  (setf  *regs-button*   (make-instance 'push-button  :font (my-findfont 13) :label "Registers" :width (scalew 100) ))
  (setf  *networks-button*  (make-instance 'push-button  :font (my-findfont 13) :label *net-name*  :width (scalew 100)))
  (setf  *lexicon-button*   (make-instance 'push-button  :font (my-findfont 13) :label *lex-name* :width (scalew 100)))
  (setf  *parse-sentence-button* (make-instance 'push-button :font (my-findfont 13)  :label "Parse Current Sentence"  ))
  
  
  (set-button  *exit-tool-button*    *atn-tool-window*
	       :left (- (width *atn-tool-window*) (width *exit-tool-button* ) (scalew 10))
	       :bottom (scaleh 10)
	       :action '(lambda nil
			 (donedrawnet)
			 (donegraphics)
			 (close-display *atn-tool-window* )
			 (close-display *atn-output-window*)
			 (close-display *step-window*)
			 (setq  *execution-terminated* t)
			 ))
  (set-button   *help-tool-button*   *atn-tool-window*
		:left (scalew 10)
		:bottom (bottom *exit-tool-button* )
		)
  
  
  
  (set-button *parse-sentence-button*  *atn-tool-window*
	      :left   (center *parse-sentence-button*   *atn-tool-window*)
	      :bottom  (bottom *exit-tool-button* ) 
	      :action '(lambda ()
			(setq *got-step-command* nil)
			(clear-scroll *atn-output-window*)
			(doatn *subnet*(string-downcase *last-sentence*))
			(reset-button  *parse-sentence-button* )))
  
  (set-button  *step-button* *atn-tool-window*  
	       :left (+ (scalew 10) (* 0 (floor (-(width *atn-tool-window* ) (scalew 130)) 3)))
	       :bottom (-(height *atn-tool-window*) (scaleh  30))
	       :action '(lambda nil 
			 (setf *single-step* (status  *step-button*))
			 (if *single-step* 
			     (progn (enable-step-window) 
				    (setf *got-step-command* :next-step))
			   (disable-step-window))))
  (set-button  *diag-button* *atn-tool-window* 
	       :left (+ (scalew 10) (* 1 (floor (-(width *atn-tool-window* )  (scalew 130)) 3)))
	       :bottom (-(height *atn-tool-window*) (scaleh  30))
	       :action '(lambda nil (setf *show-diagnostics* (status *diag-button*))))
  (set-button  *trace-button* *atn-tool-window* 
	       :left   (+(scalew 10)0 (* 2 (floor (-(width *atn-tool-window* )  (scalew 130)) 3)))
	       :bottom (-(height *atn-tool-window*) (scaleh  30))
	       :action '(lambda nil (setf *show-status* (status   *trace-button*))))
  (set-button  *graph-button* *atn-tool-window*
	       :left (+ (scalew 10) (* 3 (floor (-(width *atn-tool-window* )  (scalew 130)) 3)))
	       :bottom (-(height *atn-tool-window*) (scaleh  30))
	       :action '(lambda nil 
			 (disable-button *graph-button*)
			 (setf *show-graphics* (status *graph-button* ))
			 (format-display *atn-output-window* "Graphics output ~a.~%"
			  (if *show-graphics* "on" "off"))
			 (if *show-graphics*
			     (if (not (null *net*))
				 (drawnetwork *net*))
			   (progn (donedrawnet)
				  (donegraphics)))
			 (enable-button  *graph-button*)))
  
  (set-button *new-sentence-button*    *atn-tool-window*
	      :left  (- (width  *atn-tool-window*)(width  *new-sentence-button*) 10)
              :bottom  (+ (bottom *sentence-window*)
                          (height *sentence-window*)  30)
	      :action '(lambda ()(get-new-sentence)
			(reset-button *new-sentence-button*  )))
  (set-button   *lookup-button*     *atn-tool-window*
		:left  (+ (scalew 10) (* 0 (floor (-(width *atn-tool-window* )   (scalew 120)) 3)))
		:bottom  (- (bottom *step-button*) (scaleh 50))
		:action '(lambda ()
			  (let   ((word
				   (ask  "Look for word: "
					 :title "Lookup: please type a word (<enter> for current word)"
					 :width (scalew 400)
					 :left (scalew 200)    :font (my-findfont 10)
					 :bottom (+ (scaleh 20) (bottom *lookup-button*))
					 :height (scaleh 50))))
			    (if (string= word "")
				(print-lookup *)
			      (print-lookup word)))
			  (reset-button *lookup-button*)))
  
  (set-button *hold-button* *atn-tool-window*
	      :left  (+ (scalew 10) (* 1 (floor (-(width *atn-tool-window* ) (scalew 120)) 3)))
	      :bottom (- (bottom *step-button*) (scaleh 50))
	      :action '(lambda ()(reset-button *hold-button*  )
		        (print-registers *registers* t 1)))
  (set-button *stack-button* *atn-tool-window*
	      :left  (+ (scalew 10) (* 2 (floor (-(width *atn-tool-window* ) (scalew 120)) 3)))
	      :bottom (bottom *hold-button*)
	      :action '(lambda ()(reset-button *stack-button*)
			(print-stack)))
  (set-button *regs-button*  *atn-tool-window*
	      :left  (+ (scalew 10) (* 3 (floor (-(width *atn-tool-window* ) (scalew 120)) 3)))
	      :bottom (bottom *hold-button*)
	      :action '(lambda ()(reset-button *regs-button*  )
		        (print-registers *registers* nil 1)))
  
  (write-display    *atn-tool-window*  "Using Lexicon:  " 
		    (left *sentence-window*) 
		    (- ( bottom *hold-button*) (scaleh 80)))
  (set-button *lexicon-button*  *atn-tool-window*
	      :left  (left *hold-button*)
	      :bottom  (- ( bottom *hold-button*) (scaleh 80))
	      :action '(lambda () (lex-button-action)))
  (set-button *networks-button*  *atn-tool-window*
	      :left  (- (width *atn-tool-window*)
			(width *networks-button* ) 
			(scalew 10))
	      :bottom (- (bottom *hold-button*) (scaleh 80))
	      :action (function net-button-action))
  (write-display    *atn-tool-window*  "Using Net:  "
		    (- (left   *networks-button*) (scalew 120))
		    (- ( bottom *hold-button*) (scaleh 80))) 
  
  (write-display    *atn-tool-window*  "Current word:  " 
		    (left *sentence-window*) 
		    (+ (bottom *sentence-window*) 
		       (height *sentence-window*) 30))
  )
  
		 
    
(defun center (but disp)
   (floor (- (width disp) (width but)) 2))
     

;;========================================================
;; a window for driving  the step mode

(defun show-step-window()
  (setq *step-window*
    (make-instance 'display 
     ;;; :parent *atn-output-window* 
      :width (scalew 120) :height (scaleh 110)
      :left (+ *atn-tool-left* *atn-tool-width*):bottom (scaleh 760) :title "Stepping"))
  (setf *step-exit-button* (make-instance 'push-button :label "Exit" :font (my-findfont 13)
					  :width (scalew  100)))
  (setf *step-go-button* (make-instance 'push-button :label "Skip" :font (my-findfont 13)
					:width (scalew  100 ))) 
  (setf *step-step-button* (make-instance 'push-button :label "Next Step" :font (my-findfont 13)
					  :width (scalew 100)))
  (set-button *step-exit-button* *step-window*  :left (scalew 10) :bottom (scaleh (+ 10 (* 0 30)))
	      :action '(lambda nil
			(protect-display *step-window* t)
			(reset-button *step-exit-button*)
			(setq *got-step-command* :quit)
			(protect-display *step-window* nil)))
  (set-button *step-go-button* *step-window*  :left (scalew  10) :bottom (scaleh (+ 10 (* 1 30)))
	      :action '(lambda nil
			(protect-display *step-window* t)
			(reset-button *step-go-button*)
			(setq *got-step-command* :go)
			(protect-display *step-window* nil)
			))
  (set-button *step-step-button* *step-window*  :left (scalew 10) :bottom (scaleh  (+ 10 (* 2 30)))
	      :action '(lambda nil 
			(protect-display *step-window* t)
			(setq *got-step-command* :next-step)
			(reset-button  *step-step-button*)
			(protect-display *step-window* nil))))

(defun disable-step-window()
  ( disable-button *step-exit-button*)  
  ( disable-button *step-step-button* )
  ( disable-button *step-go-button*))
					  

(defun enable-step-window()
  (enable-button *step-exit-button*)  
  (enable-button *step-step-button* )
  (enable-button *step-go-button*))

;;========================================================
;; This routine reads  new sentence from the user

(defun get-new-sentence ()
  (let ((new-sent  (ask  "Please enter a new sentence:
"                         :font (my-findfont 10)
			 :title "ATN"
			 :width (scalew 350)
			 :left (scalew 200) :bottom (scaleh 400)
			 :height (scaleh 150)))
	)
    (format t new-sent)
    (if (not (string= "" new-sent))
	(progn
	  (clear-scroll *sentence-window*)
	  (setq *last-sentence* new-sent)
	  (format-display  *sentence-window* *last-sentence*)))))
	 
  



;;;=======================================================
;;  Here we handle the selection of networks or lexicons
;;

(defun make-directory (string)
  (remove-if #'(lambda (filename)
		 (or
		  (not (begins-with string filename))
		  (terminates-with "~" filename)))
	     (sort
	      (mapcar #'file-namestring
		      (directory (namestring   *atn-path*)))
	      #'(lambda ( x y) ( string< x y)))))



(defun begins-with (prefix string)
  (string= prefix string :start1 0 :end1  ( length prefix)
	   :start2 0 :end2  ( length prefix)))


(defun terminates-with (substr str)
 (head-equal  (reverse (coerce str 'list))
	      (reverse (coerce substr 'list))))

(defun head-equal (list sublist)
  (cond ((null sublist) t)
	( (null list) nil)
	((equal (car list) (car sublist)) (head-equal (cdr list)( cdr  sublist)))))


   
(defun set-net (filename)
  (setq *network* nil)
  (if (ignore-errors (load  filename))
      (progn (setq *net* *network*)
	     (setf (label *networks-button*) (file-namestring filename))
	     (if (not (ignore-errors (setq *subnet* (caar *net*))))
		 (format-display *atn-output-window* "** Warning: Bad network format.~%"))
	    
	     (if *show-graphics*
		 (progn (donedrawnet)
			(drawnetwork *net*))))
    (format-display *atn-output-window* "** No network exists with the name ~a.~%" filename)))



(defun set-lex (filename)
  (setq *lexicon* nil)
  (if (ignore-errors (load filename))
      (progn 
	(store-lexicon *lexicon*)
	(setf (label *lexicon-button*) (file-namestring filename)))
      (format-display *atn-output-window* "** No lexicon exists with the name ~a.~%" filename)))
  
       
(defun lex-button-action ()
  (let ((lexicon (choose-file-dialog 
		  :directory (or *lexicon-filename*
				 pail-lib::*user-dir*
				 (concatenate 'string (namestring *atn-path*)
							     "lexicae/lex-1"))
		  :button-string "Load")))
    (when lexicon (set-lex lexicon)(setq *lexicon-filename* lexicon))
    (reset-button *lexicon-button*)))

(defun net-button-action ()
  (let ((network (choose-file-dialog 
		  :directory (or *network-filename* pail-lib::*user-dir*
				 (concatenate 'string (namestring *atn-path*)
							     "networks/net-1"))
		  :button-string "Load")))
    (when network   (set-net network)(setq *network-filename* network))
    (reset-button *networks-button*)))



;;;===========================================================



;; The main user routine for user;;;==== interaction.
;;
;; -- Note on hooks for later graphical UI: the calls for
;; a given action can be straightforwardly read from this
;; routine or from process-cmd...e.g. a "parse" command
;; should set *show-status* to nil to turn off trace
;; output, and call (doatn *subnet* lastsent) with
;; current values for starting subnet and sentence to parse.
;;

(defun atn ()
  (setq *show-status* nil)
  (setq *show-diagnostics* nil)
  (setq *show-graphics* nil)
  (setq *single-step* nil)
  (setq *subnet* (ignore-errors (caar *net*)))

  (let ((str "")
	(cmd nil)
	(lastsent ""))
    (loop
     (format t "~%atn> ")
     (setq str (read-line t))
     (setq cmd (read-symbols str))

     (if (not (process-cmd cmd))
	 (case (car cmd)
	   ((q quit)
	    (if *show-graphics*
		(progn (donedrawnet)
		       (donegraphics)))
	    (return-from atn nil))

	   ((p parse)
	    (progn
	      (if (cdr cmd)
		  (setq lastsent
			(string-downcase
			 (subseq str
				 (1+ (length (format nil "~A" (car cmd))))))))
	      (setq *show-status* nil)
	      (doatn *subnet* lastsent)))
	   ((t trace)
	    (progn	  
	      (if (cdr cmd)
		  (setq lastsent
			(string-downcase
			 (subseq str
				 (1+ (length (format nil "~A" (car cmd))))))))
	      (setq *show-status* t)
	      (doatn *subnet* lastsent)))

	   ((s step)
	    (progn (setq *single-step* t)
		   (setq *show-status* t)
		   (if (cdr cmd)
		       (setq lastsent
			     (string-downcase
			      (subseq str
				      (1+ (length
					   (format nil "~A" (car cmd))))))))
		   (doatn *subnet* lastsent)
		   (setq *single-step* nil)))

	   ((nil) t)
	   (otherwise
	    (format t "I'm sorry, I don't recognize the command `~a'.~%"
		    (car cmd)))
       )))))

;;
;; Called from ATN by examine-node when single-stepping.
;; Returns nil to signal the machine to abort its parse;
;; t otherwise.
;;

(defun get-step-command ()
  (if *gin*
      (gin-get-step-command )
    (loop
      (format t "~%step# ")
      (setq str (read-line t))
      (setq cmd (read-symbols str))
      (if (not (process-cmd cmd))
	  (case (car cmd)
	    ((nil) (return t))
	    ((q quit abort) (return nil))
	    
	    ((go) (progn (setq *single-step* nil)
			 (return t)))
	    
	    (otherwise
	     (format t "What?~%")))))))


;;===============================================
;; This works with the graphical interface


(defun gin-get-step-command ()
  (mp::process-wait "step"'(lambda () *got-step-command*))
  (let ((choosen-action  *got-step-command*))
    (setq  *got-step-command* nil)
    (case choosen-action
      ((:next-step) t)
      ((:quit) nil)
      ((:go) (progn 
		  (draw-filled-rectangle (display *step-button*) (+ (left *step-button*) 2)
					 (+ (bottom *step-button*) 2)
				  (- (width *step-button*) 3)
				  (- (height *step-button*) 3) :color gi::white)
	           (setf (status *step-button*) nil)
		   (setq *single-step* nil)
		   (funcall (action *step-button*))
	            
		     t)))))
  



;;
;; Returns t if cmd is handled; nil otherwise.
;;

(defun process-cmd (cmd)
  (case (car cmd)
	
    ((net network)
     (if (cdr cmd)
	 (let* ((newnetname (cadr cmd))
		(newnet (ignore-errors (eval newnetname))))
	   (if newnet
	       (progn (setq *net* newnet)
		      (if (not (ignore-errors (setq *subnet* (caar *net*))))
			  (format t "** Warning: Bad network format.~%"))
		      (if *show-graphics*
			  (progn (donedrawnet)
				 (drawnetwork *net*))))
	     (format t "** No network exists with the name ~a.~%" newnetname)))
       (princ *net*)))
    ((lex lexicon)
     (if (cdr cmd)
	 (let* ((newlexname (cadr cmd))
		(newlex (ignore-errors (eval newlexname))))
	   (if newlex
	       (store-lexicon newlex)
	     (format t "** No lexicon exists with the name ~a.~%" newlexname)))
       (princ *lex*)))
    ((subnet)
     (if (cdr cmd)
	 (setq *subnet* (cadr cmd))
       (princ *subnet*)))

    ((reg regs registers)
     (print-registers *registers* nil 1))
    ((hold)
     (print-registers *registers* t 1))
    ((stack)
     (print-stack))
    ((star *)
     (princ *))
    ((lookup)
     (let ((word (cadr cmd)))
       (if (or (null word)
	       (eql word '*))
	   (print-lookup *)
	 (print-lookup word))))
    ((diag)
     (progn
       (setq *show-diagnostics* (not *show-diagnostics*))
       (format t "Diagnostic output ~a.~%"
	       (if *show-diagnostics* "on" "off"))))
    ((graph)
     (progn
       (setq *show-graphics* (not *show-graphics*))
       (format t "Graphics output ~a.~%"
	       (if *show-graphics* "on" "off"))
       (if *show-graphics*
	   (if (not (null *net*))
	       (drawnetwork *net*))
	 (progn (donedrawnet)
		(donegraphics)))))
    ((help ?)
     (show-commands))

    (otherwise
     (return-from process-cmd nil)))

  t)

;;
;; Help display
;;

(defun show-commands ()
  (let ((margin 20))
    (format t "~%Command~VTAction~%-------~VT------~%" margin margin)
    (format t "parse [string]~VTParses <string> without displaying status messages, using the current network and lexicon.  If <string> is not specified, the most recently parsed string is used.~%~%" margin)
    (format t "trace [string]~VTParses <string> with status messages giving the current subnetwork, node, and arc being examined.~%~%" margin)
    (format t "step [string]~VTParses <string> with status messages and enters Step Mode, pausing whenever a new node is reached to wait for input from the user.~%~%" margin)
    (format t "net [network]~VTMakes <network> the currently active network; or prints out the currently active network if none is specified.~%~%" margin)
    (format t "lex [lexicon]~VTMakes <lexicon> the currently active lexicon; or prints out the currently active lexicon if none is specified.~%~%" margin)
    (format t "diag~VTToggles diagnostic output for all parses of input, output, registers, and stack when a new node is reached.~%~%" margin)
    (format t "regs~VTDisplays the current contents of the registers.~%~%" margin)
    (format t "stack~VTDisplays the current contents of the stack.~%~%" margin)
    (format t "*~VTDisplays the current value of the * register.~%~%" margin)
    (format t "quit~VTQuit the program.~%~%" margin)
    (format t "--Step Mode Commands~%~%")
    (format t "<return>~VTSingle step to next node.~%~%" margin)
    (format t "go~VTFinish the parse in trace mode without halting.~%~%" margin)
    (format t "abort~VTCancel the parse and return to main command mode.~%~%" margin)
    (format t "diag, regs, stack, * -- as above.~%")
    ))

;;
;; *************************************
;; *Also: new commands
;;   hold
;;   lookup <str>
;;   graph
;;   subnet <sn>  ; is this documented yet?
;;

;;
;; These Handle printing of register list, stack, and
;; the result of word-lookups.
;;

(defun print-registers (regs holdonlyp indent)
  (if (and (null regs) (not holdonlyp))
      (format-display *atn-output-window*  "~&~VTRegisters: None~%" indent)
    (progn (format-display *atn-output-window*  "~&~VTRegister    Value~%" indent)
	   (format-display *atn-output-window*  "~VT--------    -----~%" indent)
	   (if holdonlyp
	       (let ((reg (getr HOLD)))
		 (format-display *atn-output-window*  "~VT~a~VT~a~%"
			 indent (car reg) (+ indent 12) (cadr reg)))
	     (dolist (reg regs)
		     (format-display *atn-output-window*  "~VT~a~VT~a~%"
			     indent (car reg) (+ indent 12) (cadr reg)))))))

;;

(defun print-stack ()
  (format-display *atn-output-window*  "~&  Top of stack~%")
  (dolist (entry *stack*)
    (format-display *atn-output-window*  "-----------------~%")
    (format-display *atn-output-window*  "  Network    ~a~%" (pop-network (list entry)))
    (format-display *atn-output-window*  "  Node       ~a~%" (pop-node (list entry)))
    (format-display *atn-output-window*  "  Output     ~a~%" (pop-output (list entry)))
    (format-display *atn-output-window*  "  Action     ~a~%" (pop-action (list entry)))
    (print-registers (pop-registers (list entry)) nil 2))
  (format-display *atn-output-window*  "-----------------~%"))

;;

(defun print-lookup (word)
  (let ((lex (if (stringp word) (lookup-string (string-downcase word))
	       (lookup-string (string-downcase (format nil "~A" word))))))
    (if lex
	(dolist (entry lex)
          (format-display *atn-output-window*  "~&Word:      ~a~%" word)
          (format-display *atn-output-window*  "Category:  ~a~%" (car entry))
	  (format-display *atn-output-window*  "Features:  ~a~%" (cdr entry))
	  (format-display *atn-output-window*  "--------~%"))
      (format-display *atn-output-window*  "-- No entries for ~a --~%" word))))


;;
;; Utility functions
;;
;; Returns a list of objects as read from a string
;; i.e. reading "Pack my dog"
;; returns list of atoms (Pack my dog).
;; Note "" returns (nil).
;;

(defun read-symbols (str)
  (let ((len (length str))
	(output nil))
    (loop
      (multiple-value-bind
	  (word pos)
	 (read-from-string str nil)  
	;;   (format-display *atn-output-window*  "Word ~a pos ~a len ~a output ~a~%" word pos len output)
	(setq word (intern word :atn))
	(setq output (ngrab output word))
	(if (= pos len)
	    (return output))
	(setq str (subseq str pos))
	(setq len (- len pos))))))



;;
;; Returns the result of appending atm onto the end of alist
;; but does NOT make a copy of alist. (i.e. destructive)
;;

(defun ngrab (alist atm)
  (if (null alist)
      (list atm)
    (nconc alist (list atm))))