;;; -*- Mode: LISP; Package: atp; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   infer.cl
;;; Short Desc: this file contains the implementations of
;;                all the inference rules 
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   8.9.91 - FB
;;; Author:     Fabio Baj
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :atp)



;;--------------------------------------------------------------
;; The generic function mk-deduct takes two clauses and deduces
;; and stores new clauses, according to some inference rules.
;; These inference rules are specializations of the generic function
;; mk-deduct on according to different subclasses of clauses or user options

(defmethod mk-deduct ( x xx y yy))
(defmethod mk-deduct (x (x-clause clause-class)
		      y (y-clause clause-class))
  (if  $resolution$ 
      (make-resolutions x x-clause y  y-clause))
  (if (and $factoring$ (eq x y)) 
      (make-factors  x x-clause y  y-clause))
   (if (and $p-bin-sup$ (eq x y))
       (deduce-with-identity  x x-clause y  y-clause)))

(defmethod  mk-deduct   (x  (x-clause all-negative-clause)
			 y  (y-clause clause-class))
  (if $prolog-mode$ (prolog-deduction  x x-clause y  y-clause))
  (if $neg-resolution$      (all-neg-deductions x x-clause y  y-clause))
  (call-next-method))

 
;;--------------------------------------------------------------
;; RESOLUTION: this is the most general form of the inference
;;             rules used by the prover. Although it is not enabled
;;             by default (since there are more effective and specialized 
;;             inference rules), it is included in the ATP module for
;;             historical and didactical reasons, and can be selected 
;;             by user. The algorithm may appear complicate, but there
;;             is the problem of finding all resolvents between two clauses
;;             trying to unify EVERY negative atoms of the first clause
;;             with EVERY positive atom of the second. To find all the resolvent
;;             of two given clauses c1 ,c2 there is the need of two call:
;;             (make-deduction c1 c2) and  (make-deduction c2 c1)
;; 
;;    A Alpha => Beta,  Alpha1 => A1 Beta1
;; --------------------------------------- exists sub (A = A1)sub
;;    add (Alpha Alpha1 => Beta Beta1)sub
;;
;; Example:
;;
;;   c1   (axiom)                       C(d) & D(x) & C(e)   =>  E(x) 
;;   c2   (axiom)                       A(x)   =>  C(x) 
;; --------------------------------------------------------------------
;;   c3   (c1 resolution c2)            A(d) & D(x) & C(e)   =>  E(x) 
;;   c4   (c1 resolution c2)            A(e) & C(d) & D(x)   =>  E(x) 
;; 
;;   c3 is obtained unifying C(x) and   C(d) 
;;   c4 is obtained unifying C(x) and   C(e) 


    
(defmethod make-resolutions (x (x-clause clause-class)
		      y (y-clause clause-class))
  (let* ((nterm (negative-atoms  x-clause))
	     (posterm (positive-atoms x-clause))
	     (nterm-buff nterm)
	     (todo nil)
	     (literal nil)
	     (res nil)
	     (retry t)
	     (new-id nil)
	     (new-clause nil))
	(loop
	  (setq literal (pop nterm-buff))
	  (if (null literal) (return))
	  (setq todo (positive-atoms y-clause))
	  (loop
	    (if (null todo) (return))   
	    (setq res (resolution y-clause nterm posterm  literal todo))
	    (if (eq 'fail res) (return)) 
	    (setq new-clause (cadr res))
	    (if *overlap*
		(let* ((sigma (car res))		      
		       (to-overlap (apply-sub (append (negative-atoms  y-clause)
						      (positive-atoms y-clause))sigma))
		       (teta (overlap (append (remove literal nterm :test 'equalatom)
					      posterm)
				      to-overlap)))
		  (if (not (eq 'fail teta))
		      (setq new-clause (norm (apply-sub new-clause (car teta)))))))
	    (setq retry t)
	    (setq new-id (store  new-clause (list x 'resolution y)))
	    (if (and(not (null new-id))
		    (not (member  new-id $simps$))
		    (not (eq 'fail (gen-subsumes (get-clause new-id)(get-clause y)))))
		(progn 
		  (format-display *out-disp* "~A SUBSUMED BY ~A"
		   y
		  new-id)
		  (remove-from-sets y (delete-sets (eval $strategy$)))))
	    (setq todo (cddr res))
	    (if (not (active-clauses x y)) (return))
	    (if (not $en-backtrack$) (return)))
	  (if (not retry) (return)))))
  



(defun resolution (clause nterm posterm literal todo)
  (let ((res (first-p-mgu literal todo)))
    (cond ((eq 'fail res) 'fail)
	  (t (let ((s-clause-positive (apply-sub (positive-atoms clause) (car res)))
		   (s-clause-negative (apply-sub (negative-atoms clause) (car res)))
		   (s-nterm (apply-sub  nterm (car res)))
		   (s-posterm (apply-sub  posterm (car res)))
		   (s-literal (apply-sub literal (car res))))
	       (cons
		(car res)
		(cons
		 (make-clause
		  (append (remove s-literal s-clause-positive :test 'equalatom)
			       s-posterm)
		  (append s-clause-negative
			       (remove s-literal s-nterm :test 'equalatom)))
		  (cdr res))))))))



;;--------------------------------------------------------------
;; FACTORS   : Factors are needed for completness of the resolution
;;             method, furthermore they are very useful to simplify proofs.
;;             This is an inference rule that uses with only one clause,
;;             so, in order to achieve uniformity with the other rules
;;             it is called by make-deductions when the two clauses are 
;;             the same.
;; n-factor
;;
;;    A A1 Alpha => Beta, 
;; --------------------------------------- exists sub (A = A1)sub
;;    add (A Alpha => Beta)sub
;;
;; p-factor
;;
;;    Alpha => A A1 Beta
;; --------------------------------------- exists sub (A = A1)sub
;;    add (Alpha => A Beta)sub
;;
;; Example:
;;
;;   c1   (axiom)              P(x) & D(d) & D(y) & D(e)   =>  H(c,x) v H(y,f) 
;; ---------------------------------------------------------------------------
;;   c2   (c1 p-factor c1)     P(f) & D(d) & D(c) & D(e)   =>  H(c,f) 
;;   c3   (c1 n-factor c1)     P(x) & D(d) & D(e)   =>  H(c,x) v H(d,f) 
;;   c4   (c1 n-factor c1)     P(x) & D(d) & D(e)   =>  H(c,x) v H(e,f) 


(defmethod make-factors (x (x-clause clause-class)
			 y (y-clause clause-class))
  
  (let* ( (new-id nil)
	  (todo (positive-atoms x-clause))
	  (res (factor x-clause todo)))
    (loop 
      (if (eq 'fail res) (return))
      (setq new-id (store (car res) (list x 'p-factor x)))
      (setq res (factor x-clause (cdr res)))
      (if (and(not (null new-id))
	      (not (member  new-id $simps$))
	      (not (eq 'fail (gen-subsumes (get-clause new-id)(get-clause y)))))
	  (progn 
	    (format-display *out-disp* "~A SUBSUMED BY P_FACTOR ~A" y new-id)
	    (remove-from-sets y (delete-sets (eval  $strategy$)))))
      (if (not (active-clauses x y)) (return))
      (if (not $merge-backtrack$) (return))))
    (if (active-clauses x y)
      (let* ((clause (get-clause x))(new-id nil)
	     (todo (negative-atoms  clause))
	     (res (factor clause todo)))
	(loop 
	  (if (not (active-clauses x y)) (return))
	  (if (eq 'fail res) (return))
	  (setq new-id(store (car res) (list x 'n-factor x)))
	  (setq res (factor clause (cdr res)))
	  (if (and(not (null new-id))
		  (not (member  new-id $simps$))
		  (not (eq 'fail (gen-subsumes (get-clause new-id)
					       (get-clause y)))))
	      (progn 
		(format-display *out-disp* "~A SUBSUMED BY N-FACTOR ~A" y new-id)
		(remove-from-sets y (delete-sets (eval  $strategy$)))))
	  (if (not (active-clauses x y)) (return))
	  (if (not $merge-backtrack$) (return))))))
  


(defmethod factor ((clause clause-class)  todo)
  (let ((s-todo (merge1 todo)))
    (cond ((eq 'fail s-todo) 'fail)
	  (t (cons
	      (make-clause
	       (apply-sub (positive-atoms clause)  (car s-todo) )
	       (apply-sub (negative-atoms clause)  (car s-todo) ))
	      (cdr s-todo))))))
	     
(defun merge1 (nterm)
  (let (( s nil))
    (cond (( null nterm )'fail)
	  ((not(eq 'fail (setq s (merge2 (car nterm)(cdr nterm)))))
	   (cons  s (cdr nterm)))
	  (t (merge1  ( cdr nterm))))))

(defun merge2  (at nterm)
  (let (( s nil))
    (cond ((null nterm) 'fail)
	  ((not(eq 'fail (setq s (mgu at (car nterm)))))
	   s)
	  (t (merge2 at (cdr nterm))))))
      
	  


;;--------------------------------------------------------------
;; P-RESOLUTION: resolution with an atomic positive clause
;; 
;;    => P,  Alpha P1  => Delta
;; ---------------------------- exists sub (P = P1)sub
;;    add (Alpha => Delta)sub
;;

;; Example      =>A(a),  A(x)B(x)A(y)=>C(y)
;;              --------------------------
;;               add B(a)A(y) => C(y)   (with Alpha := B(x)A(y))
;;               add A(x)B(x) => C(a)   (with Alpha := A(x)B(y))


(defmethod mk-deduct  (x-id (Pclause      positive-atom)
		       y-id (AlphaP1Delta clause-class))
  (if  $p-resolution$
      (let* ((P (car (positive-atoms Pclause)))
	     (AlphaP1 (negative-atoms   AlphaP1Delta))
	     (result-AlphaP1 nil))
	(loop
	  (setq result-AlphaP1 (p-resolution P AlphaP1Delta AlphaP1))
	  (if (eq 'fail result-AlphaP1) (return))
	  (setq AlphaP1 (cdr  result-AlphaP1))
	  (store  (car result-AlphaP1) (list x-id 'p-res y-id))
	  (if (not (active-clauses x-id y-id)) (return))
	  (if (not $backtrack$)(return)))))  
  (call-next-method))

(defmethod p-resolution (P (clause clause-class) AlphaP1)
  (let ((sub-rest (first-p-mgu P AlphaP1)))
    (cond ((eq 'fail sub-rest) 'fail)
	  (t (cons 	      
	      (make-clause 
	       (apply-sub (positive-atoms clause)  (car sub-rest))
	       (remove (apply-sub  P (car sub-rest) )
		       (apply-sub (negative-atoms  clause)
				  (car sub-rest)) :test 'equalatom))
	         (cdr sub-rest))))))


;;-------------------------------------------------------------------
;; ALL-NEGATIVE-RESOLUTION: it is a inference rule in which one of the given clauses
;;           must contain only negative atoms. It is the basic inference
;;           rule of this ATP. It combines togheter ideas from resolution
;;           and from rewriting based theorem proving. It is isomorphic
;;           to the EN-superposition inference rule (Hsiang 87). 
;;           It is the same of a resolution and some factoring steps 
;;           made simultaneously. It requires unification of lists of atoms
;;           (modulo permutations). By collapsing Alpha and Alpha1(the largest
;;           unifiable sets of atoms) we obtain a shorter result if compared 
;;           with resolution.
;;
;;     Alpha1 A1 Gamma =>,  
;;     Alpha Beta => A Delta
;;   --------------------------------  Exists sub (A Alpha = A1 Alpha1)sub
;;     add (Alpha Beta Gamma => Delta)sub 
;;
;;  Example :
;;
;;   c1   (axiom)            A(a) & B(x) & R(y)  =>  
;;   c2   (axiom)            A(x) & D(x) & R(a)  =>  B(b) v F(x) 
;;----------------------------------------------------------------
;;   c3   (c1 n-res c2)      D(a) & A(a) & R(a)  =>  F(a) 
;;
;; Notice that if you use only resolution + factoring there is a lot
;; of intermediate steps to be stored :
;;   c1   (axiom)                       A(a) & B(x) & R(y)   =>  
;;   c2   (axiom)                       A(x) & D(x) & R(a)   =>  B(b) v F(x) 
;;   c3   (c1 resolution c2)            A(x) & D(x) & R(a) & A(a) & R(y)   =>  F(x) 
;;   c4   (c3 n-factor c3)              D(a) & R(a) & A(a) & R(x)   =>  F(a) 
;;   c5   (c3 n-factor c3)              A(x) & D(x) & A(a) & R(a)   =>  F(x) 
;;   c3 SUBSUMED BY N-FACTOR c5
;;   c6   (c5 n-factor c5)              D(a) & A(a) & R(a)   =>  F(a) 

(defmethod all-neg-deductions  (x  (x-clause all-negative-clause)
				y  (y-clause clause-class))
  (let* ((nterm (negative-atoms  x-clause))
	 (nterm-buff nterm) (todo nil)
	 (literal nil) (res nil) 
	 (retry t) (new-id nil)(new-clause nil))
    (loop
      (setq literal (pop nterm-buff))
      (if (null literal) (return))
      (setq todo (positive-atoms y-clause))
      (loop
	(values 
	 (if (null todo) (return))   
	 (setq res (resolution y-clause nterm nil  literal todo))
	 (if (eq 'fail res) (return)) 
	 (setq new-clause (cadr res))
	 
	 (let* ((sigma (car res))
		(to-overlap (apply-sub (negative-atoms  y-clause) sigma))
		(teta (overlap (remove literal nterm :test 'equalatom)
			       to-overlap)))
	   (if (not (eq 'fail teta))
	       (setq new-clause (norm (apply-sub new-clause (car teta))))))
	 (setq retry nil)
	 (setq new-id (store  new-clause (list x 'n-res y)))
	 (if (and(not (null new-id))
		 (not (member  new-id $simps$))
		 (not (eq 'fail (gen-subsumes (get-clause new-id)
					      (get-clause y)))))
	     (progn 
	       (format-display *our-disp* "~A SUBSUMED BY ~A" y  new-id)
	       (remove-from-sets y (delete-sets (eval $strategy$)))))
	 (setq todo (cddr res))
	 (if (not (active-clauses x y)) (return))
	 (if (not $en-backtrack$) (return)))
	(if (not retry) (return))))))  


;;-------------------------------------------------------------------
;; PROLOG-RESOLUTION: resolution between an all-negative clause
;;                    (subgoal) and a horn clause. It is the resolution 
;;                    rule implicitly used by prolog interpreters
;;
;;  H Alpha => , Beta => H1
;; ---------------------------- exists sub (H = H1)sub
;;    add (Beta Alpha =>)sub
;;
;;  Example :
;;
;;  uncle(X,john) mother(Y,john)=>, 
;;  brother(U,Y1) father(Y1,Z) => uncle(U,Z)
;; ----------------------------------------- 
;;  add  brother(X,Y1) father(Y1,john) mother(Y,john)=>, 
;; (with Z-->john, U-->X)

(defmethod prolog-deduction  (x-id  (goal all-negative-clause)
			      y-id  (horn-clause clause-class))
  (let ((new-goal nil))
   
    (cond
     ((eq '! (car (negative-atoms goal))) 
      (cut-the-stack) 
      (setq new-goal
	(make-clause 
	 nil
	 (cdr (negative-atoms goal))))
      (store (prolog-sem-red new-goal) 
	     (list x-id 'prolog-res 'cut)))
     (t (setq new-goal (prolog-res goal horn-clause))
	(if (not (eq 'fail new-goal)) 
	    (progn
	      (setq *succeded-goal*  x-id)
	      (if (contains-the-cut (get-clause  y-id))
		  (put-a-cut-marker-in-the-stack))
	      (store (prolog-sem-red new-goal)
		     (list x-id 'prolog-res y-id))))))))

(defmethod prolog-res ((goal all-negative-clause) 
		       (horn-clause clause-class))
  (let ((sigma (mgu (car (positive-atoms horn-clause))
		    (sem-re-duce (car (negative-atoms goal))))))
    (cond ((eq 'fail sigma) 'fail) 
	  (t  (apply-sub (remake-clause nil
			  (append
			   (negative-atoms  horn-clause)
			   (cdr (negative-atoms goal)))
			  horn-clause)
			 sigma)))))


;;-------------------------------------------------------------------
;; PARAMODULATION:    This inference rule explicitly deals with the equality
;;                    predicate. Hsiang (Hsiang87) showed that it can be
;;                    restricted by the use of a complete simplification ordering
;;                    of first order terms
;;
;; Alpha1 => (s = t) Beta1,
;; (Alpha2 => Beta2)[r]
;; ---------------------------- exists sub (s=r)sub AND (t >/= s)sub
;; Alpha1_sub Alpha2_sub[t_sub] => Beta1sub Beta2_sub[t_sub] =>  
;;
;;  Example :
;;
;;   c1   (axiom)                   Q(x)   =>  (f(x) -> x) v K(x) 
;;   c2   (axiom)                   P(f(a))   =>  R(f(a)) 
;;--------------------------------------------------------------------
;;   c3   (c1 para-res c2)          Q(a) & P(a)   =>  K(a) v R(a) 
;;
;;   where s is f(x), t is x, r is f(a) and  (f(a) >lro a).
;;   sub is (x/a)
;;

(defmethod mk-deduct  (x (x-clause para-clause)
		       y (y-clause clause-class)) 
  (if  (or  $paramodulation$ $restricted-paramodulation$)
      
      (let* ((eqt-set  (collect-equations (positive-atoms x-clause)))
	     (todo nil)
	     (x-clause-rest nil)  
	     (newclause-todo nil) (eqt nil))
	(loop
	  (if (null eqt-set) (return)
	    (setq eqt (pop eqt-set)))
	  (setq todo (append (negative-atoms  y-clause)
			     (positive-atoms  y-clause)))
	  (setq x-clause-rest
	    (change-class 
	     (make-clause
	      (remove eqt (positive-atoms x-clause) :test 'equal)
	      (negative-atoms  x-clause))
	     'para-clause))
	  (dolist (sstt (two-senses eqt))
	    (loop
	      (setq newclause-todo 
		(paramodulation sstt  x-clause-rest y-clause  todo))
	      (if (eq 'fail newclause-todo) (return))
	      (setq todo (cdr newclause-todo))
	      
	      (store  (car newclause-todo) (list x 'para-res y)))))))
  (call-next-method))
    
(defmethod  paramodulation (ss-tt (para-clause para-clause)
			    (clause clause-class) todo)
  (if (is-int-var (car  ss-tt)) 'fail
    (let* ((ss (car ss-tt)) (tt (cadr ss-tt))
	   (sigma-todo (first-deep-mgu ss todo)))
      
      (if (eq 'fail sigma-todo) 'fail
	(let ((ss1 (apply-sub ss (car sigma-todo)))
	      (tt1 (apply-sub tt (car sigma-todo))))
	  
	  (if (order-is-violated ss1 tt1) 'fail
	    (cons 
	     (make-clause
	      (append (apply-sub  (positive-atoms para-clause) (car sigma-todo))
		      (subst tt1 ss1(apply-sub  (positive-atoms clause) (car sigma-todo)) :test 'equal))
	      (append (apply-sub  (negative-atoms  para-clause) (car sigma-todo))
		      (subst tt1 ss1(apply-sub  (negative-atoms  clause) (car sigma-todo)) :test 'equal))
	      )
	     (cdr sigma-todo))))))))
		 
	
(defun order-is-violated (ss1 tt1)
    (let ((order (lro ss1 tt1)))
      (if $restricted-paramodulation$
	   (or  (eq order '<>)(eq order '<))
	(eq order '<))))
	      
	  

;;-------------------------------------------------------------------
;; RESOLUTION-WITH (x=x)
;;               :The equality axiom all (x=x) must not explicitly given
;;                by the user. This inference rules takes in account of it.
;;                
;;   (s = t) Alpha => Beta 
;; ---------------------------- exists sub (s = t)sub
;;    add(Alpha => Beta )sub
;;
;;  Example :
;;
;;   c1   (axiom)                   (x = A) & R(x)   =>  Q(x) 
;;   c2   (c1 identity-res c1)      R(A)   =>  Q(A) 
;;
;; Note: this inference rule is exacplt a p-resolution
;;       with the positive clause    => (x=x)

(defmethod deduce-with-identity (x (x-clause clause-class)
				 y (y-clause clause-class))
   (let* ((identity '(= -1 -1))
	     (todo (negative-atoms  y-clause))
	     (res nil))
	(loop
	  (setq res (p-resolution identity y-clause todo))
	  (if (eq 'fail res) (return))  (setq todo (cdr res))
	  (store  (car res) (list x 'identity-res y))
	  (if (not (active-clauses x y)) (return))
	  (if (not $backtrack$)(return)))))

 
