;;; -*- Mode: LISP; Package: cky; Parstax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   cky-parser.cl
;;; Short Desc: CKY parser: 
;;;            
;;; Version:    1.0
;;; Status:     Review
;;; Last Mod:   18.3.92 - FB
;;; Author(s):  Rod Johnson, Fabio Baj
;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;; Rod Johnson wrote a character oriented version 
;;; Fabio Baj   wrote  the graphic user interface
;;; --------------------------------------------------------------------------


;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :cky)


; Selectors for parse lists

(defmacro t-wait (c) `(car ,c))
(defmacro t-done (c) `(cdr ,c))

; Selectors for states

(defmacro s-root(s) `(car ,s))
(defmacro s-tree(s) `(cadr ,s))
(defmacro s-tail(s) `(caddr ,s))
(defmacro s-next(s) `(caaddr ,s))

; Selectors for rules

(defmacro r-lhs(r) `(car ,r))
(defmacro r-rhs(r) `(cdr ,r))

; Selectors for complex categories

(defmacro c-cat(c) `(car ,c))
(defmacro c-eqn(c) `(cdr ,c))


; Pattern matchers

(defmacro match (a b) `(equal ,a ,b))
(defmacro match* (a b) `(member ,a ,b :test #'equal))

(defun parse (sent)
  (if (equal *parser* "ckydot") 
      (setq *chart* (ckydot sent nil 0))
    (setq *chart* (cky sent nil 0))))
  
				      
(defun cky (sent chart pos)
  (cond (sent
         (let   ((nextpos (1+ pos)))
	   (inform (format nil "" ))
	   (format-display *cky-subdisp* "~D ~A ~D" pos  (car sent) nextpos )
	   (cond ((null (cdr sent))  (inform (format nil "~A"  nextpos))))
	   (let ((column (build (car sent) chart nextpos)))
                   (cky (cdr sent) (cons column chart) nextpos))))
        (t chart)))

(defun ckydot (sent chart pos)
  (cond (sent
         (let ((nextpos (1+ pos)))
	   (inform (format nil "" ))
	   (format-display *cky-subdisp* "~D ~A ~D" pos  (car sent) nextpos )
	   (cond ((null (cdr sent)) (inform (format nil "~A"  nextpos))))
	   (let ((column (build-dot (nextword-dot (car sent)) chart nextpos)))
	     (ckydot (cdr sent) (cons column chart) nextpos))))
        (t chart)))

(defun build (word chart pos)
  (if *show-chart* (make-a-parse-step ))
  (inform (format nil "~%Cell [ ~A , ~A ]" (1- pos ) pos))
  (goto-graph-chart (1- pos) pos)
  (let ((column (make-array pos )))
    (setf (aref column (1- pos)) (closure (nextword word) (1- pos) pos))
    (buildrest column chart (- pos 2) pos)))

(defun buildrest (column chart prev pos)
  (cond ((minusp prev) column)
        (t
         (inform (format nil  "~%Cell [ ~A , ~A ]"  prev pos))
	 (goto-graph-chart  prev pos)
         (let ((cell (mkcell nil column chart prev (1- pos) pos)))
	   (cond (cell (setf (aref column prev) (closure cell prev pos))   ))
	   (buildrest column chart (1- prev) pos)))))


(defun build-dot (entry chart pos)
(if *show-chart* (make-a-parse-step))
   (let ((column (make-array pos)))
    (if *trace* (progn
		  (inform (format nil "~%Cell [ ~A ,  ~A ] ~A"
				  (1- pos ) pos  (sayword (1- pos) pos)))
		  (mapc #'(lambda (word)
			    (inform (format nil "   ~a" (showitem word))))
			entry)))
    (setf (aref column (1- pos)) (closure-dot (cons nil entry) (1- pos) pos))
    (show-graph-cell (aref column (1- pos)) (1- pos) pos)
    (buildrest-dot column chart (- pos 2) pos)))

(defun buildrest-dot (column chart prev pos)
;  (format t "sono in buildrest-dot ~D ~d ~%" prev pos)
  (cond ((minusp prev)
         column)
        (t
	 (inform (format nil  "~%Cell [ ~A , ~A ] ~A"  prev pos (sayword prev pos)))
	 
	 (let ((cell (mkcell-dot nil column chart prev (1- pos) pos)))
	   (cond (cell (setf (aref column prev) cell)))
	   (buildrest-dot column chart (1- prev) pos)))))


(defun nextword (word)
  (inform (format nil   "  Look up '~A' in the lexicon" word))
  (mapcar #'(lambda (entry)
	    (inform (format nil  "    ~A -> ~A" (car entry) word ))
            (chart-inform (format nil  " ~A -> ~A" (car entry) word ))
	    (list (car entry) (list word)))
          (lookup word)))

(defun nextword-dot (word)
  (mapcar #'(lambda (entry)
                   (list (car entry) (list (list word)) nil))
          (lookup word)))


(defun mkcell (cell column chart from mid to)
  (cond ((eq from mid)
         cell)
        (t
         (inform (format nil	 "  Cell [ ~A , ~A ] * Cell [ ~A , ~A ]"
				 from  mid  mid  to))
         (let ((more (star (aref (car chart) from) (aref column mid))))
	   (mkcell (append more cell)
		   column
		   (cdr chart)
		   from
		   (1- mid)
		   to)))))

(defun mkcell-dot (cell column chart from mid to)
;(format t "sono in MKCELL-DOT CHART= ~a  ~D ~d ~d ~%"  chart from mid to)
  (let ((closed-cell nil))
    (cond ((equal from mid)
	   (setq closed-cell (closure-dot cell from to))
	   (show-graph-cell closed-cell from to) closed-cell)
	  (t
	   (inform 
	    (format nil	 "  Cell [ ~A , ~A ] * Cell [ ~A , ~A ] ~A"
		    from  mid  mid  to  (format nil "~A & ~A" (sayword from mid)(sayword mid to))))
	   
	   (let* ((temp (star-dot (t-wait (aref (car chart) from))
			      (aref column mid)))
		  (more (cond ((null cell) temp)
			      ((null temp) cell)
			      (t (cons (append (car temp) (car cell))
				       (append (cdr temp) (cdr cell)))))))
	     (mkcell-dot more column (cdr chart) from (1- mid) to))))))



(defun star (cell1 cell2)
  (mapcan #'(lambda (prod-rule)
                   (cond ((eq (length prod-rule) 3)
                          (combine cell1
                                   cell2
                                   (r-lhs prod-rule)
                                   (first (r-rhs prod-rule))
                                   (second (r-rhs prod-rule))
                                   nil))))
          *rules*))

(defun star-dot (cell1 cell2)
  (do ((result nil)
       (list1 cell1 (cdr list1)))
      ((null list1) result)
      (do ((item1 (car list1))
           (list2 (t-done cell2) (cdr list2)))
          ((null list2))
          (let ((item2 (car list2)))
               (cond ((match (s-root item2) (c-cat (s-next item1)))
                      (let ((newitem (list (s-root item1)
                                           (cons item2 (s-tree item1))
                                           (cdr (s-tail item1)))))
                           (setq result (complete newitem result)))))))))


(defun combine (cell1 cell2 root left right result)
  (mapc 
   #'(lambda (item1)
       (and 
	(eq (s-root item1) left)
	(mapc #'(lambda (item2)
		  (and 
		   (eq (s-root item2) right)
		   (let ((new (list root item1 item2)))
		     (and 
		      (not (member new result :test #'equal))
		      (setq result (cons new result))
		      (inform (format nil "    ~A  -> ~A ~A " root left  right ))
		      (chart-inform (format nil " ~A  -> ~A ~A " root left  right ))))))
	      cell2)))
   cell1)
  result)

(defun closure (cell from to)
  (inform (format nil "  Close Cell [ ~A , ~A ]" from  to))
  (closecell cell))

(defun closure-dot (cell from to)
  (inform (format nil "  Close Cell [ ~A , ~A ]  ~A" from  to  (sayword from to)))
  (do ((newcell (star-dot *rules* cell) (star-dot *rules* newcell)))
      ((null newcell) cell)
      (setq cell (cons (append (car newcell) (car cell))
                       (append (cdr newcell) (cdr cell))))))


(defun closecell (cell)
  (let ((newcell (singleton cell)))
       (cond ((equal cell newcell)
              cell)
             (t (closecell newcell)))))

(defun singleton (cell)
  (mapc
   #'(lambda (prod-rule)
       (and (= (length prod-rule) 2)
	    (mapc #'(lambda (item)
		      (and (eq (s-root item) (first (r-rhs prod-rule)))
			   (let ((new (list (r-lhs prod-rule) item)))
			     (and (not (member new cell :test #'equal))
				  (setq cell (cons new cell))
				  (inform (format nil  "    ~A -> ~A"
						  (r-lhs prod-rule)
						  (s-root item)))
				  (chart-inform (format nil  " ~A -> ~A"
						  (r-lhs prod-rule)
						  (s-root item)))
))))
		  cell)))
   *rules*)
  cell)

(defun cky-answer ()
  (if (equal *parser* "ckydot")
      (and *chart*
	   (mapcar #'clean-tree 
		   (sentence (t-done (aref (car *chart*) 0)))))
    (and *chart* (sentence (aref (car *chart*) 0)))))

(defun sentence (cell)
  (cond ((null cell) nil)
        ((member (s-root (car cell)) *initial*)
         (cons (car cell) (sentence (cdr cell))))
        (t (sentence (cdr cell)))))

(defun chart (from to)
  (cond ((or (< from 0) (>= from (length *chart*))) nil)
        ((= to -1) (aref (car *chart*) from))
        ((or (<= to from) (> to (length *chart*))) nil)
        (t (aref (nth (- (length *chart*) to) *chart*) from))))

(defun showcell (i j)
  (if (equal *parser* "ckydot")
      (progn (let ((cell (chart i j)))
	       (format t "ECCO CELL ~A~%" cell) 
	       (cond (cell
		      (inform (format nil "Cell [ ~d , ~d ]" i j))
		      (cond ((t-done cell)
			     (inform "  Complete")
			     (mapc #'(lambda (item)
				       (inform  (showitem item)))
				   (t-done cell))))
		      (cond ((t-wait cell)
			     (inform "  Incomplete")
			     (mapc #'(lambda (item)
				       (inform  (showitem item)))
				   (t-wait cell))))))))
    (progn (let ((cell (chart i j)))
	     (cond (cell
		    (format-display *cky-subdisp*  "Cell [" i "," j "]" N)
		    (mapc
		     #'(lambda (item)
			 (format-display *cky-subdisp*  (B 4) "[" (s-root item) " -> "
					 (s-root (first (s-tree item))))
			 (cond ((cdr (s-tree item))
				(format-display *cky-subdisp*  B
						(s-root (second (s-tree item))))))
			 (format-display *cky-subdisp*  "]" N))
		     cell)))))))

(defun sizeof (cell)
  (let ((size 0))
       (mapc #'(lambda (item)
                      (setq size (+ size (length item))))
             cell)
       size))

(defun sizeof-dot (cell)
  (let ((size 0))
       (mapc #'(lambda (item)
                      (setq size (+ 1
                                    size
                                    (length item)
                                    (cond ((s-tree item) 1) (t 0)))))
             cell)
       size))



(defun chartstats ()
  (if (equal *parser* "ckydot")
      (progn 
	(let ((cells 0)
	      (size (length *chart*))
	      (conses 0)
	      (completes 0)
	      (incomplete 0))
	  (mapc #'(lambda (column)
		    (do ((i 0 (1+ i)))
			((= i (car (array-dimensions column))))
		      (let ((cell (aref column i)))
			(cond (cell
			       (setq cells (1+ cells))
			       (setq conses (+ conses
					       (sizeof-dot (t-done cell))
					       (sizeof-dot (t-wait cell))))
			       (setq completes
				     (+ completes
					(length (t-done cell))))
			       (setq incomplete 
				     (+ incomplete 
					(length (t-wait cell)))))))))
		*chart*)
	  (format-display *cky-subdisp*  "~D/~D NON-EMPTY CELLS~%~D COMPLETED ITEMS~%~D CONSES~%" 
			  cells  (/ (* size (1- size)) 2)  completes  conses )))
    (let ((cells 0)
	  (size (length *chart*))
	  (conses 0)
	  (completes 0))
      (mapc #'(lambda (column)
		(do ((i 0 (1+ i)))
		    ((= i (car (array-dimensions column))))
		  (let ((cell (aref column i)))
		    (cond (cell
			   (setq cells (1+ cells))
			   (setq conses (+ conses (sizeof cell)))
			   (setq completes
				 (+ completes (length cell))))))))
	    *chart*)
      
      (format-display *cky-subdisp*  "~D/~D NON-EMPTY CELLS~%~D COMPLETED ITEMS~%~D CONSES~%" 
		      cells  (/ (* size (1- size)) 2)  completes  conses ))))



(defun badrule ( string prod-rule)
  (if (equal *parser* "ckydot")
      (progn  (clear-scroll  *cky-msg-disp*)
	      (format-display *cky-msg-disp* "~% Illegal rule in Grammar (check which parsing algorithm are you using):")
	      (format-display *cky-msg-disp* (showrule prod-rule))
	      (setq *rules* nil))
    (progn (clear-scroll  *cky-msg-disp*)
	   (format-display *cky-msg-disp* "~% Illegal rule in Grammar (check which parsing algorithm are you using)):")
	   (format-display *cky-msg-disp* (showrule prod-rule))
	   (setq *rules* nil))))
 

(defun initialise () 
  (if (equal *parser* "ckydot")
      (initialize-dot)
    (initialize)))

(defun initialize-dot ()
  (catch
   (setq *rules*
         (mapcar #'(lambda (prod-rule)
                          (list (car prod-rule)
                                nil
                                (mapcar #'(lambda (cat)
                                                 (cond ((equal (car cat) 'self)
                                                        (cdr cat))
                                                       (t (badrule "" prod-rule))))
                                        (cdr prod-rule))))
                 *grammar*)))
  (cond ((not (boundp '*initial*))
         (setq *initial* nil)
         (mapc #'(lambda (prod-rule)
                        (setq *initial*
                              (ucons (s-root prod-rule) *initial*)))
	       *grammar*))))

(defun initialize ()
  (catch
   (setq *rules*
         (mapcar #'(lambda (prod-rule)
                          (cond ((> (length prod-rule) 3)
                                 (badrule "" prod-rule)))
                          (cons (car prod-rule)
                                (mapcar #'(lambda (cat)
                                                 (cond ((eq (car cat) 'self)
                                                        (cadr cat))
                                                       (t
                                                        (badrule "" prod-rule))))
                                        (cdr prod-rule))))
                 *grammar*)))
  (cond ((not (boundp '*initial*))
         (setq *initial* nil)
         (mapc #'(lambda (prod-rule)
                        (setq *initial* 
                              (ucons (car prod-rule) *initial*)))
	       *grammar*))))


;;;====================================
;; REDEFINED FOR PAIL INTERFACE
;;;====================================
(defun inform (string)
     (if *trace* (format-display *cky-subdisp* string)))

(defun flatc (expr)
  (length (format nil "~a" expr)))

;;;====================================
;;interface with the stepper
;;;====================================
(defun make-a-parse-step ()
  (enable-button (pop  *sentence-but-list*))
  (mp::process-wait "Step parser"  #'(lambda nil *next-step*))
  (setq *next-step* nil))


(defun complete (item cell)
  (cond ((s-tail item)
         (cond ((member item (t-wait cell) :test #'equal) cell)
               (t (inform (showstate item))
                  (cons (cons item (t-wait cell)) (t-done cell)))))
        ((member item (t-done cell) :test #'equal) cell)
        (t (inform (showstate item))
           (cons (t-wait cell) (cons item (t-done cell))))))

(defun showstate (state)
  (let ((root (s-root state))
        (done (mapcar #'(lambda (node)
			  (s-root node))
                      (reverse (cdr (s-tree state)))))
        (tail (mapcar #'(lambda (node)
			  (c-cat node))
                      (s-tail state)))
        (down (cons (s-root (first (s-tree state)))
                    (mapcar #'(lambda (node)
				(s-root node))
                            (reverse (s-tree (first (s-tree state))))))))
    (concatenate 'string
      (format nil "   [~a -> " root)
      (map-conc #'(lambda (cat) (format nil "~A " cat )) done)
      (format nil "." (car down))
      (map-conc #'(lambda (cat) (format nil "~A " cat)) tail)
      (format nil "] & [~A -> " (car down))
      (map-conc #'(lambda (cat) (format nil "~A " cat)) (cdr down))
      (format nil ".] => [~a -> " root)
      (map-conc #'(lambda (cat) (format nil "~A " cat )) done)
      (format nil " ~a . " (car down))
      (map-conc #'(lambda (cat) (format nil "~A " cat)) tail)
      (format nil "]"))))

(defun showitem (item)
  (concatenate 'string
	      (format nil "[~A->"  (s-root item) )
	      (map-conc #'(lambda (tree)
			    (format nil "~A " (s-root tree)))
			(reverse (s-tree item)))
	      "."
	      (map-conc #'(lambda (cat)
			    (format nil "~A " (c-cat cat)))
			(s-tail item))
	      "]"))

(defun sayword (from to)
  (let ((size (- to from)))
    (cond ((= size 1)
	   (format nil "< ~A >" (nth from *current-sentence*)))
	  ((= size 2)
	   (format nil "<~A ~A>" (nth from *current-sentence*)
		   (nth (1+ from) *current-sentence*)))
	  ((> size 2)
	   (format nil  "<~A ... ~A>" (nth from *current-sentence*)
		   ( nth (1- to) *current-sentence*) )))))

(defun show-graph-cell(cell  i j)
 ; (format t "sono in show-graph-cell ~D ~d ~%" i j)
  (goto-graph-chart i j)
  (cond (cell  (cond ((t-done cell)
		      (chart-inform "  Complete")
		      (mapc #'(lambda (item)
				(chart-inform  (showitem item)))
			    (t-done cell))))
	       (cond ((t-wait cell)
		      (chart-inform "  Incomplete")
		      (mapc #'(lambda (item)
				(chart-inform  (showitem item)))
			    (t-wait cell)))))))


;;===================================

(defun clean-tree (tr)
   (cond ((null tr) nil)
	 ((atom tr) tr)
	 ((= 1 (length tr)) tr)
	 ((= 3 (length tr))
	  (cons  (clean-tree (first tr))
		 (clean-treeson (second  tr))))))


(defun clean-treeson (tr)
 (reverse (mapcar #'clean-tree tr)))