;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:RULES; Base:10; -*-
;;; ************************************************************************
;;; WIN-SHELL
;;; ************************************************************************
;;;
;;; Filename:   win-back
;;; Short Desc: WIN-SHELL - a tiny shell from the WINSTON-HORN book
;;; Version:    0.1
;;; Status:     Experimental
;;; Last Mod:    5.11.90 15:00:00 SK
;;; Author:     WINSTON, typed in: Hauser-Fischer
;;;
;;; ------------------------------------------------------------------------
;;; Change History:
;;; HF  11.11.87  first code, debugged
;;; SK  16.10.89  improved
;;; SK   5.11.90  some pail standards applied
;;; ------------------------------------------------------------------------
;;;
;;; DESCRIPTION
;;; -----------
;;; backward-chaining functions
;;; ------------------------------------------------------------------------
;
;This program section extends the small expert system in Winston/Horn's
;LISP 2nd Ed., so that simple backward-chaining becomes possible. 
;
;While FORWARD-CHAINING tries to deduce as much as possible from a given set
;of data and rules, BACKWARD-CHAINING only tries to prove a fact, i.e. trace
;it back to the given data by applying the rules backwards. The program pro-
;ceeds as follows:
;
; 1. It checks whether the fact matches the consequent of a rule. 
;       - if there's no match, it tries the next rule, until the rules are
;         exhausted.
;       - if there is a match, it replaces the pattern variables in the an-
;         tecedents as far as possible and starts to prove the antecedents
;         one by one by looking at the antecedent as a fact and reapplying 1.
; 2. A fact is proved if it can be found in the database. The facts in the 
;    database were either given or deduced by the rules.
; 3. The rules can only fire forwards, never backwards. Thus the backward-
;    relies on the forward-chaining component to produce new facts.

;---------------------------------------------------------------------------

(in-package :rules)

(defun backward-chain (fact &optional first-run)

     ;This is used for backward-chaining. If the backward-chaining is the
     ; the first run on the assertions, enter a "1" (or anything else that
     ; isn't NIL) as second argument when you call BACKWARD-CHAIN, e.g.
     ;
     ;      (backward-chain '(rex is a albatross) 1)
     ;
     ; That will tell the function to initialize RULES-USED to NIL. Omit
     ; the second argument on further runs (or if you used FORWARD-CHAIN
     ; before the backward-chaining) to prevent the loss of information
     ; stored in RULES-USED.
     ; Losing that information means that the program "forgets" how it
     ; deduced a fact, it will only tell you ... WAS GIVEN.

  (cond (first-run
         (setq rules-used nil)))       ;initializes RULES-USED if FIRST-RUN
	 			       ; is not-NIL. 

  (setq last-two-facts-to-proof nil    ;this variable is used to prevent
                                       ;  circular backward-chaining in
                                       ;  BACKWARD.
        *chaining-type* 'backward)     ;*CHAINING-TYPE* is a flag for
                                       ;  trace-functions
                                       
                                       
                                       
  (cond
   ((backward fact)                    ;if the fact can be proved
    (format t "BACKWARD CHAIN finished.")
    (format t "~%problem ~a solved." fact) 
    ;(how fact)                        ;tell us how
    )
   (t                                  ;else failure!
    (format t "~%problem ~a could not be solved." fact) nil)))
 
;---------------------------------------------------------------------------

(defun bc (fact &optional first-run)   ;If BACKWARD-CHAIN is too long to
   (backward-chain fact first-run))    ; to type, use BC.

;---------------------------------------------------------------------------

  
(defun backward (fact)

					;This function does the actual backward-chaining. It is called recursively
					; from PROOF.

  (prog (alist)
	 (cond (*trace-chaining* 
		(format t "### trying to prove: ~a~%" fact)))

					;Regarding the first two SETQ in BACKWARD:
					;
					;I encountered a problem when using a rule with a similar ANTECEDENT
					;and CONSEQUENT. The specific rule was CONTRAPOSITION1 which I used
					;in my little rule-base treating the Aristotelian syllogisms:
					;       
					;  (rule contraposition1
					;        (if (all (> a) (> x) are not (> b)))
					;        (then (all (< b) (< x) are not (< a))))
					;
					;This rule for instance deduces ALL MARRIED WOMEN ARE NOT EMANCIPATED
					;                          from ALL EMANCIPATED WOMEN ARE NOT MARRIED.
					;In backward-chaining this rule can turn into a cat that's chasing its
					;tail. The two assertions are symmetrical with respect to the rule -
					;whichever you feed as antecedent, you'll get the other as consequent.
					;So you backward-chain in circles until the stack overflows.
					;
					;To interrupt such a circle I have introduced a variable LAST-TWO-
					;FACTS-PROOF. As its name indicates it stores the last two facts BACK-
					;WARD tried to prove. If the newest fact is among them, I know that
					;I am circling and can stop the backward-chaining in that direction by
					;returning NIL.
      
	 (setq old-last-two-facts-to-proof ;I'm not sure whether this SETQ busi-
	   last-two-facts-to-proof)	; ness would work in Common Lisp. Some-
	 (setq last-two-facts-to-proof	; how I think that OLD-LAST-.. would
	   (list fact			; change, too, if LAST-.. were altered.
		 (car last-two-facts-to-proof)))
					;OLD-LAST-TWO-FACTS is needed for the
					; OR below, since the relevant fact for
					; the circling could have been thrown 
					; out by the new assignment of LAST-TWO-
					; FACTS-TO-PROOF
					;LAST-TWO-FACTS-TO-PROOF in the first
					; SETQ statement is initialized either
					; in FORWARD-CHAIN or in BACKWARD-CHAIN,
					; so it shouldn't be unbound here.
	 (return (cond ((or (equal fact	;The OR expression
				   (car old-last-two-facts-to-proof)) ;checks whether FACT
			    (equal fact	;is equal to the
				   (cadr old-last-two-facts-to-proof))) ;first or the second
			nil)		;(and last) element 
					;of OLD-LAST-..
					;If it is, we have a
					; circular run and
					; stop searching
					; this branch by re-
					; turning NIL.

		       ((setq alist (filter-assertions fact nil)) ;if FACT is already pre-
			(how fact)
			alist)		; sent (given or deduced in
					; earlier runs), tell us
					; how.
		       (t 
			(do ((rules-to-try ;parameter 1        ;This DO does two
			      rules	;  initial value 1  ; things: Take a
			      (cdr rules-to-try)) ;  update form 1    ; a list of rules
			     (progress-made nil)) ; parameter/init. 2 ; and hand them
			    ((null rules-to-try) ; termination test  ; down to TRY-
			     progress-made) ; return value      ; RULE until TRY-
			  (cond		; DO body           ; RULE is suc-
			   ((setq progress-made (try-rule (car rules-to-try) fact)) ; cessful or the
			    (setq rules-to-try nil) ; rules are ex-hausted.
			    (print progress-made)
			    )))))))) 
;---------------------------------------------------------------------------




(defun sort-by-determiners (clauses) 
  (if (null clauses) nil
    (if (presentp '< (car clauses)) 
	(append (sort-by-determiners (cdr clauses)) (list (car clauses))) 
      (cons (car clauses) (sort-by-determiners (cdr clauses))))))


(defun try-rule (rule fact)
  
  ;TRY-RULE separates the different parts of the rules. If there are pattern
  ; variables left that contain "<", they are replaced by ">", because the
  ; pattern-matcher can only match ">" freely. For rules without pattern
  ; variables this needn't worry you.
  ; After separating and replacing TRY-RULE tries to match FACT to a the
  ; consequents of the rule (the work is done here by CASCADE-THROUGH-
  ; ACTIONS).                               

  (let* ((rule-name
                    (rule-header rule))
         (if-list
                    (change-<-to-> (sort-by-determiners (rule-antecedents rule))))
         (then-list
                    (change-<-to-> (rule-conclusions rule)))
         (a-list
                    (cascade-through-actions fact then-list))
	 new-binding)
        
     (cond
       (a-list                                          ;If the A-LIST of the
                                                        ; match isn't NIL,
                                                        
           (cond ((and (use-rule rule a-list)           ; check if the FACT
                       (setq new-binding (car (filter-assertions fact nil)))) new-binding) ; can be obtained di-
                                                        ; rectly by applying
                                                        ; the rule. This 
                                                        ; would be the case
                                                        ; if all antecedents
                                                        ; are given. Return
                                                        ; T, because FACT is
                                                        ; thus proved

                 (t (setq if-list                       ;Else replace the 
                          (replace-pattern-variables    ; pattern variables
                                             if-list    ; in the IF-LIST and
                                             a-list))   ; try to prove the
                    (cond ((proof if-list)              ; elements in it.
                           (use-rule rule)
			   a-list)))))))) ; If so, apply the
                                                        ; rule, so that this
                                                        ; proven deduction
                                                        ; is stored in RULES-
                                                        ; USED via FEED-TO-
                                                        ; RULES-USED. We need
                                                        ; to have it in 
                                                        ; RULES-USED, so that
                                                        ; we can get the 
                                                        ; chain with HOW.
                                                        ; USE-RULE fires the
                                                        ; RULE forward.
;---------------------------------------------------------------------------

(defun lessp (x y) (< x y))


(defun proof (if-list)

;;; PROOF tries to construct a proof for the elements in the if-list by processing
;;; the first element directly (or with BACKWARD) and handing the rest of
;;; the elements back to PROOF for a recursive call.

  (cond ((null if-list) t)		;proving nothing returns T. This is
					; necessary to stop the recursion:
					; A call to PROOF with a NIL
					; if-list can come from the line
					; marked ";**%%"
        (t (do ((if-list-to-try		;parameter 1
                 if-list		; initial value 1
                 (cdr if-list-to-try))	; update form 1
                (success nil)		;parameter 2 and initial value 2 
                (stop nil))		;parameter 3 and initial value 3
               ((or success		;termination 
                    stop		;  test on
                    (null if-list-to-try)) ;    three lines
                success)		;return form

             (let*
		 ((fact-being-proved (car if-list-to-try))
		  (facts-awaiting-proof (cdr if-list-to-try))
		  (a-list (car (filter-assertions fact-being-proved nil))))
					;Association list:      
					;Tries to match FACT-BEING-PROVED to
					; the assertions in the database. The
					; CAR is used because FILTER-ASSERTIONS
					; returns one level of parentheses too
					; many. FILTER-ASSERTIONS can return
					; NIL, (T) or (A-LIST): NIL if there is
					; no match, (T) if there is a direct match
					; without pattern variables and (A-LIST)
					; if the match includes pattern variables.
                
               (cond (*trace-chaining* 
		      (format t "Facts that we are attempting to prove is ~a~%" if-list)))
	       (cond
                ((equal a-list 't)	;if the antecedent
                 (cond (*trace-chaining* 
                        (format t ">>>fact ~a matched directly~%" 
                                fact-being-proved))) ;SK
                 (setq success		; matched directly
		   (null facts-awaiting-proof))) ; and there are no
					; more facts to be
					; proved, stop be-
					; cause SUCCESS = T.
					; Else the DO-LOOP
					; will loop again 
					; with SUCCESS = NIL.
                                                         
                (a-list                           
                 (cond (*trace-chaining* 
                        (format t ">>> fact ~a matched ~%" 
                                fact-being-proved))) ;SK
					;if the antecedent 
                 (setq new-list		; matched and produced
		   (replace-pattern-variables ; A-LIST (A-LIST has a
		    if-list-to-try 	; a value, but not T!),
		    a-list))		; replace all the
					; pattern variables in
					; IF-LIST-TO-TRY that
					; A-LIST contains.
                 (cond ((proof (replace-pattern-variables (cdr new-list) new-list)) ; In addition to that
                        (cond (*trace-chaining* 
                               (format t ">>>next fact ~a matched too~%" 
                                       fact-being-proved))) ;SK
                        (setq success t)))) ; try to prove the re-
					; maining antecedents.
					; (NEW-LIST is IF-LIST-
					; TO-TRY with replaced
					; pattern variables.)
                  
                (t 
                 (cond ((setq a-list (backward fact-being-proved))
					; try to prove it.         
			(cond ((proof (replace-pattern-variables facts-awaiting-proof a-list)) ;**%%
			       (cond (*trace-chaining* 
				      (format t ">>>next fact ~a found ~%" 
					      fact-being-proved))) ;SK
			       (setq success t))))
					;If successful, prove the
					; the rest.
		       (t
			(cond (*trace-chaining* 
			       (format t "...proof failed ~a~%" 
				       fact-being-proved) ;SK
			       (format t "...asking user: ~a = " 
				       fact-being-proved))) ;SK

			(setq answer (r-query 
				      fact-being-proved))
			(cond ((equal answer nil)
			       (format t "false~%" 
				       fact-being-proved) ;SK
			       (setq stop t) nil)
			      (t
			       (format t "true~%" 
				       fact-being-proved) ;SK
			       (remember fact-being-proved)
			       (setq success
				 (null facts-awaiting-proof)))
			      )
			nil)))
		))))))
                                                ;If there is a pattern
                                                ; variable, the fact can't 
                                                ; be proved.
                                                
;---------------------------------------------------------------------------
                               
(defun r-query (fact)			; asks user about a fact;  don't ask user about facts with
					; variables in.

  (if (not (has-variable fact)) (mydialog fact)))

(defun cascade-through-actions (fact then-list)

  ;This function works its way through the THEN-LIST, trying to find a
  ; consequent that matches FACT. If successful, it returns the association
  ; list A-LIST.

  (do* ((then-list            ;parameter 1
          then-list           ; initial value 1
          (cdr then-list))    ; update value 1
          (a-list nil))       ;parameter 2 and initial value 2
       ((or (null then-list)  ;termination test: Terminates if THEN-LIST is
            a-list)           ; NIL or MATCH is successful, i.e. A-LIST is
                              ; not NIL.
        a-list)               ;return form
       (setq a-list           ;DO-body
             (match (car then-list) fact nil))))

;---------------------------------------------------------------------------

(defun presentp (item s)

  ;Checks whether ITEM is an atom of S or contained in an atom of s. There
  ; would be a more elegant version of PRESENTP with MAPCAR, but Winston/
  ; Horn's version in LISP 2nd Ed. doesn't work.

  (cond ((equal s item) t)
        ((atom s) nil)
        (t (or (presentp item (car s))
               (presentp item (cdr s))))))

;---------------------------------------------------------------------------

(defun change-<-to-> (list)
  
  ;Replaces all < with >. Needed to make replacement of pattern variables
  ; easier.

  (subst '> '< list))

;---------------------------------------------------------------------------

(defun replace-pattern-variables (s a-list)

  ;replace pattern-variables whenever they exist.

  (cond ((null s) nil)
	((eq t a-list) (break) s)
        ((variablep s)
         (cond ((assoc  s a-list)
                (cadr (assoc  s a-list)))
               (t s)))
	((atom s) s)
        (t (cons (replace-pattern-variables (car s) a-list)
                 (replace-pattern-variables (cdr s) a-list)))))

;---------------------------------------------------------------------------
; end of program
;---------------------------------------------------------------------------

