;;; -*- Mode: LISP; Package: GTRE; Syntax: Common-lisp; -*-
;;;
;;; *******************************************************
;;;
;;; PORTABLE AI LAB - EPFL
;;;
;;; *******************************************************
;;;
;;; Filename:   gtre.cl
;;; Short Desc: Trivial Rule Engine (version g)
;;;             (portions copyright (c) 1987 K. Forbus
;;;             University of Illinois.
;;; Version:    1.0
;;; Status:     experimental
;;; Last Mod:   08.07.91 - Fatma FEKIH-AHMED
;;; Author:     Ken FORBUS & Fatma FEKIH-AHMED
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; -------------------------------------------------------
;;; RCS $Log$
;;; -------------------------------------------------------

;;; =======================================================
;;; PACKAGE DECLARATIONS
;;; =======================================================


(in-package gtre)
;(export '(...))

;________________________________________________________________________
;			     structures.
;________________________________________________________________________

(defstruct (tre-class :named
                  (:print-function (lambda (d st ignore)
				     (format st "<tre-class ~d>"
					     (tre-class-name d)))))
  name	 ;a symbol
  facts  ;facts of this class
  rules) ;rules applicable to this tre-class

(defstruct (rule :named
	       (:print-function (lambda (r st ignore)
					    (format st "<rule ~d>"
						    (rule-counter r)))))
  counter ;unique "name"
  class   ;class it is linked to.
  matcher ;function that performs the match.
  body)   ;function that does the rule's work.

(defstruct (assertion :named
		   (:print-function (lambda (a st ignore)
				      (format st "<assertion ~d>"
					      (assertion-counter a)))))
  counter     ; unique "name"
  lisp-form   ; external expression
  tms-node    ; tms node for this fact
  class	      ; class of this piece of data
  plist)      ; local property list


;________________________________________________________________________
;		      global parameters & initialization
;________________________________________________________________________

(proclaim '(special *tms-node-printer*))

(defvar *queue* nil)           ;rules to be run.

(defvar *tre-class* nil)       ;holds all tre-class structures created so far.

(defvar *rule* nil)            ;holds all rule structures created so far.

(defvar *assert* nil)          ;holds all assert structures created so far.

(defvar *assert-counter* 0)    ;unique id for asserts. 

(defvar *rule-counter* 0)      ;unique id for rules.

(defvar *rules-run* 0)         ;keeps track of the number of rules which have 
			       ;been run, for debugging purpose.

(defvar *end-forms* nil)       ;an accumulator for the internal functions 
			       ;created when defining rules.

(defvar *rule-indexing* nil)   ;an accumulator for indexing forms created when
			       ;defining rules.

(defvar *bound-vars* nil)      ;used during rule definitions to list the
                               ;variables which will be bound at run time.

(defvar *rule-name-counter* 0) ;provides a suffix to help prevent collisions in
                               ;names of functions created internally to define
                               ;rules.

;________________________________________________________________________
;			  user interface
;________________________________________________________________________

;;the top level driver function run is provided to incrementally empty the 
;;queue as a side effect of a read-eval-print loop.
;;every time a fact is added, jtre must find all the rules which could operate
;;on that fact and queue the pair of them up.
;;it reads a form from the terminal and evaluates it. the form can thus be any 
;;lisp-form or a call to some tre function. 
;;typically, rules are defined and facts are asserted. during the evaluation, 
;;of these forms, the variable *queue* is augmented with rules to run. these
;;are run after the evaluation.
;;one can exit with quit, exit or stop.
;
(defun run () ;; toplevel driver function
    (format t "~%>>")
    (do ((form (read) (read)))
        ((member form '(quit stop exit)) nil)
        (format t "~%~a" (eval form))
        (run-rules)
        (format t "~%>>")))

(defun quotize (pattern)
  (cond ((null pattern) nil)
	((variable? pattern) pattern)
	((not (listp pattern)) (list 'quote pattern))
	(t `(list ,@ (mapcar #'quotize pattern)))))

;;enqueue pushes a pair (rule-body rule-environment) onto *queue* 
;
(defun enqueue (new) (push new *queue*))

;;dequeue pops *queue* 
;
(defun dequeue () (pop *queue*))

;________________________________________________________________________
;			 database system
;________________________________________________________________________

;;facts and rules must be administrated somehow, so they are kept in a 
;;database that is implemented as a list of classes, kept in the variable 
;;*tre-class*. a class is a structure with three slots:
;;
;;  symbol, facts and rules. 
;;
;;each symbol is associated with a class. facts and rules are lists of facts
;; (= assertions) and rules that are involved with the symbol.
;;when a fact is asserted, its class must be found. the fact itself is just a
;;lisp-form.
;;
;;the tre database indexing is based on using the leftmost constant symbol:
;;
;;so, the algorithm for finding the class is to find the leftmost constant
;;variable or symbol in the fact. if this happens to be a variable, its 
;;symbol-value is used. when the symbol is finally found, it should have
;;property tre-class, which is its class. if it doesn't have any, then a 
;;new class is created and stored under this property and in the database 
;;*tre-class*. 
;;
;;this method has two advantages:
;;
;;  1) it allows storing the class on property lists which is important
;;     for those implementations of common lisp which don't have hash-tables.
;;
;;  2) it also allows forms which contain variables in their car to be indexed.
;;
;;classes are used to store rules and data.
;;;

;;get-class performs the mapping from lisp forms to class structures,creating 
;;classes as needed. notice that an error is signaled if an unbound tre 
;;variable is the leftmost symbol of the form.
; 
(defun get-class (fact &aux class val)
  (cond ((null fact)
	 (ferror "~%nil cannot be a tre-class or fact!"))
	((listp fact) (get-class (car fact)))
	((variable? fact)
	 (cond ((boundp fact) (get-class (symbol-value fact)))
	       (t (error "~%tre-class unbound: ~a" fact))))
	((symbolp fact)
	 (cond ((setq class (get fact 'tre-class)) class)
	       (t (setq class 
		    (make-tre-class :name fact :facts nil :rules nil))
		  (setf (get fact 'tre-class) class)
		  (push class *tre-class*)
		  class)))
	(t (error "bad tre-class type: ~a" fact))))

;;the contents of the database can be inspected with fetch, which returns all
;;assertions whose lisp form unifies with the given pattern. so, a problem 
;;solver can access the database using fetch and then, filter the results
;;to select the appropriate belief states.
;;as in rules, if the leftmost symbol is a tre variable, fetch won't work.
; 
(defun fetch (pattern &aux bindings unifiers)
  (dolist (candidate (tre-class-facts (get-class pattern)) unifiers)
    (setq bindings (unify pattern (assertion-lisp-form candidate)))
    (unless (eq bindings 'fail)
      (push (sublis bindings pattern) unifiers))))

;;referent retrieves specific assertions. referent simply does a linear search
;;down the list of assertions associated with a class.
;;the optional argument to referent determines whether or not facts are 
;;installed in the database (via insert) if they don't already exist.
;;this option is provided so that failed queries using referent don't fill up 
;;the database with useless assertions.
;
(defun referent (fact &optional (virtual? nil))
  (if virtual? (insert fact) (referent1 fact)))

(defun view-node (node) (assertion-lisp-form (tms-node-datum node)))

(defun referent1 (fact)
  ;special case of constant data
  (dolist (candidate (tre-class-facts (get-class fact)))
    (when (equal (assertion-lisp-form candidate) fact)
      (return candidate))))

;________________________________________________________________________
;		        variables and unification
;________________________________________________________________________


;;remember that a tre variable is any string whose first character is a "?".
;;the predicate variable? performs this test. 
; 
(defun variable? (x)
  (and (symbolp x) ;a symbol whose first character is "?"
       (string-equal "?" (symbol-name x) :end2 1)))


;;the unifier listed in the following code (unify, with helpers unify1, 
;;unify-variable, and free-in?) is suitable for fetching constant data, but is
;;not recommended for general use. unify is not used to compute whether a 
;;trigger matches an assertion.

;;a and b being fact-patterns, unify consists in a recursive algorithm, so 
;;unify1 is called with null bindings.
;
(defun unify (a b) (unify1 a b nil))

;;a and b being fact patterns, unify1 is a recursive function that will build 
;;up the bindings-parameter to an a-list with variables and values.
;;the function returns either fail or this a-list.
;
(defun unify1 (a b bindings)
   (cond ((equal a b) bindings)
	 ((variable? a) (unify-variable a b bindings))
	 ((variable? b) (unify-variable b a bindings))
	 ((or (not (listp a)) (not (listp b))) 'fail)
	 ((not (eq 'fail (setq bindings
			       (unify1 (car a) (car b) bindings))))
	  (unify1 (cdr a) (cdr b) bindings))
	 (t 'fail)))

;;unify-variables unifies a variable with an arbitrary expression. the value 
;;of the variable is found by assoc in the bindings.
;;
;;  1) if there is a value, the value has to be unified with the expression.
;;
;;  2) if there is no value, then the variable has to be free in the 
;;     expression.
;;
;;     2a) if it isn't free, then the wole unification fails.
;;
;;     2b) if it is free, the variable is then associated with the expression
;;         and pushed onto the bindings.
;
(defun unify-variable (var exp bindings &aux val)
  (setq val (cdr (assoc var bindings)))
  (cond (val (unify1 val exp bindings))
	((free-in? var exp bindings) (cons (cons var exp) bindings))
	(t 'fail)))

;;in free-in?, a variable is free in an expression if it doesn't refer back to
;;any quantified variable.
;;in other words, a bound variable is one that occurs in an enclosing
;;quantifier.
;;if there are no free variables, then the expression/formula is closed.
;;by adding universal or existential quantifiers for all free variables in an
;;expression, one obtains the universal or existential closure.
; 
(defun free-in? (var exp bindings)
  (cond ((null exp) t)
	((equal var exp) nil)
	((variable? exp) (free-in? var (cdr (assoc exp bindings)) bindings))
	((not (listp exp)) t)
	((free-in? var (car exp) bindings)
	 (free-in? var (cdr exp) bindings))))

;________________________________________________________________________
;		       building and installing rules
;________________________________________________________________________ 

;; here things get tricky, because we want to do as much work
;; as possible at rule expansion time to allow the lisp compiler
;; to do as much optimization as possible.

;;rules are written with a trigger-fact-pattern and a body. the rule should
;;be fired when a fact that unifies with the pattern is asserted (entered into
;;the database). when a rule is inserted, it's also tried to run it on the 
;;facts that already exist in the database.

;;each rule has a unique number and is stored in a rule-structure. all 
;;rule-structures are stored in the variable *rules*. the numbers are 
;;administered by *rule-counter*. 

;;rules and facts are both associated to classes. the class of a rule is found
;;with the first constant in the trigger-part. the rule structure has a slot 
;;for the class, for easy cross-reference (back-pointer). the class structure
;;also has a slot for a list of rules.


;;the macro rule is used for easy definition of rules:
;;
;;if the rule has more than one trigger, it is converted into a sequence
;;of nested rules. this conversion is made using the function make-nested-rule.
;;in all the new created rules, the macro internal-rule is substituted 
;;for rule, another macro, in the body of the rule, to keep trace of the 
;;fact that they are not top level rules.
;;
;;the functions defined within a rule have to migrate to a top-level form,
;;in order to be compiled. this means nested rules have to be treated 
;;differently than rules which appear at the top-level of a file.
;;
;;*end-forms* accumulates the functions generated during the processing of 
;;rule definition. (see the build-rule function)
;;
;;*rule-indexing* holds the forms (results of build-rule) to index top-level
;;rules. 
; 
(defmacro rule (triggers &rest body)
  (let ((*end-forms* nil)
	(*rule-indexing* nil))
    (add-rule (car triggers)
	      (subst 'internal-rule
		     'rule
		     (make-nested-rule (cdr triggers) body)))
    `(progn 'compile ,@ *end-forms* ,@ *rule-indexing*)))

;;rule and internal-rule are virtually the same. the only difference is that 
;;rule places its result, the indexing form for the rule, onto *rule-indexing*
;;and internal-rule simply returns the form so it will appear in the 
;;appropriate scope.
; 
(defmacro internal-rule (triggers &rest body)
  `(add-internal-rule ,(car triggers)
	      ,(make-nested-rule (cdr triggers) body)))

;;the function make-nested-rule performs the "unrolling" of the triggers.
;;the real work is done by build-rule. 
; 
(defun make-nested-rule (triggers body)
  (cond ((null triggers) body)
	(t `((add-internal-rule ,(car triggers)
			    ,(make-nested-rule (cdr triggers) body))))))

;;add-rule is called by the macro rule to define and add a rule (for each 
;;trigger of the principal rule) to the database.
; 
(defun add-rule (trigger body)
  (push (build-rule trigger body) *rule-indexing*)
  nil)

;;add-internal-rule works exactly in the same way as add-rule except that
;;instead of pushing the result onto *rule-indexing* , it simply returns it.
; 
(defmacro add-internal-rule (trigger body) (build-rule trigger body))


;; building a rule involves analyzing the trigger to see what
;; special-purpose unifier is needed, constructing a function
;; to do the work of the body, and building a form to index it.

;;build-rule makes the necessary calls to build the match function (which 
;;open-codes the unification) and the body function (which carries out the 
;;necessary bindings and tests for executing the body of the rule), as well as
;;creating the indexing form.
;
;;three kinds of arguments are passed to the body function. 
;;
;; 1) first, there are any variables which are newly bound by the current 
;;    trigger's pattern. 
;;
;; 2) second, there can be variables passed in from outer rules, stored in
;;    *bound-vars*. 
;;
;; 3) finally, depending on the trigger condition, the tms node associated
;;    with the fact might need to be passed in. recall that under the :intern
;;    condition, the rule body is executed no matter what the belief status 
;;    of the fact which matched the trigger pattern. however, the :in and :out
;;    conditions require the right belief status of the fact before running 
;;    the body.
;;
;;    the way used here to do that is to always queue the body for execution.
;;    if the belief conditions aren't right, the rule is stored under the 
;;    tms node to be re-queued when its status changes again.
;;    accessing the node when doing the matching is simply saving it away as an
;;    argument to the body function for later use. 
 
(defun build-rule (trigger body &aux match-function
			          body-function)
  (multiple-value-bind (pattern condition var test)
    (parse-rule-trigger trigger)
  (setq match-function (generate-match-function pattern var test condition))
  (setq body-function (generate-body-function pattern condition var body))
  (push match-function *end-forms*)
  (push body-function *end-forms*)
  `(insert-rule (get-class ,(get-trigger-class pattern))
		(function
		  ,(if *bound-vars*
		      `(lambda (p)
			 (,(cadr match-function) p ,@ *bound-vars*))
		       (cadr match-function)))
		(function
		  ,(if *bound-vars*
		       (let ((tv (nreverse (pattern-free-variables trigger))))
			 (unless (eq condition ':intern)
			   (push '*the-node* tv))
		       `(lambda ,tv
			  (,(cadr body-function) ,@ tv
			   ,@ (scratchout tv *bound-vars*))))
		       (cadr body-function))))))

(defun get-trigger-class (trigger)
  (cond ((variable? trigger)
	 (if (member trigger *bound-vars*) trigger
	     (error "~%trigger tre-class is unbound -- ~a." trigger)))
	((not (listp trigger)) (list 'quote trigger))
	(t (get-trigger-class (car trigger)))))


;________________________________________________________________________
;		     generating the body function
;________________________________________________________________________ 


;;for more explanations about this part of the code, please refer to the
;;jtre-expl3.txt file that contains a little traced and commented example.

(defmacro with-pushed-variable-bindings (new-bindings &rest body)
  `(let ((*bound-vars* (append ,new-bindings
			       (scratchout ,new-bindings *bound-vars*))))
     ,@ body))

;;the function scratchout removes from the list l2 all the elements that 
;;appear in the list l1.
;
(defun scratchout (l1 l2)
  ;non-destructive and order-preserving
  (dolist (el1 l1 l2) (setq l2 (remove el1 l2))))

;;the function generate-rule-function-name simply takes the pattern symbol
;;concatenated with the current value of *rule-name-counter* to create a new 
;;function name.
; 
(defun generate-rule-function-name (pattern)
 (intern (format nil "~a-~a" pattern (incf *rule-name-counter*))))

;________________________________________________________________________
;		      open-coding unification
;________________________________________________________________________

;;"open-coding" means "putting something in-line".  so instead of calling a
;;general-purpose unify procedure, jtre calculate a special-purpose procedure
;;to do the pattern-matching for each rule. so, open-coding the unification 
;;is done by generating specialized match functions.

;;for more explanations about this part, see the jtre-expl2.txt file that 
;;holds a little traced and commented example.

;;the generate-match-function has to build the "match-function" corresponding
;;to the given trigger (composed by all pattern, var, test, and condition 
;;arguments).
; 
(defun generate-match-function (pattern var test condition)
  (multiple-value-bind (tests binding-specs)
    (generate-match-body pattern (pattern-free-variables pattern) test)
    `(defun ,(generate-rule-function-name pattern) (p ,@ *bound-vars*)
       (if (and ,@ tests)
	   (values t (list ,@ (if var '(p)) ,@ (reverse binding-specs))
		   ,(unless (eq condition ':intern) t))))))


(defun generate-match-body (pattern vars extra-test
			       &aux structure-tests equal-tests
				    binding-specs var-alist)
  (dolist (test (generate-unify-tests pattern vars nil 'p)
		(values (append structure-tests equal-tests
				(if extra-test
				    (list (sublis var-alist extra-test))))
			binding-specs))
    (cond ((variable? (car test))
	   (setq equal-tests (append (generate-pairwise-tests (cdr test))
				     equal-tests))
	   (push (car (last test)) binding-specs)
	   (if test (push (cons (car test) (car (last test))) var-alist)))
	  (t (push test structure-tests)))))

;;generate-pairwise-tests deals with the case where a variable appears more
;;than once in a pattern and generates one single match test bound to it.
; 
(defun generate-pairwise-tests (tests)
  (cond ((or (null tests) (null (cdr tests))) nil)
	(t (cons (list 'equal (car tests) (cadr tests))
		 (generate-pairwise-tests (cdr tests))))))

;;generate-unify-tests builds the necessary conditions to match the pattern.
;;generate-unify-tests is called recursively to generate the intermediate 
;;tests.
; 
(defun generate-unify-tests (pattern vars tests path)
  (cond ((null pattern) (cons `(null ,path) ;this is the end.
			      (cons `(not (null ,(cadr path)))
				    tests))) ;and it did go out this far.
	((member pattern vars) ;must see if the pattern has been bound
	 ;elsewhere, and if it has, test to see if the element here is
	 ;consistent with that earlier binding.
	 (let ((previous (assoc pattern tests)))
	   (cond (previous ;add this position to test it
		   (setf (cdr previous) (cons path (cdr previous)))
		   tests)
		 (t (cons (list pattern path) tests)))))
	;if variable, it must be bound so test against the current value.
	((variable? pattern) (cons `(equal ,pattern ,path) tests))
	;if not a list, then see if equal (should be number check as well...)
	((not (listp pattern)) (cons `(equal ',pattern ,path) tests))
	(t (generate-unify-tests (cdr pattern) vars
             (generate-unify-tests (car pattern) vars
		(cons `(listp ,path) tests) ;avoid lisp errors
		(list 'car path)) ;extend the path.
             (list 'cdr path))))) ;extend path in other direction

;;pattern-free-variables calls pattern-free-variables1 to return the list 
;;of the pattern's variables that are free.
;;
;;note:
;;
;; a variable is free if it doesn't appear in the *bound-vars* list.
;;
;;*bound-vars* represents the variables that will be bound at run-time for
;;any stage of the analysis.
; 
(defun pattern-free-variables (pattern) (pattern-free-vars1 pattern nil))

;;pattern-free-variables1 is just an auxiliary function to pattern-free-vars
;;that allows a recursive implementation.
; 
(defun pattern-free-vars1 (pattern vars)
  (cond ((null pattern) vars)
	((variable? pattern)
	 (if (or (member pattern vars)
		 (member pattern *bound-vars*)) vars (cons pattern vars)))
	((not (listp pattern)) vars)
	(t (pattern-free-vars1 (cdr pattern)
			       (pattern-free-vars1 (car pattern) vars)))))

;________________________________________________________________________
;         recursive macroexpansion & clause expansion
;________________________________________________________________________

;knowing what forms to skip over is still very implementation
;dependent.  it gets hairier when one allow assertions to
;have procedural side-effects.
;blecherous, but necessary due to bugs in symbolics compiler
;when dealing with certain nested closures.  sigh.

;;; since rules can contain other rules, it is important to expand
;;; the internal rules in an environment where the what will be known
;;; at run-time is marked.  this means fully expanding all rules when
;;; first encountered. 
 
(defun fully-expand-body (body)
  (cond ((null body) nil)
	((not (listp body)) body)
	((symbolp (car body))
	 (case (car body)
	   (lambda `(lambda ,(cadr body)
		      ,@ (fully-expand-body (cddr body))))
	   (do `(do ,(cadr body) ,(caddr body)
		  ,@ (cdddr body)))
	   ((dotimes dolist let prog progn progv)
	    `(,(car body) ,(cadr body)
	      ,@ (fully-expand-body (cddr body))))
	   (t (let ((new-body (macroexpand body)))
		(cons (fully-expand-body (car new-body))
		    (fully-expand-body (cdr new-body)))))))
	(t (let ((new-body (macroexpand body)))
	     (cons (fully-expand-body (car new-body))
		 (fully-expand-body (cdr new-body)))))))

;________________________________________________________________________ 
;			    running rules
;________________________________________________________________________

;;the class structure is the "meeting-ground" for assertions and rules which
;;might match. so, whenever a rule is added, insert-rule tests it against 
;;all the assertions in its class.
; 
(defun insert-rule (class matcher body &aux rule)
  (setq rule (make-rule :matcher matcher
			:body body
			:class class
			:counter (incf *rule-counter*)))
  (push rule *rule*)
  (push rule (tre-class-rules class)) ;indexing
  (setf (rule-class rule) class) ;backpointer
  (dolist (candidate (tre-class-facts class))
    (try-rule-on rule candidate)))

;;and, whenever an assertion is added, try-rules tests it against all the 
;;rules in its class.
; 
(defun try-rules (fact)
  (dolist (rule (tre-class-rules (assertion-class fact)))
    (try-rule-on rule fact)))

;;the test is actually carried out by try-rule-on, which runs the rule's match
;;function on the given pattern.
;;if the matcher is satisfied, the rule's body function is added to the 
;;bindings generated by the matcher, and queued for eventual execution. 
; 
(defun try-rule-on (rule assert)
  (multiple-value-bind (okay? bindings node?)
    (funcall (rule-matcher rule) (assertion-lisp-form assert))
    (when okay?
;      (let ((*package* (find-package "GTRE-EXAMPLE")))
;	(format t "~&NODE?: ~S BINDINGS: ~S BODY: ~S"
;	      node? bindings (rule-body rule)))
      (when node? 
	(setq bindings (cons (assertion-tms-node assert) bindings)))
      (enqueue (cons (rule-body rule) bindings)))))

;;the function run-rules empties the queue, and should be called whenever
;;a fact, justification or rule is added.
;;
;;whenever a rule is triggered on a fact, the environment specified by the 
;;unification of the trigger pattern with the fact is stored.
;;the code which comprises the body of the rule is then executed in the 
;;environment given by the unification.
;;
;;run-rules runs the rules in *queue*. if *debug*, then the number of rules
;;is displayed.
; 
(defun run-rules ()
    (do ((form (dequeue) (dequeue))
         (counter 0 (1+ counter)))
        ((null form)
         (if *debug-tms*
	     (let ((*readable* t))
	       (format-display *debug-tms*  "~%    ~a assertions and/or rules run."  counter))))
        (incf *rules-run*)
      (apply (car form) (cdr form))))

;;the variant run-rules-exhaustively turns off contradiction checking while
;;the rules are being run, and should be used very carefully.
; 
(defun run-rules-exhaustively ()
  (let ((*disable-contradiction-checking* t))
    (run-rules)))


(defun show-data (&optional (stream t) &aux counter)
  (setq counter 0)
  (dolist (class *tre-class* counter)
    (dolist (datum (tre-class-facts class))
      (incf counter)
      (format stream "~%~a" (assertion-lisp-form datum))))) 

(defun all-contradictory-consequences? (node)
  (and (not (null (tms-node-consequences node)))
       (dolist (just (tms-node-consequences node) t)
	 (when (not (tms-node-contradictory?
		     (justification-consequence just)))
	   (return nil)))))

(defun printable-node-list (nodes)
  (remove-if-not
   #'(lambda (node)
       (or (and (in-node? node) 
		(null (tms-node-consequences node)))
	   (all-contradictory-consequences? node)))
   nodes))

;;; =======================================================
;;; END OF FILE
;;; =======================================================
