;;; -*- Mode: LISP; Package: GTRE-EXAMPLE; Syntax: Common-lisp; -*-
;;;
;;; *******************************************************
;;;
;;; PORTABLE AI LAB - EPFL
;;;
;;; *******************************************************
;;;
;;; Filename:   nmjinterface.cl
;;; Short Desc: Completes GTRE (Please refer to gtre.cl file)
;;;             into NMJTRE.
;;;             Portions copyright (c) 1987 K. FORBUS
;;;             University of Illinois.
;;; Version:    1.0
;;; Status:     experimental
;;; Last Mod:   10.02.92 - Fatma FEKIH-AHMED
;;; Authors:    Kenneth D. FORBUS & Fatma FEKIH-AHMED
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; -------------------------------------------------------
;;; RCS $Log$
;;; -------------------------------------------------------

;;; =======================================================
;;; PACKAGE DECLARATIONS
;;; =======================================================

(in-package gtre)
;(export '(tre-init ...))

;; allows rules to be triggered on specific truth conditions,
;; adds variables for whole fact of trigger, and escape to lisp
;; code within triggers.

;;;;general view:
;;
;;;
;;in tre, data is indexed by dividing it into classes. 
;;given the form of an assertion, its class is the leftmost symbol.
;;whenever an assertion is added to the database, its class is retrieved (or
;;created if necessary), and the assertion is stored in a list under that 
;;class.
;;
;;whenever a rule is created, it's stored under the class corresponding to its 
;;trigger. a trigger and an assertion which are not in the same class have no
;;chance of matching.
;;
;;so, the class data structure helps reducing the work required to find 
;;compatible collections of rules and assertions.
;;;
;
;;;note:
;;
;; the notion of class doesn't imply any form of inheritance or 
;; hierarchical structure.
;;;
;
;;;since rules doesn't have unique symbolic names, the rule structure includes
;;a counter to provide a unique indicator for printing and retrieving 
;;particular rules during debugging.
;;although the user (or program) can write rules with multiple triggers,
;;internally jtre turns them into an equivalent sequence of nested rules.
;;
;;then, jtre "open-codes" the unification required for matching a trigger to
;;provide speed-up. "open-coding" means "putting something in-line". so instead
;;of calling a general-purpose unify procedure, jtre calculate a 
;;special-purpose procedure to do the pattern-matching for each rule.
;;
;;assertions are implemented with the "assert" structure. the lisp-form is what
;;the user sees. the "assert" structure corresponding to a particular lisp form
;;can be retrieved by calling "referent". the "tms-node" field provides a place
;;to store the corresponding jtms node, just as the "datum" field in the 
;;"tms-node" structure provides a place to the corresponding "assert".
;;the "p-list" field provides a place for the problem solver to store other
;;useful information about a particular assertion.
;;;
 
(proclaim'(special *contradiction-handler* *disable-contradiction-checking*))

;________________________________________________________________________
;;;;initialization
;________________________________________________________________________

;;tre-init is an init and reset function. it clears database, variables and
;;property lists.
;;tre-init must be called once before operations and each time the system has 
;;to be cleared.
;
 
(defvar *assert-list* nil) ;used for the graphical interface
(intern "GOD" :gtre)
(intern "USER" :gtre)


(defun tre-init ()
  (setq *queue* nil)
  (setq *rules-run* 0)
  (setq *rule-counter* 0)
  (setq *end-forms* nil)
  (setq *rule-indexing* nil)
  (setq *bound-vars* nil)
  (setq *rule-name-counter* 0)
  (setq *assert-counter* 0)
  (jtms-init) 
  (dolist (cl *tre-class*) (remprop (tre-class-name cl) 'tre-class))
  (setq *tre-class* nil *rule* nil *assert* nil)
  (setq *tms-node-printer* 
	#'(lambda (n) (format nil "~a"
			      (assertion-lisp-form
				(tms-node-datum n)))))
  (dolist (cl *tre-class*)
    (remprop (tre-class-name cl) 'tre-class))
  (setq *tre-class* nil *rule* nil *assert* nil))

;________________________________________________________________________
;;;; user interface
;________________________________________________________________________
 
;;in the jtre version, assert! takes a fact statement and an optional 
;;justification. it's used to install premises, assumptions, and justifications
;;into the tms at the jtre level.
;;in all cases, it first retrieves the assertion corresponding to the fact
;;statement (or builds it if necessary).
;;what happens next depends on the form of the optional justification:
;;
;;  1) if the corresponding node is a premise, the fact is treated as a
;;     premise.
;;
;;  2) if the justification is not a list, the fact is treated as an 
;;     assumption, whose support will be the given justification.
;;     the default case is to view assert! as installing user-provided
;;     assumptions.
;;
;;  3) if the justification is a list, then the call to assert! is interpreted 
;;     as installing a new tms justification. the consequence of this new
;;     justification will be the fact statement, the symbolic type of the
;;     justification will be the car of the input list, and the antecedents
;;     will be the cdr of the input list.
;

(defun assert! (fact &optional (just 'user) &aux assertion node)
  (let* ((assertion (referent (expand-lisp fact) t))
	 (node (assertion-tms-node assertion)))
  (cond ((not (listp just))		
         ;making something a ground support
	 (if (ground-node? node)
	     (format t "~%~a is already grounded, by ~a."
		     (assertion-lisp-form assertion)
		     (tms-node-support node))
	     (install-ground-node node just))) 
	(t (let* ((in-out-list (separate-in-out (cdr just))))
	 (install-justification (car just) 
				  node
				  (mapcar #'(lambda (f) 
					      (assertion-tms-node (referent f t)))
					  (car in-out-list))
				  (mapcar #'(lambda (f)
					      (assertion-tms-node (referent f t)))
					  (cadr in-out-list)))
	 ))))
  (push assertion *assert-list*)
  assertion)
 
(defun separate-in-out (justifications)
  (let ((correct-format t)
	(in-list '()) (out-list '()))
    (when (pair-list-p justifications)
      (dolist (just justifications (if correct-format
				       (list (nreverse in-list)
					     (nreverse out-list))
				     (list justifications)
				     ))
	(cond ((and (equal (car just) ':in) correct-format)
	       (push (cadr just) in-list)
	       )
	      ((and (equal (car just) ':out) correct-format)
	       (push (cadr just) out-list)
	       )
	      (t (setq correct-format nil)))))))

(defun pair-list-p (l)
  (let ((result t))
    (dolist (elt l result)
      (unless (and (consp elt)
		   (equal (length elt) 2))
	(setq result nil)))))
	  
 
; attempt to eval certain conclusions  - dta

(defun expand-lisp (fact)
  (cond ((or (atom fact) (not (symbolp (car fact)))) fact)
	((equal (string-downcase (car fact)) "lisp")
	 (eval (cadr fact)))
	(t (mapcar 'expand-lisp  fact))))

; end of dta changes 

(defun quiet-assert! (fact &optional (just 'user)) 
  (let ((*disable-contradiction-checking* t))
    (assert! fact just)))

;;the macro rassert! is designed to make writing rules more convenient. the 
;;majority of calls to assert! within the scope of a rule involve expressions 
;;which are a combination of constants and variables bound in the scope of the 
;;rule's trigger patterns. when using assert! in the body of a rule, the rule 
;;author must be sure to use the appropriate backquoted expressions. this is 
;;often a source of typing mistakes. so, rassert! makes it instead. in 
;;particular, the arguments to rassert! are expanded into a form which will 
;;evaluate any variables at run-time. this processing is done by quotize.

(defmacro rassert! (fact &optional (just 'user))
  ;avoids quoting arguments
  `(assert! ,(quotize fact) ,(quotize just)))
 

;;the function retract! and macro rretract! play the symmetric role to assert!
;;and rassert!. notice that retract respects the interpretation of support, by
;;only allowing retraction of assumptions. it further enforces the condition
;;that the retractor must be the "same-source" as originally made the 
;;assumption. 
;;this restriction is useful in avoiding timing errors if the same assumption 
;;is made and retracted by several different parts of a program.

(defun retract! (fact &optional (just 'user) &aux assert node)
  (setq assert (referent fact t)
	node (assertion-tms-node assert))
  (cond ((assumption-node? node)
	 (cond ((equal just (tms-node-support node)) 
		;require same source 
		(retract-assumption node))
	       (t (format t "~%~a not source of assumption for ~a"
			  just fact))))
	(t (format t "~%~a isn't an assumption." fact)))
  (push assert *assert-list*)
  node)

(defmacro rretract! (fact &optional (just 'user))
  `(retract! ,(quotize fact) ,(quotize just)))

;________________________________________________________________________
;;; database system
;________________________________________________________________________

;;insert inserts the fact into the database.
;
(defun insert (fact &aux assert)
  (setq assert (referent1 fact))
  (cond (assert (values assert t))
	(t (setq assert (make-assertion :counter (incf *assert-counter*)
				     :lisp-form fact
				     :class (get-class fact)))
	   (setf (assertion-tms-node assert) (install-tms-node assert))
	   (push assert (tre-class-facts (assertion-class assert)))
	   (push assert *assert*)
           (try-rules assert) 
	   (values assert nil))))
 
;________________________________________________________________________
;;; building and installing rules
;________________________________________________________________________


;;parse-rule-trigger simply decomposes the trigger, using the common lisp 
;;support of keywords for facility.

(defun parse-rule-trigger (trigger)
  ;; trigger syntax is now (<condition> <pattern>), with
  ;; optional :var and :test keywords after <pattern>.
  (values (cadr trigger)
	  (cond ((member (car trigger) '(:intern :in :out)) (car trigger))
		(t (error "~%~a bad condition in ~a." (car trigger) trigger)))
	  (cadr (member ':var (cddr trigger)))
	  (cadr (member ':test (cddr trigger)))))

;________________________________________________________________________
;;;; generating the body function
;________________________________________________________________________


;;for more explanations about this part of the code, please refer to the
;;jtre-expl3.txt file that contains a little traced and commented example.

    
;;the function generate-body-function first decomposes the given trigger into:
;;	- the pattern argument
;;	- the condition argument
;;	- the var argument (known by the keyword :var)
;;
;;generate-body-function has then to build a function such that:
;;
;;	- when the corresponding tms-node conforms to the condition, the rule
;;	  is queued to be run (by the insert-rule primitive).
;;	- and when it is not, the rule is pushed onto the corresponding field
;;    of the node to be executed when it satisfies the condition.
; 
(defun generate-body-function (pattern condition var body
				       &aux fname newly-bound env)
  (setq newly-bound 
    (if var 
      (cons var (pattern-free-variables pattern))
      (pattern-free-variables pattern)))
  (setq body 
    (with-pushed-variable-bindings newly-bound (fully-expand-body body)))
  (setq env 
    (append newly-bound (scratchout newly-bound *bound-vars*)))
  (unless (eq condition ':intern) 
    (push '*the-node* env))
  (setq fname 
    (generate-rule-function-name pattern)) 
  `(defun ,fname ,env
     ,@ (cond ((eq condition ':intern) body) ;just execute the body.
	      (t `((cond ((eq (tms-node-status *the-node*) ',condition)
			 ,@ body)
			(t (push (list ',fname ,@ env)
				 ,(cond ((eq condition ':in)
					 '(tms-node-in-rules *the-node*))
					((eq condition ':out)
					 '(tms-node-out-rules *the-node*))
					(t (error "~%g-b-f error")))))))))))
 
;________________________________________________________________________
;		      interface to the jtms
;________________________________________________________________________

(defun in? (fact &aux r)
  (setq r (referent fact t))
  (when r (in-node? (assertion-tms-node r))))

(defun my-in? (fact &aux r)
  (setq r (referent fact nil))
  (when r (in-node? (assertion-tms-node r))))
  
(defun out? (fact &aux r)
  (setq r (referent fact t))
  (when r (out-node? (assertion-tms-node r))))

(defun grounding (fact &aux r)
  (setq r (referent fact t))
  (when r (ground-for-node (assertion-tms-node r))))

(defun why (fact &aux r)
  (setq r (referent fact t))
  (when r (why-node (assertion-tms-node r))))

(defun contradiction (fact &aux r)
  (setq r (referent fact t))
  (install-contradiction (assertion-tms-node r)))
 
;;; =======================================================
;;; END OF FILE
;;; =======================================================
