;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:PAIL-LIB; Base:10; -*-
;;;
;;; ******************************
;;; *  PORTABLE AI LAB - UNI ZH  *
;;; ******************************
;;;
;;; Filename:   browser.cl
;;; Short Desc: implements class for browsers
;;; Version:    1.0
;;; Status:     Experimental
;;; Last Mod:   12.5.92 dta
;;; Author:     DTA
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;; --------------------------------------------------------------------------
;;; Change History: 
;;; Written 4.1.91
;;; Extended 10.6.91 to support some default menues
;;; Extended 22.10.91 to add scrollbars and multiple trees
;;; 12.5.92 Added overall browser functions (title items) 
;;; --------------------------------------------------------------------------

;;; ==========================================================================
;;; DESCRIPTION
;;; ==========================================================================
;;;
;;;


(in-package :pail-lib)
(use-package :cwex)

(eval-when (compile load eval)
 (ensure-loaded (add-path "global" *pail-lib-dir*))
 (export '(browser recompute-browser starting-tree close-browser
	  *max-browser-width* *max-browser-height*)))


(when (boundp 'cw::*root-window*)
       (setf pail-lib::*max-browser-width* (round (width *root-window*) 1.3))
       (setf pail-lib::*max-browser-height* (round (height *root-window*) 1.3)))

(defclass browser ()
	  ((display :initarg :display
		    :initform nil
		    :accessor display
		    :type display)
	   (bottom :initarg :bottom :accessor bottom)
	   (left :initarg :left :accessor :left)
	   (width :accessor width)
	   (height :accessor height)
	   (starting-tree :initarg :starting-tree
			  :initform nil
			  :accessor starting-tree
			  :type tree)	; can also be a list of trees
	   (font :initarg :font
		 :initform (open-font :times :roman 14)
		 :accessor font
		 :type cw::font)
	   (offset :initarg :offset
		   :initform 0
		   :accessor offset
		   :type integer)
	   (border :initarg :border
		   :initform 0
		   :accessor border
		   :type integer)
	   (title :initarg :title
		  :initform "Browser "
		  :accessor title
		  :type string)
	   (buttons 
	    :initform nil
	    :accessor buttons
	    :type list)
	   (items
	    :initform nil
	    :accessor items
	    :type list)
	   (left-menu :initarg :left-menu
		      :initform (make-instance 'menu
				  :items '(("Inspect" inspect-tree "Calls the inspector on the selected item.")
   					   
					   
					   ))
		      :accessor left-menu
		      :type menu)
	   (title-menu :initarg :title-menu
		      :initform (make-instance 'menu
				  :items '(
   					   ("Recompute" recompute-me "Recomputes the browser")
					   ("Close" close-browser "Closes the browser display")
					   ))
		      :accessor title-menu
		      :type menu))
  )

;;; MENU SUPPORT FUNCTIONS

;;; This function is called when a button is pressed.  It brings up a
;;; menu, and calls the function selected.  The selected function is
;;; called with two arguments, the item that was selected and the
;;; browser itself. 

(defmethod browser-select-item ((b browser) item mousestate button)
  (let ((ans (accept-items (left-menu b))))
    (if ans (apply ans (list item b))))
  (reset-button button)
  )

(defmethod browser-title-item ((b browser)  mousestate button)
  (let ((ans (accept-items (title-menu b))))
    (if ans (apply ans (list b))))
  (reset-button button)
  )

;;; the following three functions are provided in the default menu.
;;; They are called by browser-select-item when a button is pressed.

;;; closes the display associated with the browser

(defmethod close-browser ((b browser))
  (close-display b)
  (setq b nil)) ;;; OKKIO

;;; recomputes the current browser.  This unsets and resets all
;;; buttons. 

(defmethod recompute-me ((browser browser))
  )
;;;  (recompute-browser browser)) far too dangerous does not work properly OKKIO

;;; Calls the line inspector on the selected item.
(defmethod inspect-tree ((item tree) (b browser))
  )
;;;  (inspect item)) far too dangerous OKKIO
  


;;; WHOLE BROWSER SUPPORT FUNCTIONS
;;; Closes the display associated with the browser.
(defmethod close-display ((b browser))
  (if (status (display b))
      (if (eq (display-parent (display b)) *root-window*)
	  (close-display (display b))
	(flush (display-parent (display b))))))


;;; Recomputes the browser.  All buttons are unset and then reset.
;;; The layout is recomputed.
(defmethod recompute-browser ((b browser))
  (setf (buttons b) nil)
  (setf (items b) nil)
  (clear-display (display b))
  (let* ((tree-spec (size-tree-vertical b (starting-tree b) :offset (offset b) ))
	 (main-disp (parent (display b)))
	 (wid (+ (* 2 (border b)) (width tree-spec)))
	 (hei (+ (* 2 (border b)) (height tree-spec))))
    (close-display (display b))
    (setf (display b) (make-instance 'display
			:parent main-disp :title nil
			:width (- (width main-disp)
				  *static-scroll-bar-width*)
			:height (- (height main-disp)
				   *static-pann-bar-height*)
			:left *static-scroll-bar-width* :bottom *static-pann-bar-height*
			:inner-width wid
			:inner-height hei
			:font (font b)))
    (make-static-scroll-bar (window (display b)))
    (make-static-pann-bar (window (display b)))

    )
  (push (make-instance 'push-button
	  :label "browser functions"
	  :font (font b))
	(buttons b))
  (set-button (car (buttons b)) (display b)
	      :left (- (inner-width (display b)) 5 (width (car (buttons b))))
	      :bottom (- (inner-height (display b)) 5 (height (car (buttons b))))
	      :border 2
	      :action `(lambda (  )
			 (browser-title-item ',b nil ',(car (buttons b)))))
  (display-tree-vertical b (starting-tree b)))


;;; Recomputes only the labels of the browser.  No layout information
;;; is changed.  Thus, if the labels have changed enough to warrant a
;;; new layout, recompute-browser should be used instead.
(defmethod recompute-labels ((b browser))
  (do ((buttons (buttons b) (cdr buttons))
       (items (items b) (cdr items)))
      ((or (null items) (null buttons)) nil)
    (setf  (label (car buttons)) (label (car items))))
  )


;;; Returns the descendants of an item.  For a plain browser, this is
;;; simply the value of the descendants slot.  It could be specialized
;;; to allow a different view of an object.
(defmethod find-descendants ((b browser) item)
  (descendants item))


;;; Initialization of a browser involves computing an offset and
;;; border (if none are supplied) from the font information.  The size
;;; of the browser is computed, then the display is opened to that
;;; size.  Finally the browser is displayed.  All keyword args that
;;; were passed to the make-instance call that are not defined by
;;; the browser class are passed on to the make-display.


(defconstant make-display-keyword-arguments
    '(:left :bottom :width :height :borders :icon :title :active
      :x-scrollbar :y-scrollbar :inner-width :inner-height :parent
      :frame-menu :font :display-type :reshape-method :flush-method
      :button-region :from-button :filename))

(defmethod initialize-instance  :after ((b browser)  &rest keys)
  (if (zerop (offset b)) (setf (offset b) (font-character-height (font b))))
  (if (zerop (border b)) (setf (border b) (* 2 (font-character-height (font b)))))
  (if (listp (starting-tree b))
      (setf (starting-tree b) (remove-if 'null (starting-tree b)))) ; null trees are tricky!
  (if (null (display b))
      (let* ((tree-spec (size-tree-vertical b (starting-tree b) :offset (offset b) ))
             (wid (+ (* 2 (border b)) (width tree-spec)))
             (hei (- (+ (* 2 (border b)) (height tree-spec)) (offset b)))
             (main-disp nil))
                
        (if (or (> wid *max-browser-width*)
                (> hei *max-browser-height*))
	    (progn
	      (setf main-disp
		(apply #'make-instance 'display
		       :title (title b)
		       :width (+ (min wid *max-browser-width*) *static-pann-bar-height*)
		       :height (+ (min hei *max-browser-height*) *static-scroll-bar-width*)
		       (filter-keys keys (set-difference
					  make-display-keyword-arguments
					  '(:verbose :display
					    :starting-tree :font :offset
					    :border :title :width
					    :height :wm :ruleset)))))
	      (setf (display b)
		(apply #'make-instance 'display
		       :parent main-disp :title nil
		       :width (- (width main-disp) *static-scroll-bar-width*)
		       :height (- (height main-disp) *static-pann-bar-height*)
		       :left *static-scroll-bar-width* :bottom *static-pann-bar-height*
		       :inner-width wid
		       :inner-height hei
		       :font (font b)
                     
		       (filter-keys keys
				    (set-difference make-display-keyword-arguments
						    '(:icon :title
						      :verbose :display
						      :starting-tree
						      :font :offset
						      :left :bottom
						      :inner-width
						      :inner-height
						      :border :title :wm
						      :ruleset)))))
	      (make-static-scroll-bar (window (display b)))
	      (make-static-pann-bar (window (display b)))
	      (setf (slot-value (display b) 'width) (inner-width (display b)))
	      (setf (slot-value (display b) 'height) (inner-height (display b))))
          (setf (display b)
            (apply #'make-instance 'display
                   :width wid
                   :height hei
                   :title (title b)
                   :font (font b)
                   (filter-keys keys
                                (set-difference make-display-keyword-arguments
                                                '(:verbose :display
                                                  :starting-tree :font :offset
                                                  :border :title :wm
                                                  :ruleset))))))))
  (push (make-instance 'push-button
	      :label "browser functions"
	      :font (font b))
	  (buttons b))
  (set-button (car (buttons b)) (display b)
		:left (- (inner-width (display b)) 5 (width (car (buttons b))))
		:bottom (- (inner-height (display b))  5 (height (car (buttons b))))
		:border 2
		:action `(lambda (  )
			   (browser-title-item ',b nil ',(car (buttons b)))))
  
  (display-tree-vertical b (starting-tree b)))



(defun filter-keys (keys goods) 
  (cond ((null (evenp (length keys))) (error "Odd key list ~a." keys))
        ((null keys) nil)
        ((not (member (car keys) goods))
         (filter-keys (cddr keys) goods))
        (t (prog1 keys (setf (cddr keys) (filter-keys (cddr keys) goods))))))







;;; Finally you can also move the browser window
;;; and shadow the size and location...
(defmethod (setf bottom) (arg (b browser))
  (if (eq (display-parent (display b)) *root-window*)
      (setf (bottom (display b)) arg)
    (setf (bottom (display-parent (display b))) arg)))

(defmethod (setf left) (arg (b browser))
  (if (eq (display-parent (display b)) *root-window*)
      (setf (left (display b)) arg)
    (setf (left (display-parent (display b))) arg)))

(defmethod (setf width) (arg (b browser))
  (if (eq (display-parent (display b)) *root-window*)
      (setf (width (display b)) arg)
    (setf (width (display-parent (display b))) arg)))

(defmethod (setf height) (arg (b browser))
  (if (eq (display-parent (display b)) *root-window*)
      (setf (height (display b)) arg)
    (setf (height (display-parent (display b))) arg)))

(defmethod bottom ((b browser))
  (if (eq (display-parent (display b)) *root-window*)
      (bottom (display b))
    (bottom (display-parent (display b)))))

(defmethod left ((b browser))
  (if (eq (display-parent (display b)) *root-window*)
      (left (display b))
    (left (display-parent (display b)))))

(defmethod width ((b browser))
  (if (eq (display-parent (display b)) *root-window*)
      (width (display b))
    (width (display-parent (display b)))))

(defmethod height ((b browser))
  (if (eq (display-parent (display b)) *root-window*)
      (height (display b))
    (height (display-parent (display b)))))


;;; Default browser title is extended by given title.  Thus all
;;; browsers have titles that begin with "Browser"  
(defmethod (setf title) ((browser browser) value)
  (setf (slot-value browser 'title)
    (concatenate 'string (title browser) value)))

	      
;;; support function for passing keys along.  Any keys that appear in
;;; the 'bads' list are removed from the keyword list.
(defun remove-keys (keys bads) 
  (cond ((null (evenp (length keys))) (error "Odd key list ~a." keys))
	((null keys) nil)
	((member (car keys) bads)
	 (remove-keys (cddr keys) bads))
	(t (prog1 keys (setf (cddr keys) (remove-keys (cddr keys) bads))))))

;;; If you browse more than one tree, the trees are laid out in such a
;;; way that the sum of the vertical and horizontal extent is
;;; minimized.  Size-tree-vertical computes what the dimensions are
;;; for that minimum extent.  

;;; Tries to pack trees into a convenient space
(defmethod size-tree-vertical ((browser browser) trees  &key
			       (left (border browser))
			       offset)
  (let* ((bestlayout 0)
	 (bestsum 0)
	 (thisy 0)
	 (subbuttons (loop for tree in trees
		      collect (let ((tr (size-tree-vertical browser tree :left left :offset offset)))
				(setf bestsum (+ 1 bestsum (height tr) (width tr)))
				tr))
		     )
	 subbutton)
    (loop for tree in trees as layout in subbuttons do
      (setf thisy (+ thisy (height layout))) ; try y dimension to be trees so far
      (setf subbutton (packall thisy  subbuttons))
      (when (< (+ (height subbutton) (width subbutton)) bestsum)
	(setf bestsum (+ (height subbutton) (width subbutton)))
	(setf bestlayout subbutton))
      )

    bestlayout
    )
  )
    
      
;;; finds size of the display yielded by packing all given layouts
;;; into a display with y dimension given by ydim.
      
(defun packall (ydim layouts)
  (let ((ysofar 0)
	(xsofar 0)
	(xmax 0)
	(ymax ydim)
	)
    (loop for layout in layouts do
      (cond ((> (+ ysofar (height layout)) ydim)
	     (setf ysofar (height layout))
	     (setf xsofar (+ xsofar xmax))
	     (setf xmax (width layout))
	     (if (> (height layout) ydim) (setf ymax (height layout)))
	     )
	    (t (setf ysofar (+ ysofar (height layout)))
	       (if (> (width layout) xmax) (setf xmax (width layout)))
	       )))
    (make-instance 'button-spec
      :height ymax
      :width (+ xsofar xmax))))
    

;;; Computes the size of a browser for a list of trees, displayed vertically.
#| (defmethod size-tree-vertical ((browser browser) trees  &key
			       (left (border browser))
			       offset)
  (let ((height 0)
	(maxwidth 0)
	(subbutton))
    (loop for tree in trees do
      (setf subbutton (size-tree-vertical browser tree :left left :offset offset))
      (when (> (width subbutton) maxwidth)
	  (setf maxwidth (width subbutton)))
      (setf height (+ height (height subbutton)))
      )
    (make-instance 'button-spec
      :width maxwidth
      :height height
      ))) |#



;;; Computes the size of a browser for a single tree.  This is done by
;;; virtually laying out the buttons, and doing all the computations
;;; required.   
(defmethod size-tree-vertical ((browser browser) (tree tree)  &key
			       (left (border browser))
			       offset bottom)
  (let ((fullwidth 0)
	(maxheight 0)
	(*default-push-button-size-p* nil))
        
    (if (find-descendants browser tree)
	(prog (subbutton)
	  (dolist (desc (find-descendants browser tree) fullwidth) 
	    (setf subbutton (size-tree-vertical browser desc :left (+ fullwidth left)
						:offset offset))
	    (setf fullwidth (+  fullwidth offset (width subbutton)))
	    (setf maxheight (max  maxheight  (height subbutton)))
	    )
	  (setf fullwidth (max
			   (+ 8 (font-string-width
				 (font browser)
				 (if (stringp (label tree)) (label tree) (write-to-string (label tree)))
				 ))
			   (- fullwidth offset)))
	  )
      (setf fullwidth (+ 8 (font-string-width (font browser)
					      (if (stringp (label tree)) (label tree) (write-to-string (label tree)))
					      ))))
    (make-instance 'button-spec
      :width fullwidth
      :height (+ offset maxheight (+ 6 (font-character-height (font browser) )))
      )))


;;; Display a list of trees (vertically).
#| (defmethod display-tree-vertical ((browser browser) trees  &key
				  (left (border browser) )
				  (bottom nil)
				  (connect-list nil)
				  (*default-push-button-size-p* nil)
				  )
  (let ((bot (- (height (display browser)) (border browser) (font-character-height (font browser))))
	subbutton)
    (loop for tree in trees do
      (progn 
	     (setf subbutton (display-tree-vertical browser tree :left left :bottom bot))
	     
	     (setf bot (- bot (height subbutton)))
	     
	     )))) |#


;;; Display a list of trees in a matrix.
(defmethod display-tree-vertical ((browser browser) trees  &key
				  (left (border browser) )
				  (bottom nil)
				  (connect-list nil)
				  (*default-push-button-size-p* nil)
				  )
  (let ((bot (- (inner-height (display browser)) (border browser) (font-character-height (font browser))))
	(ysofar 0)
	(xsofar (offset browser))
	(xmax 0)
	(ymax (inner-height (display browser)))
	subbutton)
    (loop for tree in trees do
      (setf subbutton (size-tree-vertical browser tree :offset (offset browser)))
      (cond ((> (+ ysofar (height subbutton)) (height (display browser)))
	     (setf xsofar (+ xsofar xmax (offset browser)))
	     (setf ysofar 0)
	     (display-tree-vertical browser tree :left xsofar :bottom
				    (- bot ysofar))
	     (setf ysofar (height subbutton))
	     (setf xmax (width subbutton))
	     (if (> (height subbutton) (height (display browser))) (setf ymax (height subbutton)))
	     
	     )
	    (t (display-tree-vertical browser tree :left  xsofar :bottom (- bot
									    ysofar))
	       (setf ysofar (+ ysofar (height subbutton)))
	       (if (> (width subbutton) xmax) (setf xmax (width subbutton)))
	       
	       )))
    (make-instance 'button-spec
      :height ymax
      :width (+ xsofar xmax))))
	     
	     
      


;;; Display a tree (vertically).  The active items are push-buttons.
(defmethod display-tree-vertical ((browser browser) (tree tree)  &key
				  (left (border browser) )
				  (bottom nil)
				  (connect-list nil)
				  (*default-push-button-size-p* nil)
				  )
  (let ((fullwidth 0)
	(maxheight 0)
	button 
	(offset (offset browser)))
    (if (null bottom)
	(setf bottom
	  (- (inner-height (display browser)) (border browser) (font-character-height (font browser)))))
    (push (setf button			; keeps track of all buttons in the browser
	    (make-instance 'push-button
	      :label (if (stringp (label tree)) (label tree) (write-to-string (label tree)))
	      :font (font browser)))
	  (buttons browser))
    (push tree (items browser))		;keeps track of items in the browser
    ;; The general strategy is to recursively lay out all the
    ;; descendants of a node first, then to add up the space that they
    ;; need.  This subtree will require as much width as the maximum
    ;; of the current button and the combined subs. The button for the
    ;; current node is centered in this horizontal space. 
    (if (find-descendants browser tree)
	(prog (subbutton)
	  (dolist (desc (find-descendants browser tree) fullwidth) 
	    (setf subbutton (display-tree-vertical browser desc :left (+ fullwidth left)
						   :bottom (- bottom offset (height button))
						    ))
	    (setf fullwidth (+  fullwidth offset (width subbutton)))
	    (setf maxheight (max  maxheight  (height subbutton)))
					; keep track
					; of the centerpoints of the
					; subs, for connecting with
					; lines later on.
	    (push (centerpoint subbutton)  connect-list))
	  (setf fullwidth (max (- fullwidth offset) (width button)))
	  )
      (setf fullwidth (width button)))
    ;; set the button.  The action item is to call the
    ;; browser-select-item (see above).
    (set-button button (display browser)
		:left (floor (- (+ left (/ fullwidth 2) ) (/ (width button) 2)))
		:bottom bottom
		:border nil
		:action `(lambda (  )
			   (browser-select-item ',browser ',tree nil ',button)))
    (dolist (connection connect-list)	; connect the nodes
      (draw-line (display browser) (car connection) (cadr connection) (+ left (/ fullwidth 2.0) ) bottom))
    ;; return a width and a centerpoint to the next level up
    (make-instance 'button-spec
      :height (+ offset maxheight (+ 6 (font-character-height (font browser) )))
      :width fullwidth
      :centerpoint (list (floor (+ left (/ fullwidth 2) )) (+ (height button)  bottom))
      )))

;;; Horizontal displays are not yet supported. 
;(defmethod display-tree ((d display) (tree tree) &key
;			 (left 0)
;			 (bottom (- (height d) 50))
;			 (connected-left -1)
;			 (connected-bot -1)
;			 (offset 10))
;  (let ((b (make-instance 'push-button :label (write-to-string (label tree)))))
;    (let ((connected-left-from (+ left (width b)))
;	  (connected-bot-from (+ bottom (/ (height b) 2))))
;      (if (and (> connected-left 0)
;	       (> connected-bot 0))
;	  (draw-line d connected-left connected-bot left (+ bottom (/ (height b) 2))))
;      (set-button b d :left left :bottom bottom :border nil :action `(lambda ()
;							   (format t (write-to-string ',(label tree)))))
;      (if (find-descendants browser tree)
;	  (progn
;	    (setf left (+ offset left (width b)))
;	    (dolist (desc (find-descendants browser tree) bottom)
;	      (setf bottom (min (- bottom (height b) offset)(display-tree d desc :left left :bottom bottom
;					 :connected-left connected-left-from
;					 :connected-bot connected-bot-from
;					 :offset offset)))))
;	(- bottom (height b) offset)))))
;


;;; The return value from a recursive call to display-tree or
;;; display-tree-horizontally has three things that must be returned.
;;; This is a structure that holds them.  
(defclass button-spec ()
	  ((width :initarg :width
		  :initform 0
		  :accessor width
		  :type integer)
	   (height :initarg :height
		  :initform 0
		  :accessor height
		  :type integer)
	   (centerpoint :initarg :centerpoint
			:initform '(0 0)
			:accessor centerpoint
			:type list)))



;;; ==========================================================================
;;; END OF FILE
;;; ==========================================================================
