;;; -*- Mode: LISP; Package: pail-lib; Syntax: Common-lisp; -*-
;;;
;;; **************************************************************************
;;;
;;; PORTABLE AI LAB IDSIA - Lugano
;;;
;;; **************************************************************************
;;;
;;; Filename:   file-browser.cl
;;; Short Desc: File Browser
;;; Version:    1.0
;;; Status:     Experimental
;;; Last Mod:   22.4.91 - PC
;;; Author:     Paolo Cattaneo
;;;
;;;
;;; Modification history
;;; 10.02.93   
;;;

(in-package :pail-lib)
(ensure-loaded (add-path "size-indep-utils" (add-subdir *pail-path* "pail-lib")))

(eval-when (compile load eval)
  (export '(choose-file-dialog choose-new-file-dialog)))

;;;OKKIO
(load (add-path    "dirp.o"   (add-subdir *pail-path* "pail-lib")))

(foreign-functions::defforeign 'directory_p 
  :arguments '(string) 
  :return-type :integer)

;;; ===========================================================================
;;; VARS
;;; ===========================================================================

(defvar *button-sep* 20)
(defvar *button-size* 100)

;;; ===========================================================================
;;; CLASSES
;;; ===========================================================================

(defclass file-browser (display)
	  ((filter            :accessor filter
			      :initarg :filter
			      :initform "")
	   (file-button       :accessor file-button
	   		      :initarg :file-button
	   		      :initform nil)
	   (file-window       :accessor file-window
			      :initarg :file-window
			      :initform nil)
	   (ok-button         :accessor ok-button
			      :initarg :ok-button
			      :initform nil)
	   (cancel-button     :accessor cancel-button
			      :initarg :cancel-button
			      :initform nil)
	   (help-button       :accessor help-button
			      :initarg :help-button
			      :initform nil)
	   (current-directory :accessor current-directory
			      :initarg :current-directory
			      :initform nil)
	   (count-clicks      :accessor count-clicks
			      :initarg :count-clicks
			      :initform "")
	   (returned          :accessor returned
			      :initarg :returned
			      :initform nil)))

(defclass file-browser-push-button (push-button)
	  ((d-position      :accessor d-position
			    :initarg :d-position
			    :initform 0)))

(defclass file-browser-select-button (select-button)
  ())

(defclass file-browser-value-button (value-button)
  ())

;;; ===========================================================================
;;; DIRECTORY LISTING
;;; ===========================================================================

(defun directory-listing (path-or-string filter)
  (let ((ls (mapcar #'enough-namestring (files-in-directory path-or-string))))
    ( only-filtered-and-dirs (cons "../" (sort (mapcar #'show-dir ls) #'string<)) filter)))

(defun show-dir (pathname)
  (if (directoryp pathname)
    (concatenate 'string (file-namestring pathname) "/")
    (file-namestring pathname)))

(defun directoryp (s) 
 (not (zerop (directory_p s))))

(defun files-in-directory (string)
    (directory (make-pathname :directory string
			      :name :wild 
			      :type :wild)))

(defun only-filtered-and-dirs(list filter)
  (remove-if #'(lambda (name)
		 (and (not (terminates-with filter name))
		     (not (terminates-with "/" name))))
	     list))

(defun terminates-with (substr str)
 (head-equal  (reverse (coerce str 'list))
	      (reverse (coerce substr 'list))))
(defun head-equal (list sublist)
  (cond ((null sublist) t)
	( (null list) nil)
	((equal (car list) (car sublist)) (head-equal (cdr list)( cdr  sublist)))))


;;; ===========================================================================
;;; GUI
;;; ===========================================================================
(defmethod initialize-instance ((f file-browser) 
				&key 
				(pathname (namestring *default-pathname-defaults*))
				(prompt "Please select a file or change directory:")
				(ok-text "OK")
				(filename "Untitled")
				(filter ""))
				
  (call-next-method)
  (setq gin::*default-display-border* 1)
  (setf (current-directory f) (strip-/ pathname))
  (setf (file-button f)
	(make-instance 'file-browser-value-button
		       :name "File:"
		       :value filename
		       :width 310
		       :border t))
  (setf (file-window f)
	(make-instance 'file-browser-select-button
		       :label prompt
		       :height 210
		       :items (directory-listing pathname filter)
		       :exclusive t))
  (setf (ok-button f)
	(make-instance 'file-browser-push-button 
	  :width *button-size* 
		       :label ok-text
		       :d-position 1))
  (setf (cancel-button f)
	(make-instance 'file-browser-push-button 
		       :width *button-size* 
		       :label "Cancel"
		       :d-position 2))
  (setf (help-button f)
	(make-instance 'file-browser-push-button 
		       :width *button-size* 
		       :label "Help"
		       :d-position 3)))

(defmethod initialize-instance :after ((f file-browser) &key pathname)
  (declare (ignore pathname))
  (protect-display f t)
  (setf (title f) "Pail: File Browser")
  (setf (height f) (senza-scalah 380))
  (setf (width f) (senza-scalaw 400))
  (display-chdir f)
  (position-file-browser f)
  (protect-display f nil))

(defmethod position-file-browser ((f file-browser))
  (set-button (file-button f)
	      f
	      :left 58
	      :bottom 300)
  (set-button (file-window f) f 
	      :left 10 
	      :bottom 60)
  (set-button (help-button f)
	      f
	      :left (+ (* (1- (d-position (help-button f))) *button-size*) (* 3 *button-sep*))
	      :bottom  *button-sep*)
  (set-button (cancel-button f)
	      f
	      :left (+ (* (1- (d-position (cancel-button f)))  *button-size*) (* 2 *button-sep*))
	      :bottom  *button-sep*)
  (set-button (ok-button f)
	      f
	      :left (+ (* (1- (d-position (ok-button f)))  *button-size*) *button-sep*)
	      :bottom  *button-sep*)
  (setf (action (cancel-button f))
    (function (lambda () 
		(progn (close-display f)
		       (setf (returned f)
			 'canceled)))))
  (setf (action (ok-button f))
    (function (lambda () 
		(let ((current-selection (button-value (file-button f))))
		  (if (directoryp (concatenate 'string (current-directory f)
					     "/"
					     current-selection))
		    (progn (protect-display f t)
		      (change-directory f (strip-/ current-selection))
		      (protect-display f nil)
		      (reset-button (ok-button f)))
		  (progn (close-display f)
			 (setf (returned f)
			   (concatenate 'string (current-directory f)
					"/"
					current-selection)))))))) ;;;;;  OKKIO PAOLO ha corretto questo
  (setf (action (file-button f))
    (function (lambda nil 
		(check-file-button f (button-value (file-button f))))))
  (setf (action (file-window f)) (function (lambda () (file-select f)))))

	     
(defmethod file-select ((f file-browser))
  (let ((current-selection (item-label)))
    ;;(when (not (directoryp current-selection))
      (set-value-f (file-button f) current-selection)
;;      )
    (if (not (string= (count-clicks f) current-selection))
	      (setf (count-clicks f) current-selection)
      (progn (setf (count-clicks f) "")
	     (if (string-directoryp current-selection)
		   (change-directory f (strip-/ current-selection))
	       (close-and-return f current-selection))))))

(defmethod set-value-f ((vb value-button) value)
  (setf (button-value vb) value))

(defmethod close-and-return ((f file-browser) selection)
  (close-display f)
  (setf (returned f)
	(concatenate 'string (current-directory f)
		     "/"
		     selection)))

(defmethod change-directory ((f file-browser) directory-name)
  (protect-display f t)
  (chdir f (if (string= directory-name "..")
	       (strip-/ (directory-namestring (strip-/ (current-directory f))))
	     (concatenate 'string (current-directory f)
				"/"
				directory-name)))
  (set-value-f (file-button f) "Untitled")
  (protect-display f nil))

(defmethod display-chdir ((f file-browser))
  (clear-display f :left 0 :bottom 330 :height 20 :width (width f))
  (write-display f (concatenate 'string "Directory: "
				(current-directory f))
		 14 340 :font *small-font*))


(defmethod chdir ((f file-browser) pathname)
  (let ((current-prompt (label (file-window f))))
    (unset-button (file-window f))
    (setf (current-directory f) pathname)
    (setf (file-window f)
      (make-instance 'file-browser-select-button
	:height 210
	:items (directory-listing pathname (filter f))
	:label current-prompt
	:exclusive t
	))
    (set-button (file-window f) f :left 10 :bottom 60)
    (setf (action (file-window f)) (function (lambda nil (file-select f))))
    (display-chdir f)))

(defmethod check-file-button ((f file-browser) filename)
  (let* ((work-file (if (probe-file filename)
			(namestring (probe-file filename))
		      nil))
	 (directory (if (absolute filename) 
			(if (directoryp filename) filename
			  (strip-/ (namestring (probe-file (directory-namestring filename)))))
			(strip-/ (namestring (probe-file (concatenate 'string
							   (current-directory f)
							   (directory-namestring filename)))))))
	 (file (if (directoryp filename) nil
		 (progn
		   (if work-file (file-namestring work-file)
		     (file-namestring filename))))))
    (if file (set-value-f (file-button f) file)
      (set-value-f (file-button f) "Untitled"))
    (when (read-from-string directory)
      (when (not (string= directory (current-directory f)))
      (chdir f directory)))))

;;; ===========================================================================
;;; UTILITIES
;;; ===========================================================================

(defun strip-/ (string)
  (if (string= string "/") string (string-right-trim '(#\/) string)))

(defun string-directoryp (string)
  (not (= (length string)
	   (length (strip-/ string)))))

(defun g-yes-or-no-p (query &key (title "Please Click"))
  (let ((dw (make-instance 'display   :width (if (> (cw::font-string-width *small-font* query) 195)
                                       (+ (cw::font-string-width *small-font* query) 20)
                                     205)
                            :font gin::*small-font-11*
                            :left (senza-scalaw 200)
                            :bottom (senza-scalah 400)
                    :height (senza-scalah  100)
                            :title title))
        (done nil)
        (yb (make-instance 'push-button  :font gin::*bold-font* :label "  Yes "))
        (nb (make-instance 'push-button  :font gin::*bold-font* :label "  No  ")))
    (write-display dw query (senza-scalaw 10)(senza-scalah  80))
    (set-button yb dw :left (senza-scalaw 10)
                 :bottom (senza-scalah 10) :action (function (lambda (&rest cw-internals)
                                                              (declare (ignore cw-internals))
                                                              (setf done :yes))))
    (set-button nb dw :left (- (width dw)(width nb) (senza-scalaw 10)) :bottom (senza-scalah 10)
                 :action (function (lambda (&rest cw-internals)
                                                               (declare (ignore cw-internals))
                                                               (setf done :no))))
    (mp:process-wait "Exit"   #'(lambda nil done))
    (cw:flush (window dw))
    (if (equal (status yb) 1)
	t nil)))

(defun absolute (string)
  (char= #\/ (car (coerce string 'list))))

(defun relative (string)
  (not (absolute string)))

;;; ===========================================================================
;;; USER INTERFACE
;;; ===========================================================================

;;; The choose-file-dialog function displays the standard pail file browser window
;;; allowing you to select a file for reading. Unless the dialog is cancelled, this
;;; function returns a pathname string.
;;; Arguments: :directory 
;;;                       a pathname string. Specifies the directory shown when the
;;;                       dialog first appears. It defaults to the value of
;;;                       *default-pathname-defaults*.
;;;            :prompt
;;;                        A string. Specifies the text that appears in the button
;;;                        that opens the chosen file. The default is "OK".


(defun choose-file-dialog (&key (directory (namestring *default-pathname-defaults*))
				(button-string "OK")
				(prompt "Please select a file or change directory:")
				(left 300)(bottom 100)
				(filter ""))
  (let* ((directory-part (if directory
			     (if (directoryp (namestring  directory))
				 (namestring directory)
			       (directory-namestring directory))
			   (namestring *default-pathname-defaults*)))
	 (file-part (if directory 
			(if (directoryp (namestring directory)) "Untitled"
			  (file-namestring directory))
		      "Untitled"))
	 (prompt-string (if (> (length prompt) 40)
			     prompt
			   (concatenate 'string 
					prompt
					(make-string (- 40 (length prompt))
						     :initial-element #\Space))))
	 (dialog (make-instance 'file-browser
		   :left left 
		   :bottom bottom
		   :height (senza-scalah 380)
		   :width  (senza-scalaw 400)
		   :filter filter
		   :prompt prompt-string
		   :pathname directory-part
		   :ok-text button-string
		   :filename file-part
		   :returned nil)))
    (mp::process-wait "file dialog" (function (lambda () (returned dialog))))
    (when (not (eq 'canceled (returned dialog)))
      (if (probe-file (returned dialog))
	  (returned dialog)
	(file-does-not-exist (returned dialog)
			     :reason "Selected file does not exist")))))

;;; The choose-new-file-dialog function displays the standard pail file browser window
;;; allowing you to select a file for writing. Unless the dialog is cancelled, this
;;; function returns a pathname string.
;;; Arguments: :directory 
;;;                       a pathname string. Specifies the directory shown when the
;;;                       dialog first appears. It defaults to the value of
;;;                       *default-pathname-defaults*. The filename component of
;;;                       :directory is used as the default filename in the
;;;                       top file button.
;;;            :button-string
;;;                       A string. Specifies the text that appears in the button
;;;                       that opens the chosen file. The default is "OK".
;;;            :prompt    Specifies the text that appears above the area where the
;;;                       files are listed.
;;;                      

(defun choose-new-file-dialog (&key (directory nil)
				    (prompt "Please select a file or change directory:")
				    (button-string "OK")
				    (left 300)
				    (bottom 100)
				    (filter ""))
  (let* ((directory-part (if directory
			     (if (directoryp (namestring directory))
				(namestring directory)
			       (directory-namestring directory))
			   (namestring *default-pathname-defaults*)))
	 (file-part (if directory 
			(if (directoryp (namestring directory)) "Untitled"
			  (file-namestring  directory))
		      "Untitled"))
	  (prompt-string (if (> (length prompt) 40)
			     prompt
			   (concatenate 'string 
					prompt
					(make-string (- 40 (length prompt))
						     :initial-element #\Space))))
	  (dialog (make-instance 'file-browser
				 :pathname directory-part
				 :left left 
				 :bottom bottom
				 :height (senza-scalah 380)
				 :width  (senza-scalaw 400)
				 :ok-text button-string
				 :prompt prompt-string
				 :filename file-part
				 :filter filter
				 :returned nil)))
    (mp::process-wait "new file dialog" (function (lambda () (returned dialog))))
    (cond ((eq (returned dialog) 'canceled)
	   nil)
	  ((not (probe-file (returned dialog)))
	   (returned dialog))
	  (t (if (g-yes-or-no-p (format nil "Do you want to overwrite file: ~% ~A"
					     (returned dialog))
				:title "File Browser Warning")
		 (returned dialog)
	       nil)))))
	   



(defun senza-scalah (x)x)
(defun senza-scalaw (x)x)
