;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:PAIL-LIB; Base:10; -*-
;;;
;;; ******************************
;;; *  PORTABLE AI LAB - UNI ZH  *
;;; ******************************
;;;
;;; Filename:   pool.cl
;;; Short Desc: Implements a pool of objects
;;; Version:    0.1
;;; Status:     Experimental
;;; Last Mod:   14.6.91
;;; Author:     DTA
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;; --------------------------------------------------------------------------
;;; Change History: 
;;; 
;;;	
;;; --------------------------------------------------------------------------

;;; ==========================================================================
;;; DESCRIPTION
;;; ==========================================================================
;;;



(in-package :pail-lib)

(export '(*pail-pool*
	  pool
	  after-get
	  known-items
	  pool-button
	  pool-find-object
	  pool-item
	  put-pool
	  get-from-pool
	  start-object
	  show-function
	  maintain-pool
	  ))


(defclass pool () 
	  ((known-items :accessor known-items
			:initform nil
			:type list)
	   ))


(defclass pool-item ()
	  ((name-part :accessor name-part
		      :initarg :name-part
		      :initform (ask "Under what name should this item be registered in the pool?~%")
		      :type string)
	   (start-object :accessor start-object
			 :initarg :start-object
			 :initform nil
			 )
	   (from-button :accessor from-button
			 :initarg :from-button
			 :initform nil
			 )
	   ))


(defmethod (setf start-object) :after (value (item pool-item))
  (setf (name-part item) "New item")
  (when (from-button item)
    (setf (button-value (from-button item)) (button-value (from-button item))))
  )

;;; Search the pool for an item with the given name.  If none exists,
;;; return nil.
(defmethod pool-find-object ((pool pool) name)
  (if (known-items pool)
      (loop for item in (known-items pool)
       thereis (when (equal (name-part item) name) (clos-copy (start-object item))))))


;;; bring up menu of pool items.
(defmethod pool-menu-get ((pool pool) query)
  (if (known-items pool)
      (accept-items (make-instance 'menu
		      :items (let ((result nil))
			       (dolist (item (known-items pool) result)
				 (push `(,(name-part item)
					 ,item)
				       result)))

		      :query query))
    (display-error "The pool is empty.  Please get a file with a `Files' button.")
    ))


(defmethod put-pool ((pool pool) (item pool-item) &key name)
  ;; CLOS does not seem to provide a way to make a copy of instances,
  ;; but if we start to save the instance to a file, and just eval the
  ;; thing that the file was going to produce, then we get a real
  ;; copy.  Tangles in trees are preserved, but not cycles.
  (cond ((member (name-part item)
		 (known-items pool)
		 :test #'(lambda (a b) (equalp (name-part b) a)))
	 (if (equal :yes
		    (y-or-n-dialog (format nil
					   "There is already an item named ~a in the pool.~%Do you want to remove the old version?"
					   (name-part item))
				   :cancel-button nil))
	     (progn
	       (setf (known-items *pail-pool*)
		 (remove-if #'(lambda (old) (equal (name-part old) (name-part item)))
			    (known-items *pail-pool*)))
	       (let ((save-button (from-button item)))
		 (setf (from-button item) nil) 
		 (push (make-instance 'pool-item
			 :start-object (clos-copy (start-object item))
			 :name-part (name-part item))
		       (known-items pool))
		 (setf (from-button item) save-button)))
	     
	   ))
	((or (equalp "New item" (name-part item))
	     (equalp "no name" (name-part item)))
	 (display-error (format nil
				"Please give this item a name (use Rename option)."
				(name-part item))))
	(t (let ((save-button (from-button item)))
	     (setf (from-button item) nil) 
	     (push (make-instance 'pool-item
		     :start-object (clos-copy (start-object item))
		     :name-part (name-part item))
		   (known-items pool))
	     (setf (from-button item) save-button))))
  item)


;;; things like copy-list and copy-tree don't make copies of CLOS
;;; objects, so I use the save-object code to do it.  This is a
;;; separate method because save-object can't save structures, so if
;;; something contains structures, they must be treated specially.  We
;;; don't use structures, except for the graphic objects, which often
;;; needn't be copied anyway.

(defmethod clos-copy (item)
  (eval (database::get-fasd-form item :reset t)))

(defmethod put-pool ((pool pool) entry &key (name (ask "Under what name should this item be registered in the pool?~%")))
  (push (make-instance 'pool-item :name-part name
		       :start-object entry)
	(known-items pool)))
  

(defmethod get-from-pool ((pool pool) prototype old-value)
  (let* ((response (pool-menu-get pool "Source object"))
	 answer)
    (if response
	(progn (documentation-print (format nil "Converting ~a . . ." (name-part response)))
	       (setf answer (make-instance 'pool-item
			      :start-object (clos-copy (change-type (start-object response) prototype))
			      :name-part (name-part response)
			      ))
	       (if (not (null (start-object answer)))
		   (progn (documentation-print "Conversion completed")
			  answer)
		 old-value))
      old-value
      )
    )
  )

(defmethod change-type (a b) (documentation-print
			      (format nil "Conversion not possible from type ~a to ~a." (class-name (class-of a)) (class-name (class-of b)))
			      nil))




(defmethod dump-editable ((item pool-item))
  (let ((dumpform (intern-all (dump-editable (start-object item))
			      :dump)))
    (if (eq dumpform nil)
	(database::instance-fasd-form-plain item) ; treat this like any instnace
      (progn (push (list (class-name (class-of (start-object item))) (name-part item) dumpform)
		   *editable-things*)
	     `(make-instance 'pail-lib::pool-item
		:name-part ,(name-part item)
		:start-object (get-file-object ,(name-part item)))))))
  

(database:define-fasd-form pool-item ((item pool-item)) :body
			   (dump-editable item))


(defmethod dump-editable (anything) nil)

(defclass pool-button (value-button)
	  ((pool  :initarg :pool
		  :initform nil
		  :type pool
		  :accessor pool)
	   (target-class :initarg :target-class
			 :initform nil
			 :type class
			 :accessor target-class)
	   (after-get :initarg :after-get
		      :initform #'(lambda (item) nil)
		      :type function
		      :accessor after-get)
	   (show-function :initarg :show-function
			  :accessor show-function
			  :initform #'(lambda (item)
					(documentation-print item)))))


(defmethod initialize-instance :after ((button pool-button) &rest keys)
  (setf (menu button) (make-instance 'menu
			:items `(("Get from pool"
				  (let ((answer (get-from-pool
						 ,(pool button)
						 ,(make-instance (target-class button))
						 (button-value ,button))))
				    (setf (from-button answer) ,button)
				    (apply (after-get ,button) (list answer))
				    answer)
				  "Gets an item from the pool")
				 ("Put to pool"
				  (put-pool
				   ,(pool button)
				   (button-value ,button))
				  "Puts this item into the pool")
				 ("Show" (progn (apply (show-function ,button)
						       (list (start-object (button-value ,button))))
						(button-value ,button)))
				 ("Edit" (edit-object (button-value ,button) (make-instance (target-class ,button))))
				 ("Rename" (rename-pool-item
					    (button-value ,button)))
				 ("Inspect" (inspect (button-value ,button)))
				 )))
  (setf (display-function button)
    `(lambda (x) (if (and x (not (stringp x)))
		     (rectify-string (name-part x))
		   (rectify-string x)
		   )))
  (setf (button-value button) (make-instance 'pool-item
				:start-object nil
				:name-part "no name"
				:from-button button))
  )


(defun rectify-string (s) (if (stringp s) s (symbol-name s)))

(defparameter *pail-pool*  (make-instance 'pail-lib::pool))




(defmethod rename-pool-item ((i pool-item))
  (setf (name-part i) (ask "By what name will this be known in the pool?~%"))
  i)


(defmethod maintain-pool ((pool pool))
  (cond ((not (known-items *pail-pool*)) (display-error "No items are in the pool"))
	(t (let* ((disp (make-instance 'display
			  :left (truncate (width *root-window*) 5)
			  :bottom (truncate (height *root-window*) 4)
			  :height (+ (font-character-height *default-font*) 155)
			  :title "Marked items will be deleted from the pool"))
		  (abort-button (make-instance 'push-button :label "Abort"))
		  (done-button (make-instance 'push-button :label "Delete"))
		  (selection (make-instance 'select-button
			       :label "Pool items" 
			       :height 150 :width 150
			       :items (mapcar #'name-part (known-items *pail-pool*))
			       :action #'(lambda nil nil)
			       )))
	     (set-button abort-button disp :left 260 :bottom 5
			 :action `(lambda nil (close-display ,disp)))
	     (set-button done-button disp
			 :left 260
			 :bottom (+ (height abort-button) 10)
			 :action #'(lambda nil
				     (let ((good nil))
				       (setf (known-items *pail-pool*)
					 (do ((i (items selection) (cdr i))
					      (pooli (known-items *pail-pool*) (cdr pooli)))
					     ((null i) good)
					   (when (not (status (car i)))
					     (push (car pooli) good)
					     )))
				       (close-display disp)
				       )))
	     (set-button selection disp)
	     ))
	))


;;; ==========================================================================
;;; END OF FILE
;;; ==========================================================================
