;;; -*- Mode: LISP; Package: PAIL-LIB; Syntax: Common-lisp;              -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; ************************************************************************
;;;
;;; Filename:   table-ed.cl
;;; Short Desc: tiny editor for tables
;;; Version:    2.0
;;; Status:     Experimental
;;; Last Mod:   17.10.91 - TW
;;; Author:     Thomas Wehrle
;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;; 14.10.91 Additional options "cell-font", "font" & "cell-width" (TW)
;;; 17.10.91 Bug fixed with frames for cells with small fonts (TW)
;;; 10.1.91  Symbols are now interned in the dump package (TW)
;;; 3.2.91   Bug in refresh fixed (refresh worked sometimes on wrong display
;;;          when having several editors open)
;;; --------------------------------------------------------------------------
;;; 
;;; Known bugs:
;;;
;;; Sometimes the following message occurs:
;;; Error: No methods applicable for gf: #<Generic Function font @ #x8b7c6e>
;;;        with args (nil)
;;;        of classes (null)
;;;
;;; This bug may have to do with GIN or COMMON WINDOWS
;;; --------------------------------------------------------------------------
;;;
;;; Usage:
;;;
;;; (make-editor-fcn <fcn-name> [:obj-var <var>]
;;;                             [:default-menu-entries <bool>]
;;;                             [:menu <menu-items-or-nil>])
;;; where 
;;;  <var> is a symbol that you can use in the menu-items for the table object
;;;  <bool> is t or nil depending on whether you want the default menu enties or not
;;;  <menu-items-or-nil> is nil or a list of lists with 3 elements:
;;;      1 = menu item's name
;;;      2 = function to be called (you can use <var> here to refer to the table object
;;;          and you can use here a function "refresh" to update the display
;;;      3 = help string
;;;
;;;  Note that if you set :default-menu-entries to true but specify a menu, then
;;;  you get a menu containing the default menu as well as your entries!
;;;
;;;  eg. (make-editor-fcn id3-table-editor)
;;;               
;;;     
;;;  or (make-editor-fcn my-ed
;;;                       :obj-var tab
;;;                       :default-menu-entries nil
;;;                       :menu `(("My function"
;;;                                ,(function (lambda ()
;;;   				      (do-something-with tab)
;;;				      (refresh tab)))))
;;;     
;;;  or (make-editor-fcn my-ed
;;;                       :obj-var tab
;;;                       :default-menu-entries t
;;;                       :menu `(("My function"
;;;                                ,(function (lambda ()
;;;   				      (do-something-with tab)
;;;				      (refresh tab)))))
;;;
;;;
;;; This gives you 2 functions:
;;;
;;; 1. (<fcn-name> <table-obj> [:left <nil-or-int>]
;;;			       [:bottom <nil-or-int>]
;;;		               [:title <string>]
;;;                            [:font <font>]
;;;                            [:cell-font <font>]
;;;                            [:cell-width <chars>]
;;;			       [:offset-x <int>]
;;;			       [:close-all <bool>])
;;; where:
;;; <table-obj>    is your table-object, if it is not a table one is created
;;; <fcn-name>     is your function to invoke your editor
;;; <nil-or-int>   coordinates of left end of the display or nil for default
;;; <nil-or-int>   coordinates of bottom of the display or nil for default
;;; <string>       a title for the display
;;; <font>         is a font, default is *default-font*
;;; <chars>        number of visible characters per cell
;;; <int>          set that to 0 to avoid the numbering of the examples
;;; <bool>         t means that all other editor displays invoked with your
;;;                function! are closed so that this display will be the only one.
;;;
;;; 2. ("close-"<fcn-name> <table-obj> [:close-all <bool>])
;;;
;;; where:
;;; <table-obj>    is a table-object that is edited
;;; <bool>         close all other editor window or just the one for <table-obj> 
;;;



;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :pail-lib)

(export '(make-editor-fcn refresh))


(defclass editor-class ()
  ((display    :initarg :display
	       :accessor display)
   (cell-font  :initarg :cell-font
	       :accessor cell-font)
   (cell-width :initarg :cell-width
	       :accessor cell-width)
   (offset-x   :initarg :offset-x
	       :accessor offset-x)
   (close-all  :initarg :close-all
	       :accessor close-all)))


(defmacro make-editor-fcn (name &key (obj-var 'tab)
				     (default-menu-entries t)
				     (menu nil))
  (let ((display-collector-name (intern (symbol-name (read-from-string (format nil "~a-disps" name)))))
	(close-fcn-name (intern (symbol-name (read-from-string (format nil "close-~a" name))))))
    `(progn
       (defparameter ,display-collector-name (make-hash-table))

       (defun ,close-fcn-name (tab &key (close-all nil))
	 (if close-all
	     (progn
	       (maphash #'(lambda (key value)
			    (declare (ignore key))
			    (close-display (display value)))
			,display-collector-name)
	       (clrhash ,display-collector-name))
	   (progn
	     (let ((d (gethash tab ,display-collector-name)))
	       (when d (close-display (display d))))
	     (remhash tab ,display-collector-name))))

       (defmethod ,name (symbol
			 &key (left nil)
			      (bottom nil)
			      (title "Table Editor")
			      (font *default-font*)
			      (cell-font *default-font*)
			      (cell-width 12)
			      (offset-x (* 10 (font-character-width cell-font)))
			      (close-all nil))
	 (declare (ignore symbol))
	 (let ((x (read-from-string
		   (ask "How many attributes ? "
			:condition 
			(function (lambda (string)
				    (let ((x (read-from-string string nil)))
				      (and (numberp x) (plusp x) (integerp x)))))
			:error-message "Please enter a positive number")))
	       (y (read-from-string
		   (ask "How many examples ? "
			:condition 
			(function (lambda (string)
				    (let ((x (read-from-string string nil)))
				      (and (numberp x) (plusp x) (integerp x)))))
			:error-message "Please enter a positive number"))))
	   (,name (make-table :attributes 
			      (let ((l nil))
				(dotimes (i x (nreverse l))
				  (setf l (cons (intern-all (read-from-string (format nil "a~a" i)) :dump)
						l))))
			      :rows 
			      (let ((ll nil))
				(dotimes (j y (nreverse ll))
				  (setf ll 
				    (cons
				     (let ((l nil))
				       (dotimes (i x (nreverse l))
					 (setf l (cons (intern-all (read-from-string (format nil "v~a.~a" i j)) :dump)
						       l))))
				     ll)))))
		  :left left
		  :bottom bottom
		  :title title
		  :font font
		  :cell-font cell-font
		  :cell-width cell-width
		  :offset-x offset-x
		  :close-all close-all)))

       (defmethod ,name ((,obj-var table)
			 &key
			 (left nil)
			 (bottom nil)
			 (title "Table Editor")
			 (font *default-font*)
			 (cell-font *default-font*)
			 (cell-width 12)
			 (offset-x (* 10 (font-character-width cell-font)))
			 (close-all nil))
  
	 (let ((*value-button-border-p* t)
	       (x (length (attributes ,obj-var)))
	       (y (number-of-rows ,obj-var))
	       (gab 6)
	       (dist 20)
	       (choice-width 10)
	       (b-width (* cell-width (font-character-width cell-font)))
	       (b-height (if (eq cell-font *small-font*) ; strange behavior with small fonts
			     (+ 2 (+ 6 (font-character-height cell-font)))
			   (+ 6 (font-character-height cell-font))))
	       (offset-y (* 2 (+ 6 (font-character-height font))))
	       (disp nil))
	   (defun refresh (table-obj)
	     (let ((ed-obj (gethash table-obj ,display-collector-name)))
	       (when ed-obj
		 (,name table-obj
			:left (left (display ed-obj))
			:bottom (bottom (display ed-obj))
			:title (title (display ed-obj))
			:font (font (display ed-obj))
			:cell-font (cell-font ed-obj)
			:cell-width (cell-width ed-obj)
			:offset-x (offset-x ed-obj)
			:close-all (close-all ed-obj)))))
	   (let* ((default-items (if ,default-menu-entries
				     (list
				      (list
				       "Add Column" (function (lambda ()
								   (let ((attr (intern-all
										(read-from-string (ask "Please enter the attribute.~%=> ")) :dump)))
								     (if (member attr (attributes ,obj-var) :test #'equal)
									 (display-error "Attribute already exists!")
								       (progn
									 (setf (attributes ,obj-var)
									   (append
									    (attributes ,obj-var)
									    (list attr)))
									 (setf (rows ,obj-var)
									   (let ((nr 0))
									     (mapcar (function
										      (lambda (row)
											(incf nr)
											(append row
												(list
												 (intern-all
												  (read-from-string
												   (ask 
												    (format nil "Enter value for example ~a.~%=> " nr)))
												  :dump)))))
										     (rows ,obj-var))))
									 (refresh ,obj-var))))))
				       "Add a new attribute and it's values to the table")
				      (list
				       "Delete Column" (function (lambda ()
								      (unless (null (attributes ,obj-var))
									(let* ((attr (accept-items (make-instance 'menu
												     :items
												     (mapcar (function (lambda (attr)
															 (list (all-symbol-names attr)
															       (write-to-string attr))))
													     (attributes ,obj-var)))))
									       (pos (search
										     (list attr)
										     (attributes ,obj-var)
										     :test #'(lambda (y x ) (equal x (write-to-string y))))))
									  (progn
									    (setf (attributes ,obj-var)
									      (remove-nth pos (attributes ,obj-var)))
									    (setf (rows ,obj-var)
									      (mapcar (function (lambda (row)
												  (remove-nth pos row)))
										      (rows ,obj-var)))
									    (refresh ,obj-var))))))
				       "Delete an attribute and it's values")
				      (list
				       "Add Row" (function (lambda ()
								 (setf (rows ,obj-var)
								   (append
								    (rows ,obj-var)
								    (list
								    (mapcar (function (lambda (attr)
											(intern-all
											 (read-from-string
											  (ask (format nil "Enter value for attribute ~a.~%=> " attr))) :dump)))
									    (attributes ,obj-var)))))
								 (refresh ,obj-var)))
				       "Add a new example (a row) to the table")
				      (list "Delete Row" (function (lambda ()
									 (let ((pos nil))
									   (setf pos
									     (read-from-string
									      (ask "Please enter the number of the example.~%=> "
										   :condition 
										   (function (lambda (string)
											       (let ((x (read-from-string string nil)))
												 (and (numberp x) (integerp x) (>= x 0)))))
										   :error-message "The example has to be identified with a number.")))
									   (if (and (plusp pos) (<= pos (number-of-rows ,obj-var)))
									       (progn
										 (setf (rows ,obj-var)
										   (remove-nth (1- pos) (rows ,obj-var)))
										 (refresh ,obj-var))
									     (display-error "Example does not exist!")))))
					    "Delete an example (a row)"))
				   nil))
		  (user-items ,menu)
		  (items (append default-items user-items))
		  (menu (make-instance 'menu :items (if items items '(("No menu" ,(function (lambda ())))))))
		  (help-button (make-instance 'push-button :label "Help"
					      :font font
					      :width (* choice-width (font-character-width font))
					      :height (+ 6 (font-character-height font))))
		  (menu-button (make-instance 'pop-up-button :label "Menu"
					      :font font
					      :width (* choice-width (font-character-width font))
					      :height (+ 6 (font-character-height font))))
		  (refresh-button (make-instance 'push-button :label "Refresh"
						 :font font
						 :width (* choice-width (font-character-width font))
						 :height (+ 6 (font-character-height font))))
		  (exit-button (make-instance 'push-button :label "Exit"
					      :font font
					      :width (* choice-width (font-character-width font))
					      :height (+ 6 (font-character-height font))))
		  (names (let ((l nil))
			   (dotimes (i y (reverse l))
			     (setf l (cons (1+ i) l))))))

	     (,close-fcn-name ,obj-var :close-all close-all)

	     (setf disp (make-instance 'display :title title
				      :width (max
					      (+ offset-x 1 (* x b-width))
					      (+ (* 4 (* choice-width (font-character-width font)))
						 (* 5 dist)))
				      :height (+ offset-y 1 (* (1+ y) b-height) gab 1)
				      :font font
				      :left (if left left 10)
				      :bottom (if bottom bottom
						(- 574 (+ offset-y 1 (* (1+ y) b-height) gab 1)))))

	     (setf (gethash ,obj-var ,display-collector-name)
	       (make-instance 'editor-class
		 :display disp
		 :cell-font cell-font
		 :cell-width cell-width
		 :offset-x offset-x
		 :close-all close-all))

	     (dotimes (i x)
	       (let ((button (make-instance 'value-button
			       :numeric nil
			       :font cell-font
			       :width b-width
			       :value (all-symbol-names (nth i (attributes ,obj-var))))))
		 (set-button button disp 
			     :left (+ offset-x (* i b-width))
			     :bottom (- (height disp)
					offset-y 1 b-height 1)
			     :action 
			     `(lambda ()
				(let ((new-val (intern-all (read-from-string (button-value ,button) nil) :dump)))
				  (if (and (member new-val (attributes ,,obj-var) :test #'equal)
					   (not (equal new-val (nth ,i (attributes ,,obj-var)))))
				      (progn
					(display-error "Attribute already exists!")
					(setf (button-value ,button) (nth ,i (attributes ,,obj-var))))
				    (setf (nth ,i (attributes ,,obj-var)) 
				      new-val)))))))
	     (dotimes (j y)
	       (dotimes (i x)
		 (let ((button (make-instance 'value-button
				 :numeric nil
				 :font cell-font
				 :width b-width
				 :value (all-symbol-names (get-xy-value i j ,obj-var)))))
		   (set-button button disp 
			       :left (+ offset-x (* i b-width))
			       :bottom (- (height disp)
					  offset-y 1
					  (* (1+ j) b-height)
					  b-height gab 1)
			       :action 
			       `(lambda ()
				  (set-xy-value ,i ,j ,,obj-var
						(intern-all
						(read-from-string (button-value ,button) nil) :dump)))))))

	     (when (plusp offset-x)
	       (write-display disp "Examples" 8 (- (height disp)
						   offset-y
						   b-height -5) :font cell-font)

	       (dotimes (j y)
		 (write-display disp
				(format nil "~6@a" (nth j names))
				10 (- (height disp) offset-y  1 (* (+ j 2) b-height) 1) :font cell-font))

	       (draw-line disp 
			  (- offset-x 3)
			  0
			  (- offset-x 3)
			  (- (height disp) offset-y)))

	     (draw-line disp 
			0
			(- (height disp)
			   offset-y -1)
			(width disp)
			(- (height disp)
			   offset-y -1))
	     (draw-line disp 
			0
			(- (height disp)
			   offset-y 1
			   (+ b-height (/ gab 2)) 1)
			(width disp)
			(- (height disp)
			   offset-y 1
			   (+ b-height (/ gab 2)) 1))

	     (set-button help-button disp 
			 :left (+ (* 0 (+ (* choice-width (font-character-width font)) dist)) dist)
			 :bottom (- (height disp)
				    (round (/ (- offset-y b-height) 2))
				    b-height)
			 :action #'(lambda nil 
				     (make-instance 'help-stream
				       :filename (add-path
						  "table-ed.asc"
						  (add-subdir *pail-path* "pail-lib"))
				       :from-button help-button)))
	     (setf (menu menu-button) menu)
	     (set-button menu-button disp
			 :left (+ (* 1 (+ (* choice-width (font-character-width font)) dist)) dist)
			 :bottom (- (height disp)
				    (round (/ (- offset-y b-height) 2))
				    b-height)
			 :action #'(lambda nil))
	     (set-button refresh-button disp 
			 :left (+ (* 2 (+ (* choice-width (font-character-width font)) dist)) dist)
			 :bottom (- (height disp)
				    (round (/ (- offset-y b-height) 2))
				    b-height)
			 :action #'(lambda () (refresh ,obj-var)))
	     (set-button exit-button disp 
			 :left (+ (* 3 (+ (* choice-width (font-character-width font)) dist)) dist)
			 :bottom (- (height disp)
				    (round (/ (- offset-y b-height) 2))
				    b-height)
			 :action #'(lambda ()
				     (close-display disp)
				     (remhash ,obj-var ,display-collector-name)))

	     (values ,obj-var disp)))))))


;;; ========================================================================
;;; END OF FILE
;;; ========================================================================
