;;; -*- Mode: LISP; Package: PLANNING; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - IDSIA LUGANO
;;;
;;; ************************************************************************
;;;
;;; Filename:   planning-dialog.cl
;;; Short Desc: dialog handling for Planning
;;; Version:    1.0
;;; Status:     Experimental
;;; Last Mod:   24.1.92 DTA
;;; Author:     Mike Rosner
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------
 

;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :planning)





;;; ==========================================================================
;;; GLOBAL VARIABLE DECLARATIONS
;;; ==========================================================================




(defparameter *verbose* *verbose-default*)
(defparameter *animate* nil)
(defvar *verbose-disp* nil)
(defvar *movie-window* nil)
(defvar *param-display* nil)
(defvar *execution-terminated* nil)
(defvar *which-planner* strips::*strips-planner*)
(defvar *which-search* search::*depth-first-search*)
(defvar *which-rank* 'snlp::rank)
(defvar *known-rank-fns* '(snlp::rank snlp::rank1 snlp::rank2 snlp::rank3))
(defvar *planning-loaded* nil)
(defvar rules::*bc-loaded* nil)


;;==================================================================
;; The Planning Dialog

(defun start-planning-dialog (&optional (from-button nil))
  (setq *demo-in-execution* nil)
;  (format t "planning: ~a - ~a" planning::*verbose*
;	  (not (typep planning::*verbose-disp* 'gin::scroll-display)))
  (if (and *verbose* (not (typep *verbose-disp* 'gin::scroll-display)))
      (progn  (setf *verbose-disp* (make-instance 'scroll-display
						 :title "Planning:Verbose Comments"
						 :width (round (width *root-window*) 3)
						 :height (round (height *root-window*) 3)
						 :borders 1
						 :active nil
						 :left (round (width *root-window*) 2)
						 :bottom (round (height *root-window*) 3)))
	      
	      (setf *verbose* *verbose-disp*)
	      )
    (setf *verbose-disp* *verbose*))
;  (deactivate-display pail-lib::*main-window*) ; especially during
					;  debugging, I don't
					;  recommend this. -dta
  (let* ((disp (make-instance 'display :title "Planning:Main Window"
			     :width 300
			     :height 150
			     :borders 1
			     :left (+ (width pail-lib::*main-window*) (left pail-lib::*main-window*))
			     :bottom (bottom pail-lib::*main-window*)))
	 (menu-button (make-instance 'pop-up-button
			:label "Demos"
			:width 100))
	 (exit-button (make-instance 'push-button :label "Exit" :width 70))
	 (help-button (make-instance 'help-button
			:technical (add-path "planning-desc.tec" *planning-path*)
			:general (add-path "planning-desc.gen" *planning-path*)
			:subject "Planning"))
	 (tool-button (make-instance 'push-button :label "Tool" :width 100))
	 (menu (make-instance 'menu
		 :items  (let ((mlist nil))
			   (dotimes (i *number-of-demos* (nreverse mlist))
			     (setf mlist 
			       (cons (list (format nil "Demo ~a" (1+ i)) 
					   `(lambda () 
					      (demo ,(1+ i) ,menu-button ,exit-button)) 
					   (make-title (1+ i)))
				     mlist))))
		 
       		 :query " DEMOS ")))
    
    (setf (menu menu-button) menu)
    
    (copy-mask *pail-logo* 0 0 disp
	       (- (width disp) (cw:bitmap-width *pail-logo*) 15)
	       (- (height disp) (cw:bitmap-height *pail-logo*) 15))
    
;    (setf (font disp) (cw:open-font :courier :italic 20 :weight :bold))
;    (write-display disp
;		   "Welcome to ..."
;		   15 (- (height disp) 15 (cw:font-ascent (font disp))))
    (setf (font disp) (cw:open-font :courier :italic 45 :weight :bold))
    (write-display disp
		   "Planning"
		   (/ (width disp) 12)
		   (- (height disp) 40 (cw:font-ascent (font disp))))
    (setf (font disp) *default-font*)
    (set-button exit-button disp 
		:left (- (width disp) (width exit-button) 15)
		:bottom 15
		:action `(lambda nil
			   (when ,disp (close-display ,disp))
			   (when *verbose-disp* (close-display *verbose-disp*))
			   (when *planner-window* (close-display *planner-window*))
			   (when *node-info-window* (close-display *node-info-window*))
			   (when *movie-window* (close-display *movie-window*))
;			   (activate-display pail-lib::*main-window*)
;			   (sleep 1)
			   
			   
			   (if ,from-button (reset-button ,from-button))))
    
    (set-button tool-button disp
		:left (/ (- (width disp) (width menu-button)) 2)
		:bottom 15
		:action `(lambda nil 
			   (disable-button ,exit-button)
			   (disable-button ,menu-button) 
;			   (deactivate-display ,disp)
			   (planning-tool  ,tool-button) 
			   (setq  *execution-terminated* nil)
			   (mp:process-wait "running" 
					    '(lambda nil *execution-terminated*))  
			   (activate-display ,disp)(sleep 1)
			   (reset-button ,tool-button)
			   (enable-button ,exit-button)
			   (enable-button ,menu-button)))
    

    (set-button help-button disp :left 15 :bottom 15)
    
    (set-button menu-button disp 
		:left (/ (- (width disp) (width menu-button)) 2)
		:bottom 45)
    disp))





(defun make-title (nr)
  (format nil "~:R Demo on~%   Planning~%" nr))

(defun make-title-w (nr)
  (format nil "~:R Demo on Planning" nr))


(defun demo (nr menu-button other-exit-button)
  (disable-button menu-button)
  (disable-button other-exit-button)
  (let* ((dispwidth (min 500 (round (width *root-window*) 3)))
	(dispheight (min 260 (round (height *root-window*) 3)))
	(*default-font* (findfont (width *root-window*) 1140 13))
	(disp (make-instance 'display :title (concatenate 'string "Planning:" (make-title nr))
			    :width dispwidth
			    :height dispheight
			    :borders 1
			    :left (- (width *root-window*) dispwidth 15)
			    :bottom (- (height *root-window*) dispheight 20)))
	(exit-button (make-instance 'push-button :label "Exit" :width (round (width disp) 5)))
	(help-button (make-instance 'help-button
		       :width (round (width disp) 5)
		       :subject (concatenate 'string "PLANNING:" (make-title nr))
		       :technical (add-path
					"demo-desc.tec"
					*planning-path*)
		       :general (add-path
					(concatenate 'string
					  "demo" (write-to-string nr) "-desc.gen")
					*planning-path*) ))
	(start-button (make-instance 'push-button :label "Start" :width (round (width disp) 5)))
	(cont-button (make-instance 'push-button :label "Continue" :width (round (width disp) 5)))
	)

    (if *verbose*
	(progn  (setf *verbose-disp* (make-instance 'scroll-display
						   :title "Planning:Verbose Comments"
						   :width (round (width *root-window*) 3)
						   :height (round (height *root-window*) 3)
						   :borders 1
						   :active t
						   :left (round (width *root-window*) 2)
						   :bottom (round (height *root-window*) 3)))
	      
		(setf *verbose* *verbose-disp*)
		)
      (setf *verbose-disp* *verbose*))
    (setf (font disp) (cw:open-font :courier :italic 20 :weight :bold))
    (write-display disp 
		   (make-title nr)
		   15 (- (height disp) 15 (cw:font-ascent (font disp))))
    (setf (font disp) *default-font*)


    (copy-mask *pail-logo* 0 0 disp 
	       (- (width disp) (cw:bitmap-width *pail-logo*) 15)
	       (- (height disp) (cw:bitmap-height *pail-logo*) 15))

    (set-button exit-button disp 
		:left (- (width disp) (width exit-button) 15) 
		:bottom 15
		:action `(lambda ()
			   (close-display *verbose-disp*)
			   (close-display *planner-window*)
			   (close-display *node-info-window*)
			   (close-display *movie-window*)
			   (setf *quit* t)
			   (enable-button ,menu-button)
			   (enable-button ,other-exit-button)
			   (close-display ,disp)))

    (set-button help-button disp :left 15 :bottom 15)
    

    (set-button start-button disp 
		:left (round (+ 15
				(width start-button)
				(round (- (width disp)
				      30
				      (* 4 (width start-button)))
				   3)))
		:bottom 15
		:action
		#'(lambda ()
		    (disable-button exit-button)
		    (dialog-handler nr disp start-button cont-button exit-button)))
		
    (set-button cont-button disp 
		:left (round (+ 15
				(* 2 (width start-button))
				(* 2 (round (- (width disp) 
					   30 
					   (* 4 (width start-button)))
					3))))
		:bottom 15
		:action 
		#'(lambda ()
		    (disable-button exit-button)
		    (setf *pause* (not *pause*))))
    (disable-button cont-button)

    
    
    
		    
    
    ))



(defun dialog-handler (nr parent start-button cont-button exit-button)
  (let ((text-disp (make-instance 'scroll-display
                                  :title "PLANNING:Comments"
				  :font (findfont (- (width parent) 25) 360 10)
				  :parent parent
				  :width (- (width parent) 25)
				  :height (- (round (height parent) 2) 20)
				  :borders 1
				  :left 12
				  :bottom 50))
	
	(problem-disp (make-instance 'scroll-display
                                  :title "Planning: sample problem"
				  :font (findfont (- (round (width *root-window*) 2.5) 10) 360 10)
				  :width (- (round (width *root-window*) 2.5) 10)
				  :height (- (round (height *root-window*) 1.5) 40)
				  :borders 1
				  :left (- (+ (left parent) (width parent)) (- (round (width *root-window*) 2.5) 10))
				  :bottom (- (bottom parent) (- (round (height *root-window*) 1.5) 20) ))))
    (labels ((clean-up ()
	       (close-display text-disp)
	       (close-display problem-disp)
	       
	       
	       
	       (unless *quit*
		 (enable-button exit-button)
		 (reset-button cont-button)
		 (disable-button cont-button)
		 (reset-button start-button))))
      (defun wait ()
	(unless *quit*
	  (enable-button cont-button)
	  (reset-button cont-button)
	  (enable-button exit-button)
	  (format-display text-disp "~&~%[Press <Continue> Button]")
	  (mp:process-wait "wait-on-click" (function (lambda () (or *quit* *pause*))))
	  ;(do () ((or *quit* *pause*)))
	  (if *quit* (clean-up)
	    (setf *pause* nil)))
	(not *quit*))
      (setf *pause* nil)
      (setf *quit* nil)
      (let ((fun-sym (find-symbol (symbol-name (read-from-string (format nil "dialog-~a" nr))) :planning)))
	(if fun-sym
	    (funcall fun-sym text-disp nil problem-disp)
	  (format t "Demo not implemented")))
      (wait)
      (clean-up))))



;;====================================================================






(defclass planning-tool-display (display)
	  ((plannerw :accessor plannerw
		     :initarg :plannerw
		     :initform nil)
	   (nifw :accessor nifw
		 :initarg :nifw
		 :initform nil)
	   (movie :accessor movie
		  :initarg :movie
		  :initform nil)
	   (param :accessor param
		  :initarg :param
		  :initform nil)
	   (help-button :accessor help-button
		  :initarg :help-button
		  :initform nil)))

(defmethod close-display :before ((disp planning-tool-display))
  (close-display (plannerw disp))
  (close-display (nifw disp))
  (close-display (movie disp))
  (close-display (param disp))
  (close-display (technical-window (help-button  disp)))
  (close-display (general-window (help-button  disp)))
  )


(defun planning-tool (&optional (from-button nil))
  (setf  *verbose* *verbose-default*)
  (setf *planning-loaded* t)
  (setf *verbose-disp* (make-instance 'scroll-display
			 :title "Planning:Verbose Comments"
			 :width (round (width *root-window*) 3)
			 :height (round (height *root-window*) 3)
			 :borders 1
			 :active *verbose*
			 :left (round (width *root-window*) 2)
			 :bottom (round (height *root-window*) 3)))

  (setf *trace-stream* *verbose-disp*)
  (let* ((dispwidth (min (round (width *root-window*) 2) 800))
	 (dispheight (min (round (height *root-window*) 3) 260))
	 (*default-font* (findfont (width *root-window*) 1140 13))
	 (disp (make-instance
		 'planning-tool-display
		:title "PLANNING: Search Spaces"
		:width dispwidth
		:height dispheight
		:left (- (width *root-window*) dispwidth)
		:bottom (- (height *root-window*) dispheight 20)))
	 
	 
	 (exit-button (make-instance 'push-button :label "Exit" :width (round (width disp) 8)))
	 (help-button (make-instance 'help-button
			:technical (add-path "tool-desc.tec" *planning-path*)
			:general (add-path "tool-desc.gen" *planning-path*)
			:width (round (width disp) 8)))
	 (run-button (make-instance 'push-button :label "Run" :width (round (width disp) 8)))
	 (stop-button (make-instance 'push-button :label "Stop" :width (round (width disp) 8)))
	 (param-button (make-instance 'push-button :label "Parameters" :width (round (width disp) 6)))
	 (verb-button (make-instance 'radio-button :label "Verbose mode" :status *verbose*))
	 (anima-button (make-instance 'radio-button :label "Animation" :status *animate*))
	 (limit-button (make-instance 'value-button :name "Search Limit:" :width (round (width disp) 8) 
				      :numeric t :value *limit*))
	 
	 

	 (menu-button-ops (make-instance 'pool-button
			    :name "Operator Set:"
			    :value ""
			    :width (round (width disp) 5)
			    :target-class 'planning::op-set
			    :pool *pail-pool*
			    :border t
			    :show-function #'(lambda (item)
					       (documentation-print (dump-editable item )))
			    ))
	 (menu-button-wms (make-instance 'pool-button
			    :name "Initial state:"
			    :pool *pail-pool*
			    :value ""
			    :width (round (width disp) 5)
			    :target-class 'rules::working-memory
			    :border t
			    ))
	 (goal-button (make-instance 'pool-button
			:name "Goal state:"
			:pool *pail-pool*
			:value ""
			:width (round (width disp) 5)
			:target-class 'rules::working-memory
			:border t
			))

;;; If you want to try to use a graphic system for showing the plans,
;;; it needs to know how to treat the objects you are using.  The
;;; planner just knows them as symbols, but the graphics needs to know
;;; what they look like.  This is a list of list pairs, the first is
;;; an object mentioned in the initial state (like block names), and
;;; the second is a graphic class to which they belong. 

	 (world-button (make-instance 'pool-button
			 :name "Graphics World:"
			 :pool *pail-pool*
			 :value ""
			 :width (round (width disp) 5)
			 :target-class 'world-desc
			 :border t
			 :show-function #'(lambda (item)
					    (documentation-print (dump-editable item )))
			 ))
	 
	 (file-button (make-instance 'file-button
			:w-directory *planning-path*))
	 )				
					
					
					
					
					
					

    (setf (help-button disp) help-button)

    (copy-mask *pail-logo* 0 0 disp
	       (- (width disp) (width *pail-logo*) 15)
	       (- (height disp) (height *pail-logo*) 15))
    (set-button file-button disp
		:left (- (width disp) (width file-button) 7)
		:bottom (- (height disp) (height file-button) (cw:bitmap-height *pail-logo*) 20))
  
    (setf (font disp) (open-font :courier :italic 20 :weight :bold))
    (write-display disp
		   "Welcome to ..."
		   15 (- (height disp) 15 (cw:font-ascent (font disp))))
    (setf (font disp) (open-font :courier :italic 45 :weight :bold))
    (write-display disp
		   "Planning"
		   (round (width disp) 10)
		   (- (height disp) 30 (cw:font-ascent (font disp))))
    (setf (font disp) *default-font*)
    
    

    (set-button help-button disp :left 15 :bottom 15)
    

    (set-button run-button disp 
		:left (+  (width help-button) 15 20)
		:bottom 15
		:action `(lambda nil
			   (cond ((null (start-object (button-value ,menu-button-ops)))
				  (display-error "Please specify an operator set."))
				 (t (let ((goal (assertions
						 (start-object (button-value ,goal-button)))))
				      (setf rules::*backward-chainer* nil)
				      (use-planner *which-planner*)
				      (use-ops (start-object (button-value ,menu-button-ops)))
				      (use-world (start-object (button-value ,world-button)))
				      (setf *limit* (button-value ,limit-button))
				      (setf *multiple-value-goal* nil)
				      (setf *multiple-values* nil)
				      (multiple-value-bind (plan planning-tree)

					  (solve (assertions (start-object (button-value ,menu-button-wms)))
						 (assertions (start-object (button-value ,goal-button))))
					(documentation-print (final-solution plan))
					
					(draw-tree planning-tree
						 
						   )
					(when (and plan (eq *current-planner* snlp::*tweak-planner*))
					  (compute-snlp-graphics-data plan
								      (width *node-info-window*)
								      (height *node-info-window*))
					  (compute-snlp-graphics-data
					   plan
					   (width *node-info-window*)
					   (height *node-info-window*))
					  (setf *h-gap*
					    (/ (width *node-info-window*)
					       (+ 2 (max-level (snlp::snlp-plan-graphics plan)))))
					  (setf (snlp::snlp-plan-graphics plan) nil))
					))))
				    (reset-button ,run-button)
				    ))

    (set-button stop-button disp 
		:left (+  (width run-button) (left run-button)  20)
		:bottom 15
		:action `(lambda nil
			   (setf search::*user-stop* t)
			   (reset-button ,stop-button)
			   ))
    (set-button param-button disp 
		:left (+  (width stop-button) (left stop-button)  20)
		:bottom 15
		:action `(lambda nil
			   
			   (get-parameter-settings ,param-button)
			   ))


  
    (set-button menu-button-ops disp 
		:left (- (width disp) (width menu-button-ops) 60)
		:bottom 55

		)
    
    (set-button menu-button-wms disp 
		:left (- (width disp) (width menu-button-ops) 60)
		:bottom (+ 55 (height menu-button-ops))

		)
    (set-button goal-button disp
		:left (- (width disp) (width menu-button-ops) 60)
		:bottom (+ 55 (* 2 (height menu-button-ops)))
		 )

    (set-button world-button disp
		:left (- (width disp) (width menu-button-ops) 60)
		:bottom (+ 55 (* 3 (height menu-button-ops)))
		 )

    (set-button verb-button disp
		:left 15
		:bottom (bottom menu-button-ops)
		:action #'(lambda nil (setf *verbose* (not *verbose*))
				  (if *verbose* (activate-display *verbose-disp*)
				    (deactivate-display *verbose-disp*))))

    

    (set-button anima-button disp
		:left 15
		:bottom (+ 10 (bottom verb-button) (height anima-button))
		:action #'(lambda nil (setf *animate* (not *animate*))
				  (if (and *animate* (start-object (button-value world-button)))
				      (let ((world (start-object (button-value world-button))))
					(init-movie-window (animation-type world)  (assertions (start-object (button-value menu-button-wms))))
					(activate-display *movie-window*))
				    (deactivate-display *movie-window*))
				  ))

    (set-button limit-button disp
		:left (+ 15 (font-string-width (font disp) "Search Limit: "))
		:bottom (+ (bottom anima-button) 5 (height anima-button)))


    (set-button exit-button disp 
		:left (- (width disp) (width exit-button) 15)
		:bottom 15
		:action `(lambda nil
			   (setf (nifw ,disp) *node-info-window*)
			   (setf (movie ,disp) *movie-window*)
			   (setf (param ,disp) *param-display*)
			   (setf (plannerw ,disp) *planner-window*)
			   (setf *planning-loaded* nil)
			   (if (not rules::*bc-loaded*) (close-display *verbose-disp*))
			   (close-display ,disp)
			   (setf *execution-terminated* t)
			   ))
    ))


(defun final-solution (plan)
  (cond ((snlp::snlp-plan-p plan)
	 (if (not (or (snlp::snlp-plan-unsafe plan)
            (snlp::snlp-plan-open plan)))
	     (snlp::plan-steps plan)
	   "No solution found.")
	 )
	((strips::plan-p plan)
	 (if (null (strips::plan-pending-goals plan))
	     (reverse (strips::plan-steps-taken-stack plan))
	   "No solution found.")
	   )))



;;; ==========================================================================
;;; Setup of the search control to be used. 
;;; ==========================================================================

(defclass parameter-display (display)
	  ((help-button :accessor help-button
			:initform nil
			:initarg :help-button)))
			

(defmethod close-display :before ((disp parameter-display))
  (close-display (technical-window (help-button disp)))
  (close-display (general-window (help-button disp))))


(defun get-parameter-settings (called-by-button)
  
  (let* ((parmfont (findfont (width *root-window*) 1140 13))
	 (*default-font* parmfont)
	 (strips-button (make-instance 'radio-button :label "Linear (Strips)"
				       :status (equal *which-planner* strips::*strips-planner*)))
	 (tweak-button (make-instance 'radio-button :label "Non-linear"
				      :status (equal *which-planner* snlp::*tweak-planner*)))
	
	 (help-button (make-instance 'help-button
			:technical (add-path "planning-p.tec" *planning-path*)
			:general (add-path "planning-p.gen" *planning-path*)
			:subject "Planning Parameters"))
	 (depth-button (make-instance 'radio-button :label "Depth First"
				      :status (equal *which-search* search::*depth-first-search*)))
	 (breadth-button (make-instance 'radio-button :label "Breadth First"
					:status (equal *which-search* search::*breadth-first-search*)))
	 (best-button (make-instance 'radio-button :label "Best First"
				     :status (equal *which-search* search::*best-first-search*)))
	 #| (limit-button (make-instance 'value-button :name "Search Limit:" :width 75 
				      :numeric t :value *limit*)) |#
	 window-height
	 (vb-height (+ 3 (height strips-button)))
	 (lineheight (font-character-height *default-font*))
	 (rb-height (max (height help-button) (font-character-height *default-font*))))

    

    
    (setf *parameter-done* (make-instance 'push-button :label "Exit"))
    (setf *new-f* (make-instance 'push-button :label "New"))
    (setf *defined-f* (make-instance 'select-button :label "Rank functions"
				     :exclusive t
				     
				     :width (- (+ (font-string-width *default-font* "Plan Representation")  10)
					       (font-string-width *default-font* "New")
					       25)
				     :height 90
				     :action `(lambda ()
						(change-ff (item-label)))
				     :items (mapcar #'(lambda (s) (format nil "~a" s))
						    *known-rank-fns*)))
    (setf window-height (+ (* 5 vb-height) 
			   (height *defined-f*) (* 9 lineheight) 10))

    
    (setf *param-display*
      (make-instance 'parameter-display  :title "PLANNING:Parameters"
		     :font parmfont
		     :left (- (width *root-window*) (+ (font-string-width parmfont "Plan Representation")  10) 30)
		     :bottom (- (height *root-window*) window-height 20)
		     :width (+ (font-string-width *default-font* "Plan Representation")  10)
		     :height window-height))

    (setf (help-button *param-display*) help-button)

    (write-display *param-display*
		   "Plan Representation"
		    5  (- window-height lineheight))
		   

    (set-button strips-button *param-display*
		:left 10
		:bottom (- window-height lineheight vb-height 4)
		:action #'(lambda nil
			    (mp:process-wait-with-timeout "strips-wait" 0.2 (function (lambda () (status strips-button))))
			    (when (status strips-button)
			      (setf *which-planner* strips::*strips-planner*)
			      (setf *which-search* (planner-default-search-fn *which-planner*))
			      (loop for style in (list
						  search::*depth-first-search*
						  search::*breadth-first-search*
						  search::*best-first-search* )
				  as button in (list
						depth-button breadth-button best-button)
				  do
				    (if (not (eq (eq *which-search* style) (status button)))
					(toggle-button button))
				     
				))
			    
			    ))


    (set-button tweak-button *param-display*
		:left 10
		:bottom (- window-height lineheight (* 2 vb-height) 4)
		:action #'(lambda nil
			    (mp:process-wait-with-timeout "tweak-wait" 0.2 (function (lambda () (status tweak-button))))
			    (when (status tweak-button)
			      (setf *which-planner* snlp::*tweak-planner*)
			      (setf *which-search*
				(planner-default-search-fn *which-planner*))
			      (loop for style in (list
						  search::*depth-first-search*
						  search::*breadth-first-search*
						  search::*best-first-search* )
				  as button in (list
						depth-button breadth-button best-button)
				  do
				    (if (not (eq (eq *which-search* style) (status button)))
					(toggle-button button))
				     
				))
			    
			    ))

    (set-exclusive strips-button tweak-button)

    (write-display *param-display*
		   "Search Strategy"
		   5  (- window-height (* 2 lineheight) (* 3 vb-height)))
    
    (set-button depth-button *param-display*
		:left 10
		:bottom (- window-height (* 2 lineheight) (* 4 vb-height) 8)
		:action #'(lambda nil
			    (when (status depth-button)
			      (setf *which-search* search::*depth-first-search*)
			      (setf (planner-default-search-fn *which-planner*)
				*which-search*))))
    (set-button breadth-button *param-display*
		:left 10
		:bottom (- window-height (* 2 lineheight) (* 5 vb-height) 8)
		:action #'(lambda nil
			    (when (status breadth-button)
			      (setf *which-search* search::*breadth-first-search*)
			      (setf (planner-default-search-fn *which-planner*)
				*which-search*))))
    (set-button best-button *param-display*
		:left 10
		:bottom (- window-height (* 2 lineheight) (* 6 vb-height) 8)
		:action #'(lambda nil
			    (when (status best-button)
			      (setf *which-search* search::*best-first-search*)
			      (setf (planner-default-search-fn *which-planner*)
				*which-search*))))


    (set-exclusive best-button breadth-button depth-button)

    
    
    (draw-line *param-display* 0 (- (bottom tweak-button) 5)
	       (width *param-display*) (- (bottom tweak-button) 5))
    (draw-line *param-display* 0 (- (bottom best-button) 5)
	       (width *param-display*) (- (bottom best-button) 5))
    (draw-line *param-display* 0 (+ rb-height 10) (width *param-display*) (+ rb-height 10))
    

    #| (set-button limit-button *param-display*
		:left (+ 10 (font-string-width *default-font* "Search Limit"))
		:bottom (- (bottom best-button) 15 lineheight)) |#
    
    (set-button *defined-f* *param-display*
		:left 5
		:bottom (+ 16 (height *parameter-done*)))
    (set-button *new-f* *param-display*
		:left (- (width *param-display*) (width *new-f*) 5)
		:bottom (bottom *defined-f*)
		:action `(lambda () (new-ff)))
    (if *which-rank* (dolist (i (items *defined-f*) t)
			     (if (equal (write-to-string (intern *which-rank*)) (label i))
				 (unless (status i) (toggle-button i))
			       (clear-button i))))
    (set-button *parameter-done* *param-display*
		:left (- (width *param-display*) 5 (width *parameter-done*))
		:bottom 5
		:action `(lambda nil
			   (close-display ,*param-display*)
			   (reset-button ,called-by-button)))
    (set-button help-button *param-display* :left 5 :bottom 5)
    ))

(defun change-ff (function-name)

  (setf *which-rank* (loop for fn in *known-rank-fns*
      thereis (if (equal function-name (format nil "~a" fn)) fn)))
  (setf (planner-default-rank-fn *which-planner*) *which-rank*)
  
  )

(defun new-ff ()
  (let ((function-name (ask "What is the name of the new function? ")))
    (if (not (fboundp (intern function-name :planning)))
	(display-error (format nil
			       "The Function `~a' is not defined~
                  ~&in the Planning Package."
			       (intern function-name)) :wait nil)
      (progn
	(push (intern function-name :planning) (planner-rank-fns *which-planner*))
	(push (intern function-name :planning) *known-rank-fns*)
	(add-item  *defined-f* function-name :action (action *defined-f*))
      ))
  (reset-button *new-f*)
  ))







;;; ========================================================================
;;; END OF FILE
;;; ========================================================================
