;;; -*- Mode: LISP; Package: SNLP; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; ************************************************************************
;;;
;;; Filename:   unif.cl
;;; Short Desc: Unifier for SNLP
;;; Version:    0.1
;;; Status:     Provisional
;;; Last Mod:   12.5.92 DTA
;;; Author:     Wan
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;
;;;
;;; --------------------------------------------------------------------------
;;; Change History:
;;;
;;;
;;; --------------------------------------------------------------------------
" (c) 1990, 1991 Copyright (c) University of Washington
  Written by Stephen Soderland, Tony Barrett and Daniel Weld.

  All rights reserved. Use of this software is permitted for non-commercial
  research purposes, and it may be copied only for that use.  All copies must
  include this copyright message.  This software is made available AS IS, and
  neither the authors nor the University of Washington make any warranty about
  the software or its performance.

  When you first acquire this software please send mail to 
  bug-snlp@cs.washington.edu; the same address should be used for problems."

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Program   : "SNLPUnify"
;;;  Author    : Stephen Soderland (modifed by Dan Weld)
;;;  Date      : Summer/Autumn 1990
;;;
;;;  Description:
;;; 
;;;    Unification functions used by SNLP
;;;

(in-package :snlp)

;;;;;;;;;;;;;;;;;;;;;;;;                                                  
;;; Test whether two conditions can be unified consistent with plan bindings
;;; Returns a list of the list of bindings needed to unify, i.e. has an extra
;;; set of parentheses.  Returns nil if they cannot be unified.  Returns (nil)
;;; if they unify with no additional bindings.
(defun unify (cond1 
	      cond2 
	      bindings
	      &optional 
	      (unify-bind '(nil))        ; (NIL) is no bindings needed
	      (stream *standard-output*))
  (if (> *trace* 2) (format stream "~%     Unifying ~a ~a" cond1 cond2))
  (if (and (eql 'not (car cond1))               ; if both are negative remove
	   (eql 'not (car cond2)))              ;   'not and unify remainder
      (unify (cadr cond1) (cadr cond2) bindings unify-bind)
      (if (and                                  ; else 
	   (eql (list-length cond1)             ;   if same length and 
		(list-length cond2))            ;      same predicate
	   (eql (car cond1) (car cond2)))
	  (dolist (b1 (cdr cond1) unify-bind)  ; test each matching terms
	    (setf b1 (bind-variable b1 bindings))  ;   bind to constant value
	    (setf cond2 (cdr cond2))
	    (let ((b2 (bind-variable (car cond2) bindings)))
              (if (> *trace* 2) 
                  (format stream "~%      Unifying ~a and ~a  w/ plan-binding  ~a" 
                          b1 b2 unify-bind))
              (incf *unify-count*)
              (if *computing-rank*
                  (incf *compute-rank-unifies*))
              (if (not (equal-term? b1 b2))
                  (if (consistent? b1 b2 bindings (car unify-bind))
                      (let ((new-bind (list b1 b2)))
                        (if (not (variable? b1))
                            (setf new-bind (list b2 b1)))
                        (setf unify-bind (list (cons new-bind 
                                                     (car unify-bind)))))
                      (return nil))))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Returns true if equal constants or both variables with equal ID's
(defun equal-term? (a b)
  (or (equal a b)
      (and (variable? a)
	   (variable? b) 
	   (equal (variable-id a) (variable-id b)))
      (and (listp a)
	   (listp b)
	   (equal-term? (car a) (car b))
	   (equal-term? (cadr a) (cadr b)))))
  

;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Replace variable with associated constant from plan-bindings
;;; Leave as a variable if not bound to a constant
(defun bind-variable (var &optional (bindings *bindings*))
  (let ((bound var))
    (if (listp var)
	(setf bound (loop for v in var collect (bind-variable v bindings)))
	(if (variable? var)
	    (let ((id (variable-id var))
		  (e1  nil))
	      (setf e1 (gethash id bindings))
	      (if e1
		  (if (bind-value e1)
		      (setf bound (bind-value e1))
		      (if (bind-syn e1)
			  (setf bound (car (bind-syn e1)))))))))
    (if (stillvars bound) bound (expand-eval bound))))


;;;;;;;;;;;;;;;;;;;;;;;;                                                  
;;; Test if B1 B2 are consistent with existing bindings
;;; If either is not found in bindings, they are ok.
;;; If one is found in the not list of the other,
;;; they are not ok.
(defun consistent? (b1 b2 bindings bind-list 
		       &optional (stream *standard-output*))
  (if (> *trace* 3) (format stream "~%         Consistent? ~a ~a" b1 b2))
  (let ((id1 b1)
	(id2 b2)
	(e1 nil)
	(e2 nil)
	(ok? t))
    (if (variable? b1)
	(setf id1 (variable-id b1)))
    (setf e1 (gethash id1 bindings))         ; e1 is entry for b1
    (if (variable? b2)
	(setf id2 (variable-id b2)))
    (setf e2 (gethash id2 bindings))         ; e2 is entry for b2
    (cond
      ((and (not (variable? b1))     ; not ok if two constants
	    (not (variable? b2)))    ; since not equal
       (setf ok? nil))
      ((or (null e1) (null e2))      ; ok if either is new term
       (setf ok? t))
      ((or (gethash id1 (bind-not e2))  ; not ok if b1 is in not(e2)
	   (gethash id2 (bind-not e1))) ; or b2 is in not(e1)
       (setf ok? nil))              
      (t                             ; otherwise need to check bind-list
       (setf ok? (check-bind-list b1 bindings bind-list))
       (when ok?
	   (setf ok? (check-bind-list b2 bindings bind-list)))))
    ok?))



;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Test if B1 or B2 is consistent with list of bindings from unifying
;;; earlier terms.
(defun check-bind-list (b1 bindings bind-list
			   &optional (stream *standard-output*))
  (let ((id1 b1)
	(id2 nil)
	(ok? t)
	(e nil))
    (if (variable? b1)
	(setf id1 (variable-id b1)))
    (dolist (pair bind-list ok?)
      (if (> *trace* 4) (format stream "~%     Check ~a with ~a" b1 pair))
      (cond
	((equal b1 (car pair))             ; if b1 = (car pair)
	 (if (variable? (cadr pair))       ; get entry for (cadr pair)
	     (setf id2 (variable-id (cadr pair)))
	     (setf id2 (cadr pair)))
	 (setf e (gethash id2 bindings))   ; not ok if b1 is in
	 (if (gethash id1 (bind-not e))    ;  not of (cadr pair)
	     (return nil)))                
	((equal b1 (cadr pair))            ; if b1 = (cadr pair)
	 (if (variable? (car pair))        ; get entry for (car pair)
	     (setf id2 (variable-id (car pair)))
	     (setf id2 (car pair)))
	 (setf e (gethash id2 bindings))   ; not ok if b1 is in
	 (if (gethash id1 (bind-not e))    ;  not of (car pair)
	     (return nil)))))))            


;;;;;;;;;;;;;;;;;;;;;;;;                                                  
;;;  Add new entries to bindings
;;;  Input is list of pairs of two terms, each pair possibly preceded by 'not
;;;  Finds their entries E1 and E2 in binding hash table.  If necessary, 
;;;  entries are created.  If an entry has a constant in its value field,
;;;  the entry for that value is used.  If the original pair has 'not, 
;;;  then add-constraint is called.  Otherwise add-bind2 is called.
(defun add-bind (new-bind
		 bindings
		 &optional
		 (not-flag nil)
		 (stream *standard-output*))
  (dolist (pair new-bind bindings)        ; 
    (if (> *trace* 3) (format stream "~%       Add bind ~a" pair))
    (setf *add-bind-count* (+ *add-bind-count* 1))
    (cond ((eql 'not (car pair))
	   (setf not-flag t)
	   (setf pair (cadr pair))))
    (let ((b1 (car pair))
	  (id1 (car pair))
	  (b2 (cadr pair))
	  (id2 (cadr pair))
	  (e1 nil)
	  (e2 nil))
      (if (variable? b1)
	  (setf id1 (variable-id b1)))
      (if (variable? b2)
	  (setf id2 (variable-id b2)))
      (setf e1 (get-bind-entry b1 id1 bindings))  ; get entry in hash tbl
      (setf e2 (get-bind-entry b2 id2 bindings))  ; for each term of pair

; added to check consistency.  Causes trouble if nil result not expected --Hank
      (if (or (and not-flag (already-bound e1 e2))
	      (and (not not-flag) (not (consistent? b1 b2 bindings nil))))
	  (return nil))

      (unless (already-bound e1 e2)
	(if not-flag                            
	    (add-constraint e1 e2 bindings)   ; if pair has 'not, add constraint
	    (add-bind2 e1 e2 bindings)))      ; else add binding
      bindings)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Returns true if two keys are already synonyms, or if
;; one is the value of the other.
(defun already-bound (e1 e2)
  (let ((b1 (bind-key e1))
	(b2 (bind-key e2)))
    (or (equal b1 (bind-value e2))
	(equal b2 (bind-value e1))
	(equal b1 b2)
	(member b1 (bind-syn e2))
	(member b2 (bind-syn e1)))))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Find entry in bindings hash table
;; If necessary, create entry.  If entry points to
;; another entry as its 'value', get that entry.
(defun get-bind-entry (b1 id1 bindings)
  ; (format t "~%Get-Bind-Entry with ...  b1 ~a,  id2 ~a~%" b1 id1)
 
  (let ((e1 (gethash id1 bindings)))       ; e1 is entry for term b1
    (cond ((null e1)                       ; create e1 if necessary
	   (setf (gethash id1 bindings)
		 (make-bind :key b1
			    :value (unless (variable? b1) b1)
			    :not (make-hash-table :test #'equal)))
	   (setf e1 (gethash id1 bindings))))
    (if (and (variable? b1) (bind-value e1))             ; if e1 points to val(e1)
	(setf e1 (gethash (bind-value e1) bindings)))    ; use entry of value(e1)
    e1))


;;;;;;;;;;;;;;;;;;;;;;;;                                                  
;; Update binding to add new constraint.
;; Add key(e1) to not(e2) and also to not of any syn(e2)
;; Do the same to key(e2).
(defun add-constraint (e1 e2 bindings &optional (stream *standard-output*))
  (cond                          
    ((not (and (bind-value e1) (bind-value e2)))     ; if both are constants
     (let ((key1 (bind-key e1))                      ; no constraint needed  
	   (key2 (bind-key e2))
	   (id1 nil)
	   (id2 nil)
	   (syn-id nil))
       (if (> *trace* 3)
	   (format stream "~%         Add-constraint ~a  ~a" key1 key2))
       (if (variable? key1)
	   (setf id1 (variable-id key1))
	   (setf id1 key1))
       (setf (gethash id1 (bind-not e2)) key1)       ; add key1 to not(e2)
       (dolist (syn2 (bind-syn e2))
	 (if (variable? syn2)
	     (setf syn-id (variable-id syn2))
	     (setf syn-id syn2))                     ; for each syn2 of syn(e2)
	 (let ((syn-e (gethash syn-id bindings)))       
	   (setf (gethash id1 (bind-not syn-e)) key1)) ; add key1 to not(syn2)
	 (setf (gethash syn-id (bind-not e1)) syn2))   ; add syn2 to not(e1)
       
       (if (variable? key2)
	   (setf id2 (variable-id key2))
	   (setf id2 key2))
       (setf (gethash id2 (bind-not e1)) key2)        ; add key2 to not(e1)
       (dolist (syn1 (bind-syn e1))
	 (if (variable? syn1)
	     (setf syn-id (variable-id syn1))
	     (setf syn-id syn1))                      ; for each syn1 of syn(e1)
	 (let ((syn-e (gethash syn-id bindings)))       
	   (setf (gethash id2 (bind-not syn-e)) key2))  ; add key2 to not(syn1)
	 (setf (gethash syn-id (bind-not e2)) syn1))))) ; add syn1 to not(e2)
  bindings)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Update entries of binding to add new binding
;; Add all not's from e1 to not of e2, and vice versa
;; If e1 or e2 has value, set other to that value
;; If both are unbound, add e2 to each syn of e1 and vice versa
(defun add-bind2 (e1 e2 bindings &optional (stream *standard-output*))
  (if (> *trace* 3)
   (format stream "~%         Add-bind2 ~a ~a" (bind-key e1) (bind-key e2)))
  (let ((not1 (list-hash (bind-not e1)))
	(not2 (list-hash (bind-not e2)))
	(id nil))
    (dolist (n2 not2)
      (if (variable? n2)
	  (setf id (variable-id n2))
	  (setf id n2))
      (setf (gethash id (bind-not e1)) n2)  ; add each not(e2) to not(e1)
      (dolist (existing-syn1 (bind-syn e1))   
	(let ((e-syn (gethash (if (variable? existing-syn1)       
                                  (variable-id existing-syn1)
                                  existing-syn1)
                              bindings)))     ; add each not(e2) 
	  (setf (gethash id (bind-not e-syn)) n2)))) ; to each syn(e1)
    (dolist (n1 not1)
      (if (variable? n1)
	  (setf id (variable-id n1))
	  (setf id n1))
      (setf (gethash id (bind-not e2)) n1) ; add each not(e1) to not(e2)
      (dolist (existing-syn2 (bind-syn e2))   
	(let ((e-syn (gethash (if (variable? existing-syn2)       
                                  (variable-id existing-syn2)
                                  existing-syn2)
                              bindings)))     ; add each not(e1) 
	  (setf (gethash id (bind-not e-syn)) n1)))) ; to each syn(e2)
    (add-val e1 e2 bindings)   ; add val(e1) to e2 and its syn if nec.
    (add-syn e1 e2 bindings)   ; add syn(e1) to e2 and its syn, vice versa
    ))

;;;;;;;;;;;;;;;;;;;;;;;;                                                  
;;  When one entry has a value (i.e. we know which constant it is bound to)
;;  The other entry should be set to have the same value.  Also the list of
;;  synomyms and constraints are deleted, since the same information is 
;;  in the entry for the associated constant value.
(defun add-val (e1 e2 bindings)
  (cond 
    ((bind-value e1)                        ; if e1 has value
     (setf (bind-value e2) (bind-value e1)) ;   set val(e2) to val(e1)
     (clrhash (bind-not e2))                ;   clear not(e2)
     (dolist (existing-syn2 (bind-syn e2))
       (let ((e (gethash (if (variable? existing-syn2)
                             (variable-id existing-syn2)
                             existing-syn2)
                         bindings)))
	 (setf (bind-value e) (bind-value e1))   ; set val of each syn(e2)
	 (clrhash (bind-not e))                  ; clear its not hash tbl
	 (setf (bind-syn e) nil)))               ; and clear its syn
     (setf (bind-syn e2) nil))              ; clear syn(e2) when e1 has value 

    ((bind-value e2)                        ; if e2 has value
     (setf (bind-value e1) (bind-value e2)) ;   set val(e1) to val(e2)
     (clrhash (bind-not e1))                ;   clear not(e1)
     (dolist (existing-syn1 (bind-syn e1))  
       (let ((e (gethash (if (variable? existing-syn1)
                             (variable-id existing-syn1)
                             existing-syn1)
                         bindings)))     
	 (setf (bind-value e) (bind-value e2))   ; set val of any syn(e1)
	 (clrhash (bind-not e))                  ; clear its not hash tbl
	 (setf (bind-syn e) nil)))               ; and clear its syn
     (setf (bind-syn e1) nil))))            ; clear syn(e1) when e2 has value 
  

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; If both are unbound, add e2 to each syn of e1 and vice versa
(defun add-syn (e1 e2 bindings)
  (let ((key1 (bind-key e1))
	(key2 (bind-key e2)))
    (if (bind-value e1)                        ; if e1 has value delete its syn
	(setf (bind-syn e1) nil)
	(dolist (existing-syn1 (bind-syn e1))  ; else for each syn(e1)
	  (let ((e (gethash (if (variable? existing-syn1) ; add syn(e2) & key2 
                                (variable-id existing-syn1)
                                existing-syn1)
                            bindings)))
	    (if (bind-syn e2)                  
		(setf (bind-syn e) (append (bind-syn e2) (bind-syn e))))
	    (setf (bind-syn e) (cons key2 (bind-syn e))))))
    (if (bind-value e2)                        ; if e2 has value delete its syn
	(setf (bind-syn e2) nil)               
	(dolist (existing-syn2 (bind-syn e2))  ; else for each syn(e2)
          (let ((e (gethash (if (variable? existing-syn2) ; add syn(e1) & key1
                                (variable-id existing-syn2)
                                existing-syn2)
                            bindings)))
	    (if (bind-syn e1)
		(setf (bind-syn e) (append (bind-syn e1) (bind-syn e))))
	    (setf (bind-syn e) (cons key1 (bind-syn e))))))
    (cond ((null (bind-value e1))
	   (if (bind-syn e2)                  ; add syn(e2) if any to syn(e1)
	       (setf (bind-syn e1) (append (bind-syn e2) (bind-syn e1))))
	   (setf (bind-syn e2) (bind-syn e1)) ; set syn(e2) to syn(e1)
	   (setf (bind-syn e1) 
		 (cons key2 (bind-syn e1))))) ; add key2 to syn(e1)
    (if (null (bind-value e2))
	(setf (bind-syn e2)
	      (cons key1 (bind-syn e2))))     ; add key1 to syn(e2)
    ))

;;;;;;;;;;;;;;;;;;;;;;;;                                                  
;;; Makes a copy of the Bindings hash table with the same entries as
;;; that of the current plan.  A new hash table is also created for
;;; the "not" constraints for each entry in bindings.
;;;
;;; For bizarre reasons this has been moved to snlp.lisp
(defun copy-bindings (old-bind)
   (let ((new-bind (make-hash-table :test #'equal)))
    (dolist (old-e (list-hash old-bind) new-bind)  ; for each entry of oldbind
      (let ((key (bind-key old-e))                 ; create entry in new-bind
	    (id  nil)
	    (new-e nil))
	(if (variable? key)
	    (setf id (variable-id key))
	    (setf id key))
	(setf new-e
	      (setf (gethash id new-bind)
		    (make-bind
		     :key (bind-key old-e)
		     :value (bind-value old-e)
		     :syn (bind-syn old-e)
		     :not (make-hash-table :test #'equal))))
	(dolist (not-key (list-hash (bind-not old-e)) new-bind)
	  (let ((not-id not-key))                ; for each not of old entry
	    (if (variable? not-key)              ; add not to new entry
		(setf not-id (variable-id not-key)))
	    (setf (gethash not-id (bind-not new-e)) not-key)))))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Make a list from hash table
(defun list-hash (tbl)
  (let ((not-list nil))
      (maphash #'(lambda (id val)
		   (declare (ignore id))
                   (push val not-list))
               tbl)
      not-list))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; mapcar-la acts like mapcar except that it expects the function to
;;; take two arguments and it makes a second list (same length as l)
;;; by repeating the atom
(defun mapcar-la (funct l atom)
  (mapcar funct l (make-sequence 'list     ; was '(list)
                                 (length l)
                                 :initial-element atom)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Add new high step number to each variable id for every term in 
;;; step from template.
(defun instantiate-step (step num)
  (make-snlp-step
   :id num
   :action (instantiate-term (snlp-step-action step) num)
   :precond (mapcar-la #'instantiate-term (snlp-step-precond step) num)
   :add (mapcar-la #'instantiate-term (snlp-step-add step) num)
   :dele (mapcar-la #'instantiate-term (snlp-step-dele step) num)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Adds the integer high-step to the id of any variables
;;; Returns constants unchanged
#+not-PAIL
(defun instantiate-term (term num)
  (cond
    ((variable? term)
     (make-variable
      :id (list (variable-id term) num)))
    ((listp term)
     (mapcar-la #'instantiate-term term num))
    (t
     term)))


;;;;;;;;;;;;;;;;;;;;;;;;
;;; Add new high step number to each variable id for every term in
;;; bindings from template.
(defun instantiate-bind (bind num)
  (mapcar-la #'instantiate-term bind num))

