;;; -*- Mode: LISP; Package: RPG; Syntax: Common-lisp;                   -*-
;;;
;;; ************************************************************************
;;;
;;; PORTABLE AI LAB - UNI ZH
;;;
;;; ************************************************************************
;;;
;;; Filename:   rpg-table-ed
;;; Short Desc: tiny editor for tables
;;; Version:    2.0
;;; Status:     Experimental
;;; Last Mod:   6.9.91 - TW
;;; Author:     Thomas Wehrle, adpated by ThE
;;;
;;; Copyright (c) 1992 Istituto Dalle Molle (IDSIA), University of
;;; Zurich, Swiss Federal Institute of Technology Lausanne.
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all
;;; copies and supporting documentation.
;;;
;;; IDSIA provides this software "as is" without express or implied
;;; warranty.  
;;;

;;;
;;; --------------------------------------------------------------------------
;;;
;;; Usage:
;;;
;;; (make-editor-fcn <fcn-name> [:obj-var <var>]
;;;                             [:default-menu-entries <bool>]
;;;                             [:menu <menu-items-or-nil>])
;;; where 
;;;  <var> is a symbol that you can use in the menu-items for the table object
;;;  <bool> is t or nil depending on whether you want the default menu enties or not
;;;  <menu-items-or-nil> is nil or a list of lists with 3 elements:
;;;      1 = menu item's name
;;;      2 = function to be called (you can use <var> here to refer to the table object
;;;          and you can use here a function "refresh" to update the display
;;;      3 = help string
;;;
;;;  eg. (make-editor-fcn id3-table-editor
;;;                       :default-menu-entries t
;;;                       :menu nil)
;;;  or (make-editor-fcn my-ed
;;;                       :obj-var tab
;;;                       :default-menu-entries nil
;;;                       :menu `(("My function"
;;;                                ,(function (lambda ()
;;;   				      (do-something-with tab)
;;;				      (refresh)))))
;;;
;;; This gives you 2 functions:
;;;
;;; (<fcn-name> <table-obj> [:left <nil-or-int>]
;;;			    [:bottom <nil-or-int>]
;;;		            [:title <string>]
;;;			    [:offset-x <int>]
;;;			    [:close-all <bool>])
;;; where:
;;; <table-obj>    is your table-object, if it is not a table one is created
;;; <fcn-name>     is you function to invoke you editor
;;; <nil-or-int>   coordinates of left end of the display or nil for default
;;; <nil-or-int>   coordinates of bottom of the display or nil for default
;;; <string>       a title for the display
;;; <int>          set that to 0 to avoid the numbering of the examples
;;; <bool>         t means that all other editor displays invoked with your
;;;                function! are closed so that this display will be the only one.
;;;
;;; ("close-"<fcn-name> <table-obj> [:close-all <bool>])
;;;
;;; where:
;;; <table-obj>    is a table-object that is edited
;;; <bool>         close all other editor window or just the one for <table-obj> 
;;;



;;; ==========================================================================
;;; PACKAGE DECLARATIONS
;;; ==========================================================================


(in-package :rpg)

(export '(make-editor-fcn refresh))


(defmacro make-editor-fcn (name &key (obj-var 'tab)
				     (default-menu-entries t)
				     (cell-font nil)
				     (menu nil))
  (let ((display-collector-name (intern (format nil "~a-disps" name)))
	(close-fcn-name (intern (format nil "close-~a" name))))
    
    `(progn
       (defparameter ,display-collector-name (make-hash-table))

       (defun ,close-fcn-name (tab &key
				   (close-all nil))
	 (if close-all
	     (progn
	       (maphash #'(lambda (key value)
			    (declare (ignore key))
			    (close-display value))
			,display-collector-name)
	       (clrhash ,display-collector-name))
	   (progn
	     (let ((d (gethash tab ,display-collector-name)))
	       (when d (close-display d)))
	     (remhash tab ,display-collector-name))))

       (defmethod ,name (symbol
			 &key (left nil)
			      (bottom nil)
			      (title "Table Editor")
			      (offset-x 100)
			      (close-all nil))
	 (declare (ignore symbol))
	 (let ((x (read-from-string
		   (ask "How many attributes ? "
			:condition 
			(function (lambda (string)
				    (let ((x (read-from-string string nil)))
				      (and (numberp x) (plusp x) (integerp x)))))
			:error-message "Please enter a positive number")))
	       (y (read-from-string
		   (ask "How many examples ? "
			:condition 
			(function (lambda (string)
				    (let ((x (read-from-string string nil)))
				      (and (numberp x) (plusp x) (integerp x)))))
			:error-message "Please enter a positive number"))))
	   (,name (make-table :attributes 
			      (let ((l nil))
				(dotimes (i x (nreverse l))
				  (setf l (cons (read-from-string
						 (format nil "a~a" i))
						l))))
			      :rows 
			      (let ((ll nil))
				(dotimes (j y (nreverse ll))
				  (setf ll 
				    (cons
				     (let ((l nil))
				       (dotimes (i x (nreverse l))
					 (setf l (cons (read-from-string
							(format nil "v~a.~a" i j))
						       l))))
				     ll)))))
		  :left left
		  :bottom bottom
		  :title title
		  :offset-x offset-x
		  :close-all close-all)))

       (defmethod ,name ((,obj-var table)
			 &key
			 (left nil)
			 (bottom nil)
			 (title "Table Editor")
			 (offset-x 100)
			 (close-all nil))
  
	 (let ((*value-button-border-p* t)
	       (x (length (attributes ,obj-var)))
	       (y (number-of-rows ,obj-var))
	       (gab 6)
	       (b-width  (* 15 (font-character-width ,cell-font)))
	       (b-height (+ 6 (font-character-height ,cell-font)))
	       (offset-y (* 2 (+ 6 (font-character-height ,cell-font))))
	       (disp nil))
	   (defun refresh ()
	     (,name ,obj-var
		    :left (left disp)
		    :bottom (bottom disp)
		    :title title
		    :offset-x offset-x
		    :close-all close-all))
	   (let* ((default-items (if ,default-menu-entries
				     (list
				      (list
				       "Add Attribute" (function (lambda ()
								   (let ((attr (read-from-string
										(ask "Please enter the attribute.~%=> "))))
								     (if (member attr (attributes ,obj-var) :test #'equal)
									 (display-error "Attribute already exists!")
								       (progn
									 (setf (attributes ,obj-var)
									   (cons attr
										 (attributes ,obj-var)))
									 (setf (rows ,obj-var)
									   (let ((nr 0))
									     (mapcar (function
										      (lambda (row)
											(incf nr)
											(cons (read-from-string
											       (ask (format nil "Enter value for example ~a.~%=> " nr)))
											      row)))
										     (rows ,obj-var))))
									 (refresh))))))
				       "Add a new attribute and it's values to the table")
				      (list
				       "Delete Attribute" (function (lambda ()
								      (let* ((attr (accept-items (make-instance 'menu
												   :items
												   (mapcar (function (lambda (attr)
														       (list (all-symbol-names attr)
															     (write-to-string attr))))
													   (attributes ,obj-var)))))
									     (pos (search
										   (list attr)
										   (attributes ,obj-var)
										   :test #'(lambda (y x ) (equal x (write-to-string y)))))
									     )
									(progn
									  (setf (attributes ,obj-var)
									    (remove-nth pos (attributes ,obj-var)))
									  (setf (rows ,obj-var)
									    (mapcar (function (lambda (row)
												(remove-nth pos row)))
										    (rows ,obj-var)))
									  (refresh)
									  ))))
				       #| (function (lambda ()
								      (let* ((attr (read-from-string
										    (ask "Please enter the attribute.~%=> ")))
									     (pos (search
										   (list attr)
										   (attributes ,obj-var)
										   :test #'equal)))
									(if pos
									    (progn
									      (setf (attributes ,obj-var)
										(remove-nth pos (attributes ,obj-var)))
									      (setf (rows ,obj-var)
										(mapcar (function (lambda (row)
												    (remove-nth pos row)))
											(rows ,obj-var)))
									      (refresh))
									  (display-error "Attribute does not exist!"))))) |#
				       "Delete an attribute and it's values")
				      (list
				       "Add Example" (function (lambda ()
								 (setf (rows ,obj-var)
								   (cons
								    (mapcar (function (lambda (attr)
											(read-from-string
											 (ask (format nil "Enter value for attribute ~a.~%=> " attr)))))
									    (attributes ,obj-var))
								    (rows ,obj-var)))
								 (refresh)))
				       "Add a new example (a row) to the table")
				      (list "Delete Example" (function (lambda ()
									 (let ((pos nil))
									   (setf pos
									     (read-from-string
									      (ask "Please enter the number of the example.~%=> "
										   :condition 
										   (function (lambda (string)
											       (let ((x (read-from-string string nil)))
												 (and (numberp x) (integerp x) (>= x 0)))))
										   :error-message "The example has to be identified with a number.")))
									   (if (and (plusp pos) (<= pos (number-of-rows ,obj-var)))
									       (progn
										 (setf (rows ,obj-var)
										   (remove-nth (1- pos) (rows ,obj-var)))
										 (refresh))
									     (display-error "Example does not exist!")))))
					    "Delete an example (a row)"))
				   nil))
		  (user-items ,menu)
		  (items (append default-items user-items))
		  (menu (make-instance 'menu :items (if items items '(("No menu" (function (lambda ()) "bla"))))))
		  (help-button (make-instance 'push-button :label "Help" :width 100))
		  (menu-button (make-instance 'pop-up-button :label "Menu" 
					      :height *default-push-button-height*
					      :width 100))
		  (refresh-button (make-instance 'push-button :label "Refresh" :width 100))
		  (exit-button (make-instance 'push-button :label "Exit" :width 100))
		  (names (let ((l nil))
			   (dotimes (i y (reverse l))
			     (setf l (cons (1+ i) l))))))

	     (,close-fcn-name ,obj-var :close-all close-all)

	     (setf disp (make-instance 'display :title title
				      :width (max
					      (+ offset-x 1 (* x b-width))
					      (+ offset-x 376))
				      :height (+ offset-y 1 (* (1+ y) b-height) gab 1)
				      :left (if left left 10)
				      :bottom (if bottom bottom
						(- 574 (+ offset-y 1 (* (1+ y) b-height) gab 1)))))

	     (setf (gethash ,obj-var ,display-collector-name) disp)

	     (dotimes (i x)
	       (let ((button (make-instance 'value-button
			       :numeric nil
			       :font ,cell-font
			       :width b-width
			       :value (all-symbol-names (nth i (attributes ,obj-var))))))
		 (set-button button disp 
			     :left (+ offset-x (* i b-width))
			     :bottom (- (height disp)
					offset-y 1 b-height 1)
			     :action 
			     `(lambda ()
				(let ((new-val (read-from-string (button-value ,button) nil)))
				  (if (and (member new-val (attributes ,,obj-var) :test #'equal)
					   (not (equal new-val (nth ,i (attributes ,,obj-var)))))
				      (progn
					(display-error "Attribute already exists!")
					(setf (button-value ,button) (nth ,i (attributes ,,obj-var))))
				    (setf (nth ,i (attributes ,,obj-var)) 
				      new-val)))))))
	     (dotimes (j y)
	       (dotimes (i x)
		 (let ((button (make-instance 'value-button
				 :numeric nil 
				 :width b-width
				 :font ,cell-font
				 :value (all-symbol-names (get-xy-value i j ,obj-var)))))
		   (set-button button disp 
			       :left (+ offset-x (* i b-width))
			       :bottom (- (height disp)
					  offset-y 1
					  (* (1+ j) b-height)
					  b-height gab 1)
			       :action 
			       `(lambda ()
				  (set-xy-value ,i ,j ,,obj-var 
						(read-from-string (button-value ,button) nil)))))))

	     (when (plusp offset-x)
	       (write-display disp "Examples" 8 (- (height disp)
						   offset-y
						   b-height -5))

	       (dotimes (j y)
		 (write-display disp
				(format nil "~6@a" (nth j names))
				10 (- (height disp) offset-y  1 (* (+ j 2) b-height))))

	       (draw-line disp 
			  (- offset-x 3)
			  0
			  (- offset-x 3)
			  (- (height disp) offset-y)))

	     (draw-line disp 
			0
			(- (height disp)
			   offset-y -1)
			(width disp)
			(- (height disp)
			   offset-y -1))
	     (draw-line disp 
			0
			(- (height disp)
			   offset-y 1
			   (+ b-height (/ gab 2)) 1)
			(width disp)
			(- (height disp)
			   offset-y 1
			   (+ b-height (/ gab 2)) 1))

	     (set-button help-button disp 
			 :left 15
			 :bottom (- (height disp)
				    (round (/ (- offset-y b-height) 2))
				    b-height)
			 :action #'(lambda nil 
				     (make-instance 'help-stream
				       :filename (add-path
						  "table-ed.asc"
						  (add-subdir *pail-path* "pail-lib"))
				       :from-button help-button)))
	     (setf (menu menu-button) menu)
	     (set-button menu-button disp
			 :left 130
			 :bottom (- (height disp)
				    (round (/ (- offset-y b-height) 2))
				    b-height)
			 :action #'(lambda nil))
	     (set-button refresh-button disp 
			 :left 245
			 :bottom (- (height disp)
				    (round (/ (- offset-y b-height) 2))
				    b-height)
			 :action #'refresh)
	     (set-button exit-button disp 
			 :left 360
			 :bottom (- (height disp)
				    (round (/ (- offset-y b-height) 2))
				    b-height)
			 :action #'(lambda ()
				     (close-display disp)
				     (remhash ,obj-var ,display-collector-name)))

	     ,obj-var))))))


;;; ========================================================================
;;; END OF FILE
;;; ========================================================================
