/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/*********************************************************************
 * Graphics Interface
 * ------------------
 * This is the main entry point to the Graphics Interface package
 * for the Rochester Connectionist Simulator.  
 * It is called automatically by the simulator if the 
 * gistart.o has been linked into the object code.
 * The routines here set up the GI tool and do all beginning
 * initialization of global variables.
 *
 * Kenton Lynne
 ********************************************************************/

#include "macros.h"
#include "externs.h"
#include "panel.h"

/* default tool window dimensions -- may be changed by -W flags */
#define SCREEN_TOP    100
#define SCREEN_LEFT   560
#define SCREEN_WIDTH  585
#define SCREEN_HEIGHT 725

/**************************************/
/* define the image for the tool icon */
/**************************************/
unsigned short  icon_image[256] = {
#include "icon/simtool.icon"
};
DEFINE_ICON_FROM_IMAGE(icon, icon_image);

/****************************************/
/* define the image for the MARKER icon */
/****************************************/
static short  mk_image[16] = {
#include "icon/marker"
};
mpr_static(mk, 16, 16, 1, mk_image);
static struct pixrect *marker_image={&mk};
static struct gricons marker_icon = 
    {NULL, NULL, 1, UNITSIZE, UNITSIZE, &marker_image};

/* forward referenced procedure */
static sigwinched();

/*************************************************************************/
gi_main(argc,argv)
     int argc;
     char *argv[];
{
/* called by the simulator in order to bring up the
 * GI interface tool.
 * It is passed the original command line arguments
 * in order to pull out the -W flags for explicit tool window
 * positioning, size, label and whatever else Sun comes up with
 * for these tool flags
 */

  int *argc_ptr;
  char **tool_args;
  char ***tool_args_ptr;

  /* print out banner on simulator panel */
  printf("\n... building Graphic Interface tool -- please wait ...\n");
     
  /* get the command line arguments */
  tool_args_ptr = &tool_args;
  tool_args = NULL;
  argc_ptr = &argc;

  /* parse command line args for -W flags */
  if ((tool_parse_all(argc_ptr, argv, tool_args_ptr, argv[0])) == -1)
  {
     tool_usage(argv[0]);
  }

  /* make the tool structure passing results of -W flags */
  if ((gi_tool = tool_make(
     WIN_LABEL,  argv[0], 
     WIN_TOP,        SCREEN_TOP,
     WIN_LEFT,       SCREEN_LEFT,
     WIN_WIDTH,      SCREEN_WIDTH,
     WIN_HEIGHT,     SCREEN_HEIGHT,
     WIN_ICON,       &icon,
     WIN_ATTR_LIST,  tool_args,
                        0)) == NULL)
  {      
    fputs("Can't make tool\n", stderr);
    exit(1);
  }

  /* set up the tool panels */
  /* NOTE: panels MUST be built in this order */
  gi_make_info_panel();
  gi_make_control_panel();
  gi_make_display_panel();
  gi_make_msg_panel();
  gi_make_cmd_panel();
  gi_make_mode_panel();

  /* read to accept signals from the tool system */
  signal(SIGWINCH, sigwinched);

  /* initialize needed vars before tool is installed */
  pre_init();

  /* install the tool int window hierarchy */
  tool_install(gi_tool);

  /* initialize needed vars after tool is installed */
  post_init();

  /* give control to the tool select manager */
/*  tool_select(gi_tool, 0); */
}

/*************************************************************************/
gi_end_session()

{
  tool_destroy(gi_tool);
   
  /* close the system font file */
  pw_pfsysclose();

  /* close the log file if still open */
  if (gi_log_fp) fclose(gi_log_fp);
}

/*************************************************************************/
gi_get_control()

{
  tool_select(gi_tool, 0);
}

/*************************************************************************/
gi_back_to_caller()

{
  tool_done(gi_tool);
}

/*************************************************************************/
tool_layoutsubwindows(th)
    struct tool *th;
{
/* lays out the panels in the tool window according to constraints
 * on how big each panel must be.
 * This routine is called by the the window manager whenever
 * the size of the tool is changed as well as when the tool
 * is first displayed and 
 * replaces the Suntool default library routine for
 * laying out tool subwindows
 */
    int i;
    int width, height, border, top, sep;
    struct toolsw *ts;

    /* save the previous values for comparison purposes */
    gi_old_width = gi_rects[GRAPH].r_width;
    gi_old_height = gi_rects[GRAPH].r_height;
 
    /* get the current window parameters */
    width = win_getwidth(th->tl_windowfd);
    height = win_getheight(th->tl_windowfd);
    border = tool_borderwidth(th);
    top = tool_stripeheight(th) + border;
    sep = tool_subwindowspacing(th);

    /* calculate info panel parameters */
    gi_rects[INFO].r_left = border;
    gi_rects[INFO].r_top = top;
    gi_rects[INFO].r_width = width - 2*border;
    gi_rects[INFO].r_height = INFO_HEIGHT;

    /* calculate mode panel parameters */
    gi_rects[MODE].r_left = border;
    gi_rects[MODE].r_top = top + INFO_HEIGHT + sep;
    gi_rects[MODE].r_width = gi_rects[INFO].r_width;
    gi_rects[MODE].r_height = MODE_HEIGHT;
 
    /* calculate display (graphics) panel parameters */
    gi_rects[GRAPH].r_left = border;
    gi_rects[GRAPH].r_top = gi_rects[MODE].r_top + MODE_HEIGHT + sep;
    gi_rects[GRAPH].r_width = width - (2*border + CNTL_WIDTH + sep);
    gi_rects[GRAPH].r_height = height
       - (INFO_HEIGHT + MSG_HEIGHT + CMD_HEIGHT + MODE_HEIGHT
          + 4*sep + border + top);
 
    /* calculate control panel parameters */
    gi_rects[CNTL].r_left = border + gi_rects[GRAPH].r_width + sep;
    gi_rects[CNTL].r_top = gi_rects[GRAPH].r_top;
    gi_rects[CNTL].r_width = CNTL_WIDTH;
    gi_rects[CNTL].r_height = height
          - (INFO_HEIGHT + MODE_HEIGHT + 2*sep + top + border);
 
    /* calculate message panel parameters */
    gi_rects[MSG].r_left = gi_rects[GRAPH].r_left;
    gi_rects[MSG].r_top 
           = gi_rects[GRAPH].r_top + gi_rects[GRAPH].r_height + sep;
    gi_rects[MSG].r_width = gi_rects[GRAPH].r_width;
    gi_rects[MSG].r_height = MSG_HEIGHT;
 
    /* calculate command panel parameters */
    gi_rects[CMD].r_left = gi_rects[GRAPH].r_left;
    gi_rects[CMD].r_top = gi_rects[MSG].r_top + gi_rects[MSG].r_height + sep;
    gi_rects[CMD].r_width = gi_rects[GRAPH].r_width;
    gi_rects[CMD].r_height = CMD_HEIGHT;
 
    /* set up the size and position parameters for all panels */
    for(ts = th->tl_sw,i = 0;i < MAX_PANELS && ts != NULL;i++,ts = ts->ts_next)
        win_setrect(ts->ts_windowfd,&gi_rects[i]);
 
    /* update the extent values */
    gi_extent_x = gi_origin_x + gi_rects[GRAPH].r_width;
    gi_extent_y = gi_origin_y + gi_rects[GRAPH].r_height;
 
    /* calculate where to start the "LOG:" prompt */
    gi_log_start_x = width - (CNTL_WIDTH + 2*sep) + 5;
}

/*************************************************************************/
static pre_init()
{
/* takes care of initialization of globals variables
 * needed before the tool is installed 
 */
  int i;

  /* set up the text object headers */
  gi_text_head.next = gi_text_head.prev = &gi_text_head;
  gi_text_head.flag = 0;

  /* set up the vertex object headers */
  gi_draw_head.next = gi_draw_head.prev = &gi_draw_head;
  gi_draw_head.flag = 0;

  /* initialize the off_display header */
  gi_off_display.next = gi_off_display.prev = &gi_off_display;
  gi_off_display.flag = 0;

  /* initialize the marker (ie. display header) for graphics panel */
  gi_marker.next = gi_marker.prev = &gi_marker;
  gi_marker.x_pos = 5;
  gi_marker.y_pos = 5;
  gi_marker.u_index = MARKER;
  gi_marker.gptr = &marker_icon;
  gi_marker.val = 0;
  gi_marker.flag = VALUE_OK + NOSHOW;
  gi_marker.icon_index = 0;

  /* initialize the info items array */
  for (i=0; i<MAX_INFO_COLS; i++) 
  {
    gi_info_item[i].pos = INFO_STARTX + (2*i+1)*INFO_CSPACE/2;
    gi_info_item[i].ui = -1;
  }
}

/*************************************************************************/
static post_init()
{
/* initializes global variables and structures that are best
 * done after the tool is installed
 */

    /* set up the mouse input events for inside the graphics window */

    /* custom mode: button events + negative move and window exit events */
    input_imnull(&gi_mask[MODE_CUSTOM]);
    gi_mask[MODE_CUSTOM].im_flags = IM_NEGEVENT;
    win_setinputcodebit(&gi_mask[MODE_CUSTOM], MS_LEFT);
    win_setinputcodebit(&gi_mask[MODE_CUSTOM], MS_MIDDLE);
    win_setinputcodebit(&gi_mask[MODE_CUSTOM], MS_RIGHT);
    win_setinputcodebit(&gi_mask[MODE_CUSTOM], LOC_WINEXIT);
    win_setinputcodebit(&gi_mask[MODE_CUSTOM], LOC_MOVEWHILEBUTDOWN);
    
    /* normal mode: same as for custom mode */
    gi_mask[MODE_MAIN] = gi_mask[MODE_CUSTOM];

    /* link mode: same as for main mode */
    gi_mask[MODE_LINK] = gi_mask[MODE_MAIN];
    
    /* text mode: same as main + ascii events */
    gi_mask[MODE_TEXT] = gi_mask[MODE_MAIN];
    gi_mask[MODE_TEXT].im_flags = IM_ASCII + IM_NEGEVENT;

    /* draw mode: same as main + positive move events */
    gi_mask[MODE_DRAW] = gi_mask[MODE_MAIN];
    win_setinputcodebit(&gi_mask[MODE_DRAW], LOC_MOVE);
 
    win_setinputmask(gi_gfx->gfx_windowfd, &gi_mask[MODE_MAIN],
        (struct inputmask *)NULL, WIN_NULLLINK);

    /* change the cursor on the graphics panel to a crosshair */
    win_setcursor(gi_gfx->gfx_windowfd, gi_cursor);
 
    /* open the system font file and set up default font as current font */
    gi_dft_font = pw_pfsysopen();
    gi_cur_font = gi_dft_font;

}

/*************************************************************************/
static sigwinched()

{
   tool_sigwinch(gi_tool);
}

