/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/*********************************************************************
 * Graphics Interface
 * ------------------
 * This file contains routines for setting up the control panel
 * and procedures called from that panel for the Graphics Interface
 * for the Rochester Connectionist Simulator.
 *
 * Note: variables defined here that can be accessed globally
 *       are externed in "control_panel.h"
 *
 * Kenton Lynne
 *   last update: 02/24/87
 *                04/02/87
 * Nigel Goddard
 *   hacked for X: April, May 1989
 ********************************************************************/

#include "macros.h"
#include "externs.h"
#include "panel.h"
#include "cmd_panel.h"
#include <X11/Viewport.h>
#include <X11/Form.h>
#include <X11/Text.h>
#include <X11/AsciiText.h>
#include <X11/Box.h>
#include <X11/Xmu.h>
#include <X11/Shell.h>
#include <X11/Command.h>

#define PAR_LEN 10

#undef DEBUG

/* get the icons for for the default icon images */
#include "dft_icons.h"

/* the following arrays hold the default aspect icons */

static XImage *square[DFT_NRANGES+1],
	 *triangle[DFT_NRANGES+1],
	 *circle[DFT_NRANGES+1],
	 *pentagon[DFT_NRANGES+1],
	 *diamond[DFT_NRANGES+1],
       	 *grey[DFT_NRANGES+1];

static char *font_labels[]={"fixed","variable", "6x10", "6x12", "8x13",
			      "8x13bold", "9x15", "cursor", 0};

static Arg layout_args[] = {
  { XtNfromHoriz, (XtArgVal)NULL },
  { XtNhorizDistance, (XtArgVal)10 },
  { XtNborderWidth, (XtArgVal) 0},
};

static Arg vertical_args[] = {
  { XtNfromVert, (XtArgVal)NULL },
  { XtNvertDistance, (XtArgVal)10 },
  { XtNborderWidth, (XtArgVal) 0},
};

static char custom_info[] = "Each mouse button has a popup menu.  Click and \
release in the display window to popup a menu.  Then click on the item you \
want. Left click executes the item, middle click edits the item and right click \
deletes the item.  Moving the mouse outside the menu box will popdown the \
menu.  Items in menus may be set with the command: gi b <menu#> \"command\" \
<item>, where <menu#> ranges from 1 to 3, \"command\" may have up to 240 \
characters or be part of a longer continued command, and <item> is the name (up \
to 10 characters) that appears in the menu.  There are 10 items per menu. \
Clicking the LOG MENU ITEMS button in this window will put the current settings \
in the log file and show you examples of the \"gi b ...\" command.  You may use \
the variables $u, $x and $y in a command.  They will be replaced with the index \
of the unit at the location where the custom menu is popped up, and the x and y \
coordinates of that location.  Terminating a command with -- indicates command \
continuation (see manual).";

static Arg custom_args[] = {
  { XtNfromVert, (XtArgVal)NULL },
  { XtNvertDistance, (XtArgVal)10 },
  { XtNborderWidth, (XtArgVal) 0},
  { XtNstring, (XtArgVal) custom_info},
  { XtNwidth, (XtArgVal) 400},
  { XtNeditType, (XtArgVal)XttextRead},
  { XtNtextOptions, (XtArgVal) wordBreak},
  { XtNheight, (XtArgVal) 400},
  { XtNfont, (XtArgVal) &gi_default_font }
};

/* set up the icon structures for the default icons */
struct gricons
     gi_square   = {NULL, NULL, DFT_NRANGES, UNITSIZE, UNITSIZE, square},
     gi_circle   = {NULL, NULL, DFT_NRANGES, UNITSIZE, UNITSIZE, circle},
     gi_triangle = {NULL, NULL, DFT_NRANGES, UNITSIZE, UNITSIZE, triangle},
     gi_pentagon = {NULL, NULL, DFT_NRANGES, UNITSIZE, UNITSIZE, pentagon},
     gi_diamond  = {NULL, NULL, DFT_NRANGES, UNITSIZE, UNITSIZE, diamond},
     gi_grey     = {NULL, NULL, DFT_NRANGES, UNITSIZE, UNITSIZE, grey};

/* static procedures in this file that are forward referenced */

static but_proc(),
       font_proc();

static what_proc(),
       how_proc(),
       where_proc(),
       show_proc(),
       change_proc(),
       erase_proc(),
       ldir_proc(),
       lhow_proc(),
       dtype_proc(),
       custlog_proc(),
       go_proc(),
       reshow_proc(),
       logfile_proc(),
       echo_proc(),
       dump_proc(),
       target_proc(),
       quit_proc(),
       help_proc(),
       layout_proc(),
       font_proc(),
       close_layout_proc();

Widget gi_control_panel,
       gi_layout_panel;

button_list * gi_what_buttons,
  * gi_how_buttons,
  * gi_lhow_buttons,
  * gi_font_buttons,
  * gi_ldir_buttons;
  
/* panel items that can be globally accessed */
Widget
  gi_main_shell,  /* shells to allow easy manage/unmanage of control panel sections */
  gi_link_shell,
  gi_text_shell,
  gi_draw_shell,
  gi_custom_shell,
  gi_mitem[MAIN_ITEM_LAST+1],
  gi_litem[LINK_ITEM_LAST+1],
  gi_titem[TEXT_ITEM_LAST+1],
  gi_ditem[DRAW_ITEM_LAST+1],
  gi_citem[CUST_ITEM_LAST+1],
  gi_clock_item,
  gi_origin_item,
  gi_reshow_o_item;

/* panel items only accessed here */
static Widget
  go_steps_item,
  go_update_item,
  dump_image_item,
  echo_item;

char gi_logfile_name[128];
int gi_echo_on;

/****************************************************************/
static add_arg(string, arg)
  char *string, *arg;
{
/* given a string and and an argument
 * to add to it, this concatenates the
 * two strings putting a blank inbetween
 */
  if (arg) /* make sure arg not null */
  {
    strcat(string," ");
    strcat(string,arg);
  }
}


/****************************************************************/
static set_args(cmd, arg_buf)
   char *cmd, *arg_buf;
{
/* reads the values from the control panel
 * input areas, converts and puts them into
 * the arg_buf (which it assumes is long enough
 * to hold the maximum sized command)
 */
   char buf[MAX_CMD_LENGTH+1];
   char *what_stg;

   /* initialize the buffer to "gi" */
   strcpy(arg_buf,GI_CMD);

   /* add the 1st argument: "s", "e" or "c" */
   add_arg(arg_buf,cmd);

   /* add the WHO field as next argument */
   add_arg(arg_buf,gi_who_buf);

   /* add the HOWMANY field as next arguement */
   add_arg(arg_buf,gi_howmany_buf);

   /* if not erase command: add the appropriate WHAT arg */
   if (strcmp(cmd,ERASE_CMD_SHORT)) /* not the erase command */
   {
      switch (gi_cur_unit_what) {
        case POT: 
             what_stg = POT_STG; break;
        case OUTPUT: 
             what_stg = OUTPUT_STG; break;
        case STATE: 
             what_stg = STATE_STG; break;
        case DATA: 
             what_stg = DATA_STG; break;
        case LINKIN: 
             what_stg = LINKIN_STG; break;
        case LINKOUT: 
             what_stg = LINKOUT_STG; break;
      }
      add_arg(arg_buf,what_stg);

      /* add the range arguments */
      add_arg(arg_buf,gi_lrange_buf);
      add_arg(arg_buf,gi_hrange_buf);
                     
      /* if LINKIN or LINKOUT, add the target item (else "0") */
      if (gi_cur_unit_what==LINKIN || gi_cur_unit_what==LINKOUT)
      {
        add_arg(arg_buf,gi_target_buf);
      } else add_arg(arg_buf,"0");
      
      /* get the icon value (for default icons) or users icon file name */
      if (gi_cur_image_value <= NUM_STD_ICONS)
         (void) sprintf(buf,"%1d\0",gi_cur_image_value);
      else
#if 0 /* xgi: currently don't support user icons */
         strcpy(buf,gi_subimage_buf);
#else
         strcpy(buf,"error");
#endif

      add_arg(arg_buf,buf);
       
      /* if this is the show command, add positioning arguments */
      if (strcmp(cmd,CHANGE_CMD_SHORT)) {
          add_arg(arg_buf, gi_wherex_buf);
          add_arg(arg_buf, gi_wherey_buf);
          add_arg(arg_buf, gi_spacex_buf);
          add_arg(arg_buf, gi_spacey_buf);
          add_arg(arg_buf, gi_numcols_buf);
      }
   }
}

gi_switch_button_bufs(lrange, hrange, llrange, lhrange, who, ghowmany, spacex,
		      spacey, wherex, wherey )
     char * lrange, * hrange, * llrange, * lhrange, * who, * ghowmany,
       * spacex, * spacey, * wherex, * wherey;
				/* change all the ranges */
{
  XtTextBlock text;
  int inpos;

  text.firstPos = 0;
  text.format = FMT8BIT;

  if (lrange != NULL & strlen(lrange) <= MAX_CMD_LENGTH)
    if (gi_mode == MODE_MAIN)
      {
	text.length = strlen(lrange);
	text.ptr = lrange;
	inpos =  strlen(gi_lrange_buf);
	XtTextReplace(gi_mitem[LRANGE_ITEM], 0, inpos, &text);
      }
    else
      (void) sprintf(gi_lrange_buf, lrange);

  if (hrange != NULL & strlen(hrange) <= MAX_CMD_LENGTH)
    if (gi_mode == MODE_MAIN)
      {
	text.length = strlen(hrange);
	text.ptr = hrange;
	inpos =  strlen(gi_hrange_buf);
	XtTextReplace(gi_mitem[HRANGE_ITEM], 0, inpos, &text);
      }
    else
      (void) sprintf(gi_hrange_buf, hrange);
  if (llrange != NULL & strlen(llrange) <= MAX_CMD_LENGTH)
    if (gi_mode == MODE_LINK)
      {
	text.length = strlen(llrange);
	text.ptr = llrange;
	inpos =  strlen(gi_llrange_buf);
	XtTextReplace(gi_litem[LLRANGE_ITEM], 0, inpos, &text);
      }
    else
      (void) sprintf(gi_llrange_buf, llrange);
  if (lhrange != NULL & strlen(lhrange) <= MAX_CMD_LENGTH)
    if (gi_mode == MODE_LINK)
      {
	text.length = strlen(lhrange);
	text.ptr = lhrange;
	inpos =  strlen(gi_lhrange_buf);
	XtTextReplace(gi_litem[LHRANGE_ITEM], 0, inpos, &text);
      }
    else
      (void) sprintf(gi_lhrange_buf, lhrange);
  if (who != NULL & strlen(who) <= MAX_CMD_LENGTH)
    if (gi_mode == MODE_MAIN)
      {
	text.length = strlen(who);
	text.ptr = who;
	inpos =  strlen(gi_who_buf);
	XtTextReplace(gi_mitem[WHO_ITEM], 0, inpos, &text);
      }
    else
      (void) sprintf(gi_who_buf, who);
  if (ghowmany != NULL & strlen(ghowmany) <= MAX_CMD_LENGTH)
    if (gi_mode == MODE_MAIN)
      {
	text.length = strlen(ghowmany);
	text.ptr = ghowmany;
	inpos =  strlen(gi_howmany_buf);
	XtTextReplace(gi_mitem[HOWMANY_ITEM], 0, inpos, &text);
      }
    else
      (void) sprintf(gi_howmany_buf, ghowmany);
  if (spacex != NULL & strlen(spacex) <= MAX_CMD_LENGTH)
    if (gi_mode == MODE_MAIN)
      {
	text.length = strlen(spacex);
	text.ptr = spacex;
	inpos =  strlen(gi_spacex_buf);
	XtTextReplace(gi_mitem[SPACEX_ITEM], 0, inpos, &text);
      }
    else
      (void) sprintf(gi_spacex_buf, spacex);
  if (spacey != NULL & strlen(spacey) <= MAX_CMD_LENGTH)
    if (gi_mode == MODE_MAIN)
      {
	text.length = strlen(spacey);
	text.ptr = spacey;
	inpos =  strlen(gi_spacey_buf);
	XtTextReplace(gi_mitem[SPACEY_ITEM], 0, inpos, &text);
      }
    else
      (void) sprintf(gi_spacey_buf, spacey);
  if (wherex != NULL & strlen(wherex) <= MAX_CMD_LENGTH)
    if (gi_mode == MODE_MAIN)
      {
	text.length = strlen(wherex);
	text.ptr = wherex;
	inpos =  strlen(gi_wherex_buf);
	XtTextReplace(gi_mitem[WHEREX_ITEM], 0, inpos, &text);
      }
    else
      (void) sprintf(gi_wherex_buf, wherex);
  if (wherey != NULL & strlen(wherey) <= MAX_CMD_LENGTH)
    if (gi_mode == MODE_MAIN)
      {
	text.length = strlen(wherey);
	text.ptr = wherey;
	inpos =  strlen(gi_wherey_buf);
	XtTextReplace(gi_mitem[WHEREY_ITEM], 0, inpos, &text);
      }
    else
      (void) sprintf(gi_wherey_buf, wherey);
}

/*****************************************************************/
gi_make_control_panel()
{
/* initial definition and set up of the control panel 
 */
    Widget temp, temp2, temp3, spc, loc, cols, params;
	/* used for organizing layout */
    struct toolsw *control_panel_sw;
	/* these are all size 80, but should be changed to 
		reasonable sizes when AddTextEdit supports
		a max_size argument */
    static char *what_labels[]={
		"Potential","Output","State","Data",0},
		*how_labels[]={
		"square","circle","triangle","pentagon","diamond","grey",
#if 0 /* xgi: currently don't support custom icons */
		"custom",
#endif
		0};
    static char *lhow_labels[]={
		"same","square","circle","triangle","pentagon","diamond",0},
		*ldir_labels[]={"Link/in","Link/out",0};
    static char *dtype_labels[]={"Lines","Boxes","Bounding Box",0};
static Arg no_bdr[] =
  {
    { XtNborderWidth, (XtArgVal) 0},
  };
static Arg layout_loc[] =
  {
    { XtNx, (XtArgVal) 500 },
    { XtNy, (XtArgVal) 300 },
  };

Widget layout_box;

extern Widget mode_panel;

/* xgi: make all of the icons--need to do this here because there
   is no way of statically making images under X */
#include "dft_icons.c"

     /* make the control panel itself */
     if ((gi_control_panel = XtCreateManagedWidget("CONTROL",boxWidgetClass,
						   gi_tool, NULL, 0))==NULL)
     {
       fputs("Can't create control panel\n", stderr);
       exit(1);
     }
     /* define all the panel items */

     /* items only displayed during MAIN mode */
/* NHG this is a mess, should all be done elsewhere */
  gi_layout_panel = XtCreatePopupShell("LAYOUT",transientShellWidgetClass,
				     gi_tool, layout_loc, XtNumber(layout_loc));

  gi_main_shell = XtCreateManagedWidget("MAIN_SHELL", boxWidgetClass,
					gi_layout_panel, no_bdr, 1);
  temp = XtCreateManagedWidget("WHO_HOW_BOX", formWidgetClass, gi_main_shell,
			       no_bdr, 1);
  temp3 = XtCreateManagedWidget("WHO_ITEM", formWidgetClass,temp,
					     no_bdr, 1);
  temp2 = AddText(temp3,"WHO: ",XtJustifyRight, NULL);
  strcpy(gi_who_buf,"0");
  gi_mitem[WHO_ITEM] = AddTextEdit(temp3,"WHO_TEXT",gi_who_buf, temp2,
	      MAX_CMD_LENGTH, 40);
  vertical_args[0].value = (XtArgVal) temp3;
  temp3 = XtCreateManagedWidget("HOWMANY_ITEM",formWidgetClass,temp,
						 vertical_args, XtNumber(vertical_args));
  temp2 = AddText(temp3, "HOW MANY: ", XtJustifyRight, NULL);
  strcpy(gi_howmany_buf,"all");
  gi_mitem[HOWMANY_ITEM] = AddTextEdit(temp3,"HOWMANY_TEXT", gi_howmany_buf, temp2,
	      PAR_LEN, PAR_LEN);

     /* xgi:  I know this isn't pretty, but I don't know another way */
     /* xgi: should look like:
	       WHAT  ------WHAT_RADIO------
	             from:____   to:_____
                     target:____
     */

  gi_mitem[WHAT_ITEM] = XtCreateManagedWidget("WHAT_ITEM", formWidgetClass,
					      gi_main_shell, no_bdr, 1);
  layout_args[0].value = (XtArgVal) AddText(gi_mitem[WHAT_ITEM], "WHAT", XtJustifyRight, NULL);
  temp = XtCreateManagedWidget("WHAT_RADIO_BOX", formWidgetClass, gi_mitem[WHAT_ITEM],
				layout_args, XtNumber(layout_args));
  AddRadio(temp,"WHAT_RADIO",what_proc,what_labels, 4, 0, &gi_what_buttons);
  vertical_args[0].value = (XtArgVal) temp;
  temp = XtCreateManagedWidget("WHAT_SUB_BOX", formWidgetClass, gi_mitem[WHAT_ITEM], 
				vertical_args, XtNumber(vertical_args));
  temp2 = AddText(temp,"from:",XtJustifyRight, NULL);
  strcpy(gi_lrange_buf,LRANGE_POT);
  gi_mitem[LRANGE_ITEM] = AddTextEdit(temp,"LRANGE_TEXT",gi_lrange_buf, temp2,
				      PAR_LEN, PAR_LEN);
  temp2 = AddText(temp,"to:",XtJustifyRight, gi_mitem[LRANGE_ITEM]);
  strcpy(gi_hrange_buf,HRANGE_POT);
  gi_mitem[HRANGE_ITEM]  = AddTextEdit(temp,"HRANGE_TEXT",gi_hrange_buf,temp2,
				       PAR_LEN, PAR_LEN);
  temp2 = AddText(temp,"target:",XtJustifyRight, NULL);
  strcpy(gi_target_buf,"");
  gi_mitem[TARGET_ITEM] = AddTextEdit(temp,"TARGET_TEXT",gi_target_buf, temp2,
				      MAX_CMD_LENGTH, 20);
	      /* turn it off, for now */
  XtUnmanageChild(gi_mitem[TARGET_ITEM]);

  gi_mitem[HOW_ITEM] =  XtCreateManagedWidget("HOW_ITEM",formWidgetClass,gi_main_shell,
					      no_bdr, 1);
  layout_args[0].value = (XtArgVal) AddText(gi_mitem[HOW_ITEM], "HOW", XtJustifyRight, NULL);
  temp = XtCreateManagedWidget("HOW_RADIO_BOX", formWidgetClass, gi_mitem[HOW_ITEM],
				layout_args, XtNumber(layout_args));
  AddRadio(temp,"HOW_RADIO",how_proc,how_labels,6,0,&gi_how_buttons);
#if 0 /* xgi:  don't currently support custom icons */
     AddText(gi_mitem[HOW_ITEM],"custom icon:",XtJustifyRight, NULL);
     strcpy(gi_subimage_buf,"");
     AddTextEdit(gi_mitem[HOW_ITEM],"SUBIMAGE_TEXT",gi_subimage_buf)
#endif

  gi_mitem[WHERE_ITEM]=XtCreateManagedWidget("WHERE_ITEM",formWidgetClass,gi_main_shell,
					     no_bdr, 1);
  layout_args[0].value = (XtArgVal) AddText(gi_mitem[WHERE_ITEM],"WHERE",
					    XtJustifyRight, NULL);
  params = XtCreateManagedWidget("WHERE_PARAMS", boxWidgetClass, gi_mitem[WHERE_ITEM],
			       layout_args, XtNumber(layout_args));
  loc = XtCreateManagedWidget("WHERE", formWidgetClass, params, no_bdr, 1);
  temp = AddText(loc, "start x:", XtJustifyRight, NULL);
  strcpy(gi_wherex_buf,"5");
  gi_mitem[WHEREX_ITEM] = temp = AddTextEdit(loc,"WHEREX_TEXT", gi_wherex_buf,
					     temp, PAR_LEN, PAR_LEN);
  temp = AddText(loc,"start y:",XtJustifyRight, temp);
  strcpy(gi_wherey_buf,"5");
  gi_mitem[WHEREY_ITEM] = AddTextEdit(loc,"WHEREY_TEXT",gi_wherey_buf, temp,
				      PAR_LEN, PAR_LEN );

  spc = XtCreateManagedWidget("SPACING", formWidgetClass, params, no_bdr, 1);
  temp = AddText(spc,"space x:",XtJustifyRight, NULL);
  strcpy(gi_spacex_buf,"5");
  gi_mitem[SPACEX_ITEM] = temp = AddTextEdit(spc,"SPACEX_TEXT", gi_spacex_buf, temp,
					     PAR_LEN, PAR_LEN);
  temp = AddText(spc,"space y:",XtJustifyRight, temp);
  strcpy(gi_spacey_buf,"5");
  gi_mitem[SPACEY_ITEM] = AddTextEdit(spc,"SPACEY_TEXT",gi_spacey_buf, temp,
				      PAR_LEN, PAR_LEN);

  cols = XtCreateManagedWidget("NUMCOLS_ITEM", formWidgetClass, params,
			       no_bdr, 1);
  temp = AddText(cols,"units per row:",XtJustifyRight, NULL);
  strcpy(gi_numcols_buf,"max");
  temp = AddTextEdit(cols,"NUMCOLS_TEXT",gi_numcols_buf, temp,
		     PAR_LEN, PAR_LEN);

  temp=XtCreateManagedWidget("SCE_BOX", formWidgetClass, gi_main_shell,
			     no_bdr, 1);
  gi_mitem[SHOW_ITEM] = AddButton(temp, "SHOW", show_proc, 0, NULL);
  gi_mitem[CHANGE_ITEM] = AddButton(temp, "CHANGE", change_proc, 0,gi_mitem[SHOW_ITEM]);
  gi_mitem[ERASE_ITEM] = AddButton(temp, "ERASE", erase_proc, 0, gi_mitem[CHANGE_ITEM]);
  AddButton(temp, "CLOSE", close_layout_proc, gi_layout_panel, gi_mitem[ERASE_ITEM]);

     /* items displayed only during LINK mode */

  gi_link_shell = XtCreateManagedWidget("LINK_SHELL", boxWidgetClass,
				 gi_layout_panel, no_bdr, 1);
  temp = XtCreateManagedWidget("LTARGET_BOX",formWidgetClass,gi_link_shell,
			       no_bdr, 1);
  temp2 = AddText(temp,"TARGET",XtJustifyRight, NULL);
  strcpy(gi_ltarget_buf, ANY_SITE);
  gi_litem[LSITE_ITEM] = AddTextEdit(temp,"LSITE_TEXT",gi_ltarget_buf, temp2,
				     MAX_CMD_LENGTH, 30);

  gi_litem[LHOW_ITEM] =  XtCreateManagedWidget("LHOW_ITEM",formWidgetClass,
					      gi_link_shell, no_bdr, 1);
  layout_args[0].value = (XtArgVal) AddText(gi_litem[LHOW_ITEM], "HOW", XtJustifyRight, NULL);
  temp = XtCreateManagedWidget("LHOW_RADIO_BOX", formWidgetClass, gi_litem[LHOW_ITEM],
				layout_args, XtNumber(layout_args)); 
  AddRadio(temp,"LHOW_RADIO",lhow_proc,lhow_labels,6,0,&gi_lhow_buttons);

  vertical_args[0].value = (XtArgVal) temp;
  temp = XtCreateManagedWidget("LRANGE_BOX",formWidgetClass,
			       gi_litem[LHOW_ITEM],vertical_args,
			       XtNumber(vertical_args));
  temp2 = AddText(temp,"from:",XtJustifyRight, NULL);
  strcpy(gi_llrange_buf,LRANGE_LINKIN);
  gi_litem[LLRANGE_ITEM] = AddTextEdit(temp,"LLRANGE_TEXT",gi_llrange_buf,
				       temp2, PAR_LEN, PAR_LEN);
  temp2 = AddText(temp,"to:",XtJustifyRight, gi_litem[LLRANGE_ITEM]);
  strcpy(gi_lhrange_buf,HRANGE_LINKIN);
  gi_litem[LHRANGE_ITEM] = AddTextEdit(temp,"LHRANGE_TEXT",gi_lhrange_buf,
				       temp2, PAR_LEN, PAR_LEN);

  temp = XtCreateManagedWidget("LDIR_BOX",formWidgetClass,
			       gi_link_shell, no_bdr, 1);
  layout_args[0].value = (XtArgVal) AddText(temp, "DIRECTION",
					    XtJustifyRight, NULL);
  temp2 = XtCreateManagedWidget("LHOW_RADIO_BOX", formWidgetClass, temp,
				layout_args, XtNumber(layout_args)); 
  AddRadio(temp2,"LDIR_RADIO",ldir_proc,ldir_labels, 2,0,&gi_ldir_buttons);
  AddButton(temp, "CLOSE", close_layout_proc, gi_layout_panel, temp2);
  XtUnmanageChild(gi_link_shell);

  /* items displayed only during TEXT mode */
  gi_text_shell = XtCreateManagedWidget("TEXT_SHELL", boxWidgetClass,
				 gi_layout_panel, no_bdr, 1);
  temp = XtCreateManagedWidget("FONT_RADIO_BOX", formWidgetClass, gi_text_shell,
				no_bdr, 1);
  AddRadio(temp,"FONT_RADIO",font_proc, font_labels,4,0,&gi_font_buttons);
  AddButton(gi_text_shell, "CLOSE", close_layout_proc, gi_layout_panel, NULL);
  XtUnmanageChild(gi_text_shell);

  /* items displayed only during DRAW mode */
  gi_draw_shell = XtCreateManagedWidget("DRAW_SHELL", formWidgetClass, 
					gi_layout_panel, no_bdr, 1);
  layout_args[0].value = (XtArgVal) AddText(gi_draw_shell,"TYPE",XtJustifyRight, NULL);
  temp = XtCreateManagedWidget("DTYPE_RADIO_BOX", formWidgetClass, gi_draw_shell,
			       layout_args, XtNumber(layout_args));
  gi_ditem[DTYPE_ITEM] = AddRadio(temp,"DTYPE_RADIO",
				  dtype_proc,dtype_labels,1,0, NULL);
  AddButton(gi_draw_shell, "CLOSE", close_layout_proc, gi_layout_panel, temp);
  XtUnmanageChild(gi_draw_shell);

    /* items only displayed during CUSTOM mode */
  gi_custom_shell = XtCreateManagedWidget("CUST_SHELL", formWidgetClass,
					  gi_layout_panel, no_bdr, 1);

  temp = gi_citem[CLOG_ITEM] = AddButton(gi_custom_shell,"LOG CUSTOM MENUS",
					 custlog_proc, 0, NULL);
  AddButton(gi_custom_shell, "CLOSE", close_layout_proc, gi_layout_panel, temp);


  custom_args[0].value = (XtArgVal) temp;
  XtCreateManagedWidget("CUSTOM_BOX", asciiStringWidgetClass, gi_custom_shell,
			       custom_args, XtNumber(custom_args));
  XtUnmanageChild(gi_custom_shell);

    /* items displayed during all modes */
    temp = XtCreateManagedWidget("STATUS_BOX",formWidgetClass,gi_control_panel,
				 no_bdr, 1);

    temp2 = AddText(temp,"Clock=",XtJustifyRight, NULL);
    strcpy(gi_clock_buf,"0          ");
    gi_clock_item = AddText(temp,gi_clock_buf,XtJustifyLeft, temp2);

    temp2 = AddButton(temp,"GO",go_proc,0, gi_clock_item);
    temp2 = AddText(temp,"steps:",XtJustifyRight, temp2);
    strcpy(gi_go_steps_buf,"1");
    go_steps_item = AddTextEdit(temp,"GO_STEPS_TEXT",gi_go_steps_buf,temp2,
				 PAR_LEN, PAR_LEN);
    temp2 = AddText(temp,"update:",XtJustifyRight, go_steps_item);
    strcpy(gi_go_update_buf,"1");
    go_update_item = AddTextEdit(temp,"GO_UPDATE_TEXT",gi_go_update_buf, temp2,
				  PAR_LEN, PAR_LEN);

    temp = XtCreateManagedWidget("DRQ_BOX", formWidgetClass, gi_control_panel,
				 no_bdr, 1);
#if 0
				/* use xwd instead.  this code should disappear */
    temp2 = AddButton(temp,"DUMP",dump_proc,0, NULL);
    strcpy(gi_dump_buf,DFT_IMAGE_FILE_NAME);
    dump_image_item = AddTextEdit(temp,"DUMP_TEXT",gi_dump_buf, temp2,
				   MAX_CMD_LENGTH, 20);
#endif

    temp2 = AddButton(temp, "NEW DISPLAY", gi_make_display_panel, NULL, NULL);
    temp2 = AddButton(temp,"RESHOW DISPLAYS",reshow_proc, 0, temp2);
    temp2 = AddButton(temp, "LOGFILE", logfile_proc, 0, temp2);
    echo_item = AddButton(temp, "NO ECHO", echo_proc, 0, temp2);
    temp2 = AddButton(temp, "HELP", help_proc, 0, echo_item);
    temp2 = AddButton(temp,"QUIT",quit_proc,0, temp2);
#if 0
				/* reset left mouse click so exits ok */
    XtAddActions(actionsList, XtNumber(actionsList));
    translations = XtParseTranslationTable(override_trans_table);
    XtOverrideTranslations(temp2, translations);
#endif
}

/*****************************************************************/
static layout_proc(w, client_data, call_data)
	Widget w;
        caddr_t client_data;
        caddr_t call_data;
{
/* called when the LAYOUT button is pressed
   client_data is the popup widget
 */
  XtPopup(client_data, XtGrabNone);
}
/****************************************************************/
static close_layout_proc(w, client_data, call_data)
	Widget w;
        caddr_t client_data;
        caddr_t call_data;
{
/* called when the CLOSE button is pressed in the layout window
   client_data is the popup widget to be popped down
 */

  XtPopdown(client_data);
}

/****************************************************************/
gi_raise_control_panel()
				/* put control panel on top, flush, then close */
				/* don't ask why, it works! */
{
  XRaiseWindow(XtDisplay(gi_tool), XtWindow(XtParent(gi_tool)));
  XSync(XtDisplay(gi_tool), 0);
}

static quit_proc(w, client_data, call_data)
	Widget w;
        caddr_t client_data;
        caddr_t call_data;
{
/* called when the QUIT button is pressed
 */
  gi_raise_control_panel();
  gi_return_to_caller();
}
/****************************************************************/
static close_help_item(w, popup, call_data)
	Widget w;
        caddr_t popup;
        caddr_t call_data;

{
  XtDestroyWidget(popup);
}

static help_proc(w, client_data, call_data)
	Widget w;
        caddr_t client_data;
        caddr_t call_data;
{
  Widget temp, namepopup, logfile_dialog;
  Position x,y;

  static char filenamebuf[128];

  static Arg p[] = 
    {
      { XtNx, (XtArgVal) 300 },
      { XtNy, (XtArgVal) 300 },
    };
  static Arg q[] =
    {
      { XtNborderWidth, (XtArgVal) 0 },
      { XtNdefaultDistance, (XtArgVal) 0 },
    };
  static Arg help_item_args[] = {
		{ XtNwidth, (XtArgVal)600 },
		{ XtNfile, (XtArgVal) filenamebuf },
		{ XtNeditType, (XtArgVal)XttextRead},
		{ XtNtextOptions, (XtArgVal) (scrollVertical | wordBreak)},
		{ XtNheight, (XtArgVal)400 },
		{ XtNfont, (XtArgVal) &gi_default_font }
	      };

  static Arg h[] = {
    { XtNfromVert, (XtArgVal) NULL },
    { XtNvertDistance, (XtArgVal) 1 },
    { XtNfont, (XtArgVal) &gi_default_font },
  };

  namepopup = XtCreatePopupShell("RCS HELP",transientShellWidgetClass,
				     w, p, XtNumber(p));
  temp = XtCreateManagedWidget("help_box", formWidgetClass, namepopup, q, XtNumber(q));
  (void) sprintf(filenamebuf, "%s/xgi.help", XGIDIR);
  help_item_args[0].value = (XtArgVal) ((int) (100 * gi_font_width));
  h[0].value = (XtArgVal) XtCreateManagedWidget("help_item", asciiDiskWidgetClass,
					   temp,help_item_args,
					   XtNumber(help_item_args));
  
  temp = XtCreateManagedWidget("CLOSE", commandWidgetClass, temp, h, XtNumber(h));
  XtAddCallback(temp,XtNcallback,close_help_item,namepopup);
  XtPopup(namepopup, XtGrabNone);
}

/*****************************************************************/
static how_proc(w, client_data, call_data)
        Widget w;
        int client_data;
        caddr_t call_data;
{
/* called when the user selects an default icon
 * on the HOW prompt (main mode)
 */

  /* set up the global current value of the icon selected */
  gi_cur_image_value = client_data+1;

}


/*****************************************************************/
static lhow_proc(w, value, call_data)
        Widget w;
        int value;
        caddr_t call_data;
{
/* called when the user selects an icon on the
 * HOW prompt (link mode)
 */

  /* if we are switching from the user's "same" icons to
     one of the defaults or vice versa, set the 
     gi_new_link_parms flag so that display_grobj_link
     will calculate a new link_index for each unit
  */
   if ((!value && gi_cur_link_image)
   || (!gi_cur_link_image && value))
     gi_new_link_parms = TRUE;

  /* based on the user's selection, set up the global link icon */
  switch (value)  
  {
    case 0: gi_cur_link_image = NULL;
            break;

    case 1: gi_cur_link_image = square;
            break;
   
    case 2: gi_cur_link_image = circle;
            break;
   
    case 3: gi_cur_link_image = triangle;
            break;
   
    case 4: gi_cur_link_image = pentagon;
            break;
   
    case 5: gi_cur_link_image = diamond;
            break;
   }
   gi_cur_lhow_value = value;
   /* force a reshow of all displayed units */
   gi_reshow_flag |= RESHOW_ALL;
   gi_reshow();
}


/*****************************************************************/
static dtype_proc(w, value, call_data)
        Widget w;
	int value;
        caddr_t call_data;
{
/* called when the user selects LINES or BOXES on
 * the TYPE prompt (draw mode)
 */
 
  gi_draw_type = value;
}

/*****************************************************************/
static ldir_proc(w, value, call_data)
        Widget w;
        int value;
        caddr_t call_data;
{
/* called when the user selects LINK/IN or LINK/OUT on
 * the direction prompt (link mode)
 */
   gi_link_direction = value;
}

/****************************************************************/
static what_proc(w, value, call_data)
        Widget w;
        int value;
        caddr_t call_data;
{
/* called when the user selects the aspect to be displayed
 * on the WHAT prompt (main mode)
 */

  int show=FALSE; 
  char *lrange, *hrange;

  /* set up the global variable indicating current WHAT choice */
  switch(value)
  {
    case 0: gi_cur_unit_what = POT;
            lrange = LRANGE_POT;
            hrange = HRANGE_POT;
            break;

    case 1: gi_cur_unit_what = OUTPUT;
            lrange = LRANGE_OUT;
            hrange = HRANGE_OUT;
            break;

    case 2: gi_cur_unit_what = STATE;
            lrange = LRANGE_STATE;
            hrange = HRANGE_STATE;
            break;

    case 3: gi_cur_unit_what = DATA;
            lrange = LRANGE_DATA;
            hrange = HRANGE_DATA;
            break;

    case 4: gi_cur_unit_what = LINKIN;
            lrange = LRANGE_LINKIN;
            hrange = HRANGE_LINKIN;
            show = TRUE;
            break;

    case 5: gi_cur_unit_what = LINKOUT;
            lrange = LRANGE_LINKOUT;
            hrange = HRANGE_LINKOUT;
            show = TRUE;
            break;
  }

  /* display the target prompt if LINKIN or LINKOUT */
  if (show)
	XtManageChild(gi_mitem[TARGET_ITEM]);
  else
	XtUnmanageChild(gi_mitem[TARGET_ITEM]);


#if 0
  /* set the default low and high range values */
  XtTextBlock text;
  int inpos;
  text.firstPos = 0;
  text.format = FMT8BIT;
  text.length = strlen(lrange);
  text.ptr = lrange;
  inpos =  XtTextGetInsertionPoint(gi_mitem[LRANGE_ITEM]);
  XtTextReplace(gi_mitem[LRANGE_ITEM], 0, inpos, &text);

  strcpy(gi_hrange_buf,hrange);
  text.length = strlen(gi_hrange_buf);
  text.ptr = gi_hrange_buf;
  inpos =  XtTextGetInsertionPoint(gi_mitem[HRANGE_ITEM]);
  XtTextReplace(gi_mitem[HRANGE_ITEM], 0, inpos, &text);

  strcpy(gi_lrange_buf,lrange);
  SetValue(gi_mitem[LRANGE_ITEM],XtNstring,gi_lrange_buf);
  SetValue(gi_mitem[LRANGE_ITEM],XtNlength,strlen(gi_lrange_buf));
  SetValue(gi_mitem[LRANGE_ITEM],XtNinsertPosition,0);
  strcpy(gi_hrange_buf,hrange);
  SetValue(gi_mitem[HRANGE_ITEM],XtNstring,gi_hrange_buf);
  SetValue(gi_mitem[HRANGE_ITEM],XtNlength,strlen(gi_hrange_buf));
  SetValue(gi_mitem[HRANGE_ITEM],XtNinsertPosition,0);
#endif
}


/****************************************************************/
static where_proc(w, client_data, call_data)
        Widget w;
        caddr_t client_data;
        caddr_t call_data;
{
/* called when the user enters a value in the
 * x: or y: fields of the WHERE prompt
 */
/* xgi: currently don't check this field.
	presumably, we should get called every keystroke
	or on a carriage return, from the CheckModify
	callback of the TextEdit widget.  This doesn't
	happen now. */

   int x_coord, y_coord;
   char stringx[MAX_NUM_LENGTH+1], stringy[MAX_NUM_LENGTH+1];

   /* move the marker to reflect the new coordinates */
   strcpy(stringx,gi_wherex_buf);
   strcpy(stringy,gi_wherey_buf);
   if ((gi_check_num(stringx)==OK) && (gi_check_num(stringy)==OK))
   {
     y_coord = atoi(stringy);
     x_coord = atoi(stringx);
     gi_move_marker(x_coord,y_coord);
   }

}

/*****************************************************************/
static font_proc(w, value, call_data)
        Widget w;
        int value;
        caddr_t call_data;
{
/* called when the user selects an font in the
 * TEXT shell.
 */

  XFontStruct * tfont;
  
  if ((tfont = (XFontStruct *) gi_get_font(font_labels[value])) == NULL)
    gi_cur_font = &gi_default_font;
  else
    gi_cur_font = tfont;
}

/****************************************************************/
static target_proc(item, client_data, event)
   Widget item;
   caddr_t client_data;
   caddr_t event;
{
/* called when the user enters a value 
 * target field of in Link mode
 */
#if 0 /* xgi: currently don't support character-at-a-time checks */
  int target;

   /* do default keystroke processing */
   panel_text_notify(item,event);

   /* get target unit from panel prompt */
   target = gi_get_target((char *) panel_get_value(gi_litem[LSITE_ITEM]));

   /* check that unit index was OK */
   if (target < 0)
   {
     gi_put_error("Invalid target");
   }

   else
   {
     gi_set_new_target(NULL,target);
     gi_reshow();
   }
#endif
}


/****************************************************************/
static custlog_proc(item, client_data, event)
   Widget item;
   caddr_t client_data;
   caddr_t event;
{
/* called when the LOG DEFINITIONS button is pressed
 * and creates a "button" command that would
 * define the same setting
 */

  /* if logging isn't enabled, put out error message */
  if (!gi_log_on)
  {
    gi_put_error("Logging is not on: request ignored");
    return;
  }
  
  LogCustomXMenus();

  gi_put_message("Custom menus have been logged");
}


/****************************************************************/
static go_proc(item, client_data, event)
   Widget item;
   caddr_t client_data;
   caddr_t event;
{
/* called when the GO button is pressed
 */
   char buf[MAX_CMD_LENGTH+1];
   char *steps, *update;
   int upd_steps=1, steps_left=1;

   /* get steps and update steps args from panel */
   steps = gi_strip(gi_go_steps_buf);
   update = gi_strip(gi_go_update_buf);
   
   /* verify that values are numeric (or null)  */
   if (gi_check_num(steps)==ERROR || gi_check_num(update)==ERROR)
   {
     gi_put_error("Invalid steps or update specified");
   }
   else
   {
     /* if not null, extract numeric values for args */
     if (steps && strlen(steps))
       steps_left = atoi(steps);
     if (update && strlen(update))
       upd_steps = atoi(update);

     /* step the simulator the requested number of steps */
     gi_do_steps(steps_left,upd_steps);

     /* build and log the "go" command if logging enabled */
     if (gi_log_on)
     {
       (void) sprintf(buf,"%s %s %s %s",GI_CMD,GO_CMD_SHORT,steps,update);
       gi_log(buf);
     }
     
     gi_put_message("Go completed successfully");
   }
}
/****************************************************************/
static reshow_proc(item, client_data, event)
   Widget item;
   caddr_t client_data;
   caddr_t event;
{
/* called when the RESHOW button is pressed
 */
  gi_reshow_all_displays();
  gi_update_clock();
}

/****************************************************************/
static void switch_logfile_toggle(w, popup, call_data)
     Widget w;
     Widget popup;
     caddr_t call_data;

{
  if (gi_log_on)
    {
      fflush(gi_log_fp);
      fclose(gi_log_fp);
      gi_log_fp = NULL;
      (void) sprintf(gi_logfile_name, "");
      gi_log_on = False;
    } 
  else
    {
      char * newname;
      FILE * fp;

      newname = XtDialogGetValueString(XtParent(w));
      if ((fp = fopen(newname,"w")) == NULL)
	{
	  gi_put_error("Can't create log file");
	  goto xexit;
	}
      gi_log_fp = fp;
      (void) sprintf(gi_logfile_name, "%s", newname);
      gi_log_on = True;
    }

 xexit:  
  XtDestroyWidget(popup);
}

static void change_logfile_name(w, popup, call_data)
     Widget w;
     Widget popup;
     caddr_t call_data;

{
  char * newname;
  FILE * fp;

  if (!gi_log_on)
    {				/* if log is switched off, turn it on */
      switch_logfile_toggle(w, popup, call_data);
      return;
    }
  newname = XtDialogGetValueString(XtParent(w));
  if (strcmp(newname, gi_logfile_name))
    {
      if ((fp = fopen(newname,"w")) == NULL)
	{
	  gi_put_error("Can't create log file");
	  goto xexit;
	}
      fflush(gi_log_fp);
      fclose(gi_log_fp);
      gi_log_fp = fp;
      (void) sprintf(gi_logfile_name, "%s", newname);
    }
  else
    {
      if ((fp = fopen(newname,"w")) == NULL)
	{
	  gi_put_error("Can't create log file");
	  goto xexit;
	}
      gi_log_fp = fp;
    }

 xexit:
  XtDestroyWidget(popup);
}

static void cancel_logfile_name(w, popup, call_data)
     Widget w;
     Widget popup;
     caddr_t call_data;

{
    XtDestroyWidget(popup);
}

static logfile_proc(w, client_data, call_data)
     Widget w;
     caddr_t client_data, call_data;

{
  Widget temp, namepopup, logfile_dialog;
  Position x,y;
  static Arg p[] = 
    {
      { XtNx, (XtArgVal) 0 },
      { XtNy, (XtArgVal) 0 },
    };
  static Arg q[] =
    {
      { XtNx, (XtArgVal) 0 },
      { XtNy, (XtArgVal) 0 },
    };
  static Arg d[] = 
    {
      { XtNvalue, (XtArgVal) NULL },
      { XtNwidth, (XtArgVal) 0 },
      { XtNlabel, (XtArgVal) "Logfile name" },
    };

  XWindowAttributes atts;

  q[0].value = (XtArgVal) &x;
  q[1].value = (XtArgVal) &y;
  XtGetValues(w,q,2);		/* get button location */
  p[0].value = x;		/* display dialog at same location */
  p[1].value = y;
  XGetWindowAttributes(XtDisplay(gi_tool), XtWindow(XtParent(gi_tool)), &atts);
  p[0].value += atts.x;		/* display dialog at same location */
  p[1].value += atts.y;

  namepopup = XtCreatePopupShell("logfile_dialog",transientShellWidgetClass,
				     w, p, XtNumber(p));
  d[0].value = (XtArgVal) gi_logfile_name;
  d[1].value = (XtArgVal)40 + ((int) (gi_font_width * ((strlen(gi_logfile_name) > 60) ?
			       strlen(gi_logfile_name) : 60)));
  logfile_dialog = XtCreateManagedWidget("dialog_box", dialogWidgetClass,
					 namepopup,d, XtNumber(d));
  AddButton(logfile_dialog, "CHANGE", change_logfile_name, namepopup, NULL);
  AddButton(logfile_dialog, "CANCEL", cancel_logfile_name, namepopup, NULL);
  if (gi_log_on)
    AddButton(logfile_dialog, "NO LOG", switch_logfile_toggle, namepopup, NULL);
  else
    AddButton(logfile_dialog, "LOG ON", switch_logfile_toggle, namepopup, NULL);
  XtPopup(namepopup, XtGrabNone);
}

/****************************************************************/
static echo_proc(item, client_data, event)
   Widget item;
   caddr_t client_data;
   caddr_t event;

{
  if (gi_echo_on)
    {
      SetValue(item,XtNlabel,"ECHO ON");
      gi_echo_on = False;
    }
  else
    {
      SetValue(item,XtNlabel,"NO ECHO");
      gi_echo_on = True;
    }
}    

gi_echo_command(val)
     int val;

{
  if (val == 0)
    {
      SetValue(echo_item,XtNlabel,"ECHO ON");
      gi_echo_on = False;
    }
  else
    {
      SetValue(echo_item,XtNlabel,"NO ECHO");
      gi_echo_on = True;
    }
}    
  
/****************************************************************/
static dump_proc(item, client_data, event)
   Widget item;
   caddr_t client_data;
   caddr_t event;
{
/* called when the DUMP button is pressed
 */

#if 0 /* xgi: dump currently not implemented */
   char imagefilename[MAX_ITEM_LENGTH+1];
   FILE *raster_file;

   /* get the filename from the prompts */
   strcpy(imagefilename, gi_strip(gi_dump_buf));
  
   /* check for a valid file name */
   if (strlen(imagefilename)==0)
   {
     gi_put_error("Please specify image filename");
   }
   else
   {
     /* attempt to open the image file */
     if ((raster_file=fopen(imagefilename,"w"))==NULL)
     {
       gi_put_error("Unable to open image file");
     }
     else
     {
       /* write the retained window to the file */
       if (pr_dump(gi_gfx->gfx_pixwin->pw_prretained,
                   raster_file,
                   NULL,
                   RT_STANDARD,
                   TRUE)==PIX_ERR)
  
        {
          gi_put_error("pr_dump failed");
        }
        else
        {
          fclose(raster_file);
          gi_put_message("Image dump successful");
        }
      }
   } 
#endif
     gi_put_error("Image Dump not implemented.  Try 'xwd'.");
}

/****************************************************************/
static show_proc(item, client_data, event)
   Widget item;
   caddr_t client_data;
   caddr_t event;
{
/*
 * called when the SHOW button is
 * pressed, this routine builds a command line (in command[])
 * from the panel values that have been set and
 * passes it to the command processor for 
 * execution
 */

   char buf1[MAX_CMD_LENGTH+1];
   char buf2[MAX_CMD_LENGTH+1];
   char *args[MAX_ARGS];
   int numargs;

   /* read panel prompts into buf1 to make up a show command */        
   set_args(SHOW_CMD_SHORT,buf1);

   /* parse the just created command into args array and buf2 */
   numargs = gi_parse_cmd(buf1,args,buf2);

   /* execute the show command */
   gi_do_show(numargs,args);

   /* if display panel needs to be updated call gi_reshow to do it */
   if (gi_reshow_flag) 
     gi_reshow();

   /* if logging enabled, then log the original command */
   if (gi_log_on) 
     gi_log(buf1);

   /* show the command just executed on the command panel */
   gi_show_prev(buf1);
}

/****************************************************************/
static change_proc(item, client_data, event)
   Widget item;
   caddr_t client_data;
   caddr_t event;
{
/*
 * called when the CHANGE button is
 * pressed, this routine builds a command line (in command[])
 * from the panel values that have been set and
 * passes it to the command processor for
 * execution
 */
   char buf1[MAX_CMD_LENGTH+1];
   char buf2[MAX_CMD_LENGTH+1];
   char *args[MAX_ARGS];
   int numargs;

   /* build a change command in buf1 from the panel prompts */
   set_args(CHANGE_CMD_SHORT,buf1);

   /* parse the command into args array and buf2 */
   numargs = gi_parse_cmd(buf1,args,buf2);

   /* execute the change command */
   gi_do_change(numargs,args);

   /* if display display needs to be updated, call gi_reshow to do it */
   if (gi_reshow_flag) 
     gi_reshow();

   /* if logging enabled, log the original command */
   if (gi_log_on) 
     gi_log(buf1);

   /* show the command just executed on the command panel */
   gi_show_prev(buf1);
}

/****************************************************************/
static erase_proc(item, client_data, event)
   Widget item;
   caddr_t client_data;
   caddr_t event;
{
/*
 * called when the ERASE button is
 * pressed, this routine builds a command line (in command[])
 * from the panel values that have been set and
 * passes it to the erase command processor for 
 * execution
 */
   char buf1[MAX_CMD_LENGTH+1];
   char buf2[MAX_CMD_LENGTH+1];
   char *args[MAX_ARGS];
   int numargs;

   /* build an erase command in buf1 from the panel prompts */
   set_args(ERASE_CMD_SHORT,buf1);

   /* parse the command into args array and buf2 */
   numargs = gi_parse_cmd(buf1,args,buf2);

   /* execute the erase command */
   gi_do_erase(numargs,args);

   /* if logging enabled, log this command */
   if (gi_log_on) 
     gi_log(buf1);

   /* show the command just executed on the command panel */
   gi_show_prev(buf1);
}

