/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/* -------------------------------------------------------------------
 Functional interface to gi_command commands.  Uses simulator variables
---------------------------------------------------------------------*/
#include "uniproc.h"
#include "macros.h"
#include "externs.h"

typedef struct vtx
{
  int x, y;
} Vertex;
  
extern struct gricons * gi_get_image();
extern struct grobj * gi_remember_grobj();
extern char * gi_cur_ipanel;	/* not really char* */

#define CHECK_WHAT(aspect) ((aspect) < POT || (aspect) > DATA)
#define CHECK_RANGE(low, high) (abs(low) > MAX_RANGE || \
				abs(high) > MAX_RANGE || \
				(low) == (high))
#define CHECK_COORDS(x, y) (abs(x) > MAX_COORD || abs(y) > MAX_COORD)

#define SHOW_UNIT \
    /* display a unit at the correct position */ \
    ptr = gi_remember_grobj(loc_x,loc_y, \
                        unit_image, \
                        unit_index, \
                        unit_what, \
                        unit_lrange, \
                        unit_hrange, \
                        unit_target, \
                        unit_site); \
 \
    if (aux_flag) \
      ptr->flag |= AUX; \
       \
    /* determine the next position to place an object */ \
    if (--num_cols_left) \
      { \
	loc_x += unit_image->size_x + x_space; \
      } \
    else \
      { \
	loc_x = start_x; \
	loc_y += unit_image->size_y + y_space; \
	num_cols_left = num_cols; \
      }

#define CHANGE_UNIT \
      /* attempt to change this unit */ \
      if ((ptr=gi_find_unit(unit_index,aux_flag))!=NULL) \
	{ \
	  /* is unit currently on the display */ \
	  on_display = FALSE; \
	  if (gi_overlap_grobj(ptr)) \
	    { \
	      on_display = TRUE; \
	      gi_reshow_flag |= RESHOW_NEEDED;  \
	       \
	      /* if icon image has changed */ \
	      if (ptr->gptr!=unit_image) \
		{ \
		  /* if new icon is smaller than old icon, make sure  \
		     that screen get cleared first \
		     */ \
		  if (ptr->gptr->size_x > unit_image->size_x \
		      || ptr->gptr->size_y > unit_image->size_y) \
		    { \
		      gi_reshow_flag |= CLEAR_NEEDED; \
		    } \
		} \
	       \
	      /* indicate that unit needs to be shown */ \
	      ptr->flag &= ~DISPLAYED; \
	    } \
	   \
	  /* check if unit aspect or site or target has changed */ \
	  if (ptr->u_what!=unit_what) \
	    { \
	      /* indicate that new value needs to be gotten */ \
	      if (on_display) \
		gi_reshow_flag |= SOME_VALS_NEEDED; \
	      ptr->flag &= ~VALUE_OK; \
	    } \
	  else if (unit_what==LINKIN || unit_what==LINKOUT) \
	    { \
	      /* if link is being shown, check for target or site change */ \
	      if (ptr->target!=unit_target  \
		  || diff_site(ptr->site,unit_site)) \
		{ \
		  /* if on the display, make sure value gets updated */ \
		  if (on_display) \
		    gi_reshow_flag |= SOME_VALS_NEEDED; \
		  ptr->flag &= ~VALUE_OK; \
		} \
	    } \
	   \
	  /* see if low or high range has changed */ \
	  if (ptr->lrange!=unit_lrange  \
	      || ptr->hrange!=unit_hrange) \
	    { \
	      /* if range has changed, set VALUE_CHG on so that \
		 a new index (and icon) will be calculated \
		 */ \
	      if (on_display) \
		gi_reshow_flag |= RESHOW_NEEDED; \
	      ptr->flag |= VALUE_CHG; \
	    } \
	   \
	  /* make changes in the grobj structure */ \
	   \
	  ptr->u_what = unit_what; \
	  ptr->site = unit_site; \
	  ptr->target = unit_target; \
	  ptr->lrange = unit_lrange; \
	  ptr->hrange = unit_hrange;     \
	  ptr->gptr = unit_image; \
	   \
	  /* decrement the number of units left to change */ \
	  num_units_left--; \
	} \
      else \
	{ \
	  done = TRUE; \
	  return(ERROR); \
	}
      
#define ERASE_UNIT \
    /* attempt to erase this unit */ \
    if ((ptr=gi_find_unit(unit_index,aux_flag))!=NULL); \
    { \
       save_x = ptr->x_pos;  \
       save_y = ptr->y_pos; \
     } \
             \
    /* check that erase was successful */ \
    if (gi_erase_grobj(ptr)==OK) \
    {  \
       /* if this is the first erased unit, move the marker here */ \
       if (num_units==num_units_left)  \
       { \
          gi_move_marker(save_x, save_y); \
       } \
       num_units_left--; \
    }  \
    else \
    { \
      done = TRUE; \
      return(ERROR); \
    }

static diff_site(site1, site2)
  char *site1, *site2;
{
/* returns whether or not the two
 * sites passed by the caller are different or not
 * (either or both sites could be null)
 */

  if (site1==site2) return(FALSE);
  if (site1==NULL || site2==NULL) return(TRUE);
  if (strcmp(site1,site2)) return(TRUE);
  return(FALSE);
}

GiShowNamedUnits(name, num_units, unit_what, unit_lrange, unit_hrange,
	    target, image, where, spacing, num_cols)
     char * name;		/* initial \ implies auxiliary */
     int num_units;		/* how many */
     int unit_what;		/* aspect */
     FLINT unit_lrange, unit_hrange; /* value range */
     char * target;		/* unit/site target for link in/out */
     char * image;		/* icon to use */
     Vertex * where;	/* where on display */
     Vertex * spacing;	/* spacing cols and rows */
     int num_cols;		/* units per row */

{
  int cmd, done;
  int loc_x, loc_y;
  struct gricons *unit_image;
  struct grobj *ptr;
  int num_units_left, num_cols_left;
  char *unit_site;
  int unit_target= -1;
  int unit_index;
  int aux_flag = FALSE;
  int start_x, start_y;	/* where on display */
  int x_space, y_space;	/* spacing cols and rows */

  start_x = where->x;
  start_y = where->y;
  x_space = spacing->x;
  y_space = spacing->y;

  if (*name == AUX_CHAR)
    {
      aux_flag = TRUE;
      name++;
    }
  
  /* get number of units to put */
  if (num_units<0) 
    return(ERROR);
   
  /* get what to display (potential, output, etc) */
  if (CHECK_WHAT(unit_what))
    return(ERROR);
        
  /* get lower and upper range */
  if CHECK_RANGE(unit_lrange, unit_hrange)
    return(ERROR);

  /* get unit_image for display */
  if ((unit_image = gi_get_image(image))==NULL)
    return(ERROR);

  /* get x and y position of first unit to put */
  if CHECK_COORDS(start_x, start_y)
    return(ERROR);

  /* get x and y spacing (allow negative spacings "backwards" */
  if CHECK_COORDS(x_space, y_space)
    return(ERROR);
 
  /* get number of items in each row */
  if (num_cols <= 0)
    return(ERROR);

  /* get link target if appropriate */
  if (unit_what==LINKIN | unit_what==LINKOUT)
  {
    if ((unit_target=gi_get_target(target))==ERROR)
      return(ERROR);
    unit_site = gi_get_site(target);
  }

  if ((unit_index = gi_sim_get_unit(name, GET_FIRST)) == ERROR)
    return (ERROR);

  if (unit_index < 0 || unit_index + num_units >= NoUnits)
    return (ERROR);

  /* made it this far; parameters are OK */
  num_cols_left = num_cols;
  num_units_left = num_units;
  loc_x = start_x;
  loc_y = start_y;
  done = FALSE;
  while (!done)
  {
    SHOW_UNIT
    /* now decide if we're finished yet */
    if ((num_units>0) && (--num_units_left<=0)) 
      done = TRUE;
    else 
      if ((unit_index = gi_sim_get_unit(name, GET_NEXT)) == ERROR)
	return (ERROR);
  }  /* of while loop */

  /* set the reshow flag appropriately */
  gi_reshow_flag |= (RESHOW_NEEDED+SOME_VALS_NEEDED+SOME_LINKS_NEEDED);

  return(OK);
}

GiShowNumberedUnits(unit_index, num_units, unit_what, unit_lrange, unit_hrange,
	    target, image, where, spacing, num_cols)
     int unit_index;		/* who (-ve => auxiliary)*/
     int num_units;		/* how many */
     int unit_what;		/* aspect */
     FLINT unit_lrange, unit_hrange; /* value range */
     char * target;		/* unit/site target for link in/out */
     char * image;		/* icon to use */
     Vertex * where;	/* where on display */
     Vertex * spacing;	/* spacing cols and rows */
     int num_cols;		/* units per row */

{
/* processes a "show" command.  Shows num_units starting with
   abs(unit_index) (-ve means auxiliary aspect).  Not much
   checking.  No ability to use sets or names or types.
   Returns 1 if successful, -1 if unsucessful.  If failure,
   then nothing is displayed.
*/
  int cmd, done;
  int loc_x, loc_y;
  struct gricons *unit_image;
  struct grobj *ptr;
  int num_units_left, num_cols_left;
  char *unit_site;
  int unit_target= -1;
  int aux_flag = FALSE;
  int start_x, start_y;	/* where on display */
  int x_space, y_space;	/* spacing cols and rows */

  start_x = where->x;
  start_y = where->y;
  x_space = spacing->x;
  y_space = spacing->y;

  if (unit_index < 0) 
    {
      aux_flag = TRUE;
      unit_index = 0 - unit_index;
    }  

  /* get number of units to put */
  if (num_units<0) 
    return(ERROR);
   
  if (unit_index < 0 || unit_index + num_units >= NoUnits)
    return (ERROR);

  /* get what to display (potential, output, etc) */
  if (CHECK_WHAT(unit_what))
    return(ERROR);
        
  /* get lower and upper range */
  if CHECK_RANGE(unit_lrange, unit_hrange)
    return(ERROR);

  /* get unit_image for display */
  if ((unit_image = gi_get_image(image))==NULL)
    return(ERROR);

  /* get x and y position of first unit to put */
  if CHECK_COORDS(start_x, start_y)
    return(ERROR);

  /* get x and y spacing (allow negative spacings "backwards" */
  if CHECK_COORDS(x_space, y_space)
    return(ERROR);
 
  /* get number of items in each row */
  if (num_cols <= 0)
    return(ERROR);

  /* get link target if appropriate */
  if (unit_what==LINKIN | unit_what==LINKOUT)
  {
    if ((unit_target=gi_get_target(target))==ERROR)
      return(ERROR);
    unit_site = gi_get_site(target);
  }
         
  /* made it this far; parameters are OK */
  num_cols_left = num_cols;
  num_units_left = num_units;
  loc_x = start_x;
  loc_y = start_y;
  done = FALSE;
  while (!done)
  {
    SHOW_UNIT
    /* now decide if we're finished yet */
    if ((num_units>0) && (--num_units_left<=0)) 
      done = TRUE;
    else 
      unit_index++;
  }  /* of while loop */

  /* set the reshow flag appropriately */
  gi_reshow_flag |= (RESHOW_NEEDED+SOME_VALS_NEEDED+SOME_LINKS_NEEDED);

  return(OK);
}

GiChangeNamedUnits(name, num_units, unit_what, unit_lrange,
		      unit_hrange, target, image)
     char * name;		/* initial \\ implies auxiliary */
     int num_units;		/* how many */
     int unit_what;		/* aspect */
     FLINT unit_lrange, unit_hrange; /* value range */
     char * target;		/* unit/site target for link in/out */
     char * image;		/* icon to use */

{
/* processes a "change" command.  Copied from gi_do_change
 */
  int cmd, done, on_display;
  int loc_x, loc_y;
  struct gricons *unit_image;
  int num_units_left, num_cols_left;
  int unit_target= -1;
  char *unit_site;
  struct grobj *ptr;
  int unit_index;
  int aux_flag = FALSE;

  if (*name == AUX_CHAR)
    {
      aux_flag = TRUE;
      name++;
    }

  /* get number of units to change */
  if (num_units<0) 
    return(ERROR);

  /* get what to display (potential, output, etc) */
  if (CHECK_WHAT(unit_what))
    return(ERROR);
        
  /* get lower and upper range */
  if CHECK_RANGE(unit_lrange, unit_hrange)
    return(ERROR);

  /* get unit_image for display */
  if ((unit_image = gi_get_image(image))==NULL)
    return(ERROR);

  /* get link target if appropriate */
  if (unit_what==LINKIN | unit_what==LINKOUT)
  {
    if ((unit_target=gi_get_target(target))==ERROR)
      return(ERROR);
    unit_site = gi_get_site(target);
  }
         
  if ((unit_index = gi_sim_get_unit(name, GET_FIRST)) == ERROR)
    return (ERROR);
   
  if (unit_index < 0 || unit_index + num_units >= NoUnits)
    return (ERROR);

  /* made it this far; parameters are OK */

  num_units_left = num_units;
  done = FALSE;
  while (!done)
    {
      CHANGE_UNIT
      /* now decide if we're finished yet */
      if ((num_units>0) && (num_units_left<=0))
	done = TRUE;
      else
      if ((unit_index = gi_sim_get_unit(name, GET_NEXT)) == ERROR)
	return (ERROR);
    }  /* of while loop */
  if (num_units_left < num_units)
    gi_reshow_flag |= RESHOW_NEEDED;
  return(OK);   
}

GiChangeNumberedUnits(unit_index, num_units, unit_what, unit_lrange,
		      unit_hrange, target, image)
     int unit_index;		/* who (-ve => auxiliary)*/
     int num_units;		/* how many */
     int unit_what;		/* aspect */
     FLINT unit_lrange, unit_hrange; /* value range */
     char * target;		/* unit/site target for link in/out */
     char * image;		/* icon to use */

{
/* processes a "change" command.  Copied from gi_do_change
 */
  int cmd, done, on_display;
  int loc_x, loc_y;
  struct gricons *unit_image;
  int num_units_left, num_cols_left;
  int unit_target= -1;
  char *unit_site;
  struct grobj *ptr;
  int aux_flag = FALSE;

  if (unit_index < 0) 
    {
      aux_flag = TRUE;
      unit_index = 0 - unit_index;
    }  

  /* get number of units to change */
  if (num_units<0) 
    return(ERROR);
   
  if (unit_index < 0 || unit_index + num_units >= NoUnits)
    return (ERROR);

  /* get what to display (potential, output, etc) */
  if (CHECK_WHAT(unit_what))
    return(ERROR);
        
  /* get lower and upper range */
  if CHECK_RANGE(unit_lrange, unit_hrange)
    return(ERROR);

  /* get unit_image for display */
  if ((unit_image = gi_get_image(image))==NULL)
    return(ERROR);

  /* get link target if appropriate */
  if (unit_what==LINKIN | unit_what==LINKOUT)
  {
    if ((unit_target=gi_get_target(target))==ERROR)
      return(ERROR);
    unit_site = gi_get_site(target);
  }
         
  /* made it this far; parameters are OK */

  num_units_left = num_units;
  done = FALSE;
  while (!done)
    {
      CHANGE_UNIT
      /* now decide if we're finished yet */
      
      if ((num_units>0) && (num_units_left<=0))
	done = TRUE;
      else
	unit_index++;
    }  /* of while loop */
  if (num_units_left < num_units)
    gi_reshow_flag |= RESHOW_NEEDED;
  return(OK);   
}

GiEraseNumberedUnits(unit_index, num_units)
     int unit_index;
     int num_units;

{
  int cmd, done;
  int num_units_left;
  struct grobj *ptr;
  int save_x, save_y;
  int aux_flag = FALSE;

  if (unit_index < 0) 
    {
      aux_flag = TRUE;
      unit_index = 0 - unit_index;
    }  

  /* get number of units to put */
  if (num_units<0) 
    return(ERROR);
   
  if (unit_index < 0 || unit_index + num_units >= NoUnits)
    return (ERROR);

  /* made it this far: parameters are OK */
  num_units_left = num_units;
  done = FALSE;
  while (!done)
  {
    ERASE_UNIT
    /* now decide if we're finished yet */
    if ((num_units>0) && (num_units_left<=0)) 
      done = TRUE;
    else 
      unit_index++;
  }  /* of while loop */

  return(OK);
}
       
GiEraseNamedUnits(name, num_units)
     char * name;
     int num_units;

{
  int cmd, done;
  int num_units_left;
  struct grobj *ptr;
  int save_x, save_y;
  int unit_index;
  int aux_flag = FALSE;

  if (*name == AUX_CHAR)
    aux_flag = TRUE;

  /* get starting unit number */
  if ((unit_index=get_unit_erase(name,GET_FIRST))<0) 
    return(ERROR);

  /* get number of units to erase */
  if (num_units<0) 
    return(ERROR);

  /* made it this far: parameters are OK */
  num_units_left = num_units;
  done = FALSE;
  while (!done)
    {
      ERASE_UNIT
	/* now decide if we're finished yet */
	if ((num_units>0) && (num_units_left<=0)) 
	  done = TRUE;
	else 
	  {
	    unit_index=get_unit_erase(name,GET_NEXT); 
	    if (unit_index<0)
	      done = TRUE;
	  } 
    }  /* of while loop */
  return(OK);
}

GiDrawLines(count, vertices)
     int count;
     Vertex * vertices;
       
{
/* processes a draw command, with count vertices.
   count < 0 => bounding box.
 */
  
   int ret_code = OK;
   struct drobj *ptr;
   int i, num_vertices;
   short flag;

   if (count > MAX_VERTICES)
     return(ERROR);

   /* check if this is to be a bounding box and get #vertices arg */
   if (count < 0)
     {
       num_vertices = 5;
       flag = BOUND_BOX;
     }
   else
   {
     num_vertices = count;
     flag = 0;
   }

   /* convert and check that coordinates are in range */
   for (i=0; i<num_vertices; i++)
     if CHECK_COORDS((vertices+i)->x, (vertices+i)->y)
       return(ERROR);

   /* create and chain the drobj and store first 2 vertices */
   if ((ptr=gi_remember_drobj(vertices->x,vertices->y,
			      (vertices+1)->x,(vertices+1)->y,flag))==NULL)
     return(ERROR);

   /* now save the rest of the vertices (if any) */
   for (i=2, vertices+=2; i<num_vertices; i++, vertices++)
     ret_code = gi_save_vertex(ptr,vertices->x,vertices->y);
   
   /* display the object if on the screen and not RESHOW_ALL */
   if (gi_overlap_drobj(ptr) && !(gi_reshow_flag & RESHOW_ALL))
     gi_display_drobj(ptr,GXcopy);

   /* return last return code from gi_save_vertex */
   return(ret_code);
}

GiDrawText(text, where, font)
     char * text;
     Vertex * where;
     char * font;

{
/* processes a gi "text" command
 * 
 */
   XFontStruct * fptr;
   struct txobj *ptr;
   char *sptr;
   int len;

   if CHECK_COORDS(where->x, where->y)
     return(ERROR);

   /* check for valid font */
   if ((fptr=gi_get_font(font))==NULL)
     return(ERROR);
   
   /* save the text argument in string space */
   /* a function call to save each character! blech */
   sptr = text;
   for (len=0; *sptr!='\0'; sptr++, len++)
     gi_save_char(*sptr,len);

   /* build and display a text object for this string */
   if ((ptr=gi_remember_txobj(where->x, where->y,len,fptr))!=NULL)
     /* if object is within display space window, display it */
     if (gi_overlap_txobj(ptr) && !(gi_reshow_flag & RESHOW_ALL))
       gi_display_txobj(ptr,GXxor);
   /* we've gotten this far: everything must have gone ok */
   return(OK);
}

GiMoveObject(from, to)
     Vertex * from;
     Vertex * to;

{
  int i;
  int from_x, from_y, offset_x, offset_y;
  struct txobj *tptr;
  struct drobj *dptr;
  struct grobj *gptr;

  if (CHECK_COORDS(from->x, from->y) || CHECK_COORDS(to->x, to->y))
    return(ERROR);

   offset_x = to->x - from->x;
   offset_y = to->y - from->y;
   
   if ((tptr=gi_find_txobj(from->x, from->y)) != NULL)
     gi_move_txobj(tptr, offset_x, offset_y, TRUE);
   else
     if ((dptr=gi_find_drobj(from->x, from->y)) != NULL)
       {
	 /* check if this is a bounding box */
	 if (dptr->flag & BOUND_BOX)
	   {
	     /* find, mark and move all contained objects */
	     gi_mark_bound(dptr, FALSE);
	     gi_move_bound(offset_x, offset_y, FALSE);
	   }
	 else
	   gi_move_drobj(dptr, offset_x, offset_y, GXclear, GXcopy);
       }
     else
       if ((gptr=gi_find_grobj(from->x,from->y)) != NULL)
	 {
	   gi_move_grobj(gptr, offset_x, offset_y, TRUE);
	 }
       else
	 return(ERROR);
  /* made it this far; must have gone ok */
  return(OK);
}

GiDeleteObject(from)
     Vertex * from;

{
  int i;
  struct txobj *tptr;
  struct drobj *dptr;
  struct grobj *gptr;
  
  if CHECK_COORDS(from->x, from->y)
    return(ERROR);

   /* try to locate text object */
   if ((tptr=gi_find_txobj(from->x,from->y))!=NULL)
     {
       if (gi_overlap_txobj(tptr))
	 gi_display_txobj(tptr,GXxor);
       gi_erase_txobj(tptr);
     }
   else  /* if text not found, try to find drawn object */
     if ((dptr=gi_find_drobj(from->x,from->y))!=NULL)
       {
	 if (gi_overlap_drobj(dptr))
	   gi_display_drobj(dptr,GXclear);
	 gi_erase_drobj(dptr);
       }
     else /* no object found at all: put out error message */
       return(ERROR);
     
  /* made it this far; must have gone ok */
  return(OK);
}


