/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/********************************************************************
 * Graphics Interface
 * ------------------
 * This file contains most of the routines related to processing
 * done on or with the info panel.
 *
 * Kenton Lynne
 *   last update: 04/01/87
 * Michael McInerny
 *   last update: 08/19/88
 *******************************************************************/

#include "macros.h"
#include "externs.h"
#include "panel.h"
#include "info_panel.h"
#include <X11/Xmu.h>
#include <X11/Text.h>
#include <X11/AsciiText.h>
#include <X11/Command.h>

/* global declarations */
info_panel_t gi_info_panel_head;
info_panel_t gi_info_panel_list; /* keeps free list */

/* static forward references */
static void set_info_slot(), set_link_slot();

void gi_clear_info_slot();

/* formatting stings and info panel buffers */
static char wmsg[] = " CLOSE  FREEZE  ";

#ifdef FSIM

static char *info_format="Index: %d\n\
Name: %s\n\
Type: %s\n\
Potential: %-1.4g\n\
Output: %-1.4g\n\
State: %d\n\
Data: %-1.4g\n\
Link: \n";

static char *link_format="Index: %d\n\
Name: %s\n\
Type: %s\n\
Potential: %-1.4g\n\
Output: %-1.4g\n\
State: %d\n\
Data: %-1.4g\n\
Link: %-1.4g\n\
Target %d";

static char *blank_info_format="Index:\n\
Name:             \n\
Type:          \n\
Potential:     \n\
Output:        \n\
State:         \n\
Data:          \n\
Link:          \n";

#else

static char *info_format="Index: %d\n\
Name: %s\n\
Type: %s\n\
Potential: %d\n\
Output: %d\n\
State: %d\n\
Data: %d\n\
Link: \n";

static char *link_format="Index: %d\n\
Name: %s\n\
Type: %s\n\
Potential: %d\n\
Output: %d\n\
State: %d\n\
Data: %d\n\
Link: %d\n\
Target %d";

static char *blank_info_format="Index:\n\
Name:             \n\
Type:          \n\
Potential:     \n\
Output:        \n\
State:         \n\
Data:          \n\
Link:          \n";

#endif

/*************************************************************************/
static void close_info(w, ipanel, call_data)
     Widget w;
     info_panel_t ipanel;
     caddr_t call_data;
{
  gi_clear_info_slot(ipanel);
}

/*************************************************************************/
static freeze_info_panel(w, ipanel, call_data)
     Widget w;
     info_panel_t ipanel;
     caddr_t call_data;
{
  if (ipanel->frozen == True)
    {
      SetValue(w,XtNlabel," FREEZE ");
      ipanel->frozen = False;
      gi_show_info(ipanel, True);
    }
  else
    {
      SetValue(w,XtNlabel,"UNFREEZE");
      ipanel->frozen = True;
    }
}

static info_panel_t new_info_widget(index)
     int index;

{
/* Make a new info_widget shell */
  info_panel_t ipanel;
  Widget box, button;
 
  static Arg a[] = 
    {
      { XtNheight, (XtArgVal) 0 },
      { XtNstring, (XtArgVal) NULL },
      { XtNwidth, (XtArgVal) 0 },
      { XtNeditType, (XtArgVal) XttextEdit },
      { XtNlength, (XtArgVal) 150 },
      { XtNborderWidth, (XtArgVal) 0},
      { XtNtextOptions, (XtArgVal) (resizeWidth | resizeHeight)},
      { XtNfont, (XtArgVal) &gi_default_font },
      { XtNleft, (XtArgVal) XtChainLeft },
      { XtNright, (XtArgVal) XtChainRight },
    };
  static Arg b[] = 
    {
      { XtNfromVert, (XtArgVal) NULL },
      { XtNvertDistance, (XtArgVal) 2 },
      { XtNfromHoriz, (XtArgVal) NULL },
      { XtNhorizDistance, (XtArgVal) 2 },
      { XtNtop, (XtArgVal) XtChainBottom },
      { XtNbottom, (XtArgVal) XtChainBottom },
    };
  static Arg c[] = 
    {
      { XtNx, (XtArgVal) 0 },
      { XtNy, (XtArgVal) 0 },
    };
  static Arg d[] = 
    {
      { XtNfromVert, (XtArgVal) NULL },
      { XtNfromHoriz, (XtArgVal) NULL },
      { XtNvertDistance, (XtArgVal) 2 },
      { XtNhorizDistance, (XtArgVal) 2 },
      { XtNfont, (XtArgVal) &gi_default_font },
      { XtNtop, (XtArgVal) XtChainBottom },
      { XtNbottom, (XtArgVal) XtChainBottom },
    };
  static Arg form_args[] =
    {
      { XtNdefaultDistance, (XtArgVal) 1 },
    };

  ipanel = (info_panel_t)malloc(sizeof(*ipanel));
  c[0].value = (XtArgVal) index*8;
  ipanel->popup = XtCreatePopupShell("info_shell",transientShellWidgetClass,
				     gi_tool,c, XtNumber(c));
  box = XtCreateManagedWidget("info_box", formWidgetClass, ipanel->popup,
			      form_args, XtNumber(form_args));
  a[0].value = (XtArgVal) 11 * (gi_default_font.max_bounds.ascent +
			       gi_default_font.max_bounds.descent);
  strcpy(ipanel->buf, blank_info_format); 
  a[1].value = (XtArgVal) ipanel->buf;
  a[2].value = (XtArgVal) XTextWidth(&gi_default_font, wmsg, strlen(wmsg));
  ipanel->text = XtCreateManagedWidget("IPANEL",asciiStringWidgetClass,
				       box, a, XtNumber(a));
  b[0].value = (XtArgVal) ipanel->text;
  button = XtCreateManagedWidget("CLOSE", commandWidgetClass, box, b, XtNumber(b));
  XtAddCallback(button, XtNcallback, close_info, ipanel);
  d[0].value = (XtArgVal) ipanel->text;
  d[1].value = (XtArgVal) button;
  button = XtCreateManagedWidget(" FREEZE ", commandWidgetClass, box, d, XtNumber(d));
  XtAddCallback(button, XtNcallback, freeze_info_panel, ipanel);
  ipanel->free = True;
  ipanel->next = NULL;

  return(ipanel);
}

/*************************************************************************/
static info_panel_t next_info_item(x,y)
     int x, y;
{
/* returns the next free info panel item, creating and
   popping it up, if necessary, at x,y.  x < 0 => default
*/
  info_panel_t ip;
  int i;

  if (gi_info_panel_head == NULL) /* if no list for this display */
    {
      if (gi_info_panel_list == NULL) /* if none on free list */
	gi_info_panel_head = new_info_widget(0); /* generate info panel */
      else
	{				/* get an info panel from the free list */
	  gi_info_panel_head = gi_info_panel_list;
	  gi_info_panel_list = gi_info_panel_list->next;
	}
      ip = gi_info_panel_head;
    }
  else				/* find the end of the list */
    {
      for(i = 1, ip=gi_info_panel_head; ip->next != NULL; ip = ip->next, i++);
      ip->next = new_info_widget(i);
      ip = ip->next;
    }
  ip->free = False;
  ip->frozen = False;
  XtPopup(ip->popup,XtGrabNone);
  if (x >= 0)
    XtMoveWidget(ip->popup, (Position) x, (Position) y);

  return(ip);
}

/*************************************************************************/
gi_show_info(ip, all_flag)
     info_panel_t ip;
     int all_flag;
{
/* Given an gi_info_item index (column),
 * displays the detailed information 
 * that is contained in the corresponding gi_info_item structure.
 * all_flag : TRUE requests that all fields should be updated
 *          : FALSE requests only changing information
 * NOTE: if ui of info_item is < 0, then column is marked free and popped down
 * Assumes that all info in gi_info_item has been
 * brought up to date by previous call to get_unit_info
 * NB: currently all_flag is ignored--everything is updated
 * NB: currently, no boldface
 */

#if 0 /* xgi: bold not supported */
  int bold_flag[DATA+1];
#endif
  int i;
  XtTextBlock text;
  char buf[190];
  struct info_unit *iu;

  /* if invalid unit number, then blank column out */
  if (ip->iu.ui < 0)
  {
    /* blank out this column */
    gi_clear_info_slot(ip);
    return;
  }
    
#if 0 /* xgi: bold not supported */
  /* initialize all the bold flags to FALSE */
  for (i=POT; i<=DATA; i++) bold_flag[i] = FALSE;
#endif

  /* if LINK mode, set up direction and target fields */
  if (gi_mode==MODE_LINK)
    {
      iu = &(ip->iu);
      (void) sprintf(buf,link_format,
	      iu->ui, iu->name, iu->type, iu->pot,
	      iu->out, iu->state, iu->data,
	      iu->link, iu->target);
    }
  else 
    {
      iu = &(ip->iu);
      (void) sprintf(buf,info_format,
	      iu->ui, iu->name, iu->type, iu->pot,
	      iu->out, iu->state, iu->data);
    }
  for (i = strlen(buf); i <= strlen(ip->buf); i++)
    buf[i] = ' ';
  buf[i] = '\0';

  text.firstPos = 0;
  text.length = strlen(ip->buf);
  text.ptr = buf;
  text.format = FMT8BIT;
  
  XtTextReplace(ip->text, 0, strlen(ip->buf), &text);
  ForceBuildLineTable(ip->text);   /* grepped from X11/lib/Xaw/Text.c */
  XtTextDisplay(ip->text);
}

/*************************************************************************/
int gi_do_info(x, y, log_flag, sx, sy, display)
     int x, y, log_flag, sx, sy;
     char * display;		/* really not a char* */
{
/* given display coordinates, popup an info widget for
 * the unit at that position, to appear at sx, sy
 * if no unit is at that position
 * return ERROR else OK
 */
  struct grobj *ptr;
  info_panel_t ip;
  char nbuf[64];
  char tbuf[64];
  int newidthn, newidtht, defwidth;
  static Dimension width;
  static Arg a[] =
    {
      { XtNwidth, (XtArgVal) &width }
    };
  static Arg b[] =
    {
      { XtNwidth, (XtArgVal) 0 }
    };
  /* check if there is an object at this display coordinate */
  if ((ptr=gi_find_grobj(x,y))!=NULL)
  {
    strcpy(nbuf, "Name: ");
    strcpy(tbuf, "Name: ");
    ip = next_info_item(sx, sy);
    ip->iu.ui = ptr->u_index;
    ip->iu.what = ptr->u_what;
    ip->iu.target = ptr->target;
    ip->iu.site = ptr->site;
    ip->display = display;

    /* get the latest values for this unit from the simulator */
    gi_get_info(&(ip->iu),TRUE);
 
    /* show the information on the screen */
    gi_show_info(ip, TRUE);

    /* resize widthwise if necessary */
    defwidth = XTextWidth(&gi_default_font, wmsg, strlen(wmsg)) + 2;
    strcat(nbuf, ip->iu.name);
    newidthn = XTextWidth(&gi_default_font, nbuf, strlen(nbuf)) + 2;
    strcat(tbuf, ip->iu.type);
    newidtht = XTextWidth(&gi_default_font, tbuf, strlen(tbuf)) + 2;
    if (newidthn < newidtht)
      newidthn = newidtht;
    if (newidthn < defwidth)
      newidthn = defwidth;
    XtGetValues(ip->popup, a, 1);
    if (width != newidthn)
      {
	b[0].value = (XtArgVal) newidthn;
	XtSetValues(ip->popup, b, 1);
      }
    
    /* if logging enabled, log this command */
    if (gi_log_on && log_flag) gi_log_info(x, y, gi_info_next+1);
 
    return(OK);
  }
  return(ERROR);
}


/*************************************************************************/
extern char * gi_cur_ipanel; /* not really char* */

void gi_clear_info_list(ipanel)
     info_panel_t ipanel;

{
  info_panel_t ip;
  char * tdisplay;

  if (ipanel->display != (tdisplay = gi_cur_ipanel))
    gi_set_display_globals(ipanel->display, False); /* set current display */
  if (ipanel != gi_info_panel_head)
    gi_put_error("error in clear_slot");
  else
    for (; ipanel != NULL;)
      {
	XtPopdown(ipanel->popup);
	ipanel->free = True;
	ip = ipanel->next;
	ipanel->next = gi_info_panel_list;
	gi_info_panel_list = ipanel;
	ipanel = ip;
      }
  gi_info_panel_head = NULL;
  if (tdisplay != gi_cur_ipanel)
    gi_set_display_globals(tdisplay, False); /* reset current display */
}

void gi_clear_info_slot(ipanel)
     info_panel_t ipanel;

{
  info_panel_t ip;
  char * tdisplay;

  XtPopdown(ipanel->popup);
  ipanel->free = True;
  if (ipanel->display != (tdisplay = gi_cur_ipanel))
    gi_set_display_globals(ipanel->display, False); /* set current display */
  for (ip = gi_info_panel_head;
       ip != ipanel && ip != NULL && ip->next != ipanel;
       ip = ip->next);
  if (ip == NULL)
    gi_put_error("error in clear_slot");
  else
    {				/* put on free chain */
      if (ipanel == gi_info_panel_head)
	gi_info_panel_head = ipanel->next;
      else
	ip->next = ipanel->next;
      ipanel->next = gi_info_panel_list;
      gi_info_panel_list = ipanel;
      gi_put_message("");
    }
  if (tdisplay != gi_cur_ipanel)
    gi_set_display_globals(tdisplay, False); /* reset current display */
}


/*************************************************************************/
static void set_info_slot(ip)
     info_panel_t ip;
{
/* set the info panel data */
/* xgi: don't currently support boldface */
  struct info_unit *iu;

  iu = &(ip->iu);
  (void) sprintf(ip->buf,info_format,
	  iu->ui, iu->name, iu->type, iu->pot,
	  iu->out, iu->state, iu->data);
}


/*************************************************************************/
static void set_link_slot(ip)
     info_panel_t ip;
{
/* set the info panel and link data */
/* does not currently support link direction */
  struct info_unit *iu;

  iu = &(ip->iu);
  (void) sprintf(ip->buf,link_format,
	  iu->ui, iu->name, iu->type, iu->pot,
	  iu->out, iu->state, iu->data,
	  iu->link, iu->target);
}
	

