/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/***********************************************************************
 * Graphics Interface
 * ------------------
 * This file contains the routines that handle the SHOW command
 * CHANGE and ERASE commands as well as their assiciated routines.
 *
 * Kenton Lynne
 *   last update: 02/24/87
 * Nigel Goddard: 06/25/89
 ***********************************************************************/

#include "macros.h"
#include "externs.h"
#include "control_panel.h"

extern XImage * MakeImage();

/* variables globally refereced by routines in this file */
static int aux_flag;          /* indicates auxiliary unit */
static char *err_string;      /* contains as error message */

/***********************************************************************/
struct grobj *gi_remember_grobj(x, y, urect, ui, unit_what,
                       unit_lrange, unit_hrange, unit_target, unit_site)
  int x, y, ui, unit_what;
  FLINT unit_lrange, unit_hrange;
  int unit_target;
  struct gricons *urect;
  char *unit_site;
{
/* creates a grobj for some aspect of a unit and
 * puts in on the appropriate chain
 */
  struct grobj *ptr, *head_ptr;
  int flag, i;

  /* allocate a piece of memory for this object */
  ptr = (struct grobj *) malloc(sizeof(struct grobj));
  if (ptr==NULL)
  {
    gi_put_error("malloc out of space");
    exit(1);
  }

  /* initialize the unit */
  ptr->flag = 0;
  ptr->val = 0;
  ptr->link_val = 0;
  ptr->icon_index = 0;
  ptr->link_index = 0;

  /* fill in values as requested */
  ptr->x_pos = x;
  ptr->y_pos = y;
  ptr->gptr = urect;
  ptr->u_index = ui;
  ptr->u_what = unit_what;
  ptr->target = unit_target;
  ptr->lrange = unit_lrange;
  ptr->hrange = unit_hrange;
  ptr->site = unit_site;
 
  if (gi_overlap_grobj(ptr))
  {
    /* if on the screen, put it on the display chain */
    head_ptr = &gi_marker;
    flag = ON_DISPLAY;
  }
  else
  { 
    /* not on screen: put it on off_display chain */
    head_ptr = &gi_off_display;
    flag = 0;
  }
    
  /* insert onto end of doubly linked list */
  ptr->next = head_ptr;
  ptr->prev = head_ptr->prev;
  ptr->prev->next = ptr;
  head_ptr->prev = ptr;

  /* set flag indicating what chain it was put on */
  ptr->flag |= flag;

  return(ptr);
}

/***********************************************************************/
struct grobj *gi_find_unit(u,aux)
  int u, aux;
{
/* given a unit number, returns a pointer to the
 * graphic object that is displaying it
 * if none, then returns NULL
 * NOTE: uses aux to make the further check determining
 *       if the AUX flag is on in the unit
 *       aux should be either 0 (get non_auxiliary) or AUX (get auxiliary)
 */

  struct grobj *ptr;
  int unit_index;

  /* first check the display chain */
  for (ptr=gi_marker.next; ptr!=&gi_marker; ptr=ptr->next)
  {
    if (u==ptr->u_index && ((ptr->flag & AUX)==aux))
      return(ptr);
  }   

  /* if no luck, check the off_display chain */
  for (ptr=gi_off_display.next; ptr!=&gi_off_display; ptr=ptr->next)
  {
    if (u==ptr->u_index && ((ptr->flag & AUX)==aux))
      return(ptr);
  }
 
  /* if haven't found, then return NULL */
  return(NULL);
}

/***********************************************************************/
static get_unit(string, flag) 
   char *string;
   int flag;
{
/* returns the unit index of the next unit after this one 
 * string may be a pointer to a number string, a name
 * string or a set name string; index is the last one gotten
 */

  int ui, numb_flag;

  /* check that WHO string is not NULL */
  if (string==NULL)
  {               
    err_string = "WHO: no unit specified";
    return(ERROR);
  }

  /* check if the user has specified an Auxiliary unit */
  if (string[0]==AUX_CHAR)
  {
    /* ignore "\" character and set AUX flag on */
    string++;
    aux_flag = AUX;
  }
  else
    aux_flag = 0;

  /* set flag if string is a unit index */
  numb_flag = gi_check_num(string);

  /* check with the simulator that this unit exists */
  if ((ui=gi_sim_get_unit(string,flag))==ERROR)
  {
     /* no existing unit with this identity */
     ui = ERROR;
     err_string = "WHO: unit not found";
  }
   
  /* if string is not an index and not an auxiliary unit */
  else if (numb_flag!=OK && !aux_flag)  
  {
    /* loop through non-aux units looking for one not yet displayed */
    while (gi_find_unit(ui,0)!=NULL)
    {
      if ((ui=gi_sim_get_unit(string,GET_NEXT))==ERROR)
      { 
         ui = ERROR;
         err_string = "WHO: units all displayed";
         break;
      }
    } 
  }

  /* if string is an index and not an auxiliary unit, check that
     it is not already displayed 
  */
  else if (gi_find_unit(ui,0)!=NULL && !aux_flag)
  {
     ui = ERROR;  
     err_string = "WHO: unit number already displayed";
  } 

  /* return the index (or ERROR) found */
  return (ui);
}

/***********************************************************************/
char *gi_get_site(string) 
  char *string;
{
/* returns a pointer to the sitename (if any) that may have
 * been concatenated with the target name.  The returned pointer is
 * to the string that has been saved in string space 
 * NULL is returned if no site specified or site is "*any"
 */

  char *sitename = NULL;

  /* strip off the site from "ui/site" target string */
  for (; *string!='\0'; string++)
  {
    if (*string==SITE_SEP_CHAR)  /* looking for "/" */        
    {
      string++;

      /* if site is not empty and not "*any", save in string_space */
      if (strlen(string) && strcmp(string,ANY_SITE))
        sitename = gi_save_string(string);
      break;
    }
  }

  /* return pointer to site name in string space or NULL */
  return(sitename);
}

/***********************************************************************/
gi_get_target(string)
   char *string;
{
/* given a unit name, number, type or setname,
 * returns the index of the first legal unit that
 * matches it or else ERROR
 * note: the string is a concatenation of the
 * target name and the "optional" site name: target/sitename
 * so must first separate the target from sitename
 */
  int ui, numb_flag;
  int site_sep; 
  int replaced = FALSE;

  /* locate the '/' (if anywhere)  and replace with  '\0' */
  for (site_sep=1; string[site_sep]!='\0'; site_sep++)
  {
    if (string[site_sep]==SITE_SEP_CHAR) 
    {
      string[site_sep] = '\0';
      replaced = TRUE;
      break;
    }
  }
    
  /* if no target unit specified, return an error 
     else call simulator interface to get unit index 
  */
  if (string==NULL)
    ui = ERROR;
  else
    ui = gi_sim_get_unit(string,GET_FIRST);
   
  /* put back the "/" if we've removed it */
  if (replaced) string[site_sep] = SITE_SEP_CHAR;
  return(ui);
}

/***********************************************************************/
static get_number(string) 
  char *string;
{
/* returns the number of units to put or 0
 * if the string argument is "all"
 * (or if string is NULL then default of 1 is returned)
 */
    
  /* default to 1 */
  if (string==NULL)
     return(1);

  /* check for a valid integer and return it  */
  else if (gi_check_num(string)==OK) 
     return(atoi(string));

  /* check for "all" and return 0 */
  else if (strcmp(string,ALL_STRING)==0)
     return(0);

  else
     return(ERROR);
}

/***********************************************************************/
static get_what(string)
   char *string;
{
/* returns an integer indicating whether to display
 * potential, output , etc
 * or an error if not within bounds
 */

  if (string==NULL || strcmp(string,POT_STG)==0)
    return(POT);
  else if (strcmp(string,OUTPUT_STG)==0)
    return(OUTPUT);
  else if (strcmp(string,STATE_STG)==0)
    return(STATE);
  else if (strcmp(string,DATA_STG)==0)
    return(DATA);
  else if (strcmp(string,LINKIN_STG)==0)
    return(LINKIN);
  else if (strcmp(string,LINKOUT_STG)==0)
    return(LINKOUT);
  else
   return(ERROR);
}

/***********************************************************************/
static get_range(string,ptr)
  char *string;
  FLINT *ptr;
{
/* Returns the integer (or float) value of the  range parameter
 * passed in string, makes sure it is within bounds,
 * and then returns the integer (or float) value into
 * the pointer field passed by the caller.
 */

#ifdef FSIM
   if (sscanf(string,"%f",ptr) == 1)
     return(OK);
#else
   /* check for non-null and numeric string */
   if (string && gi_check_num(string)==OK)  
   {
     /* convert to integer */
     *ptr = atoi(string);
     
     /* check that it is within range */
     if (abs(*ptr) <= MAX_RANGE)
       return(OK);
   }
#endif
   return(ERROR);
} 

/***********************************************************************/
static char * read_icon_file(fp, sizex, sizey)
     FILE * fp;					  /* file to read from */
     int * sizex;				  /* return width */
     int * sizey;				  /* return height */
     
{
#define MAXPIX 256
#define ONCHAR '#'
#define OFFCHAR '-'

  char * bits;
  char buf[MAXPIX];
  char icon[MAXPIX/8][MAXPIX/8];
  char * nl, * tbuf;
  int linecnt = -1;
  int bytecnt;
  int i, j;

  for (i = 0; i < MAXPIX/8; i++)
    for (j = 0; j < MAXPIX/8; j++)
      icon[i][j] = 0;		/* initialize */
  while(1)
    {
      if (++linecnt == MAXPIX/8)
	return NULL;
      if ((tbuf = fgets(buf, sizeof(buf), fp)) == NULL)
	break;					  /* get one line */
      for (nl = tbuf; *nl != '\n' && *nl != '\0'; nl++); /* find newline or end */
      if (*nl == '\0')
	return NULL;				  /* line too long */
      while (*--nl == ' ' || *nl == '\t');	  /* remove trailing whitespace */
      *++nl = '\0';				  /* terminate line */
      while (*tbuf == ' ' || *tbuf == '\t')
	tbuf++;					  /* remove leading whitespace */
      if (*tbuf == '\0')
	break;					  /* empty line => end */
      if (*tbuf != ONCHAR && *tbuf != OFFCHAR)
	return NULL;				  /* error */
      if (*sizex < 0)
	*sizex = strlen(tbuf);
      else
	if (*sizex != strlen(tbuf))
	  return NULL;				  /* different line lengths */
      bytecnt = (strlen(tbuf) + 7) >> 3;	  /* bytes per line, padded*/
      for (i = nl-tbuf; i > (bytecnt<<3); i--, nl++)
	*nl = '-';				  /* pad with off char */
      *nl = '\0';				  /* terminate */
      for (i = 0; i < bytecnt; i++)
	for (j = 0; j < 8; j++, tbuf++)
	  switch(*tbuf)
	    {
	    case ONCHAR:
	      icon[linecnt][i] |= (char) (1 << j); /* add in one bit */
	    case OFFCHAR:
	      break;
	    default:
	      return NULL;			  /* spurious char */
	    }
    }
  if (*sizey < 0)
    *sizey = linecnt;
  else
    if (*sizey != linecnt)
      return NULL;				  /* different heights */
  nl = bits = (char *) malloc ((*sizex * (*sizey)) >> 3);
  for (i = 0; i < linecnt; i++)
    for (j = 0; j < bytecnt; j++)
      *nl++ = icon[i][j];
  return bits;
}

static struct gricons * load_custom_xicon(name, dir, ptr)
     char * name;		/* icon name - files are name_0, name_1, etc. */
     char * dir;		/* terminated with /, or "" => pwd */
     struct gricons *ptr;	/* where to load icon */

{
#define MAX_FAMILY_SIZE 256
  int i;
  FILE * fp = NULL;
  int sizex = -1, sizey = -1;
  XImage * ics[MAX_FAMILY_SIZE];
  char buf[128];

  fprintf(stdout, "Loading custom icon %s\n", name);
  for (i=0; i<MAX_RANGE; i++)
    {
      (void) sprintf(buf,"%s%s_%1d",dir,name,i);
      if ((fp = fopen(buf, "r")) == NULL)
	{
	  /* no more icon files */
	  if (i < 2)
	    {
	      if (i == 0)
		(void) sprintf(buf, "No icons %s in %s", name, dir);
	      else
		(void) sprintf(buf, "Not enough icons for %s; minimum is 2", name);
	      goto errexit;
	    }
	  else
	    if (i >= MAX_FAMILY_SIZE)
	      {
		(void) sprintf(buf, "Too many members for %s", name);
		goto errexit;
	      }
	    else
	      {	
		XImage ** s, ** d;
		int j;

		/* fill in fields as requested */
		ptr->name = gi_save_string(name);
		ptr->num = i-1;
		ptr->pix_ptr = (XImage **)malloc (i * sizeof(XImage *));
		for (j = 0, s = &(ics[0]), d = ptr->pix_ptr;
		     j < i; j++) /* really ptr->num + 1 icons */
		  *d++ = *s++;	/* copy pointers */
		ptr->size_x = (short) sizex;
		ptr->size_y = (short) sizey;
		
		return ptr;
	      }
	  break;
	}
      else
	{					  /* read this file */
	  char * bits;

	  if ((bits = read_icon_file(fp, &sizex, &sizey, bits)) == NULL)
	    return NULL;
	  ics[i] = MakeImage(gi_tool, bits, sizex, sizey);
	  fclose(fp);
	}
#if 0
	  int w, h;
	  char str1[32], str2[128], str3[128], str4[32], c1, c2;

	  (void) sprintf(str3, "%s_%d_width", name, i);
	  if (fscanf(fp, "%s %s %d", str1, str2, &w) != 3 ||
	      strcmp(str1, "#define") || strcmp(str2, str3))
	    {
	      (void) sprintf(buf, "Missing icon width in first line of %s%d",
		      name, i);
	      goto errexit;
	    }
	  else
	    if (sizex < 0)
	      sizex = w;
	    else
	      if (sizex != w)
		{
		  (void) sprintf(buf, "All icons must have same width");
		  goto errexit;
		}
	  
	  (void) sprintf(str3, "%s_%d_height", name, i);
	  if (fscanf(fp, "%s %s %d", str1, str2, &h) != 3 ||
	      strcmp(str1, "#define") || strcmp(str2, str3))
	    {
	      (void) sprintf(buf, "Missing icon height in second line of %s%d",
		      name, i);
	      goto errexit;
	    }
	    if (sizey < 0)
	      sizey = h;
	    else
	      if (sizey != h)
		{
		  (void) sprintf(buf, "All icons must have same width");
		  goto errexit;
		}

	  (void) sprintf(str3, "%s_%d_bits[]", name, i);
	  if (fscanf(fp, "%s %s %s %c %c", str1, str2, str4, &c1, &c2) != 5 ||
	      strcmp(str1, "static") || strcmp(str2, "char") || 
	      strcmp(str4, str3))
	    {
	      (void) sprintf(buf, "Missing icon bits in third line of %s%d",
		      name, i);
	      goto errexit;
	    }
	  else
	    {
	      int j, k, b;
	      char tbuf[128];

	      bits = (char *) malloc ((k = sizex*sizey/8) * sizeof(char));
	      for (j = 0; j < k; j++)
		if (fscanf(fp, "%s", tbuf) != 1)
		  {
		    free(bits);
		    (void) sprintf(buf, "Reading icon (%s%d) bit strings",
			    name, i);
		    goto errexit;
		  }
		else
		  if (sscanf(tbuf, "0x%x", &b) != 1 || b < 0 || b > 0xff)
		    {
		      free(bits);
		      (void) sprintf(buf, "Converting icon (%s_%d) strings to bits",
			      name, i);
		      goto errexit;
		    }
		  else
		    *(bits+j) = (char) b;

	      ics[i] = MakeImage(gi_tool, bits, sizex, sizey);
	      fclose(fp);
	    }	  
#endif
    }
 errexit:
  if (fp != NULL)
    fclose(fp);
  free(ptr);
  gi_put_error(buf);
  return(NULL);
}

struct gricons * AddCustomXIcon(name, dir)
     char * name;
     char * dir;

{
/* builds a gricons structure and initializes it
 * as requested, inserts in on the gricons chain
 * and returns a pointer to it.  NULL => failure.
 */
  
  struct gricons *ptr;

  /* get some space for the icon object */ 
  if ((ptr=(struct gricons *) malloc(sizeof(struct gricons)))==NULL)
    {
      gi_put_error("Malloc out of space");
      return(NULL);
    } 
  if ((ptr = load_custom_xicon(name, dir, ptr)) != NULL)
    {
      /* save on chain of gricons */
      ptr->next = gi_icon_head;
      gi_icon_head = ptr;
    }
  return(ptr);
}

/*************************************************************************/
ReloadCustomXIcon(name, dir)
     char * name;
     char * dir;
     /* Reloads a custom X icon.  Returns 1 if successful, 0 otherwise */
{
  char buf[128];
  struct gricons * gptr;
  struct gricons * tptr;
  int i;
  XImage ** pix_ptr;
  struct gricons temp;

  for (gptr=gi_icon_head; gptr!=NULL; gptr=gptr->next)
    if (!strcmp(name,gptr->name))
      {
	tptr = load_custom_xicon(name, dir, &temp); /* try to load new icon */
	if (tptr == NULL)
	  return 0;
	if (tptr != &temp)
	  {
	    gi_put_error("Major internal screw up in ReloadCustomXIcon");
	    return 0;
	  }
	for (i = gptr->num, pix_ptr = gptr->pix_ptr; i > 0; pix_ptr++, i--)
	  XDestroyImage(*pix_ptr);
	free(gptr->pix_ptr);
	tptr->next = gptr->next; /* save pointer to next gricon */
	*gptr = *tptr;		/* copy temp; lots of grobjs point to gptr */
	return 1;
      }
  if (AddCustomXIcon(name, dir) != NULL)
    {
      (void) sprintf(buf, "Icon %s wasn't loaded; now it is", name);
      gi_put_message(buf);
      return 1;
    }
  else
    return 0;
}

/*************************************************************************/
static int parse_filename(string, name, dir)
     char * string;		/* input filename */
     char * name;		/* buffer for return value icon family name */
     char * dir;		/* buffer for name of directory containing icon */

{
  char tbuf[512];
  char * tdir;

  while (*string == ' ' || *string == '\t')
    string++;					  /* get rid of leading whitespace */
  if (*string != '/')
    {
      strcpy(name, string);
      dir[0] = '\0';
      return 1;
    }
  strcpy(dir, string);
  tdir = dir + strlen(dir);			  /* get to end of string */
  while (*--tdir != '/' && tdir != dir);	  /* go back to last / */
  if (tdir == dir)				  /* no directory! */
    return 0;
  strcpy(name, ++tdir);				  /* copy name to return buffer */
  *tdir = '\0';					  /* terminate directory after last / */
  return 1;
}
    
struct gricons *gi_get_image(string)
  char *string;
{
/*
 * returns a pointer to the gricons structure
 * that contains all the icon information
 * for the image requested by string 
 */
  struct gricons *gptr = NULL;
  char name[128];
  char dir[128];
  char buf[256];
  FILE * fp;

  /* if NULL then use 1st default default */
  if (string==NULL)
  {
    gptr = &gi_square;
  }
  
  /* if a single digit, set gptr to default icon array structure */ 
  else if (strlen(string)==1)
  {
    switch(*string)
    {
      case '1': gptr = &gi_square;
                break;
      case '2': gptr = &gi_circle;
                break;
      case '3': gptr = &gi_triangle;
                break;
      case '4': gptr = &gi_pentagon;
                break;
      case '5': gptr = &gi_diamond;
                break;
      case '6': gptr = &gi_grey;
                break;
    }
  }

  if (gptr==NULL) /* user_defined icons */
    {
      /* get the directory and basename */
      if (parse_filename(string, name, dir) == 0)
	{
	  (void) sprintf(buf, "can't parse %s", string);
	  gi_put_error(buf);
	  return NULL;
	}
      
      /* see if icons have already been loaded in */
      for (gptr=gi_icon_head; gptr!=NULL; gptr=gptr->next)
	{
	  if (strcmp(name,gptr->name)==0)
	    {
	      /* found it */
	      return(gptr);
	    }
	}

      /* Not found in loaded icons, so look for it */
      /* if no directory is given, check . and ./icon */
      if (*dir == '\0')
	{					  /* no directory given */
	  (void) sprintf(buf,"%s_0",name);	  /* name of zero icon file*/
	  if ((fp = fopen(buf, "r")) == NULL)
	    {			             /* not in pwd, try pwd/icon */
	      (void) sprintf(buf,"icon/%s_0",name); /* name of zero icon file*/
	      if ((fp = fopen(buf, "r")) == NULL)
		{				  /* not in pwd/icon either, lose */
		  (void) sprintf(buf, "icon %s not found in . or ./icon", name);
		  gi_put_error(buf);
		  return NULL;
		}
	      else
		(void) sprintf(dir, "icon/"); /* icon directory */
	    }
	  fclose(fp);				  /* just an existence test */
	}      
      
      /* not loaded in yet, attempt to load and save now */
      gptr = AddCustomXIcon(name, dir);
    }
  return gptr;
}

/***********************************************************************/
static get_space(string,val)
  char *string;
  int *val;
{
/* returns the spacing (vert or horz) between units
 * into the integer pointer passed by the caller
 * (defaults to 5 if string is NULL)
 */

  if (!string) 
    *val = 5;
  else
  {
    if (gi_check_num(string)==OK)
      *val = atoi(string); 
    else
      return(ERROR);
  }
  return(OK);
}
 
/***********************************************************************/
static get_columns(string) 
   char *string;
{
/* returns the number of columns for each row given a
 * string -- if string is NULL or "max" then returns the max number
 * possible
 */
   int max_num, num;

   /* if null or "max" set to largest positive integer */
   if (string==NULL || !strcmp(string,MAX_STRING)) 
     return(MAX_INT);
   else
   {
     /* make sure it is a numeric string */
     if (gi_check_num(string)==OK)
     {
       num = atoi(string);
       if (num>0) 
         return(num);
       else
         return(ERROR);
     }
     else
       return(ERROR);
   }
}

/***********************************************************************/
get_unit_erase(string, flag)   
  char *string;
  int flag;
{
/* returns the unit index of the unit indicated by string
 * flag indicates if this is the first request for this unit
 * (FIRST) or another request (NEXT) in which case the next
 * unit after the previous request is returned
 */
  int ui;

  /* check for a NULL string */
  if (string==NULL)
  {
    err_string = "WHO: no unit specified";
    return(ERROR);
  }

  /* check for auxiliary units */
  if (string[0]==AUX_CHAR)
  {
    string++;
    aux_flag = AUX;
  }
  else
    aux_flag = 0;

  /* check for "all" */
  if (strcmp(string,ALL_STRING)==0)
  {
    /* if there are is any unit on any chain return it */
    if (gi_marker.next != &gi_marker)
      ui = gi_marker.next->u_index;
    else if (gi_off_display.next != &gi_off_display)
      ui = gi_off_display.next->u_index;
    else
    {
      ui = ERROR;
      err_string = "WHO: no units displayed";
    }
  }
   
  /* contact simulator for next unit */
  else if ((ui=gi_sim_get_unit(string,flag))==ERROR)
  {
    ui = ERROR;
    err_string = "WHO: unit not found";
  }

  /* if WHO is not a unit index then get next unit from
     simulator until a unit is found that is displayed
  */
  else if (gi_check_num(string)!=OK)
  {
    while (gi_find_unit(ui,aux_flag)==NULL)
    {
      if ((ui=gi_sim_get_unit(string,GET_NEXT))==ERROR)
      {
         ui = ERROR;
         err_string = "Named units not displayed";
         break;
      }
    }
  }    

  /* if WHO is a unit index, but not displayed then an error */
  else if (gi_find_unit(ui,aux_flag)==NULL)
  {
     ui = ERROR;
     err_string = "Unit number not displayed";
  }

  return (ui);
}

/***********************************************************************/
gi_do_erase(numargs,cmd_args)
  int numargs;
  char *cmd_args[];
{
/* processes an "erase" command string
 * assuming that the arguments are passed
 * in the cmd_args array
 */
  int cmd, done;
  int unit_index;
  int num_units;
  int num_units_left;
  struct grobj *ptr;
  int save_x, save_y;

  /* get starting unit number */
  if ((unit_index=get_unit_erase(cmd_args[CMD_WHO],GET_FIRST))<0) 
  {
     gi_put_error(err_string);
     return(ERROR);
  }  

  /* get number of units to erase */
  num_units = get_number(cmd_args[CMD_NUM]);
  if (num_units<0) 
  {
     gi_put_error("Illegal number of units");
     return(ERROR);
  }

  /* made it this far: parameters are OK */
  num_units_left = num_units;
  done = FALSE;
  while (!done)
  {
    /* attempt to erase this unit */
    if ((ptr=gi_find_unit(unit_index,aux_flag))!=NULL);
    {
       save_x = ptr->x_pos; 
       save_y = ptr->y_pos;
     }
            
    /* check that erase was successful */
    if (gi_erase_grobj(ptr)==OK)
    { 
       /* if this is the first erased unit, move the marker here */
       if (num_units==num_units_left) 
       {
          gi_move_marker(save_x, save_y);
       }
       num_units_left--;
    } 
    else
    {
      done = TRUE;
      gi_put_error("No more units to erase");
      return(ERROR);
    }

    /* now decide if we're finished yet */
    if ((num_units>0) && (num_units_left<=0)) 
    {
      done = TRUE;
      gi_put_message("Erase command successful");
    }
    else 
    {
      unit_index=get_unit_erase(cmd_args[CMD_WHO],GET_NEXT); 
      if (unit_index<0)
      {
        done = TRUE;
        gi_put_message("Erase command successful");
      } 
    }  /* get next unit to erase */
  }  /* of while loop */

  return(OK);
}

/***********************************************************************/
static diff_site(site1, site2)
  char *site1, *site2;
{
/* returns whether or not the two
 * sites passed by the caller are different or not
 * (either or both sites could be null)
 */

  if (site1==site2) return(FALSE);
  if (site1==NULL || site2==NULL) return(TRUE);
  if (strcmp(site1,site2)) return(TRUE);
  return(FALSE);
}

/***********************************************************************/
gi_do_change(numargs,cmd_args)
  int numargs;
  char *cmd_args[];
{
/* processes a "change command string 
 * assuming the arguments are passed in the
 * cmd_args array 
 */
  int cmd, done, on_display;
  int unit_index;
  int num_units;
  int unit_what;
  int start_x, start_y, loc_x, loc_y;
  struct gricons *unit_image;
  FLINT unit_lrange, unit_hrange;
  int num_units_left, num_cols_left;
  int unit_target= -1;
  char *unit_site;
  struct grobj *ptr;

  /* get number of units to change */
  num_units = get_number(cmd_args[CMD_NUM]);
  if (num_units<0)
  {
     gi_put_error("Illegal number of units");
     return(ERROR);
  }

  /* get what to display (potential, output, etc) */
  if ((unit_what=get_what(cmd_args[CMD_WHAT]))==NULL)
  {
     gi_put_error("Illegal WHAT value");
     return(ERROR);
  }

  /* get lower and upper range */
  if (get_range(cmd_args[CMD_LRANGE],&unit_lrange)==ERROR
     || get_range(cmd_args[CMD_HRANGE],&unit_hrange)==ERROR
     || unit_hrange == unit_lrange)
  {
    gi_put_error("Illegal range argument");
    return(ERROR);
  }

 /* get unit_image for display */
  if ((unit_image=gi_get_image(cmd_args[CMD_IMAGE]))==NULL)
  {
     gi_put_error("Cannot find unit image");
     return(ERROR);
  }
 
  /* get link target if appropriate */
  if (unit_what==LINKIN | unit_what==LINKOUT)
  {
    if ((unit_target=gi_get_target(cmd_args[CMD_TARGET]))==ERROR)
    {
       gi_put_error("Need to designate target unit/site");
       return(ERROR);
    }
    unit_site = gi_get_site(cmd_args[CMD_TARGET]);
  }

  /* get starting unit number of unit to change */
  /* note: cannot do this before get_target is called or
   * else the save_ui value in sim_get_unit will
   * be corrupted 
   */
  if (strcmp(cmd_args[CMD_WHO],ALL_STRING)==0)
  {
    gi_put_error("All not allowed on Change command");
    return(ERROR);
  }

  if ((unit_index=get_unit_erase(cmd_args[CMD_WHO],GET_FIRST))<0)
  {
     gi_put_error("Cannot find starting unit");
     return(ERROR);
  }
      
  /* made it this far; parameters are OK */

  num_units_left = num_units;
  done = FALSE;
  while (!done)
  {
    /* attempt to change this unit */
 
    if ((ptr=gi_find_unit(unit_index,aux_flag))!=NULL)
    {
      /* is unit currently on the display */
      on_display = FALSE;
      if (gi_overlap_grobj(ptr))
      {
        on_display = TRUE;
        gi_reshow_flag |= RESHOW_NEEDED; 

        /* if icon image has changed */
        if (ptr->gptr!=unit_image)
        {
          /* if new icon is smaller than old icon, make sure 
             that screen get cleared first
          */
          if (ptr->gptr->size_x > unit_image->size_x
          || ptr->gptr->size_y > unit_image->size_y)
          {
            gi_reshow_flag |= CLEAR_NEEDED;
          }
        }

        /* indicate that unit needs to be shown */
        ptr->flag &= ~DISPLAYED;
      }

      /* check if unit aspect or site or target has changed */
      if (ptr->u_what!=unit_what)
      {
        /* indicate that new value needs to be gotten */
        if (on_display)
          gi_reshow_flag |= SOME_VALS_NEEDED;
        ptr->flag &= ~VALUE_OK;
      }
      else if (unit_what==LINKIN || unit_what==LINKOUT)
      {
        /* if link is being shown, check for target or site change */
        if (ptr->target!=unit_target 
        || diff_site(ptr->site,unit_site))
        {
          /* if on the display, make sure value gets updated */
          if (on_display)
            gi_reshow_flag |= SOME_VALS_NEEDED;
          ptr->flag &= ~VALUE_OK;
         }
       }

      /* see if low or high range has changed */
      if (ptr->lrange!=unit_lrange 
      || ptr->hrange!=unit_hrange)
      {
        /* if range has changed, set VALUE_CHG on so that
           a new index (and icon) will be calculated
        */
        if (on_display)
          gi_reshow_flag |= RESHOW_NEEDED;
        ptr->flag |= VALUE_CHG;
      }

      /* make changes in the grobj structure */

      ptr->u_what = unit_what;
      ptr->site = unit_site;
      ptr->target = unit_target;
      ptr->lrange = unit_lrange;
      ptr->hrange = unit_hrange;    
      ptr->gptr = unit_image;

      /* decrement the number of units left to change */
      num_units_left--;
    }
    else
    {
      done = TRUE;
      gi_put_error("Unit is not displayed");
      return(ERROR);
    }

    /* now decide if we're finished yet */
 
    if ((num_units>0) && (num_units_left<=0))
    {
      done = TRUE;
      gi_put_message("Change command successful");
    }
    else
    {
      unit_index=get_unit_erase(cmd_args[CMD_WHO],GET_NEXT);
      if (unit_index<0)
      {
        done = TRUE;
        gi_put_message("Change command successful");
      }
    }  /* get next unit to change */
  }  /* of while loop */

  if (num_units_left < num_units)
    gi_reshow_flag |= RESHOW_NEEDED;

   return(OK);   
}

/***********************************************************************/
gi_do_show(numargs,cmd_args)
  int numargs;
  char *cmd_args[];
{
/* processes a "show" command string
 * assuming the arguments pointed to by the cmd_args array
 */
  int cmd, done;
  int unit_index;
  int num_units;
  int unit_what;
  int start_x, start_y, loc_x, loc_y;
  struct gricons *unit_image;
  struct grobj *ptr;
  int x_space, y_space;
  int num_cols;
  FLINT unit_lrange, unit_hrange;
  int num_units_left, num_cols_left;
  char *unit_site;
  int unit_target= -1;

  /* get number of units to put */
  num_units = get_number(cmd_args[CMD_NUM]);
  if (num_units<0) 
  {
    gi_put_error("Illegal number of units");
    return(ERROR);
  }
   
  /* get what to display (potential, output, etc) */
  if ((unit_what=get_what(cmd_args[CMD_WHAT]))==NULL)  
  {
    gi_put_error("Illegal <WHAT> value");
    return;
  }
        
  /* get lower and upper range */
  if (get_range(cmd_args[CMD_LRANGE],&unit_lrange)==ERROR
     || get_range(cmd_args[CMD_HRANGE],&unit_hrange)==ERROR
     || unit_hrange == unit_lrange)
  {
    gi_put_error("Illegal range argument");
    return;
  } 

  /* get unit_image for display */
  if ((unit_image=gi_get_image(cmd_args[CMD_IMAGE]))==NULL)
    return;

  /* get x and y position of first unit to put */
  if (gi_conv_coord(cmd_args[CMD_POSX],&start_x)==NULL 
  || gi_conv_coord(cmd_args[CMD_POSY],&start_y)==NULL)
  {
    gi_put_error("Illegal coordinates in show command");
    return(ERROR);
  } 

  /* get x and y spacing (allow negative spacings "backwards" */
  if (get_space(cmd_args[CMD_SPAX],&x_space)==ERROR 
  || get_space(cmd_args[CMD_SPAY],&y_space)==ERROR)
  {
    gi_put_error("Illegal spacing parameter");
    return(ERROR);
  }
 
  /* get number of items in each row */
  if ((num_cols=get_columns(cmd_args[CMD_CNUM])) <= 0)
  {
    gi_put_error("Illegal number columns parameter");
    return(ERROR);
  }

  /* get link target if appropriate */
  if (unit_what==LINKIN | unit_what==LINKOUT)
  {
    if ((unit_target=gi_get_target(cmd_args[CMD_TARGET]))==ERROR)
    {
      gi_put_error("Need to designate the target unit");
      return(ERROR);
    }
    unit_site = gi_get_site(cmd_args[CMD_TARGET]);
  }
         
  /* get starting unit number */
  /* note: cannot do this before get_target is called
   * or else the save_ui value in sim_get_unit will
   * be corrupted
   */

  if ((unit_index=get_unit(cmd_args[CMD_WHO],GET_FIRST))<0) 
  {
    gi_put_error(err_string);
    return(ERROR);
  }  

  /* made it this far; parameters are OK */

  num_cols_left = num_cols;
  num_units_left = num_units;
  loc_x = start_x;
  loc_y = start_y;
  done = FALSE;
  while (!done)
  {
    /* display a unit at the correct position */
    ptr = gi_remember_grobj(loc_x,loc_y,
                        unit_image,
                        unit_index,
                        unit_what,
                        unit_lrange,
                        unit_hrange,
                        unit_target,
                        unit_site);

    if (aux_flag) ptr->flag |= AUX;
      
    /* determine the next position to place an object */
    if (--num_cols_left)
    {
      loc_x += unit_image->size_x + x_space;
    }
    else
    {
      loc_x = start_x;
      loc_y += unit_image->size_y + y_space;
      num_cols_left = num_cols;
     }

    /* now decide if we're finished yet */
    if ((num_units>0) && (--num_units_left<=0)) 
      done = TRUE;
    else 
    {
      unit_index=get_unit(cmd_args[CMD_WHO],GET_NEXT); 
      if (unit_index<0)
        done = TRUE;
    }  /* get next unit to display */
  }  /* of while loop */

  /* set the reshow flag appropriately */
  gi_reshow_flag |= (RESHOW_NEEDED+SOME_VALS_NEEDED+SOME_LINKS_NEEDED);

  /* move the marker to the last position */
  gi_move_marker(loc_x,loc_y);

  gi_put_message("Show command successful");
  return(OK);
}

