% decl.pl
% declarations that use type information

:- use_module( library(flags) ).
:- use_module( library(wlists), [ wappend/3 ] ).


% PHONOLOGY/2
% phonology(-Node,+Phon)
% phonology(+Node,?Phon
% defines the phonology Phon of node Node
%
% is only neccesary if you do generation, or if your
% grammar is sign-based, and you want the string attribute
% of the top-goal to be instantiated. Note that phonology
% is called _before_ parser starts as well, hence don't
% define it as `yield'
%
% is used to 
% 1. print result of generation
% 2. instantiate phon attribute during parsing (in case of
%    sign-based grammar)
%

phonology(_,_).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%% data structures %%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% some syntactic sugar

:- op(403,fx,'\').      % not right
:- op(403,fx,'/').      % not left
:- op(403,fx,'\\').     % left
:- op(403,fx,'//').     % right
:- op(402,xfy,'#').     % Cat # SubcatList
:- op(200,fx,^).        % category

:- op(800,xfy,'=/=').   % dif

X =/= Y :-
	{ feature:eval_path(X,Xval),
	  dif(Xval,Yval),
	  feature:eval_path(Y,Yval) 
        }.

eval_a_call(_ =/= _, no).   % call it upon partial evaluation, but
                            % do not touch the terms..

user_eval({Cat#Sc},C) :-
	{ C:sc <=> Sc,
          C <=> Cat }.

user_eval({\Cat},C) :-
	{ C:dir => ~right,
	  C <=> Cat }.

user_eval({/Cat},C) :-
	{ C:dir => ~left,
	  C <=> Cat }.

user_eval({\\Cat},C) :-
	{ C:dir => left,
	  C <=> Cat }.

user_eval({//Cat},C) :-
	{ C:dir => right,
	  C <=> Cat }.

user_eval({[H|T]},C) :-
	{ C:h <=> H,
	  C:t <=> T }.

user_eval({^Cat},C) :-
	{ C:cat => Cat }.

user_defined_eval({/_}).
user_defined_eval({\_}).
user_defined_eval({//_}).
user_defined_eval({\\_}).
user_defined_eval({_#_}).
user_defined_eval([_|_]).
user_defined_eval({^_}).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% HFC/2 for head-driven processing %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

hfc(Cat0,Cat) :-
	Cat0:inv   <=> Cat:inv,
	Cat0:cat   <=> Cat:cat,
	Cat0:sem   <=> Cat:sem,
	Cat0:subj  <=> Cat:subj.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% IGNORE_SEMANTICS/2 for packing %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% ignore_semantics(Node,NodeWithoutSem).

ignore_semantics(Sa,Sb) :-
	unify_except(Sa,Sb,sem).

%%%%%%%%%%%%%%%%%%%
%%% SEMANTICS/2 %%%
%%%%%%%%%%%%%%%%%%%

% semantics(Node,Sem)
% defines the semantics Sem of node Node. Is used: 
% 
% 1. in order to determine the semantic-head of a rule 
%    (for head-driven generation).
% 2. to instantiate the semantics of a node, at the
%    start of generation. 
% 3. to print the result of parsing.

semantics(Sign,Sign:sem).

%%%%%%%%%%%
%% TOP/2 %%
%%%%%%%%%%%

% top(Name,Node)
% Name is atom
% Node is node
% if flag(top_features,Name), then top(Name,Node) is called before
% parsing or generation of Node.

top(complete, _ # []).

top(s,S # []) :-
	S:cat:vform => fin,
	S:inv => - ,
	S:f_slash => [],
	S:slash => [],
	S:rel => [],
	S:extra => [].

top(sbar,S & ^vp # []) :-
	S:cat:vform => dat,
	S:rel => [],
	S:slash => [],
	S:extra => [].

top(main,S # []) :-
	S:rel => [],
	S:slash => [],
	S:extra => [],
	S:inv => +,
	S:f_slash <=> [_],
	S:cat:vform => fin.

top(main_sj,S # []) :-
	S:rel => [],
	S:slash => [],
	S:extra => [],
	S:inv => +,
	S:f_slash <=> [S:subj],     % main sentences with subject topicalized
	S:cat:vform => fin.

top(top,Nslash # []) :-
	Nslash:slash => [],
	Nslash:rel => [],
	Nslash:extra => [].

top(inv,Inv # []) :-
	Inv:rel => [],
	Inv:slash => [],
	Inv:extra => [],
	Inv:inv => + .

top(tp,Cat) :-
	Cat:slash => [],
	Cat:extra => [].

top(np,Cat) :-
	Cat:cat => np,
	Cat:sc => [],
	Cat:slash => [],
	Cat:extra => [].

top(vp,Cat) :-
	Cat:cat => vp,
	Cat:sc => [],
	Cat:slash <=> [_Subj],
	Cat:extra => [],
	Cat:inv => + .

%%%%%%%%%%%%%%%%%%%%%
%%%% deriv_tree %%%%%
%%%%%%%%%%%%%%%%%%%%%

deriv_tree(X,Tree) :-
	X:tree ==> Tree.

%%%%%%%%%%%%%%%%
% PRETTY PRINT %
%%%%%%%%%%%%%%%%

:- del_graphic(user:gp,user:gl,user:gd).

:- use_module( [ library(prettyvars),
                 library(lists) ] ).

gp(q,Rule,Sem) :-
	copy_term(Rule:sem,Sem),
	prettyvars(Sem).

gd(q,1,Sem,Sem:arg1).
gd(q,2,Sem,Sem:arg2).
gd(q,3,Sem,Sem:arg3).
gd(q,P,Sem,Mod) :-
	Sem => c0,
	nth(P,Sem:restr,Mod).

gd(q,P,Sem,Mod) :-
	Sem => c1,
	nth(P0,Sem:restr,Mod),
	P is P0 + 1.

gd(q,P,Sem,Mod) :-
	Sem => c2,
	nth(P0,Sem:restr,Mod),
	P is P0 + 2.

gd(q,P,Sem,Mod) :-
	Sem => c3,
	nth(P0,Sem:restr,Mod),
	P is P0 + 3.

gl(q,Sem,Label) :-
	find_index(Sem:index,Label0),
	find_label(Sem,Label1),
	cola(Label0,Label1,Label).

find_index('$VAR'('_'),nolabel) :- 
	!.
find_index('$VAR'(L),'$VAR'(L)).
find_index(Term,Label) :-
	functor(Term,_,A),
	arg(A,Term,ReentPos),
	find_index(ReentPos,Label).

find_label(`ix,nolabel).
find_label(S,Label) :-
	S:fun => prime(Label).

cola(nolabel,nolabel,'').
cola(nolabel,L,L).
cola(L,nolabel,L).
cola(L0,L1,i(L0,L1)).

gp(d,X,Tree) :-
	X:tree <=> Tree.

gd(d,No,tree(_,_,Ds),D) :-
	nth(No,Ds,D).

gl(d,tree(L0,_,_),L) :-
%%	find_type(L0,[L|_]),!.
        pr(L0,L).
%% gl(d,tree(L,_,_),L).

:- add_graphic(user:gp,user:gl,user:gd).

:- usr_cmd([alias,s,s,d]).
:- usr_cmd([alias,g,s,d,gm]).
:- usr_cmd([alias,q,s,q]).
:- usr_cmd([alias,f,s,q,gm]).

%%%%%%%%%%%%%%%%%%
%% EXTERN_SEM/2 %%
%%%%%%%%%%%%%%%%%%

% extern_sem(ExtSem,IntSem)
%
% is like a reversible `portray' of semantics 
%
% ExtSem should be a list of prolog terms
% it is read in using `sequence_of_terms' => command.pl
% and written using `write_list3' => message.pl
%

extern_sem([Term],Fs) :-
	nonvar(Fs),!,        % from fs to sem
	copy_term(Fs,Fs0),   % unwise to affect fs, we dont know where
	                     % this predicate will be useful
	prettyvars(Fs0),
	extern_sem0(Term,Fs0).

extern_sem([Term0],Fs) :-     
	% var(Fs),           % from sem to fs
	un_prettyvars(Term0,Term),
	extern_sem0(Term,Fs).


extern_sem0(Index,Sem) :-
	var(Index),
	var(Sem),!,
	{ Sem => ix,
	  Sem:index <=> Index }.

extern_sem0( i(Index,Short),Sem) :-
	var(Sem),!,
	{ Sem:index <=> Index },
	extern_sem00(Short,Sem).

extern_sem0(Index,Sem) :-
	var(Index),
	nonvar(Sem),
	{ Sem => ix,
	  Sem:index <=> Ind },
	find_index(Ind,Ind1),
	( Ind1 == nolabel -> true ; Ind1 = Index ), !.

extern_sem0(i(Index,Short),Sem) :-
	nonvar(Sem),
	{ Sem:index <=> Ind },
	find_index(Ind,Index),
	\+ Index == nolabel,
	!,
	extern_sem00(Short,Sem).

extern_sem0(Term,Sem) :-
	extern_sem00(Term,Sem).

extern_sem00(Term,Sem) :-
	extract_restr(Term0,Term,Sem:restr),
	extern_sem1(Term0,Sem).

extern_sem1(Term0,Sem) :-
	var(Term0),
	!,
	Sem:fun <=> FunT,
	find_type(FunT,[prime(Fun)|_]),
	find_args(Sem,Args),
	Term0 =.. [Fun|Args].

extern_sem1(Term0,Sem) :-
	Term0 =.. [Fun|Args],
	Sem:fun => prime(Fun),
	find_args(Sem,Args).

find_args(Sem,[]) :-
	Sem => c0.
find_args(Sem,[A1]) :-
	Sem:arg1 <=> A1a,
	extern_sem0(A1,A1a),
	Sem => c1.
find_args(Sem,[A1,A2]) :-
	Sem => c2,
	Sem:arg1 <=> A1a,
	extern_sem0(A1,A1a),
	Sem:arg2 <=> A2a,
	extern_sem0(A2,A2a).
find_args(Sem,[A1,A2,A3]) :-
	Sem => c3,
	Sem:arg1 <=> A1a,
	extern_sem0(A1,A1a),
	Sem:arg2 <=> A2a,
	extern_sem0(A2,A2a),
	Sem:arg3 <=> A3a,
	extern_sem0(A3,A3a).

extract_restr(A,T,[Hr|Tr]):-
	wappend(Tail0,[A],T),
	extern_sem_l(Tail0,[Hr|Tr]).
extract_restr(A,A,[]).

:- block extern_sem_l(-,-).
extern_sem_l([],[]).
extern_sem_l([H|T],[Sem|SemT]) :-
	extern_sem0(H,Sem),
	extern_sem_l(T,SemT).

%%%%%%%%%%%%%%%%%
%%%%% flags %%%%%
%%%%%%%%%%%%%%%%%

gflags :-
	wr_flag(add_mod),
	wr_flag(push_to_slash),
	wr_flag(push_to_extra).

flags_on :-
	flag(add_mod,_,on),
	flag(push_to_slash,_,on),
	flag(push_to_extra,_,on).

flags_off :-
	flag(add_mod,_,off),
	flag(push_to_slash,_,off),
	flag(push_to_extra,_,off).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% PORTRAY of signs, etc. %%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

:- del_portray(pr).

gm_show(A,B) :-
	pr(A,B).

pr(Sign):-
	pr(Sign,Term),
	write(Term).

pr(Sign # Args,Term) :-
	find_type(Sign:cat,[C0|_]),
	combine_pt(Sign:dir,C0,Term0),
	combine_sl(Sign:slash,Term0,Term1),
	combine_ex(Sign:extra,Term1,Term2),
	combine_sc(Args,Term2,Term).

combine_sc(Var,T0,T) :-
	var(Var),!,T0=T.
combine_sc([],T,T).
combine_sc([H|T],X0,{ X0 # Slashed }) :-
	pr_list([H|T],Slashed).

combine_sl(Var,T0,T) :-
	var(Var),!,T0=T.
combine_sl([],T,T).
combine_sl([H|T],X0,X0+Slashed) :-
	pr_list([H|T],Slashed).

combine_ex(Var,T0,T) :-
	var(Var),!,T0=T.
combine_ex([],T,T).
combine_ex([H|T],X0,X0-Extra) :-
	pr_list([H|T],Extra).

combine_pt(`topic,C0,C0).   % default no mark
combine_pt(`extra,C0,C0).
combine_pt(`left,C0,{\C0}).
combine_pt(`right,C0,{/C0}).

pr_list(Var,V):-
	var(Var),!,
	Var = V.

pr_list([],[]).
pr_list([H|T],[NH|NT]):-
	pr(H,NH),
	pr_list(T,NT).
	
:- add_portray(pr).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% SIZE/2 used for generator bug_c.pl %%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

size(Term,N) :-
	size(Term,0,N).

size(T,N,N) :-
	T => ix.
size(T,N0,N) :-
	T => cx,
	T:restr <=> Restr,
	T:fun <=> Fun,
	size(Restr,N0,N1),
	size_fun(Fun,N1,N2),
	size_args(T,N2,N).

size([],N,N).
size([H|T],N0,N) :-
	size(H,N0,N1),
	size(T,N1,N).

size_fun(_,N0,N) :-
	N is N0 + 1.

size_args(T,N,N) :-
	T => c0.
size_args(T,N0,N) :-
	T:arg1 <=> A1,
	size(A1,N0,N1),
	size_args2(T,N1,N).

size_args2(T,N,N) :-
	T => c1.
size_args2(T,N0,N) :-
	T:arg2 <=> A2,
	size(A2,N0,N1),
	size_args3(T,N1,N).

size_args3(T,N,N) :-
	T => c2.
size_args3(T,N0,N) :-
	T:arg3 <=> A3,
	size(A3,N0,N).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% GET_ARGS for the generator bug_dd.pl %%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

get_args(Node,Args) :-
	general(Node:args,Args0),
	restrictors(Node:sem,Args1),
	wappend(Args0,Args1,Args).

restrictors(`ix,[]).
restrictors(S,R) :-
	restr(S:restr,R).

restr([],[]).
restr([H:sem|T0],[H|T]) :-
	restr(T0,T).

:- block general(-,-).
general([],[]).
general([H0|T0],[H|T]) :-
	unify_except_l(H0,H,{[lex, rel]}),   % because that's what is changed by lexical rules
	general(T0,T).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% tree-like pretty printing of lexicon set-up %%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% to view the lexicon as if it were a real inheritance
%% hierarchy.

:- use_module( library(call_tree) ).

call_default(lexical).

call_leaf(a_word(C),L) :-
	C:lex:stem <=> L.

call_leaf(verb_stem(C),L) :-
	C:lex:stem <=> L.

call_leaf(a_noun(C),L) :-
	C:lex:stem <=> L.

call_leaf(r(Rule),L) :-
	find_type(Rule,[L|_]).

call_clause(A,B) :-
	user_clause(A,B).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% pretty printing of trees            %%%%
%%%% whose labels are feature structures %%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

tree_label(FS) :-
%%	lexic(FS,Cat),
	shorten_label(FS,Cat),
	{latex:print_it_fs(fs(Cat))}.

% there are different versions of 
% shorten/2 from which you can choose with
% flag(shorten).
% Note that sometimes latex/dctree fails if the
% wrong nodeskip size has been chosen. Therefore
% if you have very large feature structures, choose
% a larger value for nodeskip with flag(nodeskip)
%
% off : all features 
% short : cat, inv, sc, slash, extra, rel, f_slash, f_extra, f_rel 
%         but empty list values are suppressed
% Att: feature Att 
% if shorten=cat -> nodeskip 100
% if not shortened -> nodeskip +- 1000
% if shortended -> nodeskip +- 400

shorten_label([H|T],Cat) :-
	shorten_l([H|T],Cat).

shorten_label(FS,Cat) :-
	FS => sign,
	flag(shorten,Flag),
	shorten_label(Flag,FS,Cat).

shorten_label(sc_cat,FS,Cat) :-
	FS:cat <=> Cat:cat,
	shorten_l(FS:sc,Cat:sc).

shorten_label(Att,FS,Part) :-
	feature:e(Att,_,_),
	FS:Att <=> Part:Att.

shorten_label(short,FS,Cat) :-
	simple_features(Simple),
	unify_simple(Simple,FS,Cat),
	list_features(List),
	unify_list(List,FS,Cat),
	change_features(Change),
	unify_change(Change,FS,Cat),
	sign_features(Sign),
	unify_sign(Sign,FS,Cat).

simple_features([sem]).
list_features([sc,slash,extra,rel,f_slash,f_extra,f_rel]).
change_features([inv,dir,cat]).
sign_features([subj]).

unify_change_att(cat,Type,FType) :-
	find_type(Type,[FType0|_]),
	(FType0 == top -> true ; FType0 = FType ). 
unify_change_att(lex,_ => lexical, +).
unify_change_att(lex,_ => phrasal, -).

unify_change_att(inv,` -,_).
unify_change_att(inv,` +,` +).
unify_change_att(dir,T,_) :-
	T => topic.
unify_change_att(dir,L,left) :-
	L => left.
unify_change_att(dir,R,right) :-
	R => right.
unify_change_att(dir,E,_) :-
	E => extra.

unify_change([],_,_).
unify_change([Att|T],F0,F) :-
	{ F0:Att <=> Val0,
          F:Att <=> Val },
	unify_change_att(Att,Val0,Val),
	unify_change(T,F0,F).


unify_simple([],_,_).
unify_simple([Att|T],F0,F) :-
	{ F0:Att <=> F:Att },
	unify_simple(T,F0,F).

unify_list([],_,_).
unify_list([Att|T],F0,F) :-
	{ F0:Att <=> A0,
	  F:Att <=> A },
        shorten_l(A0,A),
	unify_list(T,F0,F).

unify_sign([],_,_).
unify_sign([Att|T],F0,F) :-
	{ F0:Att <=> A0,
	  F:Att <=> A,
          latex:shorten_label(A0,A) },
	unify_sign(T,F0,F).



shorten_l(Var,V) :-
	var(Var),!,Var=V.
shorten_l([],_).      % if empty - dont mention it
shorten_l([H0|T0],[H|T]) :-
	latex:shorten_label(H0,H),
	shorten_l1(T0,T).

shorten_l1(Var,V) :-
	var(Var),!,Var=V.
shorten_l1([],[]).
shorten_l1([H0|T0],[H|T]) :-
	latex:shorten_label(H0,H),
	shorten_l1(T0,T).


:- initialize_flag(shorten,short).
:- initialize_flag(nodeskip,400).
:- initialize_flag(generator,bug_dd).
:- initialize_flag(parser,rcp3).

:- del_command(stem_cmd).

stem_cmd(show:sh(Type,Output,Thing)) -->
	[stem],
	show:type(Type),
	show:output(Output),
	stems(Thing,[]).

stems(W0,W) -->
	stem(W0,W1),
	stems(W1,W).

stems(W,W) --> [].

stem(Stems0,Stems) -->
	[Name],
	{{ all_verb_stems(Name,Stems1),
	   append(Stems1,Stems,Stems0) }}.

all_verb_stems(Name,Stems) :-
	FS:lex:stem ==> Name,
	findall(value(Name,verb_stem(FS)), verb_stem(FS), Stems).

:- add_command(stem_cmd).

user_clause(H,B) :-
	decl_clause(H,B).

user_clause(H,B) :-
	templates_clause(H,B).

user_clause(H,B) :-
	rules_clause(H,B).

user_clause(H,B) :-
	lexicon_clause(H,B).

user_clause(H,B) :-
	inflection_clause(H,B).

%%% TK interface

ct(tk,F) :-
	call_tree_bu_tk(F).

ct(latex,F) :-
	call_tree_bu_latex(F).

ct(prolog,F) :-
	call_tree_bu(F).


update_unary_preds :-
	write('Updating lexical predicates...'),ttyflush,
	tcl_eval('set unary_preds {}'),
	( setof(X,H^B^(user_clause(H,B),functor(H,X,1)),All),
	  ( member(F,All),
	    send_a_unary_pred(F),
	    fail
	  ; true 
          )
        ; true),
	write('Done.'),nl.

send_a_unary_pred(F) :-
	concat_all([set,' unary_preds',' [linsert $unary_preds 0 ',F,' ]'],Atom),
	tcl_eval(Atom).
