;;|=========================================================================|
;;|                         COPYRIGHT NOTICE                                |
;;|                                                                         |
;;|             Copyright 1990, 1991, 1992, 1993, 1994 Mark Tarver          |
;;|                                                                         |
;;|        Permission to use, copy, and distribute this software and        |
;;| its documentation for any purpose is hereby granted providing           |
;;| any such use, copying and distribution is not done                      |
;;| for money, securities or any other pecuniary benefit and that both      |
;;| the above copyright and this permission notice appear in all copies     |
;;| and in the supporting documentation.  Any modification of the software  |
;;| or documentation should be accompanied by the name of the author of the |
;;| modification, and Mark Tarver must be formally notified                 |
;;| of this modification before distributing the software.                  |
;;|                                                                         |
;;|       Any commercial use of this software or use of the names "SEQUEL", |
;;| or "Mark Tarver" in connection with any version, modified or            |
;;| unmodified, of this software, through publicity or advertising,         |
;;| requires written permission.  Mark Tarver makes no                      |
;;| representation about the suitability of this software for any purpose.  |
;;| SEQUEL is provided "as is" without express or implied warranty.         |
;;|                                                                         |
;;|       Mark Tarver disclaims all warranties with regard to               |
;;| this software, including all implied warranties of merchantability and  |
;;| fitness. In no event shall Mark Tarver be liable for any                |
;;| special, indirect or consequential damages or any damages whatsoever    |
;;| resulting from loss of use, data or profits, whether in an action of    |
;;| contract, negligence or other tortious action, arising out of or in     |
;;| connection with the use or performance of this software.                |
;;|                                                                         |
;;|=========================================================================|

(in-package :sequel)

(defun create-framework
       (fp1 fp2 fp3 fp4 fp5 fp6 fp7 fp8 fp9 fp10)
       (cond ((and (symbolp fp1)
                   (stringp fp2)
                   (stringp fp3)
                   (stringp fp4)
                   (consp fp5)
                   (every 'stringp fp5)
                   (consp fp6)
                   (every 'stringp fp6)
                   (listp fp7)
                   (every 'stringp fp7)
                   (every 'listp
                          (list fp8 fp9 fp10))
                   (every 'stringp
                          (append fp8 fp9 fp10)))
              (prog2
                (set fp1
                     (make-framework ':print_name
                                     fp2
                                     ':author
                                     fp3
                                     ':date
                                     fp4
                                     ':syntax
                                     fp5
                                     ':theories
                                     fp6
                                     ':rewrites
                                     fp7
                                     ':tactics0
                                     fp8
                                     ':tactics1
                                     fp9
                                     ':tactics2
                                     fp10))
                (list 'framework fp1 'created)))
             (t
              (raise
                (format nil
                        "code 22: framework ~A improperly specified"
                        fp1)))))

(defun call-framework
       (fp1)
       (prog (framework)
             (setq *autotypes* (union '(proof proof-object sequent t-expr)
                                      *autotypes*))
             (setq *definitions* nil)
             (setq framework (eval fp1))
             (set '*framework*
                  (framework-print_name framework))
             (set '*author*
                  (framework-author framework))
             (set '*date*
                  (framework-date framework))
             (set '*redefinition-action* ':quiet)
             (eval (list 'defun
                         'external-syntax-out
                         (list 'a)
                         'a))
             (eval (list 'defun
                         'external-syntax-in
                         (list 'a)
                         'a))
             (set '*redefinition-action* ':warn)
             (initialise-framework)
             (mapcar 'consult
                     (framework-syntax framework))
             (setq *type-check* t)
             (mapcar 'consult
                     (framework-theories framework))
             (mapcar 'consult
                     (framework-rewrites framework))
             (setq *type-check* nil)
             (mapcar 'consult
                     (framework-tactics0 framework))
             (setq *type-check* t)
             (mapcar 'consult
                     (framework-tactics1 framework))
             (mapcar 'consult
                     (framework-tactics2 framework))
             (return (list 'framework fp1 'loaded))))

(defun initialise-framework
       nil
       (set '*theories* nil)
       (set '*rewrites* nil)
       (set '*tactics0* nil)
       (set '*tactics1* nil)
       (set '*tactics2* nil))

(defun save-framework
       (fp1 fp2)
       (prog (framework funcs)
             (format t
                     "Saving framework ~A to file ~A. Please wait.~%"
                     fp1
                     fp2)
             (setq framework (eval fp1))
             (setq stream (open fp2
                                :direction
                                :output
                                :if-exists
                                :overwrite
                                :if-does-not-exist
                                :create))
             (setq funcs (set-difference *autotypes*
                                         (list 'proof
                                               'proof-object
                                               't-expr
                                               'sequent)))
             (setq funcs (append
                           (extr-funcs
                             (mapcan 'read-file
                                     (append (framework-syntax framework)
                                             (framework-rewrites framework)
                                             (framework-tactics0 framework)
                                             (framework-tactics1 framework)
                                             (framework-tactics2 framework))))
                           funcs))
             (setq funcs (append funcs
                                 (mapcan 'add-auto-rewrite
                                         (set-difference *autotypes*
                                                         (list 'proof
                                                               'proof-object
                                                               't-expr
                                                               'sequent)))))
             (setq funcs (union funcs *tactics2*))
             (setq funcs (mapcan 'add-&-funcs funcs))
             (setq funcs (append
                           (list 'external-syntax-out
                                 'external-syntax-in)
                           funcs))
             (setq funcs (remove-if-not 'fboundp funcs))
             (mapcar 'ppstream
                     (mapcar 'ppdef funcs))
             (ppstream
               (list 'defun
                     'startup-framework
                     nil
                     (list 'restore-sigs
                           (list 'quote
                                 (mapcan 'listsig funcs)))
                     (list 'restore-axioms
                           (list 'quote
                                 (mapcar 'get-theory
                                         *theories*)))
                     (list 'restore-globals
                           (list 'quote
                                 (get-globals *globals*)))
                     (list 'setq
                           '*autotypes*
                           (list 'quote *autotypes*))
                     (list 'setq
                           '*tactics0*
                           (list 'quote *tactics0*))
                     (list 'setq
                           '*tactics1*
                           (list 'quote *tactics1*))
                     (list 'setq
                           '*tactics2*
                           (list 'quote *tactics2*))
                     (list 'setq
                           '*rewrites*
                           (list 'quote *rewrites*))
                     (list 'setq
                           '*theories*
                           (list 'quote *theories*))
                     (list 'setq
                           '*framework*
                           *framework*)
                     (list 'setq '*author* *author*)
                     (list 'setq '*date* *date*)
                     (list 'setq
                           '*primitives*
                           (list 'quote *primitives*))
                     (list 'restore-primitives
                           (list 'quote
                                 (get-primitives *primitives*)))
                     (if (boundp '*constants*)
                         (list 'setq
                               '*constants*
                               (list 'quote *constants*)))
                     (save-documentation)
                     (list 'restore-framework
                           (list 'quote fp1)
                           (list 'quote
                                 (list (framework-print_name framework)
                                       (framework-author framework)
                                       (framework-date framework)
                                       (framework-syntax framework)
                                       (framework-theories framework)
                                       (framework-rewrites framework)
                                       (framework-tactics0 framework)
                                       (framework-tactics1 framework)
                                       (framework-tactics2 framework))))))
             (close stream)
             (return (list fp1 'saved))))

(defun save-documentation
       nil
       (list 'restore-documentation
             (list 'quote
                   (mapcan 'sd1
                           (append *tactics0*
                                   *tactics1*
                                   *tactics2*
                                   *rewrites*
                                   *theories*)))))

(defun restore-documentation
       (fp1)
       (cond ((null fp1) nil)
             ((and (consp fp1)
                   (consp (car fp1))
                   (= (length (car fp1)) 2))
              (put-prop (caar fp1)
                        'doc
                        (cadar fp1))
              (restore-documentation (cdr fp1)))
             (t
              (raise "code 13: No Patterns have Fired in restore-documentation"))))

(defun sd1
       (fp1)
       (cond ((get-prop fp1 'doc nil)
              (list (list fp1
                          (get-prop fp1 'doc nil))))
             (t nil)))

(defun get-theory
       (fp1)
       (list fp1
             'inter
             (mapcar 'ppxt
                     (get-prop fp1 'inter nil))))

(defun get-primitives
       (fp1)
       (mapcar 'get-primitive fp1))

(defun get-primitive
       (fp1)
       (list fp1
             'recognisor
             (get-prop fp1 'recognisor nil)))

(defun restore-primitives
       (fp1)
       (mapcar 'restore-primitive fp1))

(defun restore-primitive
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq 'recognisor (cadr fp1)))
              (put-prop (car fp1)
                        'recognisor
                        (caddr fp1)))
             (t (raise "code 13: No Patterns have Fired in restore-primitive"))))

(defun listsig
       (fp1)
       (cond ((signature fp1) (list (list fp1 (signature fp1))))
             (t nil)))

(defun add-&-funcs
       (fp1)
       (cond ((fboundp (concat fp1 '&)) (list fp1
                                              (concat fp1 '&)))
             (t (list fp1))))

(defun add-auto-rewrite
       (fp1)
       (cond ((fboundp (concat fp1 'rewrite)) (list (concat fp1
                                                            'rewrite)))
             (t nil)))

(defun ppdef
       (fp1)
       (ppdef1 (ppx fp1)))

(defun ppdef1 (fp1)
  (cond ((and (consp fp1)
              (= (length fp1) 4)
              (eq 'named-lambda (car fp1))
              (and (consp (caddr (cdr fp1)))
                   (= (length (caddr (cdr fp1)))
                      3))
              (eq 'block
                  (caadr (cddr fp1))))
         (list 'defun
               (cadr fp1)
               (caddr fp1)
               (caddr (caddr (cdr fp1)))))
        ((and (consp fp1)
              (= (length fp1) 4)
              (eq 'named-lambda (car fp1))
              (and (consp (caddr (cdr fp1)))
                   (> (length (caddr (cdr fp1)))
                      1))
              (eq 'block
                  (caadr (cddr fp1))))
         (cons 'defun
               (cons (cadr fp1)
                     (cons (caddr fp1)
                           (cddar (cdddr fp1))))))
        ((and (consp fp1)
              (= (length fp1) 5)
              (eq 'named-lambda (car fp1))
              (and (consp (caddr (cddr fp1)))
                   (= (length (caddr (cddr fp1)))
                      3))
              (eq 'block
                  (caadr (cdddr fp1))))
         (list 'defun
               (cadr fp1)
               (caddr fp1)
               (caddr (cdr fp1))
               (caddr (caddr (cddr fp1)))))
        ((and (consp fp1)
              (= (length fp1) 5)
              (eq 'named-lambda (car fp1))
              (and (consp (caddr (cddr fp1)))
                   (> (length (caddr (cddr fp1)))
                      1))
              (eq 'block
                  (caadr (cdddr fp1))))
         (cons 'defun
               (cons (cadr fp1)
                     (cons (caddr fp1)
                           (cons (caddr (cdr fp1))
                                 (cddar (cdddr (cdr fp1))))))))
        (t (raise "code 13: No Patterns have Fired in ppdef1"))))

(defun extr-funcs
       (fp1)
       (cond ((null fp1) nil)
             ((and (consp fp1)
                   (consp (car fp1))
                   (> (length (car fp1)) 1)
                   (eq 'deftactic (caar fp1)))
              (cons (cadar fp1)
                    (extr-funcs (cdr fp1))))
             ((and (consp fp1)
                   (consp (car fp1))
                   (> (length (car fp1)) 1)
                   (eq 'define (caar fp1)))
              (cons (cadar fp1)
                    (extr-funcs (cdr fp1))))
             ((and (consp fp1)
                   (consp (car fp1))
                   (> (length (car fp1)) 1)
                   (eq 'defrew (caar fp1)))
              (cons (cadar fp1)
                    (extr-funcs (cdr fp1))))
             ((and (consp fp1)
                   (consp (car fp1))
                   (> (length (car fp1)) 1)
                   (eq 'deflog (caar fp1)))
              (cons (cadar fp1)
                    (extr-funcs (cdr fp1))))
             ((and (consp fp1)
                   (consp (car fp1))
                   (> (length (car fp1)) 1)
                   (eq 'deflog (caar fp1)))
              (cons (cadar fp1)
                    (extr-funcs (cdr fp1))))
             ((and (consp fp1)
                   (consp (car fp1))
                   (> (length (car fp1)) 1)
                   (eq 'deflog (caar fp1)))
              (cons (cadar fp1)
                    (extr-funcs (cdr fp1))))
             ((and (consp fp1)
                   (consp (car fp1))
                   (eq 'mutual (caar fp1)))
              (extr-funcs (append (cdar fp1) 
                          (cdr fp1))))
             ((consp fp1) (extr-funcs (cdr fp1)))
             (t (raise "code 13: No Patterns have Fired in extr-funcs"))))

(defun ppstream
       (fp1)
       (pprint fp1 stream)
       (terpri stream))

(defun fdg (fp1) (fdg1 fp1 nil nil))

(defun fdg1
       (fp1 fp2 fp3)
       (cond ((null fp1) (reverse fp2))
             ((and (consp fp1)
                   (member (car fp1) fp3))
              (fdg1 (cdr fp1) fp2 fp3))
             ((and (consp fp1)
                   (symbolp (car fp1))
                   (fboundp (car fp1))
                   (consp (ppx (car fp1))))
              (fdg1 (append (fdg2 (ppx (car fp1)))
                            (cdr fp1))
                    (cons (car fp1) fp2)
                    (cons (car fp1) fp3)))
             ((consp fp1) (fdg1 (cdr fp1) fp2 fp3))
             (t (raise "code 13: No Patterns have Fired in fdg1"))))

(defun fdg2
       (fp1)
       (cond ((consp fp1) (remove-duplicates (flatten (cdr fp1))))
             (t nil)))

(defun p-axioms-to
       (fp1)
       (ppstream
         (cons 'defun
               (cons 'startup-framework
                     (cons nil (mapcar 'pat1 fp1))))))

(defun pat1
       (fp1)
       (list 'prog2
             (list 'splist fp1 (plist fp1))
             (list 'compile
                   (list 'plist fp1)
                   nil)))

(defun load-saved-framework
       (fp1)
       (load fp1)
       (startup-framework)
       (list 'framework 'loaded 'from fp1))

(defun load-compiled-framework
       (fp1)
       (load fp1)
       (startup-framework)
       (mapcar 'lisp-compile-axioms *theories*)
       (list 'compiled 'framework 'loaded 'from fp1))

(defun lisp-compile-axioms
       (fp1)
       (put-prop fp1
                 'inter
                 (mapcar #'(lambda (x)
                                   (compile nil x))
                         (get-prop fp1 'inter nil))))

(defun ppxt
       (fp1)
       (read-from-string (subseq (write-to-string fp1)
                                 23)))

(defun get-globals
       (fp1)
       (mapcar 'get-global fp1))

(defun get-global
       (fp1)
       (list fp1
             (get-prop fp1 'global nil)
             (get-prop fp1 'type nil)
             (if (boundp fp1)
                 (eval fp1)
                 '-no-value-)))

(defun restore-globals
       (fp1)
       (mapcar 'restore-global fp1))

(defun restore-global
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 4))
              (global1 (list (car fp1)
                             (cadr fp1)
                             (caddr fp1)))
              (if (not (eq (caddr (cdr fp1))
                           '-no-value-))
                  (set (car fp1)
                       (caddr (cdr fp1)))))
             (t (raise "code 13: No Patterns have Fired in restore-global"))))

(defun restore-sigs
       (fp1)
       (mapcar 'restore-sig fp1))

(defun restore-sig
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 2))
              (setf (gethash (car fp1) *sfht*)
                    (cadr fp1)))
             (t (raise "code 13: No Patterns have Fired in restore-sig"))))

(defun restore-axioms
       (fp1)
       (mapcar 'restore-axioms1 fp1))

(defun restore-axioms1
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3))
              (put-prop (car fp1)
                        (cadr fp1)
                        (caddr fp1)))
             (t (raise "code 13: No Patterns have Fired in restore-axioms1"))))

(defun restore-framework
       (fp1 fp2)
       (cond ((and (consp fp2)
                   (= (length fp2) 9))
              (set fp1
                   (make-framework ':print_name
                                   (car fp2)
                                   ':author
                                   (cadr fp2)
                                   ':date
                                   (caddr fp2)
                                   ':syntax
                                   (caddr (cdr fp2))
                                   ':theories
                                   (caddr (cddr fp2))
                                   ':rewrites
                                   (caddr (cdddr fp2))
                                   ':tactics0
                                   (caddr (cdddr (cdr fp2)))
                                   ':tactics1
                                   (caddr (cdddr (cddr fp2)))
                                   ':tactics2
                                   (caddr (cdddr (cdddr fp2))))))
             (t (raise "code 13: No Patterns have Fired in restore-framework"))))

(defun suppress (fp1)
  (cond ((null fp1) t)
        ((consp fp1)
         (setq *tactics0* (remove (car fp1) *tactics0*))
         (setq *tactics1* (remove (car fp1) *tactics1*))
         (setq *tactics2* (remove (car fp1) *tactics2*))
         (setq *rewrites* (remove (car fp1) *rewrites*))
         (suppress (cdr fp1)))
        (t (raise "code 13: No Patterns have Fired in suppress"))))

