;;|=========================================================================|
;;|                         COPYRIGHT NOTICE                                |
;;|                                                                         |
;;|             Copyright 1990, 1991, 1992, 1993, 1994 Mark Tarver          |
;;|                                                                         |
;;|        Permission to use, copy, and distribute this software and        |
;;| its documentation for any purpose is hereby granted providing           |
;;| any such use, copying and distribution is not done                      |
;;| for money, securities or any other pecuniary benefit and that both      |
;;| the above copyright and this permission notice appear in all copies     |
;;| and in the supporting documentation.  Any modification of the software  |
;;| or documentation should be accompanied by the name of the author of the |
;;| modification, and Mark Tarver must be formally notified                 |
;;| of this modification before distributing the software.                  |
;;|                                                                         |
;;|       Any commercial use of this software or use of the names "SEQUEL", |
;;| or "Mark Tarver" in connection with any version, modified or            |
;;| unmodified, of this software, through publicity or advertising,         |
;;| requires written permission.  Mark Tarver makes no                      |
;;| representation about the suitability of this software for any purpose.  |
;;| SEQUEL is provided "as is" without express or implied warranty.         |
;;|                                                                         |
;;|       Mark Tarver disclaims all warranties with regard to               |
;;| this software, including all implied warranties of merchantability and  |
;;| fitness. In no event shall Mark Tarver be liable for any                |
;;| special, indirect or consequential damages or any damages whatsoever    |
;;| resulting from loss of use, data or profits, whether in an action of    |
;;| contract, negligence or other tortious action, arising out of or in     |
;;| connection with the use or performance of this software.                |
;;|                                                                         |
;;|=========================================================================|

(in-package :sequel)

(defun define1 (fp1)
     (progn
           (if (not *outfile*)
               (format t ";;SEQUEL compiling ~A...~%" (car fp1))) 
           (define2 (bracket (remove-sig fp1)))
           (if (fboundp 'remove-trace)
               (untrace1 (list (car fp1))))
           (car fp1)))

(defun mutual1 (fp1)
  (progn (mapcar 'define2 
                 (mapcar 'bracket 
                         (mapcar 'remove-sig (mapcar 'cdr fp1))))
         (mapcar 'cadr fp1)))
 
(defun define2 (fp1)
  (cond
    ((and (consp fp1) (id (car fp1)))
     (evalrldef (car fp1) (brrl (cdr fp1) (car fp1))))
    (t (raise (format nil "code 1: bad function name ~A" (car fp1))))))

(defun evalrldef (fp1 fp2)
  (cond
    ((equal-arguments fp2) (build-def fp1 fp2))
    (t (raise (format nil "code 3: unequal number of parameters in ~A" fp1)))))

(defun equal-arguments (fp1)
    (equal-arguments1 (find-arity fp1) (cdr fp1)))

(defun equal-arguments1 (fp1 fp2)
  (cond
    ((null fp2) t)
    ((and (consp fp2) (= fp1 (find-arity fp2)))
     (equal-arguments1 fp1 (cdr fp2)))
    (t nil)))

(defun build-def (fp1 fp2)
  (lisp-function-of fp2 (lisp-function-shell fp1 fp2)))
 
(defun lisp-function-shell (fp1 fp2)
  (lisp-function-shell1 fp1 (find-arity fp2)))

(defun lisp-function-shell1 (fp1 fp2)
 (list 'defun fp1 (fparams-of fp2) (list 'cond '<nextrule>)))

(defun fparams-of (fp1) (fparams-of1 fp1 nil))

(defun fparams-of1 (fp1 fp2)
  (cond
    ((eq 0 fp1) fp2)
    (t (fparams-of1 (1- fp1) (cons (getfparam fp1) fp2)))))

(defun find-arity (fp1)
  (cond
    ((consp fp1)
     (length (remove-guards (car (sep-patt-action (car fp1))))))
    (t (raise "code 0: Bug"))))

(defun remove-guards (fp1)
  (cond
    ((null fp1) nil)
    ((and (consp fp1) (guard (car fp1))) (remove-guards (cdr fp1)))
    ((consp fp1) (cons (car fp1) (remove-guards (cdr fp1))))
    (t (raise "code 0: Bug"))))

(defun lisp-function-of (rules def)
  (if (null rules) 
      (eval (print-if-asked (optimise (add-errmess (cadr def) def))))
      (lisp-function-of (rest rules)
                      (compile-rule (first rules) def)))) 

(defun add-errmess
      (fp1 fp2)
        (cond ((and (consp fp2)
                    (= (length fp2) 2)
                    (and (consp (car fp2))
                         (= (length (car fp2))
                            2))
                    (eq t (caar fp2))
                    (eq '<nextrule> (cadr fp2)))
               (list (list t (cadar fp2))))
              ((eq '<nextrule> fp2)
               (list t
                   (list 'raise
                     (format nil "code 13: No Patterns have Fired in ~A" fp1))))
              ((consp fp2)
               (cons (add-errmess fp1 (car fp2))
                     (add-errmess fp1 (cdr fp2))))
              (t fp2)))

(defun rremove
       (fp1 fp2)
       (cond ((atom fp2) fp2)
             (t (mapcar #'(lambda (a) (rremove fp1 a))
                        (remove fp1 fp2 :test 'equal)))))


(defun print-if-asked (def) (if *outfile* 
                                (progn (pprint def stream)
                                       (terpri stream))
                                def))

(defun compile-rule (rule def)
 (list 'defun
        (second def)
        (third def)
        (add-compiled-rule (fourth def) (rlcode-of rule))))

(defun add-compiled-rule (code compiled-rule)
  (lsubst (list compiled-rule '<nextrule>)
                 '<nextrule>
                 code))

(defun lsubst (l x y)
  (cond ((null y) y)
        ((equal (car y) x)
         (append l (cdr y)))
        ((consp (car y))
         (cons (lsubst l x (car y))
               (lsubst l x (cdr y))))
        (t (cons (car y) (lsubst l x (cdr y))))))

(defun rlcode-of (fp1) (rlcode-of1 (sep-patt-action fp1)))

(defun rlcode-of1 (fp1)
  (cond
    ((and (consp fp1) (= (length fp1) 3) (arrow (cadr fp1)))
     (rlam/lisp (rlam fp1)))
    (t (raise "code 0: Bug"))))

(defun rlam/lisp (fp1)
  (cond
    ((and (consp fp1) (= (length fp1) 3) (f-arrow (cadr fp1)))
     (list (add-and (encode-pattern (car fp1) (car fp1)))
           (encode-action (car fp1) (caddr fp1))))
    ((and (consp fp1) (= (length fp1) 3) (b-arrow (cadr fp1)))
     (insert-setq (gentemp)
       (cons 'and
                 (append (encode-pattern (car fp1) (car fp1))
                         (list (encode-action (car fp1) (caddr fp1)))))))
    (t (raise "code 0: Bug"))))

(defun insert-setq (fp1 fp2)
  (cond ((and (consp fp2)
              (eq 'and (car fp2)))
         (list
           (cons 'and
                 (append (butlast (cdr fp2))
                         (list (get-fail fp1 (car (last fp2)))))) fp1))
        (t
         (raise "code 0: Bug!"))))

(defun get-fail (sym expr)
  (if (and (consp expr) (= (length expr) 3) (eq (car expr) 'fail-if))
      (list 'not (list (cadr (cadr expr)) (list 'setq sym (caddr expr)))) 
      (list 'not (list 'equal (list 'setq sym expr) *failure-object*))))

(defun fail-if (fp1 fp2)
  (if (funcall fp1 fp2) *failure-object* fp2))

(defun add-and (fp1)
  (cond
    ((null fp1) t)
    ((and (consp fp1) (= (length fp1) 1)) (car fp1))
    ((consp fp1) (cons 'and fp1))
    (t (raise "code 0: Bug!"))))

(defun encode-pattern (fp1 fp2)
  (cond
    ((null fp1) nil)
    ((consp fp1)
     (addlambdainstr (encode-pattern-instr (car fp1) fp2)
         (encode-pattern (cdr fp1) fp2)))
    (t (raise "code 0: Bug"))))

(defun addlambdainstr (fp1 fp2)
  (cond ((eq 'nocode fp1) fp2) (t (cons fp1 fp2))))

(defun encode-pattern-instr (fp1 fp2)
  (cond
    ((and (consp fp1) (= (length fp1) 2) (eq 'guard (car fp1)))
     (coderlguard fp2 (cadr fp1)))
    ((and (consp fp1) (= (length fp1) 3) (eq 'opl (car fp1))
          (> (length (cadr fp1)) 3))
     (list 'and (list 'consp (tterm (caddr fp1)))
           (list '> (list 'length (tterm (caddr fp1)))
                 (- (length (cadr fp1)) 3))))
    ((and (consp fp1) (= (length fp1) 3) (eq 'opl (car fp1))
          (= (length (cadr fp1)) 3))
     (list 'consp (tterm (caddr fp1))))
    ((and (consp fp1) (= (length fp1) 3) (eq 'opl (car fp1)))
     (raise "code 5: misuse of |"))
    ((and (consp fp1) (= (length fp1) 3) (eq 'cll (car fp1)))
     (list 'and (list 'consp (tterm (caddr fp1)))
           (list '= (list 'length (tterm (caddr fp1)))
                 (length (cadr fp1)))))
    ((and (consp fp1) (= (length fp1) 3) (null (cadr fp1)))
     (list 'null (tterm (caddr fp1))))
    ((and (consp fp1) (= (length fp1) 3) (eq 'name (car fp1))
          (selfeval (cadr fp1)))
         (list 'equal (cadr fp1) (tterm (caddr fp1))))
    ((and (consp fp1) (= (length fp1) 3) (eq 'name (car fp1)))
     (list 'eq (list 'quote (cadr fp1)) (tterm (caddr fp1))))
    ((and (consp fp1) (= (length fp1) 3) (eq 'fv (car fp1))) 'nocode)
    ((and (consp fp1) (= (length fp1) 3) (eq 'ftv (car fp1))) 'nocode)
    ((and (consp fp1) (= (length fp1) 3) (eq 'bv (car fp1)))
     (list 'equal (tterm (caddr fp1)) (findassfp (cadr fp1) fp2)))
    ((and (consp fp1) (= (length fp1) 3) (eq 'btv (car fp1)))
     (list 'equal (ttail (caddr fp1)) (findassfp (cadr fp1) fp2)))
    (t (raise "code 0: Bug!"))))

(defun coderlguard (fp1 fp2) (encode-action fp1 fp2))

(defun encode-action (fp1 fp2)
  (cond
    ((null fp2) nil)
    ((selfeval fp2) fp2)
    ((var fp2) (findassfp fp2 fp1))
    ((atom fp2) (list 'quote fp2))
    ((and (consp fp2) (atsign (car fp2)) (hasbar fp2))
     (raise (format nil "code 5: misuse of | in ~A" fp2))) 
    ((and (consp fp2) (atsign (car fp2)) (hasdbar fp2))
     (raise (format nil "code 6: misuse of <> in ~A" fp2))) 
    ((and (consp fp2) (atsign (car fp2)) (var (cadr fp2)))
     (raise (format nil "code 9: variable ~A used as a function" (cadr fp2))))
    ((and (consp fp2) (atsign (car fp2)) (not (symbolp (cadr fp2))))
     (raise (format nil "code 10: non-symbol ~A used as function name" 
            (cadr fp2))))
    ((hasdbar fp2)
     (cons 'append
           (mapcar #'(lambda (elem) (encode-action fp1 elem))
                   (remove '<> fp2))))
    ((hasbar fp2) (rlactioncons fp1 fp2))
    ((anon-func fp2) (ch-to-lambda fp2 fp1))
    ((and (consp fp2) (> (length fp2) 1) (atsign (car fp2)))
     (cons (cadr fp2)
           (mapcar #'(lambda (elem) (encode-action fp1 elem))
                   (cddr fp2))))
    ((is-verbatim fp2) (verbatim fp2 fp1))
    ((anon-func fp2) (ch-to-lambda fp2 fp1))
    ((listp fp2)
     (cons 'list
           (mapcar #'(lambda (elem) (encode-action fp1 elem)) fp2)))
    (t (raise "code 0: Bug!"))))

(defun anon-func (fp1) (and (consp fp1) (eq (cadr fp1) '/.)))

(defun ch-to-lambda (fp1 fp2) 
  (ins-f (list 'lambda (cdaddr fp1) 
                (substpm (cdaddr fp1) (encode-action fp2 (cadddr fp1))))))

(defun ins-f (lam) (read-from-string (format nil "(function ~A)" lam)))

(defun substpm (ps body)
  (if (null ps) 
      body 
      (substpm (cdr ps) (subst (car ps) (list 'quote (car ps)) body :test 
           'equal))))

(defun selfeval (fp1)
  (cond
    ((numberp fp1) t)
    ((stringp fp1) t)
    ((characterp fp1) t)
    ((null fp1) t)
    ((eq t fp1) t)
    (t nil)))

(defun findassfp (fp1 fp2)
  (cond
    ((eq t fp2) nil)
    ((eq fp1 (car fp2)) fp1)
    ((and (consp fp2) (consp (car fp2)) (= (length (car fp2)) 3)
          (eq 'fv (caar fp2)) (equal (cadar fp2) fp1))
     (tterm (caddr (car fp2))))
    ((and (consp fp2) (consp (car fp2)) (= (length (car fp2)) 3)
          (eq 'ftv (caar fp2)) (equal (cadar fp2) fp1))
     (ttail (caddr (car fp2))))
    ((consp fp2) (findassfp fp1 (cdr fp2)))
    (t (raise (format nil "code 4: free variable ~A" fp1))))) 

(defun ttail (fp1) (ttail1 (reverse fp1)))

(defun ttail1 (fp1)
  (cond
    ((consp fp1) (ttail2 (cdr fp1) (getfparam (car fp1))))
    (t (raise "code 0: Bug"))))

(defun ttail2 (fp1 fp2)
  (cond
    ((and (consp fp1) (= (length fp1) 1) (eq 2 (car fp1)))
     (list 'cdr fp2))
    ((and (consp fp1) (eq 1 (car fp1)))
     (ttail2 (cdr fp1) (list 'car fp2)))
    ((consp fp1)
     (ttail2 (cons (1- (car fp1)) (cdr fp1)) (list 'cdr fp2)))
    (t (raise "code 0: Bug!"))))

(defun tterm (fp1) (tterm1 (reverse fp1)))

(defun tterm1 (fp1)
  (cond
    ((consp fp1) (tterm2 (cdr fp1) (getfparam (car fp1))))
    (t (raise "code 0: Bug!"))))

(defun tterm2 (fp1 fp2)
  (cond
    ((null fp1) fp2)
    ((and (consp fp1) (eq 1 (car fp1)))
     (tterm2 (cdr fp1) (list 'car fp2)))
    ((consp fp1)
     (tterm2 (cons (1- (car fp1)) (cdr fp1)) (list 'cdr fp2)))
    (t (raise "code 0: Bug!"))))

(defun getfparam (fp1) (concat 'fp fp1))

(defun concat (x y) 
  (intern (string-upcase (format nil "~A~A" x y))))

(defun implode (fp1) (intern (coerce fp1 'string)))

(defun hasdbar (fp1)
  (cond
    ((null fp1) nil)
    ((and (consp fp1) (eq '<> (car fp1))) t)
    ((consp fp1) (hasdbar (cdr fp1)))
    (t (raise "Code 0: Bug!")))) 

(defun rlactioncons (fp1 fp2)
  (insert-conses
      (mapcar #'(lambda (elem) (encode-action fp1 elem))
              (remove (return-bar) fp2))))

(defun insert-conses (fp1)
  (cond
    ((and (consp fp1) (= (length fp1) 1)) (car fp1))
    ((consp fp1) (list 'cons (car fp1) (insert-conses (cdr fp1))))
    (t (raise "Code 0: Bug!"))))

(defun return-bar () '\|)

(defun is-verbatim (fp1)
  (cond ((and (consp fp1) (eq '% (car fp1))) t) (t nil)))

(defun verbatim (fp1 fp2)
  (cond
    ((and (consp fp1) (= (length fp1) 3)) 
     (look-for-circumflex fp2 (cadr fp1)))
    ((consp fp1) (look-for-circumflex fp2 (butlast (cdr fp1))))
    (t (raise "Code 0: Bug!"))))

(defun look-for-circumflex (fp1 fp2)
  (cond ((atom fp2) fp2)
        ((and (consp fp2) (eq (car fp2) '^) (not (consp (cadr fp2))))
         (cons (encode-action fp1 (cadr fp2))
               (look-for-circumflex fp1 (cddr fp2))))
        ((and (consp fp2) (eq (car fp2) '^))
         (cons (encode-action fp1 (car (bracket (list (cadr fp2)))))
               (look-for-circumflex fp1 (cddr fp2))))
        ((and (consp fp2) (consp (car fp2)))
         (cons (look-for-circumflex fp1 (car fp2))
               (look-for-circumflex fp1 (cdr fp2))))
        (t (cons (car fp2)
                 (look-for-circumflex fp1 (cdr fp2))))))
   
(defun rlam (fp1)
  (cond
    ((and (consp fp1) (= (length fp1) 3) (arrow (cadr fp1)))
     (list (pm (car fp1)) (cadr fp1) (caddr fp1)))
    (t (raise "code 0: Bug!"))))

(defun pm (fp1) (pm1 nil fp1 (list 1)))

(defun pm1 (fp1 fp2 fp3)
  (cond
    ((null fp2) nil)
    ((and (consp fp2) (bar (car fp2)))
     (pm1 (cons (car fp2) fp1) (cdr fp2) fp3))
    ((and (consp fp2) (anon (car fp2)) (consp fp3))
     (pm1 fp1 (cdr fp2) (cons (1+ (car fp3)) (cdr fp3))))
    ((and (consp fp2) (atom (car fp2)) (consp fp3))
     (cons (addpattinstr (car fp2) fp1 fp3)
           (pm1 (cons (car fp2) fp1) (cdr fp2)
                 (cons (1+ (car fp3)) (cdr fp3)))))
    ((and (consp fp2) (guard (car fp2)) (consp fp3))
     (cons (list 'guard (car fp2)) (pm1 fp1 (cdr fp2) fp3)))
    ((and (consp fp2) (listp (car fp2)) (consp fp3))
     (append (cons (addpattinstr (car fp2) fp1 fp3)
                   (pm1 fp1 (car fp2) (cons 1 fp3)))
             (pm1 (append (flatten (car fp2)) fp1) (cdr fp2)
                   (cons (1+ (car fp3)) (cdr fp3)))))
    (t (raise "code 0: Bug!"))))

(defun anon (fp1) (cond ((eq '_ fp1) t) (t nil)))

(defun guard (fp1)
  (cond ((and (consp fp1) (atsign (car fp1))) t) (t nil)))

(defun atsign (fp1) (cond ((eq '@ fp1) t) (t nil)))

(defun addpattinstr (fp1 fp2 fp3)
  (cond
    ((null fp1) (list 'name nil fp3))
    ((openlistp fp1) (list 'opl fp1 fp3))
    ((listp fp1) (list 'cll fp1 fp3))
    ((and (var fp1) (consp fp2) (bar (car fp2)) (member fp1 fp2))
     (list 'btv fp1 fp3))
    ((and (var fp1) (consp fp2) (bar (car fp2)))
     (list 'ftv fp1 fp3))
    ((and (consp fp2) (bar (car fp2))) 
     (raise (format nil "code 35: use of dotted list with ... | ~A" fp1)))
    ((and (var fp1) (member fp1 fp2)) (list 'bv fp1 fp3))
    ((var fp1) (list 'fv fp1 fp3))
    (t (list 'name fp1 fp3))))

(defun openlistp (fp1)
  (cond ((atom fp1) nil) ((hasbar fp1) t) (t nil)))

(defun hasbar (fp1)
  (cond
    ((null fp1) nil)
    ((and (consp fp1) (bar (car fp1))) t)
    ((consp fp1) (hasbar (cdr fp1)))
    (t (raise "Code 0: Bug!"))))

(defun bar (x) (eq x '\|))

(defun flatten (fp1)
  (cond
    ((null fp1) nil)
    ((consp (car fp1)) (append (flatten (car fp1)) (flatten (cdr fp1))))
    (t (cons (car fp1) (flatten (cdr fp1))))))

(defun sep-patt-action (fp1) (sep-patt-action1 fp1 nil))

(defun sep-patt-action1 (fp1 fp2)
  (cond
    ((and (consp fp1) (= (length fp1) 2) (arrow (car fp1)))
     (list fp2 (car fp1) (cadr fp1)))
    ((and (consp fp1) 
          (consp (car fp1))
          (not (eq (caar fp1) '@))
          (occurs '@ (car fp1)))
     (raise (format nil "code 12: guard ~A within pattern" 
            (rremove '@ (car fp1)))))
    ((consp fp1)
     (sep-patt-action1 (cdr fp1) (append fp2 (list (car fp1)))))
    (t (raise "code 0: Bug"))))

(defun occurs
       (fp1 fp2)
       (cond ((atom fp2) nil)
             ((and (consp fp2)
                   (equal (car fp2) fp1))
              t)
             ((and (consp fp2)
                   (consp (car fp2)))
              (or (occurs fp1 (car fp2))
                  (occurs fp1 (cdr fp2))))
             ((consp fp2) (occurs fp1 (cdr fp2)))
             (t (raise "No Patterns have Fired in occurs"))))

(defun brrl (fp1 fp2) (brrl1 fp1 nil fp2))

(defun brrl1 (fp1 fp2 fp3)
  (cond
    ((and (consp fp1) (consp (car fp1)) (eq 'sig (caar fp1)))
     (brrl1 (cdr fp1) fp2 fp3))
    ((and (consp fp1) (= (length fp1) 2) (arrow (car fp1)))
     (list (append fp2 (list (car fp1) (cadr fp1)))))
    ((and (consp fp1) (> (length fp1) 1) (arrow (car fp1)))
     (cons (append fp2 (list (car fp1) (cadr fp1)))
           (brrl1 (cddr fp1) nil fp3)))
    ((consp fp1) (brrl1 (cdr fp1) (append fp2 (list (car fp1))) fp3))
    (t (raise (format nil
               "code 2: No ->s or too few/many actions in final line of ~A" 
                  fp3)))))

(defun arrow (fp1)
  (cond ((f-arrow fp1) t) ((b-arrow fp1) t) (t nil)))

(defun f-arrow (fp1) (cond ((eq '-> fp1) t) (t nil)))

(defun b-arrow (fp1) (cond ((eq '<- fp1) t) (t nil)))

(defun bracket (l) (shortform (bracket2 l)))

(defun shortform (x)
  (cond ((not (consp x)) x)
        ((and (bar (cadr x))
              (consp (caddr x))
              (not (member '<> (caddr x)))
              (not (eq '@ (car (caddr x)))))
        (shortform (cons (car x) (caddr x))))
        (t (cons (shortform (car x))
                 (shortform (cdr x))))))

(defun bracket2 (l)
  (cond
    ((null l) nil)
    ((eq '[ (car l)) (bracket1 (find-rsq (cdr l))))
    ((eq '% (car l)) (except-verbatim (cdr l) nil))
    ((eq '] (car l)) (raise "code 7: unmatched ]"))
    ((consp (car l))
     (cons (cons '@ (bracket2 (car l))) (bracket2 (cdr l))))
    (t (cons (car l) (bracket2 (cdr l))))))

(defun bracket1 (l) (cons (bracket (car l)) (bracket (cadr l))))

(defun find-rsq (l) (find1-rsq nil l -1))

(defun find1-rsq (bl fl n)
  (cond
    ((null fl) (raise "code 8: unmatched ["))
    ((eq '[ (car fl)) (find1-rsq (cons '[ bl) (cdr fl) (1- n)))
    ((and (eq '] (car fl)) (= n -1)) (list (reverse bl) (cdr fl)))
    ((eq '] (car fl)) (find1-rsq (cons '] bl) (cdr fl) (1+ n)))
    (t (find1-rsq (cons (car fl) bl) (cdr fl) n))))

(defun except-verbatim (l accum)
  (cond
    ((null l) (raise "code 11: missing %"))
    ((eq (car l) '%)
     (cons (cons '% (reverse (cons '% accum))) (bracket (cdr l))))
    (t (except-verbatim (cdr l) (cons (car l) accum)))))

(defun remove-sig (fp1)
  (cond
    ((and (consp fp1) (> (length fp1) 1) (id (car fp1))
          (eq '{ (cadr fp1)))
     (cons (car fp1) (find-rcurly (car fp1) nil (cddr fp1))))
    (t (setf (get (car fp1) 'target-type) 'type?) fp1)))

(defun find-rcurly (fp1 fp2 fp3)
  (cond
    ((null fp3) (error "code 16: missing }"))
    ((and (consp fp3) (eq (quote }) (car fp3)))
     (cdr fp3))
    ((consp fp3)
     (find-rcurly fp1 (append fp2 (list (car fp3))) (cdr fp3)))
    (t (raise "code 0: Bug!"))))

(defun optimise (fp1)
  (insert-declarations 
   (optimise-eval-quotes 
    (optimise-progns (optimise-progs 
    (insert-let (optimise-evals (compress fp1))))))))

(defun optimise-evals (fp1)
  (cond ((and (consp fp1)
              (= (length fp1) 2)
              (eq 'eval (car fp1))
              (and (consp (cadr fp1))
                   (= (length (cadr fp1)) 2))
              (eq 'quote (caadr fp1)))
         (cadar (cdr fp1)))
        ((consp fp1) (mapcar 'optimise-evals fp1))
        (t fp1)))

(defun optimise-eval-quotes (fp1)
  (cond ((and (consp fp1) (= (length fp1) 2)
              (eq 'eval (car fp1))
              (consp (cadr fp1))
              (= (length (cadr fp1)) 2)
              (eq 'quote (caadr fp1)))
         (cadar (cdr fp1)))
        ((consp fp1) (mapcar 'optimise-eval-quotes fp1))
        (t fp1)))

(defun insert-let (fp1)
  (cond ((not *fastcode*) fp1)
        ((and (consp fp1)
              (= (length fp1) 4)
              (eq 'defun (car fp1)))
         (list 'defun
               (cadr fp1)
               (caddr fp1)
               (il1 (caddr (cdr fp1)))))
        (t fp1)))
 
(defun rpted-subterms
       (fp1)
       (rs1 (all-subterms fp1)))

(defun all-subterms (fp1)
  (cond ((and (consp fp1)
              (= (length fp1) 2)
              (eq 'protect (car fp1)))
         nil)
        ((and (consp fp1)
              (= (length fp1) 2)
              (eq 'quote (car fp1)))
         nil)
        ((consp fp1)
         (append (list fp1)
                 (mapcan 'all-subterms fp1)))
        (t nil)))

(defun protect (fp1) fp1)

(defun rs1
       (fp1)
       (cond ((null fp1) nil)
             ((and (consp fp1)
                   (member (car fp1)
                           (cdr fp1)
                           ':test
                           'equal))
              (cons (car fp1)
                    (rs1 (cdr fp1))))
             ((consp fp1) (rs1 (cdr fp1)))
             (t (raise "code 13: No Patterns have Fired in rs1"))))

(defun il1 (fp1)
  (cond ((and (consp fp1)
              (eq 'cond (car fp1)))
         (cons 'cond
               (mapcar #'(lambda (e)
                                 (mapcar 'il1 e))
                       (cdr fp1))))
        ((and (consp fp1)
              (strict fp1)
              (rpted-subterms fp1))
         (let-bind (rpted-subterms fp1)
                   (gentemp)
                   (list 'let nil fp1)))
        ((consp fp1) (mapcar 'il1 fp1))
        (t fp1)))

(defun strict (fp1) (and (eigen 'if fp1) (eigen 'and fp1) (eigen 'or fp1)))
 
(defun let-bind (fp1 fp2 fp3)
  (cond ((null fp1) fp3)
        ((and (consp fp1)
              (consp fp3)
              (= (length fp3) 3)
              (eq 'let (car fp3))
              (occurs (car fp1)
                      (caddr fp3)))
         (let-bind (cdr fp1)
                   (gentemp)
                   (list 'let
                         (cons (list fp2 (car fp1))
                               (cadr fp3))
                         (substop fp2
                                (car fp1)
                                (caddr fp3)))))
        ((consp fp1) (let-bind (cdr fp1) fp2 fp3))
        (t (raise "code 13: No Patterns have Fired in let-bind"))))

(defun substop (fp1 fp2 fp3)
  (cond ((and (consp fp3)
              (eq 'protect (car fp3)))
         (cons 'protect (cdr fp3)))
        ((equal fp3 fp2) fp1)
        ((consp fp3)
         (mapcar #'(lambda (e)
                           (substop fp1 fp2 e))
                 fp3))
        (t fp3)))

(defun optimise-progs (fp1)
  (optimise-progs1 fp1 (find-setqs fp1)))

(defun find-setqs (fp1)
  (cond ((and (consp fp1) (= (length fp1) 3)
         (eq 'equal (car fp1))
         (and (consp (cadr fp1))
              (= (length (cadr fp1)) 3))
              (eq 'setq (caadr fp1))
              (failure (caddr fp1)))
         (list (cadar (cdr fp1))))
        ((and (consp fp1) (= (length fp1) 2)
              (eq 'not (car fp1))
              (consp (cadr fp1))
              (= (length (cadr fp1)) 2)
              (consp (cadr (cadr fp1)))
              (= (length (cadr (cadr fp1))) 3) 
              (eq (car (cadr (cadr fp1))) 'setq))
         (list (cadr (cadr (cadr fp1))))) 
         ((consp fp1) (mapcan 'find-setqs fp1))
         (t nil)))

(defun optimise-progs1 (code progs)
  (if (null progs)
      code
      (optimise-progs2 code progs)))

(defun optimise-progs2 (fp1 fp2)
  (cond ((and (consp fp1)
              (= (length fp1) 4)
              (eq 'defun (car fp1))
              (consp (caddr (cdr fp1)))
              (eq 'cond
                  (caadr (cddr fp1))))
         (list 'defun
               (cadr fp1)
               (caddr fp1)
               (list 'prog
                     fp2
                     (list 'return
                           (cons 'cond
                                 (cdadr (cddr fp1)))))))
        (t (raise "code 13: No Patterns have Fired in optimise-progs2"))))

(defun failure (fp1) (equal fp1 *failure-object*))
   
(defun insert-declarations
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 2)
                   (eq 'defun (car fp1)))
              (id1 (cons 'defun (cdr fp1))
                   (signature (cadr fp1))))
             (t
              (raise "code 13: No Patterns have Fired in insert-declarations"))))

(defun id1
       (fp1 fp2)
       (cond ((null fp2) fp1)
             ((eq fp2 'void) fp1)
             (t (id2 (third fp1)
                     fp1
                     fp2
                     (list 'declare)))))

(defun id2
       (fp1 fp2 fp3 fp4)
       (cond ((and (null fp1)
                   (consp fp3)
                   (= (length fp3) 2)
                   (eq '-> (car fp3)))
              (id3 fp2 fp4 (cadr fp3)))
             ((and (consp fp1) (consp fp3))
              (if (registered-type (car fp3))
                  (id2 (cdr fp1)
                       fp2
                       (cdr fp3)
                       (append fp4
                               (list
                                 (list 'type
                                       (trans-type (car fp3))
                                       (car fp1)))))
                  (id2 (cdr fp1) fp2 (cdr fp3) fp4)))
             (t (raise "code 13: No Patterns have Fired in id2"))))

(defun id3
       (fp1 fp2 fp3)
       (cond ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'defun (car fp1))
                   (and (consp fp2)
                        (= (length fp2) 1)))
              (if (registered-type fp3)
                  (list 'defun
                        (cadr fp1)
                        (caddr fp1)
                        (list 'the
                              (trans-type fp3)
                              (caddr (cdr fp1))))
                  (list 'defun
                        (cadr fp1)
                        (caddr fp1)
                        (caddr (cdr fp1)))))
             ((and (consp fp2)
                   (= (length fp2) 1))
              fp1)
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'defun (car fp1)))
              (if (registered-type fp3)
                  (list 'defun
                        (cadr fp1)
                        (caddr fp1)
                        fp2
                        (list 'the
                              (trans-type fp3)
                              (caddr (cdr fp1))))
                  (list 'defun
                        (cadr fp1)
                        (caddr fp1)
                        fp2
                        (caddr (cdr fp1)))))
             ((and (consp fp1)
                   (> (length fp1) 2)
                   (eq 'defun (car fp1)))
              (cons 'defun
                    (cons (cadr fp1)
                          (cons (caddr fp1)
                                (cons fp2 (cdddr fp1))))))
             (t (raise "code 13: No Patterns have Fired in id3"))))

(defun registered-type (fp1)
  (cond ((and (consp fp1) (eq (car fp1) 'list)) t)
        (t (consp (assoc fp1 *assoctypes*)))))

(defun trans-type (fp1)
  (cond ((and (consp fp1) 
              (eq (car fp1) 'list))
         'list) 
        (t (cadr (assoc fp1 *assoctypes*)))))

(defun add-assoc-type (fp1 fp2)
  (push (list fp1 fp2) *assoctypes*))

(defun remove-assoc-type (fp1)
  (setq *assoctypes* (remove fp1 *assoctypes* :test 'equal-assoc)))

(defun equal-assoc (fp1 fp2)
  (and (consp fp2) (equal fp1 (car fp2))))

(defun optimise-progns (fp1)
  (cond ((and (consp fp1)
              (eq 'cond (car fp1)))
         (cons 'cond
               (mapcar 'optimise-progns1
                       (cdr fp1))))
        ((consp fp1) (mapcar 'optimise-progns fp1))
        (t fp1)))

(defun optimise-progns1 (fp1)
  (cond ((and (consp fp1)
              (= (length fp1) 2)
              (consp (cadr fp1))
              (eq 'progn (caadr fp1)))
         (cons (car fp1) (cdadr fp1)))
        (t fp1)))

(defun compress (fp1)
  (cond
    ((not (consp fp1)) fp1)
    ((and (consp fp1) (= (length fp1) 2) (eq 'and (car fp1)))
     (cadr fp1))
    ((and (consp fp1) (> (length fp1) 1) (eq 'and (car fp1))
          (consp (cadr fp1)) (eq 'and (caadr fp1)))
     (compress (append (cons 'and (cdadr fp1)) (cddr fp1))))
    ((and (consp fp1) (> (length fp1) 2) (eq 'and (car fp1))
          (consp (caddr fp1)) (eq 'and (caadr (cdr fp1))))
     (compress
         (append (cons 'and (cons (cadr fp1) (cdadr (cdr fp1))))
                 (cdddr fp1))))
    ((and (consp fp1) (= (length fp1) 2) (eq 'cond (car fp1))
          (and (consp (cadr fp1)) (= (length (cadr fp1)) 2))
          (eq t (caadr fp1)))
     (compress (cadar (cdr fp1))))
    ((and (consp fp1) (= (length fp1) 3) (eq 'cons (car fp1))
          (and (consp (cadr fp1)) (= (length (cadr fp1)) 2))
          (eq 'car (caadr fp1))
          (and (consp (caddr fp1)) (= (length (caddr fp1)) 2))
          (eq 'cdr (caadr (cdr fp1)))
          (equal (cadar (cddr fp1)) (cadar (cdr fp1))))
     (cadar (cdr fp1)))
    ((and (consp fp1) (= (length fp1) 3) (eq 'cons (car fp1))
          (and (consp (cadr fp1)) (= (length (cadr fp1)) 2))
          (eq 'car (caadr fp1))
          (and (consp (caddr fp1)) (= (length (caddr fp1)) 3))
          (eq 'cons (caadr (cdr fp1)))
          (and (consp (cadar (cddr fp1)))
               (= (length (cadar (cddr fp1))) 2))
          (eq 'car (caadr (caddr fp1)))
          (and (consp (cadar (cdadr (cdr fp1))))
               (= (length (cadar (cdadr (cdr fp1)))) 2))
          (eq 'cdr (caadr (cadar (cddr fp1))))
          (equal (cadar (cdadr (caddr fp1))) (cadar (cdr fp1)))
          (and (consp (caddr (caddr fp1)))
               (= (length (caddr (caddr fp1))) 2))
          (eq 'cdr (caadr (cdadr (cdr fp1))))
          (and (consp (cadar (cddar (cddr fp1))))
               (= (length (cadar (cddar (cddr fp1)))) 2))
          (eq 'cdr (caadr (caddr (caddr fp1))))
          (equal (cadar (cdadr (cdadr (cdr fp1)))) (cadar (cdr fp1))))
     (cadar (cdr fp1)))
    ((and (consp fp1) (= (length fp1) 2) (eq 'cdr (car fp1))
          (and (consp (cadr fp1)) (= (length (cadr fp1)) 2))
          (eq 'cdr (caadr fp1))
          (and (consp (cadar (cdr fp1)))
               (= (length (cadar (cdr fp1))) 2))
          (eq 'cdr (caadr (cadr fp1))))
     (list 'cdddr (compress (cadar (cdadr fp1)))))
    ((and (consp fp1) (= (length fp1) 2) (eq 'car (car fp1))
          (and (consp (cadr fp1)) (= (length (cadr fp1)) 2))
          (eq 'cdr (caadr fp1))
          (and (consp (cadar (cdr fp1)))
               (= (length (cadar (cdr fp1))) 2))
          (eq 'cdr (caadr (cadr fp1))))
     (list 'caddr (compress (cadar (cdadr fp1)))))
    ((and (consp fp1) (= (length fp1) 2) (eq 'cdr (car fp1))
          (and (consp (cadr fp1)) (= (length (cadr fp1)) 2))
          (eq 'car (caadr fp1))
          (and (consp (cadar (cdr fp1)))
               (= (length (cadar (cdr fp1))) 2))
          (eq 'cdr (caadr (cadr fp1))))
     (list 'cdadr (compress (cadar (cdadr fp1)))))
    ((and (consp fp1) (= (length fp1) 2) (eq 'cdr (car fp1))
          (and (consp (cadr fp1)) (= (length (cadr fp1)) 2))
          (eq 'cdr (caadr fp1))
          (and (consp (cadar (cdr fp1)))
               (= (length (cadar (cdr fp1))) 2))
          (eq 'car (caadr (cadr fp1))))
     (list 'cddar (compress (cadar (cdadr fp1)))))
    ((and (consp fp1) (= (length fp1) 2) (eq 'car (car fp1))
          (and (consp (cadr fp1)) (= (length (cadr fp1)) 2))
          (eq 'car (caadr fp1))
          (and (consp (cadar (cdr fp1)))
               (= (length (cadar (cdr fp1))) 2))
          (eq 'cdr (caadr (cadr fp1))))
     (list 'caadr (compress (cadar (cdadr fp1)))))
    ((and (consp fp1) (= (length fp1) 2) (eq 'car (car fp1))
          (and (consp (cadr fp1)) (= (length (cadr fp1)) 2))
          (eq 'cdr (caadr fp1))
          (and (consp (cadar (cdr fp1)))
               (= (length (cadar (cdr fp1))) 2))
          (eq 'car (caadr (cadr fp1))))
     (list 'cadar (compress (cadar (cdadr fp1)))))
    ((and (consp fp1) (= (length fp1) 2) (eq 'cdr (car fp1))
          (and (consp (cadr fp1)) (= (length (cadr fp1)) 2))
          (eq 'car (caadr fp1))
          (and (consp (cadar (cdr fp1)))
               (= (length (cadar (cdr fp1))) 2))
          (eq 'car (caadr (cadr fp1))))
     (list 'cdaar (compress (cadar (cdadr fp1)))))
    ((and (consp fp1) (= (length fp1) 2) (eq 'cdr (car fp1))
          (and (consp (cadr fp1)) (= (length (cadr fp1)) 2))
          (eq 'cdr (caadr fp1)))
     (list 'cddr (compress (cadar (cdr fp1)))))
    ((and (consp fp1) (= (length fp1) 2) (eq 'cdr (car fp1))
          (and (consp (cadr fp1)) (= (length (cadr fp1)) 2))
          (eq 'car (caadr fp1)))
     (list 'cdar (compress (cadar (cdr fp1)))))
    ((and (consp fp1) (= (length fp1) 2) (eq 'car (car fp1))
          (and (consp (cadr fp1)) (= (length (cadr fp1)) 2))
          (eq 'car (caadr fp1)))
     (list 'caar (compress (cadar (cdr fp1)))))
    ((and (consp fp1) (= (length fp1) 2) (eq 'car (car fp1))
          (and (consp (cadr fp1)) (= (length (cadr fp1)) 2))
          (eq 'cdr (caadr fp1)))
     (list 'cadr (compress (cadar (cdr fp1)))))
    (t (mapcar 'compress fp1))))

(defun raise (string) 
  (progn (format t ";; SEQUEL error message~%~A~%" string)
         (if (not (y-or-n-p "do you wish this input to be recorded? "))
             (pop *history*)) 
         (throw 'abort 'aborted)))
         
(defun id (fp1) (and (symbolp fp1) (not (var fp1))))

(defun print-delayed () (mapcar 'print-undef (reverse *delayed*)))

(defun print-undef (fp1) (write-string "* ") (write fp1) (terpri))

(defun var (x)
  (or (eq x 'w) (eq x 'x) (eq x 'y) (eq x 'z)
      (and (atom x) (equal (last (explode x)) '(#\?)))))

(defun explode (fp1)
  (cond
    ((numberp fp1) (coerce (format nil "~S" fp1) 'list))
    (t (coerce (string fp1) 'list))))
