; Newsgroups: comp.lang.scheme
; Distribution: world
; Subject: SICP: Evaluation by re-writing? (beginner)
; Summary: Another dumb question about SICP Section 3.1.2
; Keywords: scope, extent, environments
; 
; 
; Problem with SICP: What's wrong with substitution?
; 
; In SICP Section 3.1.2, the authors say that the substitution model
; is no longer valid as soon as you introduce SET!. There is
; a worked example to show what they mean on page 176. This issue is 
; used as the rationale for the environment model used later in The 
; Book.
; 
; I think there is another way to explain the indefinite extent
; of scheme objects, by using substitution, not of variable values
; as in The Book, but of variable _names_ (re-writing). My assertion 
; is that Scheme can be implemented by re-writing. Can anyone
; confirm or dis-prove this please? 
; 
; I'll explain what I mean. Consider the example:
; 
; (define (make-simplified-withdraw balance)
; 	(lambda (amount)
; 		(set! balance (- balance amount))
; 		balance)
; 
; this is the same as :
; 
; (set! make-simplified-withdraw 
; (lambda (balance)
; 	(lambda (amount)
; 		(set! balance (- balance amount))
; 		balance)
; 
; 
; so make-simplified-withdraw is equal to
; 
; (procedure (balance)
; 	(lambda (amount)
; 		(set! balance (- balance amount))
; 		balance)
; 
; Consider evaluating 
; 
; 	((make-simplified-withdraw 25) 20)
; 
; First  we evaluate the inner-most expression.
; 
; 	(make-simplified-withdraw 25)
; 
; or:
; 
; ((procedure (balance)
; 	(lambda (amount)
; 		(set! balance (- balance amount))
; 		balance)
;  25)
; 
; Taking the inner-most expression again (procedure ....)
; 
; To evaluate a procedure we re-write all the non-quoted 
; parts of the body replacing all the formals with unique symbols 
; generated with GENSYM. We also replace the formals to get a procedure
; that is ready for apply:
; 
; (applicable-procedure (g1)
; 	(lambda (amount)
; 		(set! g1 (- g1 amount))
; 		g1)
; 
; Next we apply the applicable-procedure by setting the formals
; so that 
; 	g1 = 25
; and we then evaluate the body of the applicable procedure:
; 
; 	(lambda (amount)
; 		(set! g1 (- g1 amount))
; 		g1)
; 
; Resulting in
; 
; 	(procedure (amount)
; 		(set! g1 (- g1 amount))
; 		g1)
; 
; Notice the reference to the GENSYM "g1" in this code fragment.
; In general as long as there are pieces of source code refering to
; "g1" still existing, then "g1" will still exist.
; 
; going back to: ((make-simplified-withdraw 25) 20)
; 
; this is  now
; 
; ((procedure (amount) 
; 	(set! g1 (- g1 amount)) 
; 		g1)
;  20)
; 
; You can see that when evaluated using the steps explained above 
; we should have evaluation of the sequence:
; 
; 	(set! g1 (- g1 g2))
; 	g1
; 
; Which is a lot different from the sequence:
; 
; 	(set! 25 (- 25 amount)) 
; 	20 
; 
; which is what happens when you substitute variable values rather
; that variable names, as described in The Book.
; 
; Can anyone  tell me whether evaluation-by-rewriting is a 
; correct interpretation of Scheme? 
; If so, was it left out of SICP because it cannot be compiled?
; Does this method have a correct name?
; 
; Your comments gratefully received by email,  I will summarise.
; 
; Bill
; 
; PS. Here is a Lisp version of the above method of evaluation.
; 
;;;;
;;;; Scheme in RefLisp -  Version 2.0 - 1 March 1993
;;;;
;;;;	by Bill Birch
;;;;
;;;;	The functions allow scheme programs to be run.
;;;;	Albeit very slowly.
;;;;	Environments are created by substitution of variables.
;;;;
;;
;;	Substitute  - replace all occurences of b with a,
;;				  skipping quoted expressions.
;;
(defun substitute (a b s)
   (cond
      ((eq s b) a)   
      ((atom s) s)   
      (else (cond
	  		((equal (car s) 'quote) s)
			(else  
			  (cons (substitute a b (car s))
   	          (substitute a b (cdr s))))))))
;;
;;
;; Macro to define variables and procedures.
;;
(df define (args &rest body)
	(cond ((atom args) (set args (eval (car body))))
		(else 
			(set (car args) (eval `(my-lambda ,(cdr args) ,@body))))))
;;
;;
;; Macro to generate procedures
;;
(df my-lambda (formals &rest body)
	`(lambda (&rest actuals)
		(eval (cons 'progn (subst-list (gensyms ',(defs body))
				',(defs body)
				(subst-list
					(setlis (gensyms ',formals) actuals)
					',formals
					',body))))))

;;
;; Substitute a list of items 
;;
(defun subst-list (to from on)
	(cond ((null to) on)
		(else (subst-list 
			(cdr to) 
			(cdr from) 
			(substitute (car to) (car from) on)))))

;;
;; Reurn a list of gensyms as long as the list passed in.
;;
(defun gensyms (l)
	(unless (null l)
		(cons (gensym) (gensyms (cdr l)))))
;;
;; Set a list of variables with the values given, 
;; each in turn.
;;
(defun setlis (vars vals)
	(cond ((null vars) '())
		(else
			(set (car vars) (car vals))
			(cons (car vars) 
				(setlis (cdr vars) (cdr vals))))))

;;
;; Find all first-level (define) statements in a form body
;; return a list of the names to be defined.
;;
(defun defs (form)
	(cond
		((null form) '())
		((and
			(consp (car form))
			(eq (caar form) 'define))

				(cons (cond
					((consp (cadar form))
						(caar(cdr(car form))))
					(else (cadar form)))
					(defs (cdr form))))))


;; END-OF-FILE
